/*
 * Copyright (C) 2005 Stephane Marchesin
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This code uses a lowpass filter implementation from Audiality :
 * http://audiality.org/
 * The copyright from Audiality follows :
 */
/*(LGPL)
---------------------------------------------------------------------------
	a_filters.c - Various Audio Signal Filters
---------------------------------------------------------------------------
 * Copyright (C) 2001, 2002, David Olofson
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include "config.h"

#include "xmms/i18n.h"
#include <xmms/plugin.h>
#include <gtk/gtk.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "libxmms/util.h"
#include "libxmms/configfile.h"

static void init(void);
static void cleanup(void);
static void about(void);
static void configure(void);
static int mod_samples(gpointer *d, gint length, AFormat afmt, gint srate, gint nch);

static GtkWidget *conf_dialog = NULL;
static GtkObject* max_amp;

float max_amplification=5.;

#define PLUGIN_NAME "Multiband Compressor"
#define PLUGIN_VERSION "0.1"

// a low pass filter
typedef struct A_resof12
{
	gint Ld1, Ld2;
	gint Rd1, Rd2;
	gint f;
	gint q;
} A_resof12;

// an amplification filter
typedef struct A_amp
{
	float from;
	float to;
} A_amp;


// we use 5 bands, so we need 4 low pass filters
#define BANDS 5
static float frequencies[BANDS-1]={350,600,800,1100};
static float qs[BANDS-1]={10.,10.,10.,10.};
static A_resof12 low_pass_filters[BANDS-1];
static A_amp amplification_filters[BANDS];
static gint32* buffers[BANDS];
static gint32* master_buffer;
static gint current_rate=-1;
static gint current_buffer_size=-1;


EffectPlugin mbc_ep =
{
	NULL,
	NULL,
	NULL, /* Description */
	init,
	cleanup,
	about,
	configure,
	mod_samples
};

EffectPlugin *get_eplugin_info(void)
{
	mbc_ep.description = g_strdup(_(PLUGIN_NAME" Plugin "PLUGIN_VERSION));
	return &mbc_ep;
}

static void init(void)
{
	ConfigFile *cfg;
	int i;
	for(i=0;i<BANDS;i++)
		buffers[i]=NULL;
	master_buffer=NULL;
	current_rate=-1;
	current_buffer_size=-1;
	cfg = xmms_cfg_open_default_file();
	xmms_cfg_read_float(cfg, "mbc_plugin", "max_amplification", &max_amplification);
	xmms_cfg_free(cfg);
}

static void cleanup(void)
{
	ConfigFile *cfg;
	int i;
	for(i=0;i<BANDS;i++)
	{
		g_free(buffers[i]);
		buffers[i]=NULL;
	}
	g_free(master_buffer);
	master_buffer=NULL;
	current_rate=-1;
	current_buffer_size=-1;
	cfg = xmms_cfg_open_default_file();
	xmms_cfg_write_float(cfg, "mbc_plugin", "max_amplification", max_amplification);
	xmms_cfg_write_default_file(cfg);
	xmms_cfg_free(cfg);
}

static void about(void)
{
	static GtkWidget *about_dialog = NULL;

	if (about_dialog != NULL)
		return;

	about_dialog = xmms_show_message(_("About "PLUGIN_NAME" Plugin"),
					 _(PLUGIN_NAME" Plugin for XMMS\n\nVersion "PLUGIN_VERSION"\n\nStephane Marchesin 2005"), _("Ok"), FALSE,
					 NULL, NULL);

	gtk_signal_connect(GTK_OBJECT(about_dialog), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			&about_dialog);
}

static void configuration_value_change(GtkWidget *widget, gpointer data)
{
	max_amplification = GTK_ADJUSTMENT(widget)->value;
}

static void configuration_close(GtkButton * button, gpointer data)
{
	gtk_widget_destroy(GTK_WIDGET(conf_dialog));
}

static void configure(void)
{
	GtkWidget* button, *frame, *hscale;

	if (conf_dialog != NULL)
		return;

	conf_dialog = gtk_dialog_new();
	gtk_signal_connect(GTK_OBJECT(conf_dialog), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &conf_dialog);
	gtk_window_set_title(GTK_WINDOW(conf_dialog), _("Configure "PLUGIN_NAME));

	// create an adjustment widget
	max_amp = gtk_adjustment_new((gint)max_amplification, 1, 31, 0.1, 10, 1);
	gtk_signal_connect(GTK_OBJECT(max_amp), "value-changed", GTK_SIGNAL_FUNC(configuration_value_change), NULL);

	// create an hscale widget
	hscale = gtk_hscale_new(GTK_ADJUSTMENT(max_amp));
	gtk_widget_set_usize(hscale, 300, 30);
	gtk_widget_show(hscale);

	// create a frame
	frame = gtk_frame_new("Max amplification");
	gtk_widget_show(frame);
	gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(hscale));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(conf_dialog)->vbox), frame, TRUE, TRUE, 5);

	// create a close button
	button = gtk_button_new_with_label(_("Close"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(configuration_close), NULL);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(conf_dialog)->vbox), button, TRUE, TRUE, 5);

	gtk_widget_show(conf_dialog);
}


static void init_filters(gint rate)
{
	int i;
	for(i=0;i<BANDS-1;i++)
	{
		memset(&(low_pass_filters[i]),0,sizeof(A_resof12));
		low_pass_filters[i].f = (int)(512.0 * sin(M_PI * frequencies[i] / rate));
		if(low_pass_filters[i].f > 256)
			low_pass_filters[i].f = 256;
		if(qs[i] < 1.)
			low_pass_filters[i].q = 256;
		else
			low_pass_filters[i].q = 256 / qs[i];

	}

	for(i=0;i<BANDS;i++)
	{
		amplification_filters[i].from=1.;
		amplification_filters[i].to=1.;
	}
}

#define CLIP(X) \
	if (X<-32768)\
		X=-32768;\
	else if (X>32767)\
		X=32767;

static void amplify_add(A_amp* af,gint32* buf_in,gint32* buf_out,gint length)
{
	gint32 s;
	int i;
	int f;
	float delta=af->to-af->from;
	float l=length/2.;
	for(i = 0; i < length/2; i++)
	{
		f=1024*(i*delta/l+af->from);
		s=(buf_in[i]*f)/1024;
		buf_out[i]+=s;
	}
}


static void low_pass(A_resof12* rf,gint32* buf_in,gint32* buf_out_low,gint32* buf_out_high,gint length)
{
	gint Rl, Rb, Rh;
	gint Ll, Lb, Lh;
	gint f=rf->f;
	gint q=rf->q;
	gint Ld1=rf->Ld1, Ld2=rf->Ld2;
	gint Rd1=rf->Rd1, Rd2=rf->Rd2;
	gint i;
	for(i=0;i<length/2;i+=2)
	{
		Rl = Rd2 + (f*Rd1 >> 8);
		Rh = ((buf_in[i]) << 4) - Rl - (q*Rd1 >> 8);
		Rb = (f*Rh >> 8) + Rd1;
		Rl>>=4;
		Rh>>=4;
		buf_out_low[i] = Rl;
		buf_out_high[i] = Rh;
		Rd1 = Rb;
		Rd2 = Rl;

		Ll = Ld2 + (f*Ld1 >> 8);
		Lh = ((buf_in[i+1]) << 4) - Ll - (q*Ld1 >> 8);
		Lb = (f*Lh >> 8) + Ld1;
		Ll>>=4;
		Lh>>=4;
		buf_out_low[i+1] = Ll;
		buf_out_high[i+1] = Lh;
		Ld1 = Lb;
		Ld2 = Ll;
	}

	rf->Ld1=Ld1;
	rf->Ld2=Ld2;
	rf->Rd1=Rd1;
	rf->Rd2=Rd2;
}
	
static void convert_16_to_32(gint16* buf_in,gint32* buf_out,gint length)
{
	gint i;
	for(i=0;i<length/2;i++)
	{
		buf_out[i]=(gint32)buf_in[i];
	}
}

static gint32 max_volume(gint32* buf_in,gint length)
{
	gint i;
	gint32 max=buf_in[0];
	for(i=0;i<length/2;i++)
		if (abs(buf_in[i])>max)
			max=abs(buf_in[i]);
	return max;
}

static void convert_32_to_16(gint32* buf_in,gint16* buf_out,gint length)
{
	gint i;
	gint32 max=max_volume(buf_in,length);
	if (max==0)
		max=1;
	for(i=0;i<length/2;i++)
	{
//		gint64 s=(((gint64)buf_in[i])*32767)/max;
//		if ((s<-32768)||(s>32767))
//			printf("clipping ! %ld\n",s);
		gint64 s=(((gint64)buf_in[i]));
		CLIP(s);
		buf_out[i]=s;
	}
}

static gint avg_volume(gint32* buffer,gint length)
{
	gint i;
	gint64 sum=0;
	for(i=0;i<length/2;i++)
	{
		sum+=abs(buffer[i]);
	}
	return sum/(length/2);
}

static int mod_samples(gpointer *d, gint length, AFormat afmt, gint srate, gint nch)
{
	gint16  *data = (gint16 *)*d;
	gint i;

	if (!(afmt == FMT_S16_NE ||
	      (afmt == FMT_S16_LE && G_BYTE_ORDER == G_LITTLE_ENDIAN) ||
	      (afmt == FMT_S16_BE && G_BYTE_ORDER == G_BIG_ENDIAN)) ||
	    nch != 2)
		return length;

	// check filter initialization and buffer allocation
	if (current_rate!=srate)
	{
		init_filters(srate);
		current_rate=srate;
	}
	if (current_buffer_size<length)
	{
		for(i=0;i<BANDS;i++)
		{
			buffers[i]=(gint32*)g_realloc(buffers[i],2*length);
		}
		master_buffer=(gint32*)g_realloc(master_buffer,2*length);
		current_buffer_size=length;
	}

	// first, convert the initial data to 32 bit
	convert_16_to_32(data,buffers[0],length);

	// low pass the data into different bands
	for(i=0;i<BANDS-1;i++)
		low_pass(&low_pass_filters[i],buffers[i],buffers[i],buffers[i+1],length);

	// clear the master buffer
	memset(master_buffer,0x00,2*length);

	// amplify the bands and sum up
	for(i=0;i<BANDS;i++)
	{
		float amplification,delta;
		gint vol;
		// compute the average volume for this band
		vol=avg_volume(buffers[i],length);
		amplification_filters[i].from=amplification_filters[i].to;

		// determine the amplification factor for this band
		if (vol==0)
			amplification=1.0;
		else
			amplification=3000.f/vol;

		if (amplification>max_amplification)
			amplification=max_amplification;
		if (amplification<1.0)
			amplification=1.0;

		if (amplification_filters[i].from-amplification>9.)
		{
			delta=-9.;
		}
		else if (amplification_filters[i].from-amplification<-3.)
		{
			delta=3.;
		}
		else
			delta=amplification-amplification_filters[i].from;

		amplification_filters[i].to=amplification_filters[i].from+delta;
		// amplify and accumulate
		amplify_add(&amplification_filters[i],buffers[i],master_buffer,length);
	}

	// convert everything back to 16 bit
	convert_32_to_16(master_buffer,data,length);

	return length;
}

