#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <signal.h>

#include "lwm.h"
#include "color.h"
#include "disp.h"
#include "cursor.h"
#include "shape.h"
#include "error.h"
#include "manage.h"
#include "resource.h"

Mode mode;			/* The window manager's mode. (See "lwm.h".) */
int start_x;			/* The X position where the mode changed. */
int start_y;			/* The Y position where the mode changed. */

Display * dpy;			/* The connection to the X server. */
int screen_count;		/* The number of screens. */
ScreenInfo * screens;		/* Information about these screens. */
ScreenInfo * current_screen;

XFontStruct *font;		/* Actual titlebar font. */
XFontStruct *popup_font;	/* Actual menu font. */

Bool shape;			/* Does server have Shape Window extension? */
int shape_event;		/* ShapeEvent event type. */
int screen_bpp;		

/* Atoms we're interested in. See the ICCCM for more information. */
Atom wm_state;
Atom wm_change_state;
Atom wm_protocols;
Atom wm_delete;
Atom wm_take_focus;
Atom wm_colormaps;
Atom compound_text;

/** Netscape uses this to give information about the URL it's displaying. */
Atom _mozilla_url;

char *argv0;

static void initScreens(void);
static void initScreen(int);

/*ARGSUSED*/
extern int
main(int argc, char *argv[]) {
	XEvent ev;

	struct sigaction sa;

	argv0 = argv[0];

	mode = wm_initialising;

	/* Open a connection to the X server. */
	dpy = XOpenDisplay("");
	if (dpy == 0)
		panic("can't open display.");

	parseResources();

	/* Set up an error handler. */
	XSetErrorHandler(errorHandler);

	/* See early if the server has the Shape Window extension. */
	shape = serverSupportsShapes();
	if (!shape) 
		panic("Server doesn't support shapes.");
	
	/* Set up signal handlers. */
	signal(SIGTERM, Terminate);
	signal(SIGINT, Terminate);
	signal(SIGHUP, Terminate);

	/* Ignore SIGCHLD. */
	sa.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
	sa.sa_flags = SA_NOCLDWAIT;
#else
	sa.sa_flags = 0;
#endif
	sigemptyset(&sa.sa_mask);
	sigaction(SIGCHLD, &sa, 0);

	/* Internalize useful atoms. */
	wm_state = XInternAtom(dpy, "WM_STATE", False);
	wm_change_state = XInternAtom(dpy, "WM_CHANGE_STATE", False);
	wm_protocols = XInternAtom(dpy, "WM_PROTOCOLS", False);
	wm_delete = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
	wm_take_focus = XInternAtom(dpy, "WM_TAKE_FOCUS", False);
	wm_colormaps = XInternAtom(dpy, "WM_COLORMAP_WINDOWS", False);
	compound_text = XInternAtom(dpy, "COMPOUND_TEXT", False);
	
	_mozilla_url = XInternAtom(dpy, "_MOZILLA_URL", False);
	
	/*
	 * Get fonts for our titlebar and our popup window. We try to
	 * get Lucida, but if we can't we make do with fixed because everyone
	 * has that.
	 */
	font = XLoadQueryFont(dpy, font_name);
	if (font == 0)
		font = XLoadQueryFont(dpy, "fixed");
	if (font == 0)
		panic("can't find a font for the titlebars.");
	
	popup_font = XLoadQueryFont(dpy, popup_font_name);
	if (popup_font == 0)
		popup_font = XLoadQueryFont(dpy, "fixed");
	if (popup_font == 0)
		panic("can't find a font for the popup window.");
	
	initScreens();
	
	/*
	 * Initialisation is finished, but we start off not interacting with the
	 * user.
	 */
	mode = wm_idle;
	
	/*
	 * The main event loop.
	 */
	for (;;) {
		// check for prioritary events
		// an event is prioritary if it always comes with another event
		// the most prioritary is the one that comes first
		if (XCheckTypedEvent(dpy,MapNotify,&ev)) {goto prioritary;}
		if (XCheckTypedEvent(dpy,MotionNotify,&ev)) {goto prioritary;}
		if (XCheckTypedEvent(dpy,EnterNotify,&ev)) {goto prioritary;}
		if (XCheckTypedEvent(dpy,ConfigureNotify,&ev)) {goto prioritary;}
		if (XCheckTypedEvent(dpy,Expose,&ev)) {goto prioritary;}
		
		// the events without handlers are prioritary too
		if (XCheckTypedEvent(dpy,NoExpose,&ev)) {goto prioritary;}
			
		XNextEvent(dpy, &ev);
		
		prioritary:
		switch(ev.type)
		{
			// look if we can find a similar event on the same window but more recent
			// the new one will then replace the old one
			case MotionNotify:
			case Expose:
			case NoExpose:
			case ReparentNotify:
			
			case MapRequest:
			case ConfigureRequest:
			case UnmapNotify:
			while(XCheckTypedWindowEvent(dpy,ev.xany.window,ev.type,&ev));
		};

		switch(ev.type)
		{
			// look if we can find a similar event but more recent on any window
			// the new one will then replace the old one
			case EnterNotify:
			while(XCheckTypedEvent(dpy,ev.type,&ev));
		};
		
		dispatch(&ev);
/*		
		if (ev.type==Expose)
			printf("+expose\n");
		if (ev.type==MotionNotify)
			printf("+motion\n");
		if (ev.type==PropertyNotify)
			printf("+keypress\n");
		if (ev.type==ButtonPress)
			printf("+button press\n");
		if (ev.type==ButtonRelease)
			printf("+button releas\n");
		if (ev.type==MapRequest)
			printf("+mapreq\n");
		if (ev.type==ConfigureRequest)
			printf("+config req\n");
		if (ev.type==UnmapNotify)
			printf("+unmap\n");
		if (ev.type==DestroyNotify)
			printf("+destroy\n");
		if (ev.type==ClientMessage)
			printf("+client message\n");
		if (ev.type==ColormapNotify)
			printf("+Color map\n");
		if (ev.type==PropertyNotify)
			printf("+property\n");
		if (ev.type==ReparentNotify)
			printf("+reparent\n");
		if (ev.type==EnterNotify)
			printf("+Enter\n");
		if (ev.type==CirculateRequest)
			printf("Circulate\n");
		if (ev.type==ConfigureNotify)
			printf("config notify\n");
		if (ev.type==CreateNotify)
			printf("create notify\n");
		if (ev.type==GravityNotify)
			printf("Gravity notify\n");
		if (ev.type==MapNotify)
			printf("map notify\n");
		if (ev.type==MappingNotify)
			printf("mapping nofity\n");
		if (ev.type==SelectionClear)
			printf("selection clear\n");
		if (ev.type==SelectionNotify)
			printf("selection notify\n");
		if (ev.type==SelectionRequest)
			printf("selection request\n");
		if (ev.type==NoExpose)
			printf("noexpose\n");
*/
//		sleep(1);
	}
}

void
sendConfigureNotify(Client *c) {
	XConfigureEvent ce;

	ce.type = ConfigureNotify;
	ce.event = c->window;
	ce.window = c->window;
	ce.x = c->size.x + border;
	ce.y = c->size.y + border;
	ce.width = c->size.width - 2 * border;
	ce.height = c->size.height - 2 * border;
	ce.above = None;
	ce.border_width = c->border;
	ce.override_redirect = 0;
	XSendEvent(dpy, c->window, False, StructureNotifyMask, (XEvent *) &ce);
}

extern void
scanWindowTree(int screen) {
	unsigned int i;
	unsigned int nwins;
	Client * c;
	Window dw1;
	Window dw2;
	Window * wins;
	XWindowAttributes attr;
	
	XQueryTree(dpy, screens[screen].root, &dw1, &dw2, &wins, &nwins);
	for (i = 0; i < nwins; i++) {
		XGetWindowAttributes(dpy, wins[i], &attr);
		if (attr.override_redirect || isShaped(wins[i]) || wins[i] == screens[screen].popup)
			continue;
		c = Client_Add(wins[i], screens[screen].root);
		if (c != 0 && c->window == wins[i]) {
			// we create a new window
			// init its parameters
			c->screen = &screens[screen];
			c->size.x = attr.x - border;
			c->size.y = attr.y - border;
			c->size.width  = attr.width  + 2 * border;
			c->size.height = attr.height + 2 * border;
			c->border = attr.border_width;
			c->active = true;
			c->tab_xpos=0;
			c->tab_pixmap=0;
			c->tab_needs_rendering = true;
			
			if (attr.map_state == IsViewable) {
				c->internal_state = IPendingReparenting;
				manage(c, 1);
			}
		}
	}
	XFree(wins);
}

/*ARGSUSED*/
extern void
shell(ScreenInfo * screen, int button, int x, int y) {
	char * command;
	char * sh;
	
	/* Get the command we're to execute. Give up if there isn't one. */
	command = (button == Button1) ? btn1_command : btn2_command;
	if (command == 0)
		return;
	
	sh = getenv("SHELL");
	if (sh == 0)
		sh = "/bin/sh";
	
	switch (fork()) {
	case 0:		/* Child. */
		close(ConnectionNumber(dpy));
		if (screen && screen->display_spec != 0)
			putenv(screen->display_spec);
		execl(sh, sh, "-c", command, 0);
		fprintf(stderr, "%s: can't exec \"%s -c %s\"\n", argv0, sh,
			command);
		execlp("xterm", "xterm", 0);
		exit(EXIT_FAILURE);
	case -1:	/* Error. */
		fprintf(stderr, "%s: couldn't fork\n", argv0);
		break;
	}
}

extern int
titleHeight(void) {
	return (font->ascent + font->descent + 4);
}

static void
initScreens(void) {
	int screen;
	
	/* Find out how many screens we've got, and allocate space for their info. */
	screen_count = ScreenCount(dpy);
	screens = (ScreenInfo *) malloc(screen_count * sizeof(ScreenInfo));
	
	/* Go through the screens one-by-one, initialising them. */
	for (screen = 0; screen < screen_count; screen++) {
		initialiseCursors(screen);
		initScreen(screen);
		scanWindowTree(screen);
	}
}

unsigned long getcolor(int screen,unsigned char r,unsigned char g,unsigned char b)
{
	XColor colour;
	colour.red=((unsigned short)r)<<8;
	colour.green=((unsigned short)g)<<8;
	colour.blue=((unsigned short)b)<<8;
	XAllocColor(dpy, DefaultColormap(dpy, screen), &colour);
	return colour.pixel;
}

static void
initScreen(int screen) {
	XGCValues gv;
	XSetWindowAttributes attr;
	int len,i;
	char * display_string = DisplayString(dpy);
	char * colon = strrchr(display_string, ':');
	char * dot = strrchr(display_string, '.');
	
	/* Set the DISPLAY specification. */
	if (colon) {
		len = 9 + strlen(display_string) + ((dot == 0) ? 2 : 0) + 10;
		screens[screen].display_spec = (char *) malloc(len);
		sprintf(screens[screen].display_spec, "DISPLAY=%s", display_string);
		if (dot == 0) dot = screens[screen].display_spec + len - 3;
		else dot = strrchr(screens[screen].display_spec, '.');
		sprintf(dot, ".%i", screen);
	} else {
		screens[screen].display_spec = 0;
	}
	
	/* Find the root window. */
	screens[screen].root = RootWindow(dpy, screen);
	screens[screen].display_width = DisplayWidth(dpy, screen);
	screens[screen].display_height = DisplayHeight(dpy, screen);
	
	/* Get the pixel values of the colours we use. */
	screens[screen].black = BlackPixel(dpy, screen);
	screens[screen].white = WhitePixel(dpy, screen);
	for(i=0;i<nb_colors;i++)
		x_colors[i]=getcolor(screen,colors_rgb[i*3],colors_rgb[i*3+1],colors_rgb[i*3+2]);
	
	// now create the font colors
	{
		unsigned char 
			colr1=colors_rgb[title_font_fg_color*3+0],
			colr2=colors_rgb[title_font_bg_color*3+0],
			colg1=colors_rgb[title_font_fg_color*3+1],
			colg2=colors_rgb[title_font_bg_color*3+1],
			colb1=colors_rgb[title_font_fg_color*3+2],
			colb2=colors_rgb[title_font_bg_color*3+2];
 
		// assumes col?1<col?2
		for(i=0;i<nb_title_font_colors;i++)
		{	
			x_title_font_fg_colors[nb_title_font_colors-1-i]=getcolor(screen,colr1+((colr2-colr1+1)*i)/nb_title_font_colors,colg1+((colg2-colg1+1)*i)/nb_title_font_colors,colb1+((colb2-colb1+1)*i)/nb_title_font_colors);
		}
		colr1=colors_rgb[inactive_colors[title_font_fg_color]*3+0],
		colr2=colors_rgb[inactive_colors[title_font_bg_color]*3+0],
		colg1=colors_rgb[inactive_colors[title_font_fg_color]*3+1],
		colg2=colors_rgb[inactive_colors[title_font_bg_color]*3+1],
		colb1=colors_rgb[inactive_colors[title_font_fg_color]*3+2],
		colb2=colors_rgb[inactive_colors[title_font_bg_color]*3+2];
		
		for(i=0;i<nb_title_font_colors;i++)
		{	
			x_title_font_bg_colors[nb_title_font_colors-1-i]=getcolor(screen,colr1+((colr2-colr1+1)*i)/nb_title_font_colors,colg1+((colg2-colg1+1)*i)/nb_title_font_colors,colb1+((colb2-colb1+1)*i)/nb_title_font_colors);
		}
	}

	/* Set up root (frame) GC's. */
	gv.foreground = screens[screen].black ^ screens[screen].white;
	gv.background = screens[screen].white;
	gv.font = font->fid;
	gv.function = GXcopy;
	gv.line_width = 1;
	gv.subwindow_mode = IncludeInferiors;
	screens[screen].gc_thin = XCreateGC(dpy, screens[screen].root,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	gv.line_width = 1;
	screens[screen].gc = XCreateGC(dpy, screens[screen].root,
		GCForeground | GCBackground | GCFunction |
		GCFont | GCLineWidth | GCSubwindowMode, &gv);
	
	/* Create a window for our popup. */
	screens[screen].popup = XCreateSimpleWindow(dpy, screens[screen].root,
		0, 0, 1, 1, 1, screens[screen].black, screens[screen].white);
	attr.event_mask = ButtonMask | ButtonMotionMask | ExposureMask;
	// use save_under
	attr.save_under=True;
	XChangeWindowAttributes(dpy, screens[screen].popup, CWEventMask|CWSaveUnder, &attr);
	
	/* Create menu GC. */
	gv.line_width = 1;
	gv.font = popup_font->fid;
	screens[screen].menu_gc = XCreateGC(dpy, screens[screen].popup,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	/* Create size indicator GC. */
	gv.foreground = screens[screen].black;
	gv.function = GXcopy;
	screens[screen].size_gc = XCreateGC(dpy, screens[screen].popup,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	/* Announce our interest in the root window. */
	attr.cursor = screens[screen].root_cursor;
	attr.event_mask = SubstructureRedirectMask | SubstructureNotifyMask |
		ColormapChangeMask | ButtonPressMask | PropertyChangeMask |
		EnterWindowMask;
	
	XChangeWindowAttributes(dpy, screens[screen].root, CWCursor |
		CWEventMask, &attr);
	
	/* Make sure all our communication to the server got through. */
	XSync(dpy, False);
}

/**
Find the screen for which root is the root window.
*/
ScreenInfo *
getScreenFromRoot(Window root) {
	int screen;
	
	for (screen = 0; screen < screen_count; screen++)
		if (screens[screen].root == root)
			return &screens[screen];
	
	return 0;
}
