#include "client.h"
#include "manage.h"
#include "shape.h"
#include "lwm.h"
#include "color.h"
#include "pixmaps.h"
#include "look.h"
#include "font.h"
#include "resource.h"
#include "cursor.h"
#include "mouse.h"


Client *current;
static Client *clients;

static int popup_width;	/* The width of the size-feedback window. */

Edge interacting_edge;

static void sendClientMessage(Window, Atom, long, long);

void
setactive(Client *c, int on, long timestamp) {
	int inhibit = isShaped(c->window);

	if (c == 0 || hidden(c))
		return;

	if (!inhibit) {
		XMoveResizeWindow(dpy, c->parent,
			c->size.x-lt_width, c->size.y - titleHeight(),
			c->size.width+lt_width, c->size.height + titleHeight());
		XMoveWindow(dpy, c->window, border+lt_width, border + titleHeight());
	}

	if (on && c->accepts_focus) {
		XSetInputFocus(dpy, c->window, RevertToPointerRoot, CurrentTime);
		if (c->proto & Ptakefocus)
			sendClientMessage(c->window, wm_protocols, wm_take_focus,
				timestamp);
		cmapfocus(c);
	}

	if (!inhibit)
		Client_DrawBorder(c, on);
}


// render the tab
void client_render_tab(Client* c,int active)
{
	int i;
	
	if (c->active!=active)
	{	
		c->active=active;
		c->tab_needs_rendering=true;
	}
	
	// do we really need to render the pixmap ?
	if (!c->tab_needs_rendering)
		return;

	// recycle the pixmap
	if (c->tab_pixmap!=0) 
		XFreePixmap(dpy,c->tab_pixmap);
	c->tab_pixmap=XCreatePixmap(dpy,c->parent,c->tab_width+round_tab_xsize,titleHeight(),screen_bpp);
	
	// draw the yellow tab
	XSetForeground(dpy, c->screen->gc, active?x_colors[5]:x_colors[inactive_colors[5]]);
	XFillRectangle(dpy,c->tab_pixmap,c->screen->gc,1,1,c->tab_width+round_tab_xsize-2,titleHeight());
	XSetForeground(dpy, c->screen->gc, active?x_colors[6]:x_colors[inactive_colors[6]]);
	XDrawLine(dpy,c->tab_pixmap,c->screen->gc,0,0,c->tab_width+round_tab_xsize,0);
	XDrawLine(dpy,c->tab_pixmap,c->screen->gc,0,1,0,titleHeight());
	XSetForeground(dpy, c->screen->gc, active?x_colors[9]:x_colors[inactive_colors[9]]);
	XDrawLine(dpy,c->tab_pixmap,c->screen->gc,c->tab_width+round_tab_xsize-1,1,c->tab_width+round_tab_xsize-1,titleHeight());
	
#ifndef LOOK_R51
	// draw the left & right icons
	for(i=0;i<pixmaps_size*pixmaps_size;i++)
	{
		XSetForeground(dpy,c->screen->gc, active?x_colors[left_pixmap[i]]:x_colors[inactive_colors[left_pixmap[i]]]);
		XDrawPoint(dpy,c->tab_pixmap,c->screen->gc,i%pixmaps_size+DEFAULT_BORDER,i/pixmaps_size+1+(titleHeight()-pixmaps_size)/2);
	}
	for(i=0;i<pixmaps_size*pixmaps_size;i++)
	{
		XSetForeground(dpy,c->screen->gc, active?x_colors[right_pixmap[i]]:x_colors[inactive_colors[right_pixmap[i]]]);
		XDrawPoint(dpy,c->tab_pixmap,c->screen->gc,i%pixmaps_size+c->tab_width+2,i/pixmaps_size+1+(titleHeight()-pixmaps_size)/2);
	}
#endif
	
	/* Draw window title. */
	if (c->name != 0) {
#ifdef USE_FT_FONTS		
		FTDrawString(dpy, c->tab_pixmap, c->screen->gc, border + 2
			+ 20, font->ascent + font->descent,
			c->name, strlen(c->name),active);
			
#else
		XDrawImageString(dpy, c->tab_pixmap, c->screen->gc, border + 2
			+ 20, font->ascent + font->descent,
			c->name, strlen(c->name));
#endif
	}
		
	c->tab_needs_rendering=false;
}

// draw the borders graphics
void
Client_DrawBorder(Client *c, int active) {
//	int quarter = (border + titleHeight()) / 4;
	int i;
	
	if (c->parent == c->screen->root || c->parent == 0)
		return;

#ifdef LOOK_R51
	// XXX ne pas tout clearer, c'est lent et ca fait clignoter faire juste un rectangle
	XSetWindowBackground(dpy, c->parent,x_colors[4]);
	XClearWindow(dpy, c->parent);
//	XSetForeground(dpy, c->screen->gc, active?c->screen->orange:c->screen->light_gray);
//	XFillRectangle(dpy,c->parent,c->screen->gc,0,0,c->tab_width+round_tab_xsize,titleHeight());
	// draw the window borders
	for(i=0;i<DEFAULT_BORDER;i++)
	{	
		XSetForeground(dpy, c->screen->gc,x_colors[border_colors[BORDER_UP][i]]);
		XDrawLine(dpy,c->parent,c->screen->gc,lt_width,titleHeight()+i,c->size.width+lt_width,titleHeight()+i);
	}
	XDrawImageString(dpy, c->parent, c->screen->gc, border + 2
			+ 20, font->ascent + font->descent,
			c->name, strlen(c->name));
#else
	client_render_tab(c,active);
	XCopyArea(dpy,c->tab_pixmap,c->parent,c->screen->gc,0,0,c->tab_width+round_tab_xsize,titleHeight(),0,0);
	
	// draw the window borders
	for(i=0;i<DEFAULT_BORDER;i++)
	{	
		XSetForeground(dpy, c->screen->gc,x_colors[border_colors[BORDER_UP][i]]);
		XDrawLine(dpy,c->parent,c->screen->gc,lt_width,titleHeight()+i,c->size.width+lt_width,titleHeight()+i);
	}
#endif
	
	for(i=0;i<DEFAULT_BORDER;i++)
	{	
		XSetForeground(dpy, c->screen->gc,x_colors[border_colors[BORDER_DOWN][DEFAULT_BORDER-1-i]]);
		XDrawLine(dpy,c->parent,c->screen->gc,lt_width,c->size.height+titleHeight()+i-DEFAULT_BORDER,c->size.width+lt_width,c->size.height+titleHeight()+i-DEFAULT_BORDER);
	}
	for(i=0;i<DEFAULT_BORDER;i++)
	{	
		XSetForeground(dpy, c->screen->gc,x_colors[border_colors[BORDER_LEFT][i]]);
		XDrawLine(dpy,c->parent,c->screen->gc,lt_width+i,titleHeight()+i,lt_width+i,c->size.height+titleHeight()-i);
	}
	for(i=0;i<DEFAULT_BORDER;i++)
	{	
		XSetForeground(dpy, c->screen->gc,x_colors[border_colors[BORDER_RIGHT][DEFAULT_BORDER-1-i]]);
		XDrawLine(dpy,c->parent,c->screen->gc,c->size.width+lt_width-DEFAULT_BORDER+i,titleHeight()+DEFAULT_BORDER-1-i,c->size.width+lt_width-DEFAULT_BORDER+i,c->size.height+titleHeight()-DEFAULT_BORDER+1+i);
	}

//	XSetForeground(dpy, c->screen->gc, active?x_colors[title_font_fg_color]:x_colors[inactive_colors[title_font_fg_color]]);
//	XSetBackground(dpy, c->screen->gc, active?x_colors[title_font_bg_color]:x_colors[inactive_colors[title_font_bg_color]]);

}


Client *
Client_Get(Window w) {
	Client * c;
	
	if (w == 0 || (getScreenFromRoot(w) != 0))
		return 0;
	
	/* Search for the client corresponding to this window. */
	for (c = clients; c; c = c->next)
		if (c->window == w || c->parent == w)
			return c;
	
	/* Not found. */
	return 0;
}


Client *
Client_Add(Window w, Window root) {
	Client * c;

	if (w == 0 || w == root)
		return 0;

	/* Search for the client corresponding to this window. */
	for (c = clients; c != 0; c = c->next)
		if (c->window == w || c->parent == w)
			return c;

	c = calloc(1, sizeof *c);
	c->window = w;
	c->parent = root;
	c->hidden = False;
	c->state = WithdrawnState;
	c->internal_state = INormal;
	c->cmap = None;
	c->name = c->fallback_name = 0;
	c->ncmapwins = 0;
	c->cmapwins = 0;
	c->wmcmaps = 0;
	c->accepts_focus = 1;
	c->next = clients;
        c->close_pending = False;
	
	/* Add to head of list of clients. */
	return (clients = c);
}


void
Client_Remove(Client *c) {
	Client * cc;
	
	if (c == 0)
		return;
	
	/* Remove the client from our client list. */
	if (c == clients)
		clients = c->next;
	for (cc = clients; cc && cc->next; cc = cc->next) {
		if (cc->next == c)
			cc->next = cc->next->next;
	}
	
	/* Remove it from the hidden list if it's hidden. */
	if (hidden(c)) {
		unhidec(c, 0);
		/* Al Smith points out that you also want to get rid of the menu
		 * so you can be sure that if you let go on an item, you always
		 * get the corresponding window. */
		if (mode == wm_menu_up) {
			XUnmapWindow(dpy, current_screen->popup);
			mode = wm_idle;
		}
	}
	
	/* A deleted window can no longer be the current window. */
	if (c == current) {
		current = 0;
		
		/* As pointed out by J. Han, if a window disappears while it's
		 * being reshaped you need to get rid of the size indicator. */
		if (mode == wm_reshaping) {
			XUnmapWindow(dpy, current_screen->popup);
			mode = wm_idle;
		}
	}
	
	if (getScreenFromRoot(c->parent) == 0)
		XDestroyWindow(dpy, c->parent);
	
	if (c->ncmapwins != 0) {
		XFree(c->cmapwins);
		free(c->wmcmaps);
	}
	
	if (c->name != 0)
		XFree(c->name);
	
	free(c);
}


void
Client_MakeSane(Client *c, Edge edge, int *x, int *y, int *dx, int *dy) {
	Bool	horizontal_ok = True;
	Bool	vertical_ok = True;

	if (edge != ENone) {
		/*
		 *	Make sure we're not making the window too small.
		 */
		if (*dx < c->size.min_width)
			horizontal_ok = False;
		if (*dy < c->size.min_height)
			vertical_ok = False;

		/*
		 * Make sure we're not making the window too large.
		 */
		if (c->size.flags & PMaxSize) {
			if (*dx > c->size.max_width)
				horizontal_ok = False;
			if (*dy > c->size.max_height)
				vertical_ok = False;
		}

		/*
		 *	Make sure the window's width & height are multiples of
		 *	the width & height increments (not including the base size).
		 */

		if (c->size.width_inc > 1) {
			int apparent_dx = *dx - 2 * border - c->size.base_width;
			int x_fix = apparent_dx % c->size.width_inc;

			switch (edge) {
			case ELeft:
			case ETopLeft:
			case EBottomLeft:
				*x += x_fix;
				/*FALLTHROUGH*/
			case ERight:
			case ETopRight:
			case EBottomRight:
				*dx -= x_fix;
				break;
			default: break;
			}
		}

		if (c->size.height_inc > 1) {
			int apparent_dy = *dy - 2 * border - c->size.base_height;
			int y_fix = apparent_dy % c->size.height_inc;

			switch (edge) {
			case ETop:
			case ETopLeft:
			case ETopRight:
				*y += y_fix;
				/*FALLTHROUGH*/
			case EBottom:
			case EBottomLeft:
			case EBottomRight:
				*dy -= y_fix;
				break;
			default: break;
			}
		}

		/*
		 * Check that we may change the client horizontally and vertically.
		 */

		if (c->size.width_inc == 0)
			horizontal_ok = False;
		if (c->size.height_inc == 0)
			vertical_ok = False;
	}

	/*
	 * Ensure that, were the client to lose focus, it would still be accessible.
	 */
	if (*y + border >= c->screen->display_height)
		*y = c->screen->display_height - border;

	/*
	 * Update that part of the client information that we're happy with.
	 */
	if (interacting_edge != ENone) {
		if (horizontal_ok) {
			c->size.x = *x;
			c->size.width  = *dx;
		}
		if (vertical_ok) {
			c->size.y = *y;
			c->size.height = *dy;
		}
	} else {
		if (horizontal_ok)
			c->size.x = *x;
		if (vertical_ok)
			c->size.y = *y;
	}
}

void
Client_SizeFeedback(void) {
	int x, y;
	char buf[4*2 + 3 + 1];

	/* Make the popup 10% wider than the widest string it needs to show. */
	sprintf(buf, "%i x %i", current_screen->display_width,
		current_screen->display_height);
	popup_width = XTextWidth(popup_font, buf, strlen(buf));
	popup_width += popup_width/10;

	/* Put the popup in the right place to report on the window's size. */
	getMousePosition(&x, &y);
	XMoveResizeWindow(dpy, current_screen->popup, x + 8, y + 8,
		popup_width, popup_font->ascent + popup_font->descent + 1);
	XMapRaised(dpy, current_screen->popup);
	
	/*
	* Ensure that the popup contents get redrawn. Eventually, the function
	* size_expose will get called to do the actual redraw.
	*/
	XClearArea(dpy, current_screen->popup, 0, 0, 0, 0, True);
}

void
size_expose(void) {
	int width, height;
	char buf[4*2 + 3 + 1];
	
	width = current->size.width - 2*border;
	height = current->size.height - 2*border;

	/* This dance ensures that we report 80x24 for an xterm even when
	 * it has a scrollbar. */
	if (current->size.flags & (PMinSize|PBaseSize) && current->size.flags & PResizeInc) {
		if (current->size.flags & PBaseSize) {
			width -= current->size.base_width;
			height -= current->size.base_height;
		} else {
			width -= current->size.min_width;
			height -= current->size.min_height;
		}
	}

	if (current->size.width_inc != 0)
		width /= current->size.width_inc;
	if (current->size.height_inc != 0)
		height /= current->size.height_inc;

	sprintf(buf, "%i x %i", width, height);
	XDrawString(dpy, current_screen->popup, current_screen->size_gc,
		(popup_width - XTextWidth(popup_font, buf, strlen(buf))) / 2,
		popup_font->ascent + 1, buf, strlen(buf));
}

static void
Client_OpaquePrimitive(Client *c, Edge edge) {
	Cursor cursor;
	int ox, oy;

	if (c == 0 || c != current)
		return;

	/* Find out where we've got hold of the window. */
	getMousePosition(&ox, &oy);
	ox = c->size.x - ox;
	oy = c->size.y - oy;
	
	cursor = getEdgeCursor(edge);
	XChangeActivePointerGrab(dpy, ButtonMask | PointerMotionHintMask |
		ButtonMotionMask | OwnerGrabButtonMask, cursor, CurrentTime);
	
	/*
	 * Store some state so that we can get back into the main event
	 * dispatching thing.
	 */
	interacting_edge = edge;
	start_x = ox;
	start_y = oy;
	mode = wm_reshaping;
}

void
Client_Back(Client *c)
{
	if (c == 0)
		return;

	XLowerWindow(dpy, c->window);
	XLowerWindow(dpy, c->parent);
}

void
Client_Close(Client *c) {
	if (c == 0)
		return;

	/*
	 *	Terminate the client nicely if possible. Be brutal otherwise.
	 */
	if (c->proto & Pdelete) {
		sendClientMessage(c->window, wm_protocols, wm_delete, CurrentTime);
	} else {
		XKillClient(dpy, c->window);
	}
}

void
Client_Close_Pend(Client *c) {
        if (c == 0)
                return;
        c->close_pending = True;
}
       
void
Client_SetState(Client *c, int state) {
	long	data[2];

	data[0] = (long) state;
	data[1] = (long) None;

	c->state = state;
	XChangeProperty(dpy, c->window, wm_state, wm_state, 32,
		PropModeReplace, (unsigned char *) data, 2);
}

static void
sendClientMessage(Window w, Atom a, long data0, long data1) {
	XEvent	ev;
	long	mask;

	memset(&ev, 0, sizeof(ev));
	ev.xclient.type = ClientMessage;
	ev.xclient.window = w;
	ev.xclient.message_type = a;
	ev.xclient.format = 32;
	ev.xclient.data.l[0] = data0;
	ev.xclient.data.l[1] = data1;
	mask = (getScreenFromRoot(w) != 0) ? SubstructureRedirectMask : 0L;

	XSendEvent(dpy, w, False, mask, &ev);
}

extern void
Client_FreeAll(void) {
	Client *c;
	XWindowChanges wc;

	for (c = clients; c; c = c->next) {
		int not_mapped = !normal(c);

		/* Remap the window if it's hidden. */
		if (not_mapped) {
			XMapWindow(dpy, c->parent);
			XMapWindow(dpy, c->window);
		}

		/* Reparent it, and then push it to the bottom if it was hidden. */
		XReparentWindow(dpy, c->window, c->screen->root, c->size.x, c->size.y);
		if (not_mapped)
			XLowerWindow(dpy, c->window);

		/* Give it back its initial border width. */
		wc.border_width = c->border;
		XConfigureWindow(dpy, c->window, CWBorderWidth, &wc);
	}
}

extern void
Client_ColourMap(XEvent *e) {
	int i;
	Client * c;

	for (c = clients; c; c = c->next) {
		for (i = 0; i < c->ncmapwins; i++) {
			if (c->cmapwins[i] == e->xcolormap.window) {
				c->wmcmaps[i] = e->xcolormap.colormap;
				if (c == current)
					cmapfocus(c);
				return;
			}
		}
	}
}

extern void
Client_ReshapeEdge(Client *c, Edge e) {
	Client_OpaquePrimitive(c, e);
}

extern void
Client_Move(Client *c) {
	Client_OpaquePrimitive(c, ENone);
}

extern int
hidden(Client *c) {
	return c->state == IconicState;
}

extern int
withdrawn(Client *c) {
	return c->state == WithdrawnState;
}

extern int
normal(Client *c) {
	return c->state == NormalState;
}
