#include "main.h"
#include "lwm.h"
#include "manage.h"
#include "resource.h"

static int current_item;	/* Last known selected menu item. -1 if none. */

typedef struct menuitem menuitem;
struct menuitem {
	char * label;
	Client * client;
	menuitem * next;
};

static menuitem * hidden_menu = 0;

static char * chooseLabel(Client *);
static void getMenuDimensions(int *, int *, int *);

void
getMousePosition(int * x, int * y) {
	Window root, child;
	int t1, t2;
	unsigned int b;
	
	/* It doesn't matter which root window we give this call. */
	XQueryPointer(dpy, screens[0].root, &root, &child, x, y, &t1, &t2, &b);
	current_screen = getScreenFromRoot(root);
}

int
menu_whichitem(int x, int y) {
	int width;	/* Width of menu. */
	int height;	/* Height of each menu item. */
	int length;	/* Number of items on the menu. */
	
	getMenuDimensions(&width, &height, &length);
	
	/*
	 * Translate to popup window coordinates. We do this ourselves to avoid
	 * a round trip to the server.
	 */
	x -= start_x;
	y -= start_y;
	
	/*
	 * Are we outside the menu?
	 */
	if (x < 0 || x > width || y < 0 || y >= length * height)
		return -1;
	
	return y / height;
}

static void
getMenuDimensions(int *width, int *height, int *length) {
	int w;	/* Widest string so far. */
	int i;	/* Menu item. */
	
	menuitem *m = hidden_menu;
	
	w = 0;
	for (i = 0; m != 0; m = m->next, i++) {
		int tw = XTextWidth(popup_font, m->label, strlen(m->label)) + 4;
		if (tw > w) w = tw;
	}
	
	*width = w + border;
	*height = popup_font->ascent + popup_font->descent + 1;
	*length = i;
}

void
menuhit(XButtonEvent *e) {
	int width;	/* Width of menu. */
	int height;	/* Height of each menu item. */
	int length;	/* Number of menu items. */
	
	if (hidden_menu == 0)
		return;
	
	current_screen = getScreenFromRoot(e->root);
	
	getMenuDimensions(&width, &height, &length);
	
	/*
	 * Arrange for centre of first menu item to be under pointer.
	 */
	start_x = e->x - width / 2;
	start_y = e->y - height / 2;
	
	if (start_x < 0)
		start_x = 0;
	if (start_x + width > current_screen->display_width)
		start_x = current_screen->display_width - width;
	if (start_y < 0)
		start_y = 0;
	
	current_item = menu_whichitem(e->x, e->y);
	
	XMoveResizeWindow(dpy, current_screen->popup, start_x, start_y,
		width, length * height);
	XMapRaised(dpy, current_screen->popup);
	XChangeActivePointerGrab(dpy, ButtonMask | ButtonMotionMask |
		OwnerGrabButtonMask, None, CurrentTime);
	
	mode = wm_menu_up;
}

void 
hide(Client *c) {
	menuitem *newitem;

	if (c == 0)
		return;

	/* Create new menu item, and thread it on the menu. */
	newitem = (menuitem *) malloc(sizeof(menuitem));
	if (newitem == 0)
		return;
	newitem->label = chooseLabel(c);
	newitem->client = c;
	newitem->next = hidden_menu;
	hidden_menu = newitem;

	if (current == c)
		current = 0;

	/* Actually hide the window. */
	XUnmapWindow(dpy, c->parent);
	XUnmapWindow(dpy, c->window);

	c->hidden = True;

	/* If the window was the current window, it isn't any more... */
	setactive(c, 0, 0L);
	Client_SetState(c, IconicState);
}

void
unhide(int n, int map) {
	Client *c;
	menuitem *prev = 0;
	menuitem *m = hidden_menu;

	/* Find the nth client. */
	if (n < 0)
		return;

	while (n-- > 0 && m != 0) {
		prev = m;
		m = m->next;
	}

	if (m == 0)
		return;

	c = m->client;

	/* Remove the item from the menu, and dispose of it. */
	if (prev == 0) {
		hidden_menu = m->next;
	} else {
		prev->next = m->next;
	}
	free(m);

	c->hidden = False;

	/* Unhide it. */
	if (map) {
		XMapRaised(dpy, c->parent);
		XMapWindow(dpy, c->window);
		Client_SetState(c, NormalState);
	}
}

void
unhidec(Client *c, int map) {
	int i = 0;
	menuitem *m = hidden_menu;

	if (c == 0)
		return;

	/* My goodness, how the world sucks. */
	while (m != 0) {
		if (m->client == c) {
			unhide(i, map);
			return;
		}
		m = m->next;
		i++;
	}
}

void
update_menu(Client *c) {
	menuitem *m;

	if (c == 0)
		return;

	for (m = hidden_menu; m != 0; m = m->next) {
		if (m->client == c) {
			m->label = chooseLabel(c);
			return;
		}
	}
}

char *
chooseLabel(Client * c) {
	return (c->name != 0) ? c->name :
		((c->fallback_name != 0) ? c->fallback_name : "<unknown>");
}

void
menu_expose(void) {
	int i;		/* Menu item being drawn. */
	int width;	/* Width of each item. */
	int height;	/* Height of each item. */
	int length;	/* Number of menu items. */
	menuitem *m;

	getMenuDimensions(&width, &height, &length);

	/* Redraw the labels. */
	for (m = hidden_menu, i = 0; m != 0; m = m->next, i++) {
		int tx = (width - XTextWidth(popup_font, m->label,
			strlen(m->label))) / 2;
		int ty = i * height + popup_font->ascent + 1;

		XDrawString(dpy, current_screen->popup, current_screen->menu_gc,
			tx, ty, m->label, strlen(m->label));
	}

	/* Highlight current item if there is one. */
	if (current_item >= 0 && current_item < length)
		XFillRectangle(dpy, current_screen->popup, current_screen->menu_gc,
			0, current_item * height, width, height);
}

void
menu_motionnotify(XEvent* ev) {
	int x, y;		/* Event position. */
	int old;			/* Old menu position. */
	int width;		/* Width of menu. */
	int height;		/* Height of each menu item. */
	int length;		/* Number of menu items. */
	XButtonEvent *e = &ev->xbutton;
	
	x = e->x;
	y = e->y;
	
	getMenuDimensions(&width, &height, &length);
	
	old = current_item;
	current_item = menu_whichitem(x, y);
	
	if (current_item == old) return;
	
	// Unhighlight the old position, if it was on the menu. 
	if (old >= 0 && old < length)
		XFillRectangle(dpy, current_screen->popup, current_screen->menu_gc,
			0, old * height, width, height);
	
	// Highlight the new position, if it's on the menu. 
	if (current_item >= 0 && current_item < length)
		XFillRectangle(dpy, current_screen->popup, current_screen->menu_gc,
			0, current_item * height, width, height);
			
}

void
menu_buttonrelease(XEvent *ev) {
	int x,y;		/* Pointer position. */
	int n;		/* Menu item. */

	/*
	 * Work out which menu item the button was released over.
	 */
	x = ev->xbutton.x;
	y = ev->xbutton.y;
	n = menu_whichitem(x, y);

	/* Hide the menu until it's needed again. */
	XUnmapWindow(dpy, current_screen->popup);/*BUG?*/

	/* Do the menu thing (of unhiding windows). */
	unhide(n, 1);

	if (current)
		cmapfocus(current);
}
