/* vector.c
     COPYRIGHT
          Both this software and its documentation are

              Copyrighted 1993 by IRISA /Universite de Rennes I - France,
              Copyright 1995,1996 by BYU, Provo, Utah
                         all rights reserved.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/
#include <stdio.h>
#include <stdlib.h>

#include <polylib/polylib.h>

#ifdef MAC_OS
  #define abs __abs
#endif



Value Factorial(n)
int n;
{
	Value fact;
	int i;

	value_assign( fact, VALUE_ONE );
	for (i=1;i<=n;i++)
	{
		value_multiply( fact, int_to_value(i) );
	}
	return fact;
} /* Factorial */


Value Binomial(n, p)
int n, p;
{
	int i;
	Value prod, f;

	if (n-p<p)
		p=n-p;
	if (p!=0)
	{
		value_assign( prod, int_to_value(n-p+1) );
		for (i=n-p+2;i<=n;i++)
			value_multiply( prod, int_to_value(i) );
		value_assign( f, Factorial(p) );
		value_division( prod, f );
		return prod;
	}
	else
		return VALUE_ONE;
} /* Binomial */

/*----------------------------------------------------------------------*/
/* CNP                                                                  */
/*        a!/(b!(a-b)!) = number of ways to choose b items from a items */
/*----------------------------------------------------------------------*/
Value CNP( a, b )
int a,b;
{
	int i;
	Value c;

	if( a <= b )
		return VALUE_ONE;
	value_assign( c, VALUE_ONE );
	for( i=a ; i>b ; --i )
		value_multiply( c, int_to_value(i) );
	for( i=1 ; i<=(a-b) ; ++i )
		value_division( c, int_to_value(i) );
	return c;
} /* CNP */

Value Gcd(a, b) /* V */
Value a, b;
{
	Value aux;

	while (value_notzero_p(a)) 
	{ 
		value_assign(aux,value_mod(b,a));     /* aux = b % a */
		value_assign(b,a);         /* b = a */
		value_assign(a,aux);       /* a = aux */
	}

	value_assign( aux, value_abs (b) );
	return( aux );     /* return ( abs (b) */
} /* Gcd */


Vector *Vector_Alloc(length)
unsigned length;
{
	Vector *vec;
	vec=(Vector *)malloc(sizeof(vec));
	if (!vec)
	{  errormsg1("Vector_Alloc", "outofmem", "out of memory space");
		return 0;
	}
	vec->Size=length;
	vec->p=(Value *)malloc(length * sizeof(Value));
	if (!vec->p)
	{
		errormsg1("Vector_Alloc", "outofmem", "out of memory space");
		free(vec);
		return 0;
	}
	return vec;
} /* Vector_Alloc */


void Vector_Free(vec)
Vector *vec;
{
	if (!vec) return;
	free((char *)vec->p);
	free((char *)vec);
} /* Vector_Free */


void Vector_Set(p, n, length)
Value *p, n;
unsigned length;
{
	Value *cp;
	int i;

	cp = p;
	for (i=0;i<length;i++)
	{
		value_assign( *cp, n );
		cp++;
	}
} /* Vector_Set */


void Vector_Add(p1, p2, p3, length)
Value *p1, *p2, *p3;
unsigned length;
{
	Value *cp1, *cp2, *cp3;
	int i;

	cp1=p1;
	cp2=p2;
	cp3=p3;
	for (i=0;i<length;i++)
	{
		/* *cp3++=*cp1++ + *cp2++; */
		value_assign( *cp3, value_plus(*cp1, *cp2 ) );
		cp1++; cp2++; cp3++;
	}
} /* Vector_Add */


void Vector_Sub(p1, p2, p3, length)
Value *p1, *p2, *p3;
unsigned length;
{
	Value *cp1, *cp2, *cp3;
	int i;

	cp1=p1;
	cp2=p2;
	cp3=p3;
	for (i=0;i<length;i++)
	{
		/* *cp3++= *cp1++ - *cp2++; */
		value_assign( *cp3, value_minus( *cp1, *cp2) );
		cp1++; cp2++; cp3++;
	}
} /* Vector_Sub */


void Vector_Or(p1, p2, p3, length)
Value *p1, *p2, *p3;
unsigned length;
{
	Value *cp1, *cp2, *cp3;
	int i;

	cp1=p1;
	cp2=p2;
	cp3=p3;
	for (i=0;i<length;i++)
	{
		/* *cp3++=*cp1++ | *cp2++; */
		value_assign( *cp3, value_or( *cp1, *cp2 ) );
		cp1++; cp2++; cp3++;
	}
} /* Vector_Or */


void Vector_Scale(p1, p2, lambda, length)
Value *p1, *p2, lambda;
unsigned length;
{
	Value *cp1, *cp2;
	int i;

	cp1=p1;
	cp2=p2;
	for (i=0;i<length;i++)
	{
		/* *cp2++=*cp1++ * lambda; */
		value_assign( *cp2, value_mult(*cp1, lambda ) );
		cp1++; cp2++;
	}
} /* Vector_Scale */


void Vector_AntiScale(p1, p2, lambda, length)
Value *p1, *p2, lambda;
unsigned length;
{
	Value *cp1, *cp2;
	int i;

	cp1=p1;
	cp2=p2;
	for (i=0;i<length;i++)
	{
		/* *cp2++=*cp1++ / lambda; */
		value_assign( *cp2, value_div( *cp1, lambda ) );
		cp1++; cp2++;
	}
} /* Vector_AntiScale */


Value Vector_Min(p, length)
Value *p;
unsigned length;
{
	Value *cp, min;
	int i;

	if( length==0 )
		return( VALUE_MAX );
	cp=p;
	value_assign( min, *cp );
	cp++;
	for (i=1;i<length;i++)
	{
		if ( value_gt(min,*cp) )
			value_assign( min, *cp );
		cp++;
	}
	return min;
} /* Vector_Min */


Value Vector_Max(p, length)
Value *p;
unsigned length;
{
	Value *cp, max;
	int i;

	cp=p;
	value_assign( max, *cp );
	cp++;
	for (i=1;i<length;i++)
	{
		if ( value_lt(max,*cp) )
			value_assign( max, *cp );
		cp++;
	}
	return max;
} /* Vector_Max */


Value Inner_Product(p1, p2, length)
     Value *p1, *p2;
     unsigned length;
{
	Value ip;
	int i;

	value_assign(ip,value_mult(*p1,*p2));
	p1++; p2++;
	for( i=1 ; i<length ; i++)
	{
		value_addto(ip,value_mult(*p1,*p2));
		p1++; p2++;
	}
	return ip;
} /* Inner_Product */


void  Vector_Combine(p1, p2, p3, lambda, mu, length)
Value *p1, *p2, *p3, lambda, mu;
unsigned length;
{
	Value *cp1, *cp2, *cp3;
	Value tmp1, tmp2;
	int i;

	cp1=p1;
	cp2=p2;
	cp3=p3;

	for (i=0;i<length;i++)
	{
		/* tmp1 = lambda * *cp1 */
		value_assign(tmp1,value_mult(lambda,*cp1));

		/* tmp2 = mu * *cp2 */
		value_assign(tmp2,value_mult(mu,*cp2));

		/* *cp3 = tmp1 + tmp2 */
		value_assign(*cp3,value_plus(tmp1,tmp2));

		cp1++; cp2++; cp3++;
	}
} /* Vector_Combine */


Value Vector_Min_Not_Zero(p, length, index)
Value *p;
int *index;
unsigned length;
{
	Value *cp, min, aux;
	int i;
  
	cp=p;
	/* min = VALUE_MAX */
	value_assign (min, VALUE_MAX);

	for (i=0; i<length; i++) 
	{
		int test;

		value_assign (aux, *cp);
		value_assign(aux,value_abs(aux));

		/* test = aux < min */
		test = value_lt (aux, min);
		/* test = test && (aux != 0) */
		test = test && value_notzero_p (aux);

		if (test != 0) 
		{
		  value_assign (min, aux);
		  *index=i;
		}
		cp++;
	}

	/* if (min == Top) */
	if (value_eq (min, VALUE_MAX))
		value_assign (min, VALUE_ONE);

	return min;
} /* Vector_Min_Not_Zero */




Value  Vector_Gcd(p, q, length)
     Value *p, *q;
     unsigned length;
{

  Value *cq, *cp, min;
  int i, Not_Zero, Index_Min=0;

  cp=p;
  for (cq=q,i=0;i<length;i++) 
  {
    /* aux = abs (aux) */
    value_assign(*cq,value_abs(*cp));

    cq++;
    cp++;
  }

  do 
  {
    /* min=Vector_Min_Not_Zero(q, length, &Index_Min) */
    value_assign (min, Vector_Min_Not_Zero(q, length, &Index_Min));     
    /* if (min != 1) */
    if (value_notone_p(min))
    {
      cq=q;
      Not_Zero=0;
      for (i=0;i<length;i++,cq++)
        if (i!=Index_Min)
        {
          /* Not_Zero |= (*cq %= min) */
          Not_Zero |= value_notzero_p (value_modulus(*cq,min));
        }
    } 
    else 
      break;
  } 
  while (Not_Zero);
  
  return min;
} /* Vector_Gcd */



/* gcd reduce a vector, making sure its pos-th element is positive */
void Vector_Normalize_Positive(p, tmp, length, pos)
Value *p, *tmp;
int pos, length;
{
	Value g;
	int i;

	value_assign( g, Vector_Gcd( p, tmp, length) );	/* g is always positive */
	if ( value_neg_p(p[pos]) )
		value_oppose( g );           /* make pos-th element positive */

	/* normalize */
	if( value_notone_p(g) )
		for( i=0; i<length; i++ )
			value_division( p[i], g );

} /* Vector_Normalize_Positive */



/* gcd reduce a vector */
/* making the last element positive is *not* OK for equalities */
void  Vector_Normalize(p, q, length)
Value *p, *q;
unsigned length;
{
  Value *cp, gcd;
  int i;

  /* gcd = Vector_Gcd (p, q, length) */
  value_assign(gcd, Vector_Gcd( p, q, length));

  /* if (gcd > 1) */
  if ( value_gt(gcd, VALUE_ONE) )
  {
    cp = p;
 
    for (i=0; i<length; i++)
    { 
      /* *cp /= gcd */
      value_division(*cp, gcd);

      cp++;
    }
  }
} /* Vector_Normalize */


void Vector_Map(p1, p2, p3, length, f)
Value *p1, *p2, *p3;
unsigned length;
Value (*f)();
{
	Value *cp1, *cp2, *cp3;
	int i;
  
	cp1=p1;
	cp2=p2;
	cp3=p3;
	for (i=0;i<length;i++)
	{
		value_assign( *cp3, (*f)(*cp1, *cp2) );
		cp1++; cp2++; cp3++;
	}
} /* Vector_Map */


Value Vector_Reduce(p, length, f)
Value *p;
unsigned length;
Value (*f)();
{
	Value *cp, r;
	int i;

	cp = p;
	value_assign( r, *cp );
	cp++;
	for(i=1;i<length;i++)
	{
		value_assign( r, (*f)(r, *cp) );
		cp++;
	}
	return r;
} /* Vector_Reduce */


void Vector_Print(Dst, Format, Vec)
FILE *Dst;
char *Format;
Vector *Vec;
{
	int i;
	Value *p;
	unsigned length;

	fprintf(Dst, "%d\n", length=Vec->Size);
	p = Vec->p;
	for (i=0;i<length;i++)
	{
		if( Format )
			fprintf(Dst, Format, *p++);
		else
			fprintf(Dst, P_VALUE_FMT, *p++);
	}
	fprintf(Dst, "\n");
} /* Vector_Print */


Vector *Vector_Read()
{
	Vector *Vec;
	unsigned length;
	int i;
	Value *p;

	scanf("%d", &length);
	Vec = Vector_Alloc(length);
	if (!Vec)
	{
		errormsg1("Vector_Read", "outofmem", "out of memory space");
		return 0;
	}
	p = Vec->p;
	for (i=0;i<length;i++)
		scanf(VALUE_FMT, p++);
	return Vec;
} /* Vector_Read */


int First_Non_Zero(p, length)
Value *p;
unsigned length;
{
	Value *cp;
	int i;
  
	cp = p;
	for (i=0;i<length;i++)
	{
		if ( value_notzero_p(*cp) )
			break;
		cp++;
	}
	return( (i==length) ? -1 : i );
} /* First_Non_Zero */


void Vector_Sort(Vec, n)
Value *Vec;
unsigned n;	/* length */
{
	int i, j;
	Value temp;
	Value *current_node=(Value *)0,
	      *left_son,
	      *right_son;

	for (i=(n-1)/2;i>=0;i--)
	{                                   /* phase 1 : build the heap */
		j=i;
		value_assign( temp, *(Vec+i) );
		while (j<=(n-1)/2)
		{                                /* while not a leaf         */
			current_node = Vec+j;
			left_son = Vec+(j<<1)+1;
			if ((j<<1)+2>=n)
			{                             /* only one son             */
				if ( value_lt(temp,*left_son) )
				{
					value_assign( *current_node, *left_son );
					j=(j<<1)+1;
				}
				else
					break;
			}
			else
			{                             /* two sons                 */
				right_son=left_son+1;
				if ( value_lt(*right_son,*left_son) )
				{
					if ( value_lt(temp,*left_son) )
					{
						value_assign( *current_node, *left_son );
						j=(j<<1)+1;
					}
					else
						break;
				}
				else
				{
					if ( value_lt(temp,*right_son) )
					{
						value_assign( *current_node, *right_son );
						j=(j<<1)+2;
					}
					else
						break;
				}
			}
		}
		value_assign( *current_node, temp );
	}
	for (i=n-1;i>0;i--)
	{                                   /* phase 2 : sort the heap */
		value_assign( temp, *(Vec+i) );
		value_assign( *(Vec+i), *Vec );
		j=0;
		while (j<i/2)
		{                                /* while not a leaf        */
			current_node=Vec+j;
			left_son=Vec+(j<<1)+1;
			if ((j<<1)+2>=i)
			{                             /* only one son            */
				if ( value_lt(temp,*left_son) )
				{
					value_assign( *current_node, *left_son );
					j=(j<<1)+1;
				}
				else
					break;
			}
			else
			{                             /* two sons                */
				right_son=left_son+1;
				if ( value_lt(*right_son,*left_son) )
				{
					if ( value_lt(temp,*left_son) )
					{
						value_assign( *current_node, *left_son );
						j=(j<<1)+1;
					}
					else
						break;
				}
				else
				{
					if ( value_lt(temp,*right_son) )
					{
						value_assign( *current_node, *right_son );
						j=(j<<1)+2;
					}
					else
						break;
				}
			}
		}
		value_assign( *current_node, temp );
	}
} /* Vector_Sort */


int Vector_Equal(Vec1, Vec2, n)
Value *Vec1, *Vec2;
unsigned n;
{
	int i;
	Value *p1, *p2;

	p1=Vec1;
	p2=Vec2;
	for (i=0;i<n;i++)
	{
		/* if (*p1++!=*p2++) break; */
		if ( value_ne(*p1,*p2) )
			break;
		p1++; p2++;
	}

	return (i==n);
} /* Vector_Equal */



