/*************************************************/
/*     verif_ehrhart.c                           */
/* program to compare effective number of points */
/* in a polytope with the corresponding          */
/* evaluation of the Ehrhart polynomial.         */
/* Parameters vary in range -RANGE to RANGE      */
/* (define below) by default.                    */
/* Can be overridden by specifying               */
/* -r<RANGE>, or -m<min> and -M<max>             */
/*                                               */
/* written by Vincent Loechner (c) 2000.         */
/*  loechner@icps.u-strasbg.fr                   */
/*************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <polylib/polylib.h>
#define MAXRAYS 1024

/* define this to print all the results */
/* else, only a progress bar is printed */
/* #define PRINT_ALL_RESULTS	*/


/* RANGE : normal range for evalutations (-RANGE -> RANGE) */
#define RANGE 50
/* SRANGE : small range for evalutations */
#define SRANGE 15
/* if dimension >= BIDDIM, use SRANGE */
#define BIGDIM 5
/* VSRANGE : very small range for evalutations */
#define VSRANGE 5
/* if dimension >= VBIDDIM, use VSRANGE */
#define VBIGDIM 8

Value min, max;

#ifndef PRINT_ALL_RESULTS
int st;
#endif

/****************************************************/
/* function check_poly :                            */
/* scans the parameter space from min tp max (all   */
/* directions). Computes the number of points in    */
/* the polytope using both methods, and compare them*/
/* returns 1 on success                             */
/****************************************************/

int check_poly( Polyhedron *S, Polyhedron *C, Enumeration *en,
                int nparam, int pos, Value *z )
{
	Value k;
	Value c;
	int cc;

	if( pos == nparam )
	{
		/* computes the ehrhart polynomial */
		c = compute_poly( en, &z[S->Dimension-nparam+1] );

		/* if c=0 we may be out of context. */
		/* scanning is useless in this case*/
		if( !in_domain(C, &z[S->Dimension-nparam+1]) )
		{
			/* ok */ ;
		}
		else
		{
#ifdef PRINT_ALL_RESULTS
			printf( "EP( "VALUE_FMT, z[S->Dimension-nparam+1] );
			for( k=S->Dimension-nparam+2 ; k<=S->Dimension ; ++k )
				printf( ", "VALUE_FMT, z[k] );
			printf( " ) = "VALUE_FMT" ",	c );
#endif

			/* count manually the number of points */
			cc=count_points( 1, S, z );
			if( value_ne(int_to_value(cc),c) )
			{
				printf("\n");fflush(stdout);

				fprintf( stderr, "Error !\n" );
				fprintf( stderr, "EP( "VALUE_FMT, z[S->Dimension-nparam+1] );
				for( k=S->Dimension-nparam+2 ; k<=S->Dimension ; ++k )
					fprintf( stderr, ", "VALUE_FMT, z[k] );
				fprintf( stderr, " ) should be %d,", cc );
				fprintf( stderr, " while EP eval gives "VALUE_FMT".\n",	c );
				return( 0 );
			}
#ifdef PRINT_ALL_RESULTS
			else
				printf( "OK.\n" );
#endif
		}
	}
	else
		for( value_assign(k,min) ;
				value_le(k,max) ;
				value_increment(k) )
		{
#ifndef PRINT_ALL_RESULTS
			if( !pos && !(k%st) )
			{
				printf( "o" );
				fflush(stdout);
			}
#endif
			value_assign( z[pos+S->Dimension-nparam+1], k );
			if( ! check_poly( S, C, en, nparam, pos+1, z ) )
				return( 0 );
		}
	return( 1 );
}



int main( int argc, char *argv[] )
{
	Matrix *C1, *P1;
	Polyhedron *C, *P, *S;
	Polyhedron *CC, *PP;
	Enumeration *en;
	Value *p;
	int i,j;
	int m,M;

	/******* Read the input *********/
	P1 = Matrix_Read();
	C1 = Matrix_Read();
	if( C1->NbColumns < 2 )
	{	fprintf( stderr, "Not enough parameters !\n" );
		exit(0);
	}
	P = Constraints2Polyhedron(P1, MAXRAYS);
	C = Constraints2Polyhedron(C1, MAXRAYS);
	Matrix_Free( C1 );
	Matrix_Free( P1 );


	/******* read the options: initialize min and max ********/
	if( P->Dimension >= VBIGDIM )
		M = VSRANGE;
	else if( P->Dimension >= BIGDIM )
		M = SRANGE;
	else
		M = RANGE;
	m = -M;
	if( argc != 1 )
	{
		for( i=1 ; i<argc ; i++ )
		{
			if( !strncmp(argv[i], "-m", 2) )
			{
				/* min specified */
				m = atoi( &argv[i][2] );
			}
			else if( !strncmp(argv[i], "-M", 2) )
			{
				/* max specified */
				M = atoi( &argv[i][2] );
			}
			else if( !strncmp(argv[i], "-r", 2) )
			{
				/* range specified */
				M = atoi( &argv[i][2] );
				m = -M;
			}
			else
			{
				fprintf( stderr, "Unknown option: %s\n", argv[i] );
				fprintf( stderr, "Usage: %s [-m<>][-M<>][-r<>]", argv[0] );
				return( -1 );
			}
		}
	}
	if( m > M )
	{
		fprintf( stderr, "Nothing to do: min > max !\n" );
		return( 0 );
	}
	min = int_to_value(m);
	max = int_to_value(M);

	/******* Compute true context *******/
	CC = align_context(C, P->Dimension, MAXRAYS);
	PP = DomainIntersection( P, CC, MAXRAYS);
	Domain_Free( CC );
	C1 = Matrix_Alloc( C->Dimension+1, P->Dimension+1 );
	for( i=0 ; i<C1->NbRows ; i++ )
		for( j=0 ; j<C1->NbColumns ; j++ )
			if( i==j-P->Dimension+C->Dimension )
				value_assign( C1->p[i][j], VALUE_ONE );
			else
				value_assign( C1->p[i][j], VALUE_ZERO );
	CC = Polyhedron_Image( P, C1, MAXRAYS );
	Domain_Free( C );
	C = CC;


	/******* compute EP *********/
	en = Polyhedron_Enumerate(P,C,MAXRAYS);


	/******* initializations for check *********/
	p = (Value *)malloc( sizeof(Value) * (P->Dimension+2) );
	for( i=0 ; i<=P->Dimension ; i++ )
		value_assign( p[i], VALUE_ZERO );
	value_assign( p[i], VALUE_ONE );

	/* S = scanning list of polyhedra */
	S = Polyhedron_Scan( P, C, MAXRAYS );

#ifndef PRINT_ALL_RESULTS
	if( C->Dimension > 0 )
	{
		if( max-min>80 )
			st = 1+(max-min)/80;
		else
			st=1;
		for( i=min ; i<=max ; i+=st )
			printf(".");
		printf( "\r" );
		fflush(stdout);
	}
#endif

	/******* CHECK NOW *********/
	if( ! check_poly( S, C, en, C->Dimension, 0, p ) )
	{
		fprintf( stderr, "Check failed !\n" );
		return( -1 );
	}
#ifndef PRINT_ALL_RESULTS
	printf( "\n" );
#endif
	return( 0 );
}

