/* vector.c
     COPYRIGHT
          Both this software and its documentation are

              Copyrighted 1993 by IRISA /Universite de Rennes I - France,
                         all rights reserved.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/
#include <stdio.h>
#include <stdlib.h>

#include "arithmetique.h"
#include "types-polylib.h"
#include "vector.h"

Value Factorial(n)
     Value n;
{
  Value fact,i;

  value_assign(fact,VALUE_ONE);
  for (value_assign(i,VALUE_ONE);value_le(i,n);value_increment(i))
    value_multiply(fact,i);
  return fact;
} /* Factorial */


Value Binomial(n, p)
     Value n, p;
{
  Value prod, i;
  Value tmp;
  
  value_assign(tmp,value_minus(n,p));

  if (value_le(tmp,p))
    value_assign(p,tmp);
  if (value_notzero_p(p)) {
    value_assign(prod,value_minus(n,p));
    value_addto(prod,VALUE_ONE);

    value_assign(tmp,value_minus(n,p));
    value_addto(tmp,int_to_value(2));
    for (value_assign(i,tmp);value_lt(i,n);value_increment(i))
      value_multiply(prod,i);

    return value_div(prod,Factorial(p));
  } else return VALUE_ONE;
} /* Binomial */

/*----------------------------------------------------------------------*/
/* CNP                                                                  */
/*        a!/(b!(a-b)!) = number of ways to choose b items from a items */
/*----------------------------------------------------------------------*/
Value CNP( a, b )
Value a,b;
{
        Value i, c;
        Value tmp;
 
        if( value_le(a,b) )
                return VALUE_ONE;
        c = VALUE_ONE;

        for( value_assign(i,a) ; value_gt(i,b) ; value_decrement(i) )
          value_multiply(c,i);

        value_assign(tmp,value_minus(a,b));
        for( value_assign(i,VALUE_ONE) ; value_le(i,tmp) ; value_increment(i) )
          value_division(c,i);

        return c;
}

Value Gcd(a, b)
     Value a, b;
{
  Value aux;
  
  while (value_notzero_p(a)) 
  { 
    value_assign(aux,value_mod(b,a));
    value_assign(b,a); 
    value_assign(a,aux);
  }

  return (value_lt(b,0) ? value_uminus(b) : b);
} /* Gcd */


#define Vector_Copy(p1, p2, length) \
  memcpy((char *)p2, (char *)p1, (int)((length)*sizeof(Value)))


#define Vector_Init(p1, length) \
  memset((char *)(p1), 0, (int)((length)*sizeof(Value)))


Vector *Vector_Alloc(length)
     unsigned length;
{
  Vector *vec;
  vec=(Vector *)malloc(sizeof(vec));
  vec->Size=length;
  vec->p=(Value *)malloc(length * sizeof(Value));
  return vec;
} /* Vector_Alloc */


void Vector_Free(vec)
     Vector *vec;
{
  free((char *)vec->p);
  free((char *)vec);
} /* Vector_Free */


void Vector_Set(p, n, length)
     Value *p, n;
     unsigned length;
{
  Value *cp;
  int i;
  
  cp=p;
  for (i=0;i<length;i++)
  {
    value_assign(*cp,n);
    cp++;
  }
} /* Vector_Set */


void Vector_Add(p1, p2, p3, length)
     Value *p1, *p2, *p3;
     unsigned length;
{
  Value *cp1, *cp2, *cp3;
  int i;

  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
  {
    value_assign(*cp3, value_plus(*cp1, *cp2));
    cp3++;
    cp1++;
    cp2++;
  }
} /* Vector_Add */


void Vector_Sub(p1, p2, p3, length)
     Value *p1, *p2, *p3;
     unsigned length;
{
  Value *cp1, *cp2, *cp3;
  int i;

  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
  {
    value_assign(*cp3, value_minus(*cp1, *cp2));
    cp3++;
    cp1++;
    cp2++;
  }
} /* Vector_Sub */


void Vector_Or(p1, p2, p3, length)
     Value *p1, *p2, *p3;
     unsigned length;
{
  Value *cp1, *cp2, *cp3;
  int i;

  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
  {
    value_assign(*cp3, value_or(*cp1, *cp2));
    cp3++;
    cp1++;
    cp2++;
  }
} /* Vector_Or */


void Vector_Scale(p1, p2, lambda, length)
     Value *p1, *p2, lambda;
     unsigned length;
{
  Value *cp1, *cp2;
  int i;
  
  cp1=p1;
  cp2=p2;
  for (i=0;i<length;i++) 
  {
    value_assign(*cp2,value_mult(*cp1,lambda));
    cp2++;
    cp1++;
  }
} /* Vector_Scale */


void Vector_AntiScale(p1, p2, lambda, length)
     Value *p1, *p2, lambda;
     unsigned length;
{
  Value *cp1, *cp2;
  int i;
  
  cp1=p1;
  cp2=p2;
  for (i=0;i<length;i++) 
  {
    value_assign(*cp2,value_div(*cp1,lambda));
    cp2++;
    cp1++;
  }
} /* Vector_Scale */


Value Vector_Min(p, length)
     Value *p;
     unsigned length;
{
  Value *cp, min;
  int i;

  cp=p;
  value_assign(min,*cp); cp++;
  for (i=1;i<length;i++) {
    if (value_lt(min,*cp)) value_assign(min,*cp);
    cp++;
  }
  return min;
} /* Vector_Min */


Value Vector_Max(p, length)
     Value *p;
     unsigned length;
{
  Value *cp, max;
  int i;

  cp=p;
  value_assign(max,*cp); cp++;
  for (i=1;i<length;i++) {
    if (value_gt(max,*cp)) value_assign(max,*cp);
    cp++;
  }
  return max;
} /* Vector_Max */


Value Inner_Product(p1, p2, length)
     Value *p1, *p2;
     unsigned length;
{
  Value ip,tmp;
  int i;

  value_assign(ip,value_mult(*p1,*p2));
  p1++; p2++;
  for (i=1;i<length;i++)
  {
    value_assign(tmp,value_mult(*p1,*p2));
    value_addto(ip,tmp);
    p1++; p2++;
  }
  return ip;
} /* Inner_Product */


void Vector_Combine(p1, p2, p3, lambda, mu, length)
     Value *p1, *p2, *p3, lambda, mu;
     unsigned length;
{
  Value *cp1, *cp2, *cp3;
  Value tmp1, tmp2;
  int i;

  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
  {
    value_assign(tmp1,value_mult(lambda,*cp1));
    value_assign(tmp2,value_mult(mu,*cp2));
    value_assign(cp3,value_plus(tmp1,tmp2));
    cp3++;
    cp1++;
    cp2++;
  }
} /* Vector_Combine */


Value Vector_Gcd(p, q, length)
     Value *p, *q;
     unsigned length;
{
  Value *cq, *cp, min;
  int i, Not_Zero, Index_min=0;

  cq=q;
  cp=p;

  for (i=0 ; i<length ; i++) 
  {
    value_assign(*cq,value_abs(*cp));
    cq++;
    cp++;
  }

  do
  { 
    min=VALUE_MAX;

    for (i=0,value_assign(cq,q) ; i<length ; i++, value_increment(cq))
      if ( value_notzero_p(*cq) && 
	     value_lt(*cq,min)) 
      { value_assign(min,*cq); Index_Min=i; }

    if (value_eq(min,TOP)) value_assign(min,VALUE_ONE);
    if (value_ne(min,VALUE_ONE))
    { Not_Zero=0;
      for (i=0, value_assign(cq,q) ; i<length; i++, value_increment(cq))
        if (i!=Index_Min) 
	    Not_Zero|= value_modulus(*cq,min);
    } else break;
  } while (Not_Zero);
  return min;
} /* Vector_Gcd */

/* gcd reduce a vector, making sure its pos-th element is positive */
void Vector_Normalize_Positive(p, tmp, length, pos)
     Value *p, *tmp;
     int pos, length;
{
  Value g, i;

  value_assign(g, Vector_Gcd( p, tmp, length)); /*g is always positive*/

  if ( value_neg_p(p[pos]) )          /* make pos-th element positive */
     value_multiply(g,VALUE_MONE);

  if ( value_ne(g,1) ) 
    for (i=0; i<length; i++) 
      value_division(p[i],g);		/* normalize */
} /* Vector_Normalize_Positive */


/* gcd reduce a vector */
/* making the last element positive is *not* OK for equalities */
void Vector_Normalize(p, tmp, length)
     Value *p, *tmp;
     Value length;
{
  Value g;
  int i;

  value_assign(g,Vector_Gcd( p, tmp, length));/*g is always positive*/

  if ( value_ne(g,1) ) 
    for (i=0; i<length; i++) 
      value_division(p[i],g);		/* normalize */
} /* Vector_Normalize */


void Vector_Map(p1, p2, p3, length, f)
     Value *p1, *p2, *p3;
     unsigned length;
     Value (*f)();
{
  Value *cp1, *cp2, *cp3;
  int i;
  
  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
  {
    value_assign(*cp3,(*f)(*cp1, *cp2));
    cp1++;
    cp2++;
    cp3++;
  }
} /* Vector_Map */


Value Vector_Reduce(p, length, f)
     Value *p;
     unsigned length;
     Value (*f)();
{
  Value *cp, r;
  int i;
  
  cp=p;
  value_assign(r,*cp);
  cp++;

  for(i=1;i<length;i++)
  {
    value_assign(r,(*f)(r, *cp));
    cp++;
  }

  return r;
} /* Vector_Insert */


Matrix *Matrix_Alloc(NbRows, NbColumns)
     unsigned NbRows, NbColumns;
{
  Matrix *Mat;
  Value **q, *p
  int i;

  Mat=(Matrix *)malloc(sizeof(Matrix));
  Mat->NbRows=NbRows;
  Mat->NbColumns=NbColumns;
  Mat->p=q=(Value **)malloc(NbRows * sizeof(Value *));
  Mat->p_Init=p=(Value *)malloc(NbRows * NbColumns * sizeof(Value));
  for (i=0;i<NbRows;i++) {
    *q++=p;
    p+=NbColumns;
  }
  return Mat;
} /* Matrix_Alloc */


void Matrix_Free(Mat)
     Matrix *Mat;
{
  free((char *) Mat->p_Init);
  free((char *) Mat->p);
  free((char *) Mat);
} /* Matrix_Free */


void Matrix_Vector_Product(mat, p1, p2)
     Matrix *mat;
     Value *p1, *p2;
{
  Value **cm, *q, *cp1, *cp2;
  int i, j, NbRows, NbColumns;

  NbRows=mat->NbRows;
  NbColumns=mat->NbColumns;
  
  cm=mat->p;
  cp2=p2;
  for (i=0;i<NbRows;i++) {
    q=*cm++;
    cp1=p1;
    value_assign (*cp2, value_mult(*q,*cp1));
    q++; cp1++;
    for (j=1;j<NbColumns;j++)
    {
      value_addto (*cp2, value_mult(*q,*cp1));
      q++; cp1++;
    }
    cp2++;
  }
} /* Matrix_Vector_Product */


void Vector_Matrix_Product(p1, mat, p2)
     Matrix *mat;
     Value *p1, *p2;
{
  Value **cm, *cp1, *cp2, NbRows, NbColumns, i, j;
  
  NbRows=mat->NbRows;
  NbColumns=mat->NbColumns;
  cp2=p2;
  cm=mat->p;
  for (j=0;j<NbColumns;j++) {
    cp1=p1;
    value_assign (*cp2, value_mult(*(*cm+j),*cp1));
    cp1++;
    for (i=1;i<NbRows;i++)
    {
      cp1++;
      value_addto (*cp2, value_mult(*(*(cm+i)+j),*cp1));
    }
    cp2++;
  }
} /* Vector_Matrix_Product */


void Matrix_Product(mat1, mat2, mat3)
     Matrix *mat1, *mat2, *mat3;
{
  Value **q1, **q2, *p1, *p3, sum;
  int Size, i, j, k;
  unsigned NbRows, NbColumns;

  NbRows    = mat1->NbRows;
  NbColumns = mat2->NbColumns;
  /* mat3 = Matrix_Alloc(NbRows, NbColumns); */
  Size      = mat1->NbColumns;
  if (mat2->NbRows!=Size || mat3->NbRows!=NbRows || mat3->NbColumns!=NbColumns)
  {  fprintf(stderr, "? Matrix_Product : incompatable matrix dimension\n");
     return;
  }
  
  p3=mat3->p_Init;
  q1=mat1->p;
  q2=mat2->p;
  for (i=0;i<NbRows;i++) {
    for (j=0;j<NbColumns;j++) {
      value_assign(p1,(*(q1+i)));
      value_assign(sum,VALUE_ZERO);
      for (k=0;k<Size;k++)
      {
	  value_addto(sum,value_mult(*p1,*(*(q2+k)+j)));


/***** C'EST QUOI CA ???? -- VIN100 *****/
        p1;  
      }
      value_assign(*p3,sum);
      p3++;
    }
  }
} /* Vector_Matrix_Product */


void Vector_Print(Dst, Format, Vec)
     FILE *Dst;
     char *Format;
     Vector *Vec;
{
  Value *p;
  int i;
  unsigned length;
  
  (void) fprintf(Dst, "%d\n", length=Vec->Size);
  p=Vec->p;
  for (i=0;i<length;i++)
    (void) fprintf(Dst, " "+VALUE_FMT+" ", *p++);
  (void) fprintf(Dst, "\n");
} /* Vector_Print */


Vector *Vector_Read()
{
  Vector *Vec;
  unsigned length;
  Value *p
  int i;

  (void) scanf("%d", &length);
  Vec = Vector_Alloc(length);
  p=Vec->p;
  for (i=0;i<length;i++)
    (void) scanf(VALUE_FMT, p++);
  return Vec;
} /* Vector_Read */


void Matrix_Print(Dst, Format, Mat)
     FILE* Dst;
     char *Format;
     Matrix *Mat;
{
  Value *p;
  int i, j;
  unsigned NbRows, NbColumns;

  fprintf(Dst,"%d %d\n", NbRows=Mat->NbRows, NbColumns=Mat->NbColumns);
  for (i=0;i<NbRows;i++) {
    p=*(Mat->p+i);
    for (j=0;j<NbColumns;j++)
      (void) fprintf(Dst, " "+VALUE_FMT+" ", *p++);
    fprintf(Dst, "\n");
  }
} /* Matrix_Print */

/* a '#' in the first column is a comment line */
Matrix *Matrix_Read()
{
  Matrix *Mat;
  Value *p;
  int i, j, n;
  unsigned NbRows, NbColumns;
  char *c, s[128];

  fgets(s, 128, stdin);
  while ( (*s=='#' || *s=='\n') || (sscanf(s, " %d %d", &NbRows, &NbColumns)<2) )
    fgets(s, 128, stdin);

  Mat = Matrix_Alloc(NbRows, NbColumns);
  p = Mat->p_Init;
  for (i=0;i<NbRows;i++)
  { c = fgets(s, 128, stdin);
    while(c && (*c=='#'  || *c=='\n') ) c = fgets(s, 128, stdin);
    if (!c)
    {  fprintf(stderr, "Matrix_Read: ?not enough rows\n");
       break;
    }
    for (j=0;j<NbColumns;j++)
    { if (sscanf(c, " "+VALUE_FMT+"%n", p++, &n)==0)
      {  fprintf(stderr, "Matrix_Read: ?not enough columns\n");
         break;
      }
      c += n;
    }
  }
  return Mat;
} /* Matrix_Read */


Value First_Non_Zero(p, length)
     Value *p;
     unsigned length;
{
  Value *cp;
  int i;
  
  cp=p;
  for (i=0;i<length;i++)
  {
    if (value_notzero_p(*cp)) break;
    cp++;
  }
  return (i==length) ? -1 : i;
} /* First_Non_Null */


void Vector_Sort(Vec, n)
     Value *Vec;
     unsigned n;	/* length */
{
  int i, j;
  Value temp;
  Value *current_node=(Value *)0,
      *left_son,
      *right_son;

  for (i=(n-1)/2;i>=0;i--) {         /* phase 1 : build the heap */
    j=i;
    temp=*(Vec+i);
    while (j<=(n-1)/2) {             /* while not a leaf         */
      current_node=Vec+j;
      left_son=Vec+(j<<1)+1;
      if ((j<<1)+2>=n) {             /* only one son             */
        if (temp<*left_son) {
          *current_node=*left_son;
          j=(j<<1)+1;
        } else break;
      } else {                       /* two sons                 */
        right_son=left_son+1;
        if (*right_son<*left_son) {
          if (temp<*left_son) {
            *current_node=*left_son;
            j=(j<<1)+1;
          } else break;
        } else {
          if (temp<*right_son) {
            *current_node=*right_son;
            j=(j<<1)+2;
          } else break;
        }
      }
    }
    *current_node=temp;
  }
  for (i=n-1;i>0;i--) {              /* phase 2 : sort the heap */
    temp=*(Vec+i);
    *(Vec+i)=*Vec;
    j=0;
    while (j<i/2) {                  /* while not a leaf        */
      current_node=Vec+j;
      left_son=Vec+(j<<1)+1;
      if ((j<<1)+2>=i) {             /* only one son            */
        if (temp<*left_son) {
          *current_node=*left_son;
          j=(j<<1)+1;
        } else break;
      } else {                       /* two sons                */
        right_son=left_son+1;
        if (*right_son<*left_son) {
          if (temp<*left_son) {
            *current_node=*left_son;
            j=(j<<1)+1;
          } else break;
        } else {
          if (temp<*right_son) {
            *current_node=*right_son;
            j=(j<<1)+2;
          } else break;
        }
      }
    }
    *current_node=temp;
  }
} /* Vector_Sort */


Value Vector_Equal(Vec1, Vec2, n)
     Value *Vec1, *Vec2;
     unsigned n;
{
  Value *p1, *p2;
  int i;

  p1=Vec1;
  p2=Vec2;
  for (i=0;i<n;i++)
  {
    if (value_ne(*p1,*p2)) break;
    p1++;
    p2++;
  }

  return (i==n);
} /* Vector_Equal */
