#ifndef _EHRHART_H_
#define _EHRHART_H_

#include "types.h"

/* data structures for a pseudo-polynomial */

typedef enum { polynomial, periodic, evector } enode_type;

typedef struct _evalue		/* rational number or pointer */
        { int d;		/* denominator */
          union
          {  int n;		/* numerator (if denominator != 0) */
             struct _enode *p;	/* pointer   (if denominator == 0) */
          } x;
        } evalue;

typedef struct _enode
        {  enode_type type;	/* polynomial or periodic */
           int size;            /* number of attached pointers */
           int pos;		/* parameter position */
           evalue arr [1];	/* array of rational/pointer */
        } enode;

typedef struct _enumeration
{
	struct _enumeration *next;  /* next in the list                  */
	Polyhedron *ValidityDomain; /* contraints on the parameters      */
	evalue EP;                  /* Ehrhart Polynomial, corresponding
	                               to parameter values inside the
	                               domain ValidityDomain below       */
} Enumeration;

/*-----------------------------Example Usage------------------------------*/
/* enode *e                                                               */
/*     e->type = polynomial     e->type = periodic   e->type = evector    */
/*     e->size = degree+1       e->size = period     e->size = length     */
/*     e->pos  = [1..nb_param]                                            */
/*     e->arr[i].d = denominator                                          */
/*     e->arr[i].x.p = pointer to another enode (if denominator is zero)  */
/*     e->arr[i].x.n = numerator (if denominator is non-zero)             */
/*------------------------------------------------------------------------*/

/*------------------------------------------------------------------------*/
/* This representation has the following advantages:                      */
/*   -- its dynamic, it can grow/shrink easily                            */
/*   -- it is easy to evaluate for a given context (values of parameters) */
/*   -- it allows pseudo-polynomial to be reduced with rules              */
/*   -- it can be constructed recursively                                 */
/*------------------------------------------------------------------------*/

#ifndef FOREVER
#define FOREVER for(;;)
#endif

#if __STDC__

extern Enumeration *Polyhedron_Enumerate ( Polyhedron *P, Polyhedron *C, int
                                      MAXRAYS );
extern Polyhedron *Polyhedron_Scan ( Polyhedron *D, Polyhedron *C, int
                                     MAXRAYS );

#else /* __STDC__ */

extern Enumeration *Polyhedron_Enumerate (/* Polyhedron *P, Polyhedron *C, int
                                      MAXRAYS */);
extern Polyhedron *Polyhedron_Scan (/* Polyhedron *D, Polyhedron *C, int
                                     MAXRAYS */);

#endif /* __STDC__ */

#endif /* _EHRHART_H_ */
