#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#define PCHAR 'O'		/* first-1 parameter name char. */

#define EDEBUG0		/* prints results only */

#undef EDEBUG		/* full debug */
#undef EDEBUG1		/* prints enumeration points */
#undef EDEBUG2		/* prints domains */
#undef EDEBUG3		/* prints systems of equations that are solved */
#undef EDEBUG4		/* prints message for degree reduction */
#undef EDEBUG5		/* prints result before simplification */
#define EP_EVALUATION	/* Evaluation of the ehrhart polynomial */

#include "stdansi.h"
#include "types.h"
#include "polyhedron.h"
#include "vector.h"
#include "polyparam.h"
#include "ehrhart.h"

#ifdef EP_EVALUATION
#include "eval_ehrhart.h"
#endif


#define REDUCE_DEGREE
#undef ALL_OVERFLOW_WARNINGS	/* Print all warnings on stdout, or just one */

#ifndef ALL_OVERFLOW_WARNINGS
int overflow_warning_flag = 1;
#endif

FILE *yyout = stdout;

#define Vector_Init(p1,length) memset((char *)(p1),0,(int)(length)*sizeof(int))

/*-------------------------------------------------------------------*/
/* EHRHART POLYNOMIAL SYMBOLIC ALGEBRA SYSTEM                        */
/*-------------------------------------------------------------------*/ 
/*------------------------------------------------------------*/
/* enode *new_enode(type, size, pos)                          */
/*      type : enode type                                     */
/*      size : degree+1 for polynomial, period for periodic   */
/*      pos  : 1..nb_param, position of parameter             */
/* returns a newly allocated enode                            */
/* can be freed with a simple free(x)                         */
/*------------------------------------------------------------*/
static enode *new_enode(type,size,pos)
enode_type type;
int size, pos;
{  enode *res;
   int i;
   res = (enode *) malloc( sizeof(enode) + 2*(size-1)*sizeof(int) );
   res->type = type;
   res->size = size;
   res->pos = pos;
   for (i=0; i<size; i++)
   {  res->arr[i].d = 0;
      res->arr[i].x.n = 0;
   }
   return res;
}

/*------------------------------------------------------------*/
/* free_evalue_refs(e)                                        */
/*     e : pointer to an evalue                               */
/* releases all memory referenced by e.  (recursive)          */
/*------------------------------------------------------------*/
void free_evalue_refs(e)
evalue *e;
{   enode *p;
    int i;

    if (e->d!=0) return;  /* its a constant */
    p = e->x.p;
    if (!p) return;	/* null pointer */
    for (i=0; i<p->size; i++)
    {  free_evalue_refs(&(p->arr[i]));
    }
    free(p);
}

/*------------------------------------------------------------*/
/* evalue_copy(e)                                             */
/*     e : pointer to an evalue                               */
/*------------------------------------------------------------*/
enode *ecopy(e)
enode *e;
{
	enode *res;
	int i;

	res = new_enode( e->type, e->size, e->pos );
	for( i=0 ; i<e->size ; ++i )
	{
		if( (res->arr[i].d=e->arr[i].d)==0 )
			res->arr[i].x.p = ecopy( e->arr[i].x.p );
		else
			res->arr[i].x.n = e->arr[i].x.n;
	}
	return( res );
}

/*------------------------------------------------------------*/
/* print_evalue(DST, e)                                       */
/*    DST : destination file                                  */
/*    e : pointer to evalue to be printed                     */
/* Returns nothing, prints the evalue to DST                  */
/*------------------------------------------------------------*/
void print_enode(FILE *DST, enode *p); /* forward declaration */
void print_evalue(DST, e)
FILE *DST;
evalue *e;
{  
    if (e->d != 0)
    {   if (e->d != 1) fprintf(DST, "%d/%d", e->x.n, e->d);
        else fprintf(DST, "%d", e->x.n);
        return;
    }
    else print_enode(DST, e->x.p);
}

/*------------------------------------------------------------*/
/* print_enode(DST, p )                                       */
/*    DST : destination file                                  */
/*    p : pointer to enode  to be printed                     */
/* Returns nothing, prints the enode  to DST                  */
/*------------------------------------------------------------*/
void print_enode(DST, p)
FILE *DST;
enode *p;
{   int i;

    if (!p)
    {   fprintf(DST, "NULL");
        return;
    }
    if (p->type == evector)
    {   fprintf(DST, "{ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " }\n");
    }
    else if (p->type == polynomial)
    {   fprintf(DST, "( ");
        for (i=p->size-1; i>=0; i--)
        {  print_evalue(DST, &p->arr[i]);
           if (i==1) fprintf(DST, " * %c + ", PCHAR+p->pos);
           else if ( i>1 ) fprintf(DST, " * %c^%d + ", PCHAR+p->pos, i);
        }
        fprintf(DST, " )\n");
    }
    else if (p->type == periodic)
    {   fprintf(DST, "[ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " ]_%c", PCHAR+p->pos);
    }
}

/*------------------------------------------------------------*/
/* int eequal (e1, e2)                                        */
/*      e1, e2 : pointers to evalues                          */
/* returns 1 (true) if they are equal, 0 (false) if not       */
/*------------------------------------------------------------*/
static int eequal(e1, e2)
evalue *e1, *e2;
{  int i;
   enode *p1, *p2;

   if (e1->d != e2->d) return 0;
   /* e1->d == e2->d */
   if (e1->d !=0)
   {  if (e1->x.n!=e2->x.n) return 0;
      /* e1->d == e2->d != 0  AND e1->n == e2->n */
      return 1;
   }
   /* e1->d == e2->d == 0 */
   p1 = e1->x.p;
   p2 = e2->x.p;
   if (p1->type != p2->type) return 0;
   if (p1->size != p2->size) return 0;
   if (p1->pos  != p2->pos ) return 0;
   for (i=0; i<p1->size; i++)
      if (!eequal(&p1->arr[i], &p2->arr[i]) ) return 0;
   return 1;
}

/*------------------------------------------------------------*/
/* void reduce_evalue ( e )                                   */
/*       e : pointer to an evalue                             */
/*------------------------------------------------------------*/
static void reduce_evalue ( e )
evalue *e;
{  enode *p;
   int i, j, k;

   if (e->d!=0) return;	/* a rational number, its already reduced */
   p = e->x.p;

   /* first reduce the components of p */
   for (i=0; i<p->size; i++) reduce_evalue(&p->arr[i]);

   if (p->type==periodic)
   {  /* try to reduce the period */
      for (i=1; i<=(p->size)/2; i++)
      {  if ( (p->size % i)==0 )
         {  /* can we reduce the size to i ? */
            for (j=0; j<i; j++)
               for (k=j+i; k<e->x.p->size; k+=i)
                  if (!eequal(&p->arr[j], &p->arr[k]) ) goto you_lose;
            /* OK, lets do it */
            for (j=i; j<p->size; j++) free_evalue_refs(&p->arr[j]);
            p->size = i;
            break;
you_lose:   /* OK, lets not do it */
            continue;
         }
      }

      /* try to reduce its strength */
      if (p->size == 1)
      {  e->d   = p->arr[0].d;
         e->x.p = p->arr[0].x.p;
         free(p);
      }
   }
   else if (p->type==polynomial)
   {  /* try to reduce the degree */
      for (i=p->size-1; i>=0; i--)
      {  if (p->arr[i].d==1 && p->arr[i].x.n==0)  /* zero coefficient */
            continue;
         else
            break;
      }
      if (i==-1) p->size = 1;
      else if (i+1<p->size) p->size = i+1;

      /* try to reduce its strength */
      if (p->size == 1)
      {  e->d   = p->arr[0].d;
         e->x.p = p->arr[0].x.p;
         free(p);
      }
   }
}

/*------------------------------------------------------------*/
/* void emul (e1, e2, res)                                    */
/*       e1 : pointer to an evalue                            */
/*       e2 : pointer to a constant evalue                    */
/*       res : pointer to result evalue = e1 * e2             */
/* multiplies two evalues and puts the result in res          */
/*------------------------------------------------------------*/
static void emul (e1, e2, res)
evalue *e1, *e2, *res;
{  enode *p;
   int i,g;

   if (e2->d==0)
   {  fprintf(stderr, "emul: ?expecting constant value\n");
      return;
   }
   if (e1->d!=0)
   {  /* product of two rational numbers */
      res->d = e1->d * e2->d;
      res->x.n = e1->x.n * e2->x.n;
      g = Gcd(res->x.n, res->d);
      if (g!=1)
      {  (res->d) /= g;
         (res->x.n) /= g;
      }
   }
   else /* e1 is an expression */
   {  res->d = 0;
      p = e1->x.p;
      res->x.p = new_enode(p->type, p->size, p->pos);
      for (i=0; i<p->size; i++)
      {  emul(&p->arr[i], e2, &res->x.p->arr[i]); }
   }
}

/*------------------------------------------------------------*/
/* void eadd (e1, res)                                        */
/*       e1 : an evalue                                       */
/*       res : result = res + e1                              */
/* adds one evalue to evalue 'res'                            */
/*------------------------------------------------------------*/
void eadd(e1, res)
evalue *e1, *res;
{  int i, g;

   if ((e1->d!=0) && (res->d!=0))
   {  /* add two rational numbers*/
      res->x.n = e1->x.n * res->d + res->x.n * e1->d;
      res->d = e1->d * res->d;
      g = Gcd(res->x.n, res->d);
      if (g!=1)
      {  (res->d) /= g;
         (res->x.n) /= g;
      }
      return;
   }
   else if ((e1->d!=0) && (res->d==0))
   {  if (res->x.p->type==polynomial)
      {  /* add the constant to the constant term */
         eadd(e1, &res->x.p->arr[0]);
         return;
      }
      else if (res->x.p->type==periodic)
      {  /* add the constant to all elements of periodic number */
         for (i=0; i<res->x.p->size; i++)
         {  eadd(e1, &res->x.p->arr[i]);
         }
         return;
      }
      else
      {  fprintf(stderr, "eadd: cannot add const with vector\n");
         return;
      }
   }
   else if ((e1->d==0) && (res->d!=0))
   {
		fprintf(stderr, "eadd: cannot add evalue to const\n");
		return;
   }
   else /* ((e1->d==0) && (res->d==0)) */
   {  if ( (e1->x.p->type != res->x.p->type) ||
           (e1->x.p->pos  != res->x.p->pos ) )
      {  fprintf(stderr, "eadd: ?cannot add, incompatible types\n");
         return;
      }
      if (e1->x.p->size == res->x.p->size)
      {  for (i=0; i<res->x.p->size; i++)
         {  eadd(&e1->x.p->arr[i], &res->x.p->arr[i]);
         }
         return;
      }
      /* sizes are different*/
      if (res->x.p->type==polynomial)
      {
			/* VIN100: if e1-size > res-size you have to copy e1 in a
				new enode and add res to that new node. If you do not do
				that, you lose the the upper weight part of e1 ! */
			if( e1->x.p->size > res->x.p->size )
			{
				enode *tmp;
				tmp = ecopy( e1->x.p );
				for( i=0 ; i<res->x.p->size ; ++i )
				{	eadd(&res->x.p->arr[i], &tmp->arr[i]);
					free_evalue_refs(&res->x.p->arr[i]);
				}
				res->x.p = tmp;
			}
			else
			{
				for (i=0; i<e1->x.p->size ; i++)
				{  eadd(&e1->x.p->arr[i], &res->x.p->arr[i]);
				}
				return;
			}
		}
      else if (res->x.p->type==periodic)
      {  fprintf(stderr, "eadd: ?addition of different sized periodic nos\n");
         return;
      }
      else /* evector */
      {  fprintf(stderr, "eadd: ?cannot add vectors of different length\n");
         return;
      }
   }
}

/*------------------------------------------------------------*/
/* void edot (v1, v2, res)                                    */
/*       v1 : an enode (vector)                               */
/*       v2 : an enode (vector of constants)                  */
/*       res : result (evalue) = v1.v2 (dot product)          */
/* computes the inner product of two vectors. Result = res    */
/*------------------------------------------------------------*/
void edot(v1, v2, res)
enode *v1, *v2;
evalue *res;
{  int i;
   evalue tmp;

   if ((v1->type != evector) || (v2->type != evector))
   {  fprintf(stderr, "edot: ?expecting vectors\n");
      return;
   }
   if (v1->size != v2->size)
   {  fprintf(stderr, "edot: ? vector lengths do not agree\n");
      return;
   }
   if (v1->size<=0)
   {  res->d = 1;	/* set result to 0/1 */
      res->x.n = 0;
      return;
   }

   /* vector v2 is expected to have only rational numbers in */
   /* the array.  No pointers. */

   emul(&v1->arr[0], &v2->arr[0], res);
   for (i=1; i<v1->size; i++)
   {  /* res = res + v1[i]*v2[i] */
      emul(&v1->arr[i], &v2->arr[i], &tmp);
      eadd(&tmp, res);
      free_evalue_refs(&tmp);
   }
}

/*----------------------------------------------------------------------*/
/* EXTENSIONS TO POLYHEDRAL LIBRARY                                     */
/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/
/* Polyhedron *Polyhedron_Scan(D, C, MAXRAYS)                           */
/*       D : Domain to be scanned (single polyhedron only)              */
/*       C : Context domain                                             */
/*       MAXRAYS : Workspace size                                       */
/* Returns a linked list of scan domains, outer loop first              */
/*----------------------------------------------------------------------*/
Polyhedron *Polyhedron_Scan(D, C, MAXRAYS)
Polyhedron *D, *C;
int MAXRAYS;
{
   int i, j, dim, dim2,  numP;
   Matrix *M;
   Polyhedron *C1, *C2, *D1;
   Polyhedron *res, *last, *tmp;

   numP = C->Dimension;
   dim2 = D->Dimension;
   dim = D->Dimension - numP;
   res = last = (Polyhedron *) 0;
   if (dim==0) return (Polyhedron *)0;
   M   = Matrix_Alloc(dim2, dim2+2);
   C1  = align_context(C, dim2, MAXRAYS);
   for (i=0; i<dim; i++)
   {  Vector_Init(M->p_Init, dim2*(dim2+2));
      for (j=i+1; j<dim; j++) {  M->p[j-i-1][j+1] = 1; }
      M->NbRows = dim-i-1;
      D1 = M->NbRows ? DomainAddRays(D, M, MAXRAYS) : D;
      tmp = DomainSimplify(D1, C1, MAXRAYS);
      if (!last) res = last = tmp;
      else { last->next = tmp; last = tmp; }
      C2 = DomainIntersection(C1, D1, MAXRAYS);
      if (i>0) Polyhedron_Free(C1);
      C1 = C2;
      if (M->NbRows) Polyhedron_Free(D1);
   }
   Matrix_Free(M);
   return res;
}

/*--------------------------------------------------------------------*/
/* This procedure adds additional constraints to D so that as         */
/* each parameter is scanned, it will have a minimum of 'size' points */
/* If this is not possible, an empty polyhedron is returned           */
/*--------------------------------------------------------------------*/
Polyhedron *Polyhedron_Preprocess(D, size, MAXRAYS)
Polyhedron *D;
int size, MAXRAYS;
{  int p, p1, ub, lb, a, a1, b, b1, g;
   int dim, con, new, needed;
   int **C, *temp;
   Matrix *M;
   Polyhedron *D1;


   dim = D->Dimension;
   temp=(int *)malloc((dim+1)*sizeof(int));
   con = D->NbConstraints;
   M = Matrix_Alloc(200, dim+2);
   new = 0;
   C = D->Constraint;
   for (p=1; p<=dim; p++)
   {

      for (ub=0; ub<con; ub++)
      {  if ((a=C[ub][p])>=0) continue;		/* not an upper bound */
         for (lb=0; lb<con; lb++)
         {  if ((b=C[lb][p])<=0) continue;	/* not a lower bound */
            /* check if a new constraint is needed for this (ub,lb) pair */
            /* a constraint is only needed if a previously scanned */
            /* parameter (1..p-1) constrains this parameter (p) */
            needed=0;
            for (p1=1; p1<p; p1++)
            {  if (C[ub][p1] || C[lb][p1]) {needed=1; break;}
            }
            if (!needed) continue;

            /* create new constraint: b*UB-a*LB >= a*b*size */
            g  = Gcd(abs(a), abs(b));
            a1 = a/g;
            b1 = b/g;
            M->p[new][0]=1;
            Vector_Combine(&(C[ub][1]), &(C[lb][1]), &(M->p[new][1]),
                           b1,-a1,dim+1);
            M->p[new][dim+1] += a1*b1*size;
            Vector_Normalize(&(M->p[new][1]), temp, dim+1);
            new++;
         }
      }
   }
   D1 = AddConstraints(M->p_Init, new, D, MAXRAYS);
   Matrix_Free(M);
   free(temp);
   return D1;
}

/*---------------------------------------------------------------------*/
/* PROCEDURES TO COMPUTE ENUMERATION                                   */
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/* int lower_upper_bounds(pos,P,context,LBp,UBp)                       */
/*    pos : index position of current loop index (1..hdim-1)           */
/*    P: loop domain                                                   */
/*    context : context values for fixed indices                       */
/*    LBp, UBp : pointers to resulting bounds                          */
/* returns the flag = (UB_INFINITY, LB_INFINITY)                       */
/*---------------------------------------------------------------------*/
#define LB_INFINITY 1
#define UB_INFINITY 2
int lower_upper_bounds(pos, P, context, LBp, UBp)
Polyhedron *P;
int pos, *context, *LBp, *UBp;
{  int LB=0, UB=0;
   int flag, i, n, n1, d;

   /* Compute Upper Bound and Lower Bound for current loop */
   flag = LB_INFINITY | UB_INFINITY;
   for (i=0; i<P->NbConstraints; i++)
   {  d = P->Constraint[i][pos];
      if (d==0) continue;

      n = -Inner_Product(&context[1], &(P->Constraint[i][1]),
                         P->Dimension+1 );

      /*---------------------------------------------------*/
      /* Compute n/d        n/d<0              n/d>0       */
      /*---------------------------------------------------*/
      /*  n%d == 0    floor   = n/d      floor   = n/d     */
      /*              ceiling = n/d      ceiling = n/d     */
      /*---------------------------------------------------*/
      /*  n%d != 0    floor   = n/d - 1  floor   = n/d     */
      /*              ceiling = n/d      ceiling = n/d + 1 */
      /*---------------------------------------------------*/

      /* check to see if constraint is inequality */
      /* if constraint is equality, both upper and lower bounds are fixed */
      if( P->Constraint[i][0] == 0 )	/* Equality */
      {
         /* if not integer, return 0; */
         if( n%d != 0 )
         {
         	*LBp = 1;
         	*UBp = 0;	/* empty loop */
         	return 0;
         }
         n1 = n/d;
         /* Upper and Lower bounds found */
         if( (flag&LB_INFINITY) || n1>LB ) LB = n1;
         if( (flag&UB_INFINITY) || n1<UB ) UB = n1;
         flag = 0;
      }

      if (d>0)   /* Lower Bound */
      {  /* n1 = ceiling(n/d) */
         if ((n>0) && (n%d!=0)) n1 = (n/d)+1; else n1 = n/d;
         if (flag&LB_INFINITY) { LB = n1; flag^=LB_INFINITY; }
         else if (n1>LB) LB = n1;
      }

      if (d<0)   /* Upper Bound */
      {  /* n1 = floor(n/d) */
         if ((n>0) && (n%d!=0)) n1 = (n/d)-1; else n1 = n/d;
         if (flag&UB_INFINITY) { UB = n1; flag^=UB_INFINITY; }
         else if (n1<UB) UB = n1;
      }
   }

   if ((flag & LB_INFINITY)==0) *LBp = LB;
   if ((flag & UB_INFINITY)==0) *UBp = UB;
   return flag;
}

/*---------------------------------------------------------------------*/
/* int count_points(pos,P,context)                                     */
/*    pos : index position of current loop index (0..hdim-1)           */
/*    P: loop domain                                                   */
/*    context : context values for fixed indices                       */
/* recursive procedure, recurs for each imbriquation                   */
/* returns the number of integer points in this polyhedron             */
/*---------------------------------------------------------------------*/
static int count_points (pos, P, context)
int pos;
Polyhedron *P;
int *context;
{
   int LB=0, UB=0, k, CNT;

   if ( lower_upper_bounds(pos, P, context, &LB, &UB) !=0 )
   /* Problem if UB or LB is INFINITY */
   {  fprintf(stderr, "count_points: ? infinite domain\n");
      return -1;
   }

#ifdef EDEBUG1
   if (!P->next)
   {  int i;
      for (k=LB; k<=UB; k++)
      {  fprintf(stderr, "(");
         for (i=1; i<pos; i++)
         {  fprintf(stderr, "%d,", context[i]);
         }
         fprintf(stderr, "%d)\n", k);
      }
   }
#endif

   context[pos] = 0;
   if (UB<LB) return 0;
   if (!P->next) return (UB-LB+1);

   /*-----------------------------------------------------------------*/
   /* Optimization idea                                               */
   /*   If inner loops are not a function of k (the current index)    */
   /*   i.e. if P->Constraint[i][pos]==0 for all P following this and */
   /*        for all i,                                               */
   /*   Then CNT = (UB-LB+1)*count_points(pos+1, P->next, context)    */
   /*   (skip the for loop)                                           */
   /*-----------------------------------------------------------------*/

   CNT = 0;
   for (k=LB; k<=UB; k++)
   {
      /* insert k in context */
      context[pos] = k;
      
      CNT = CNT + count_points( pos+1, P->next, context);
   }

   /* reset context */
   context[pos] = 0;
   return CNT;
}

/*------------------------------------------------------------*/
/* enode *P_Enum(L, LQ, context, pos, nb_param, dim, lcm)     */
/*     L : list of polyhedra for the loop nest                */
/*     LQ : list of polyhedra for the parameter loop nest     */
/*     pos : 1..nb_param, position of the parameter           */
/*     nb_param : number of parameters total                  */
/*     dim : total dimension of the polyhedron, param incl.   */
/*     lcm : the denominator of the polyhedron                */
/* Returns an enode tree representing the pseudo polynomial   */
/*   expression for the enumeration of the polyhedron.        */
/* A recursive procedure.                                     */
/*------------------------------------------------------------*/
static enode *P_Enum(L, LQ, context, pos, nb_param, dim, lcm)
Polyhedron *L, *LQ;
int *context;
int pos, nb_param, dim, lcm;
{  enode *res, *B, *C;
   int hdim, i, n, k, j, nexp, rank, flag, g, nLB, nUB, nlcm, noff, k1;
   int *temp;
   Matrix *A;

#ifdef EDEBUG
fprintf(stderr, "---------------------- begin P_Enum ---------------------\n");
fprintf(stderr, "Calling P_Enum with pos = %d\n", pos);
#endif

   /* Xavier Redon modification: handle the case when there is no parameter */
	if(nb_param==0)
	{
		res=new_enode(polynomial,1,0);
		res->arr[0].d=1;
		res->arr[0].x.n=count_points(1,L,context);
		return res;
	}

   /* hdim is the degree of the polynomial + 1 */
   hdim = dim-nb_param+1;		/* homogenous dim w/o parameters */

   /* code to limit generation of equations to valid parameters only */
   /*----------------------------------------------------------------*/
   flag = lower_upper_bounds(pos, LQ, &context[dim-nb_param], &nLB, &nUB);
   if (flag & LB_INFINITY)
   {  if (!(flag & UB_INFINITY))
      {  /* compute nLB such that (nUB-nLB+1) >= (hdim*lcm) */
         nLB = nUB - hdim*lcm + 1;  /* only an upper limit: set lower limit */
			if( nLB > 0 )
				nLB = 0;
      }
      else nLB = 0;	/* no upper nor lower limit: set lower limit to 0 */
   }

   /* if (nUB-nLB+1) < (hdim*lcm) then we have more unknowns than equations */
   /* We can: 1. Find more equations by changing the context parameters, or */
   /* 2. Assign extra unknowns values in such a way as to simplify result. */
   /* Think about ways to scan parameter space to get as much info out
      of it as possible */

#ifdef REDUCE_DEGREE
   if ( pos==1 && (flag & UB_INFINITY)==0 )
   /* only for finite upper bound on first parameter */
   /* NOTE: only first parameter because subsequent parameters may
            be artificially limited by the choice of the first parameter */
   {  n = nUB-nLB+1;				/* total number of samples>0 */
		if( n<0 )
			i=0;
		else 
      	i =  (n%lcm!=0) ? (n/lcm)+1 : n/lcm;	/* ceiling of n/lcm */
      /* reduce degree of polynomial based on number of sample points */
      if (i < hdim)
      {  hdim=i;
#ifdef EDEBUG4
         fprintf(stdout,
         "Parameter #%d: LB=%d UB=%d lcm=%d degree reduced to %d\n",
         pos, nLB, nUB, lcm, hdim-1);
#endif
      }
   }
#endif
   /* hdim is now set */

   /* allocate result structure */
   res = new_enode(polynomial, hdim, pos);
   for (i=0; i<hdim; i++)
   {  res->arr[i].x.p = new_enode(periodic, lcm, pos);
   }

   /* utility arrays */
   temp=(int *)malloc((2*hdim+1)*sizeof(int));	/* used for Gauss */
   A = Matrix_Alloc(hdim, 2*hdim+1);		/* used for Gauss */
   B = new_enode(evector, hdim, 0);
   C = new_enode(evector, hdim, 0);

   /*----------------------------------------------------------------*/
   /*                                                                */
   /*                                0<-----+---k--------->          */
   /* |---------noff----------------->-nlcm->-------lcm---->         */
   /* |--- . . . -----|--------------|------+-------|------+-------|-*/
   /* 0          (q-1)*lcm         q*lcm    |  (q+1)*lcm   |         */
   /*                                      nLB          nLB+lcm      */
   /*                                                                */
   /*----------------------------------------------------------------*/
   nlcm = (nLB<0)?(nLB%lcm+lcm):(nLB%lcm) ;
   noff = nLB - nlcm;			/* noff is a multiple of lcm */
   for (k=nlcm ; k<lcm+nlcm; k++)
   { 
#ifdef EDEBUG
fprintf(stderr, "Finding %d-th elements of periodic coefficients\n", k);
#endif
      for (n=k+noff, i=0; (n<hdim*lcm+nLB); n=n+lcm, i++)
      {
         /* n == i*lcm + k + noff;   */
         /* nlcm <= k < nlcm+lcm     */
         /* n mod lcm == k1 */

#ifdef ALL_OVERFLOW_WARNINGS
         if ( ((flag & UB_INFINITY)==0) && (n>nUB))
             fprintf(stdout,
                    "Domain Overflow: Parameter #%d: nLB=%d n=%d nUB=%d\n",
                             pos, nLB, n, nUB);
#else
			if ( overflow_warning_flag && ((flag & UB_INFINITY)==0) && (n>nUB))
			{
				fprintf(stdout, "\nWARNING: Parameter Domain Overflow." );
				fprintf(stdout, " Result may be incorrect on this domain.\n" );
				overflow_warning_flag = 0;
			}
#endif

         /* set parameter to n */
         context[dim-nb_param+pos] = n;
#ifdef EDEBUG1
         fprintf(stderr,"%c = %d  (hdim=%d, lcm=%d)\n",PCHAR+pos,n,hdim,lcm);
#endif
         /* setup B vector */
         if (pos==nb_param)
         {  /* call count */
            /* count can only be called when the context is fully specified */
            B->arr[i].d = 1;
            B->arr[i].x.n = count_points( 1, L, context );
#ifdef EDEBUG3
            for (j=1; j<pos; j++) fputs("   ", stdout);
            fprintf(stdout, "E(");
            for (j=1; j<nb_param; j++)
               fprintf(stdout, "%d,", context[dim-nb_param+j]);
            fprintf(stdout, "%d) = %d =", n, B->arr[i].x.n );
#endif
         }
         else /* count is a function of other parameters */
         {  /* call P_Enum recursively */
            B->arr[i].d = 0;
            B->arr[i].x.p =
                P_Enum(L,LQ->next,context,pos+1,nb_param,dim,lcm);
#ifdef EDEBUG3
            for (j=1; j<pos; j++) fputs("   ", stdout);
            fprintf(stdout, "E(");
            for (j=1; j<=pos; j++)
               fprintf(stdout, "%d,", context[dim-nb_param+j]);
            for (j=pos+1; j<nb_param; j++)
               fprintf(stdout, "%c,", PCHAR + j);
            fprintf(stdout, "%c) = ", PCHAR + j);
            print_enode(stdout, B->arr[i].x.p);
            fprintf(stdout, " =");
#endif
         }

         /* create i-th equation*/
         /* K_0 + K_1 * n**1 + ... + K_dim * n**dim | Identity Matrix */
         A->p[i][0] = 0;  /* status bit = equation */
         nexp = 1;
         for (j=1; j<=hdim; j++)
         {  A->p[i][j] = nexp;
            A->p[i][j+hdim] = 0;
#ifdef EDEBUG3
            fprintf(stdout, " + %d c%d", nexp, j);
#endif
            nexp *= n;
         }
#ifdef EDEBUG3
            fprintf(stdout, "\n");
#endif
         A->p[i][i+1+hdim] = 1;
      }
      /* assertion check */
      if (i!=hdim)
         fprintf(stderr, "P_Enum: ?expecting i==hdim\n");

#ifdef EDEBUG
fprintf(stderr, "B (enode) =\n");
print_enode(stderr, B);
fprintf(stderr, "A (Before Gauss) =\n");
Matrix_Print(stderr, " %4d", A);
#endif
      /* solve hdim (=dim+1) equations with hdim unknowns, result in CNT */
      rank = Gauss (A, hdim, 2*hdim, temp);
#ifdef EDEBUG
fprintf(stderr, "A (After Gauss) =\n");
Matrix_Print(stderr, " %4d", A);
#endif
      /* assertion check */
      if (rank!=hdim)
      {  fprintf(stderr, "P_Enum: ?expecting rank==hdim\n");
      }

      /* if (rank < hdim) then set the last hdim-rank coefficients to ? */
      /* if (rank == 0) then set all coefficients to 0 */

      /* copy result as k1-th element of periodic numbers */
      k1 = (k<lcm)? k : k-lcm;

      for (i=0; i<rank; i++)
      {  /* set up coefficient vector C from i-th row of inverted matrix */
         for (j=0; j<rank; j++)
         {  g = Gcd(A->p[i][i+1],A->p[i][j+1+hdim]);
            C->arr[j].d = A->p[i][i+1]/g;
            C->arr[j].x.n = A->p[i][j+1+hdim]/g;
         }
#ifdef EDEBUG
fprintf(stderr, "C (enode) =\n");
print_enode(stderr, C);
#endif
         /* the i-th enode is the lcm-periodic coefficient of term n**i */
         edot(B, C, &(res->arr[i].x.p->arr[k1]) );
#ifdef EDEBUG
fprintf(stderr, "B.C (evalue)=\n");
print_evalue(stderr, &(res->arr[i].x.p->arr[k1]) );
fprintf(stderr, "\n");
#endif
      }
   }
#ifdef EDEBUG
fprintf(stderr, "res (enode) =\n");
print_enode(stderr, res);
fprintf(stderr, "--------------------- end P_Enum ------------------------\n");
#endif

   /* reset context */
   context[dim-nb_param+pos] = 0;

   /* release memory */
   free(temp);
   Matrix_Free(A);
   free(B);
   free(C);

   return res;
}

/*----------------------------------------------------------------*/
/* Polyhedron_Enumerate(P, C, MAXRAYS)                            */
/*    P : Polyhedron to enumerate                                 */
/*    C : Context Domain                                          */
/*    MAXRAYS : size of workspace                                 */
/* Procedure to count points in a (fixed-parameter) polytope.     */
/* Returns a list of validity domains + evalues EP                */
/*----------------------------------------------------------------*/
Enumeration *Polyhedron_Enumerate(P, C, MAXRAYS)
Polyhedron *P;	/* Polyhedron to count */
Polyhedron *C;	/* parameter Context domain */
int MAXRAYS;	/* workspace size */
{
   Polyhedron *L, *CQ, *CQ2, *LQ, *U;
   Param_Polyhedron *PP;
   Param_Domain   *Q;
   Param_Vertices *V;
   int i, j, k, ix, hdim, dim, nb_param, lcm, nbPV;
   unsigned bx;
   int *context;
 /*  evalue e; */
	Enumeration *en, *res;

	res = NULL;

#ifdef EDEBUG2
fprintf(stderr, "C = \n");
Polyhedron_Print(stderr, " %4d", C);
#endif

   hdim		= P->Dimension + 1;
   dim		= P->Dimension;
   nb_param	= C->Dimension;

   /* Create a context vector size dim+2 */
   context = (int *) malloc ( (hdim+1)*sizeof(int) );
   
   /* Set it to all zeros */
   memset ((char *)context, 0, (hdim+1)*sizeof(int) );

   /* Set context[hdim] = 1  (the constant) */
   context[hdim] = 1;

#ifdef EDEBUG2
fprintf(stderr, "P = \n");
Polyhedron_Print(stderr, " %4d", P);
#endif

   PP = Polyhedron2Param_Domain( P, C, MAXRAYS );
	if( !PP )
	{
		fprintf(stdout, "\nEhrhart Polynomial:\n0\n");
		return( NULL );
   }
   nbPV = PP->nbV;
 
   for( Q=PP->D ; Q ; Q=Q->next )
   {
   	en = (Enumeration *)malloc( sizeof(Enumeration) );
		en->next = res;
		res = en;

		res->ValidityDomain = Q->Domain;

#ifdef EDEBUG0
fprintf(stdout, "---------------------------------------\n" );
fprintf(stdout, "Domain :\n");
Print_Domain(stdout, Q->Domain);
#endif

#ifndef ALL_OVERFLOW_WARNINGS
		overflow_warning_flag = 1;
#endif

      /* compute "denominator of P" */
      /* lcm = Least Common Multiple of the denominators of the vertices of P */
      lcm = 1;
#ifdef EDEBUG0
fprintf(stdout, "Vertices :\n");
#endif
 
      for( i=0, ix=0, bx=MSB, V=PP->V ; V && i<nbPV ; i++, V=V->next )
      {
         if (Q->F[ix] & bx)
         {
#ifdef EDEBUG0
Print_Vertex(stdout, V->Vertex );
fprintf(stdout, "\n" );
#endif
            for( j=0; j<V->Vertex->NbRows; j++)
            {  k = V->Vertex->p[j][V->Vertex->NbColumns-1];  /* a matrix */
               if ((k!=0) && (k!=1)) lcm = lcm * k / Gcd(lcm, k);
            }
         }
         NEXT(ix,bx);
      }
#ifdef EDEBUG
fprintf(stderr, "Denominator = %d\n", lcm);
#endif

      CQ = DomainIntersection(C, Q->Domain, MAXRAYS);
#ifdef EDEBUG2
fprintf(stderr, "CQ = \n");
Polyhedron_Print(stderr, " %4d", CQ);
#endif

      /* before scanning, add constraints to ensure at least hdim*lcm */
      /* points in every dimension */
      CQ2 = Polyhedron_Preprocess(CQ, hdim*lcm-1, MAXRAYS);
      if (emptyQ(CQ2))
      {  fprintf(stdout, "Degenerate Domain. Can not continue.\n");
      }

#ifdef EDEBUG2
fprintf(stderr, "CQ2 = \n");
Polyhedron_Print(stderr, " %4d", CQ2);
#endif

      /* L is used in counting the number of points in the base cases */
      L = Polyhedron_Scan(P, CQ, MAXRAYS);
      U = Universe_Polyhedron(0);
      /* LQ is used to scan the parameter space */
      LQ = Polyhedron_Scan(CQ2, U, MAXRAYS); /* bounds on parameters */
      Domain_Free(U);
      Domain_Free(CQ);
      Domain_Free(CQ2);

#ifdef EDEBUG2
fprintf(stderr, "L = \n");
Polyhedron_Print(stderr, " %4d", L);
fprintf(stderr, "LQ = \n");
Polyhedron_Print(stderr, " %4d", LQ);
#endif
#ifdef EDEBUG3
fprintf(stdout, "\nSystem of Equations:\n");
#endif
      res->EP.d = 0;
      res->EP.x.p = P_Enum(L, LQ, context, 1, nb_param, dim, lcm);
#ifdef EDEBUG5
fprintf(stdout, "\nEhrhart Polynomial (before simplification):\n");
print_evalue(stdout, &res->EP);
#endif
      reduce_evalue(&res->EP);
#ifdef EDEBUG0
fprintf(stdout, "\nEhrhart Polynomial:\n");
print_evalue(stdout, &res->EP);
fprintf( stdout, "\n" );
	/* sometimes the final \n lacks (when a single constant is printed) */
#endif

   }
	return res;
}

#ifdef EDEBUG0
int main()
{
	Matrix *C1, *P1;
	Polyhedron *C, *P;
	Enumeration *en;
	int *p, i, k;
	char s[256];

	P1 = Matrix_Read();
	C1 = Matrix_Read();
	if( C1->NbColumns < 2 )
	{	fprintf( stderr, "Not enough parameters !\n" );
		exit(0);
	}
	P = Constraints2Polyhedron(P1, 1024);
	C = Constraints2Polyhedron(C1, 1024);
	Matrix_Free( C1 );
	Matrix_Free( P1 );

	en = Polyhedron_Enumerate(P,C,1024);

#ifdef EP_EVALUATION
	if( !isatty( 0 ) )
		return 0 ;			/* no tty input -> no evaluation. */

	printf( "Evaluation of the Ehrhart polynomial :\n" );
	p = (int *)malloc( sizeof(int) * (C->Dimension) );
	FOREVER
	{
		printf( "Enter %d parameters (or 'q' to exit) : ", C->Dimension );
		if( fgets( s, 255, stdin ) == 0 )
			break;
		if( s[0]=='q' || s[0]=='Q' )
			break;
		i = 0;
		while( s[i]==' ' || s[i]=='\t' )
			++i;
		for( k=0 ; k<C->Dimension ; ++k )
		{
			if( s[i]=='\n' || s[i]==0 )
				break;
			p[k] = atoi( &s[i] );
			while( isdigit(s[i]) )
				++i;
			while( s[i]==' ' || s[i]=='\t' || ispunct(s[i]) )
				++i;
		}
		if( k != C->Dimension )
		{
			printf( "Not enough parameters!\n" );
			continue;
		}
		printf( "EP( %d", p[0] );
		for( k=1 ; k<C->Dimension ; ++k )
			printf( ", %d", p[k] );
		printf( " ) = %d\n",	compute_poly( en, p ) );
	}
#endif /* EP_EVALUATION */

	return 0;
}
#endif /* EDEBUG0 */

