/* domains.c
     COPYRIGHT
          Both this software and its documentation are
 
              Copyrighted 1997 by Vincent Loechner.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#include <polylib/polylib64.h>

#include "domains.h"
#include "gtk_ddraw.h"
#include "gtk_domain.h"
#include "gtk_properties.h"
#include "gtk_context.h"
#include "gtk_windows.h"
#include "gtk_repere.h"
#include "visutypes.h"

int contains_int_point (Polyhedron *);

int pgcd (int a, int b)
{
  int r;

  if (a == 0)
    return (abs (b));

  if (b == 0)
    return (abs (a));

  do
    {
      r = a % b;
      a = b;
      b = r;
    }
  while (r != 0);

  return (abs (a));
}



/* Parcours de polyedre */
/* Usage :
	mat = (Matrix *)Scanning_Matrix( Polyhedron *P, Matrix *order, int ws )
P is the Polyhedron to be scanned
ws the working space for the polylib calls
order a matrix containing the indices to be scanned (destroyed by this call) :
exemple :
	order = [ 0 1 0 0 0 ]
	        [ 0 0 1 0 0 ]
	        [ 0 0 0 1 0 ]
	to scan a polyhedron [i,j,k], with one parameter M

This call to Scanning_Matrix returns a matrix of the following form :
		[0    i     j   k   M  cte]
		0/1 *(!=0)  0   0   *   *
		           ...
		0/1  *!0    0   0   *   *
		0/1   *    *!0  0   *   *
		           ...
		0/1   *    *!0  0   *   *
		0/1   *     *  *!0  *   *
		           ...
		0/1   *     *  *!0  *   *
*/

static void scan_r (Polyhedron * P, Matrix * forall_ray, Matrix * C, int ws)
{
  int l, c;

  /* elimine les contraintes sur les termes restants
     en ajoutant des lines dans ces directions. */
  /* forall_ray =  0 1     c   d  d+1
     0 0..0  1 0..0   0
     ........
   */

  for (c = 1; c < forall_ray->NbColumns; ++c)
    if (forall_ray->p[0][c])
      break;
  /* elimine la premiere ligne de la matrice forall_ray */
  if (--forall_ray->NbRows)
    {
      Polyhedron *Psimp, *DU;
      memcpy (&forall_ray->p[0][0], &forall_ray->p[1][0],
	      sizeof (Value) * forall_ray->NbColumns * forall_ray->NbRows);
      DU = DomainAddRays (P, forall_ray, ws);

      /* la borne min et max en fct des precedents est donnee par DU */
      /* ajoute la matrice de contraintes de DU a C pour les lignes de
         DU contenant un !=0 a l'indice c */

      for (l = 0; l < DU->NbConstraints; ++l)
	{
	  if (DU->Constraint[l][c])
	    {
	      memcpy (&C->p[C->NbRows][0], &DU->Constraint[l][0],
		      sizeof (Value) * (DU->Dimension + 2));
	      ++C->NbRows;
	    }
	}

      /* simplifie P en donnant DU comme contexte */
      Psimp = DomainSimplify (P, DU, ws);
      Polyhedron_Free (DU);

      scan_r (Psimp, forall_ray, C, ws);

      Domain_Free (Psimp);
    }
  else
    {
      /* ajoute la matrice de contraintes restant (dans P) a C */
      for (l = 0; l < P->NbConstraints; ++l)
	memcpy (&C->p[l + C->NbRows][0], &P->Constraint[l][0],
		sizeof (Value) * (P->Dimension + 2));
      C->NbRows += P->NbConstraints;
    }
}

/* passer en parametre l'ordre dans lequel on veut executer la boucle. */
/* sous forme de vecteur ( {1,2,3,...} si ordre i->j->...) */
Matrix *Scanning_Matrix (Polyhedron * P, Matrix * order, int ws)
{
  Matrix *MRes;

  if (order->NbColumns != P->Dimension + 2)
    {
      Warn ("? Scanning_Matrix: operation on different dimensions\n");
      return NULL;
    }

/*	if( P->next )
	{
		Warn("? Scanning_Matrix: can't scan a union of domains\n");
		return NULL;
	}
*/


/*	Mtmp = Matrix_Alloc( P->Dimension, P->Dimension+2 );
 *	for( l=0 ; l<P->Dimension ; ++l )
 *	{
 *		for( i=0 ; i<Mtmp->NbColumns ; ++i )
 *			Mtmp->p[l][i] = 0;
 *		if( order->p[l]<1 || order->p[l]>P->Dimension )
 *			fprintf(stderr,"? Scanning_Matrix: incorrect order vector\n");
 *		Mtmp->p[l][order->p[l]] = 1;
 *	}
 */

  MRes = Matrix_Alloc (ws, P->Dimension + 2);
  MRes->NbRows = 0;

  scan_r (P, order, MRes, ws);

/*	Matrix_Free( Mtmp ); */

  if (MRes->NbRows == 0)
    {
      Matrix_Free (MRes);
      return NULL;
    }
  else
    {				/* trop de lignes sont allouees...
				   on recopie cette matrice dans une nouvelle */
      Matrix *M;
      M = Matrix_Alloc (MRes->NbRows, MRes->NbColumns);
      memcpy (&M->p[0][0], &MRes->p[0][0],
	      sizeof (Value) * MRes->NbRows * MRes->NbColumns);
      Matrix_Free (MRes);
      return (M);
    }
}



/* parcours d'un polyedre convexe (pas union) avec appel de fonction */
/* pour chaque pt entier */
/* USAGE :	FORALL_P_do( Polyhedron *P, int (*f)(Value *) )
		la fonction f est appelee en passant le pt courant en parametre */
/* le parcours s'arrete si f renvoie FALSE */
/* la fonction renvoie FALSE si elle s'est arretee avant la fin */
/* c = indice parcourant 0 -> NbC-2 inclus */
static int
FORALL_P_at_pt_do (Matrix * C, int c, Value * pt, int (*f) (Value *))
{
  int l, i, j;
  int min_num = 0, max_num = 0, new_num;
  int min_den = 1, max_den = 1, new_den;
  int trouve_min, trouve_max;
  int g, den;

/*
Cherche une ligne du type :
         0   1 ... c+1  c+2 .. NbC-2 NbC-1
    C =  *   * .... *    0 ....  0     *
*/

  trouve_min = trouve_max = 0;

  /* on parcourt tous les !=0 sur cette colonne */
  for (l = 0; l < C->NbRows; ++l)
    {
      if (C->p[l][c + 1] == 0)
	continue;

      /* si sur cette ligne,
         il y a un !=0 apres cet indice on ignore la ligne */
      for (j = c + 2; j < C->NbColumns - 1; ++j)
	if (C->p[l][j])
	  break;
      if (j < C->NbColumns - 1)
	continue;

      /* OK, la ligne 'l' contient bien une ref utile a cette variable */

      /* cherche la valeur de i : cte +val des ind. precedent */
      new_num = C->p[l][C->NbColumns - 1];
      for (j = 1; j <= c; ++j)
	new_num += pt[j - 1] * C->p[l][j];
      new_den = C->p[l][c + 1];
      if (new_den < 0)		/* denominateurs toujours > 0 */
	new_den = -new_den;
      else
	new_num = -new_num;
      if (C->p[l][0])
	{
	  /* inegalite */
	  if (C->p[l][c + 1] > 0)
	    {
	      /* new est un nouveau min */
	      if (trouve_min)
		{
		  if (new_num * min_den > min_num * new_den)
		    {
		      min_num = new_num;
		      min_den = new_den;
		    }
		}
	      else
		{
		  min_num = new_num;
		  min_den = new_den;
		  trouve_min = 1;
		}
	    }
	  else
	    {			/* new est un nouveau max */
	      if (trouve_max)
		{
		  if (new_num * max_den < max_num * new_den)
		    {
		      max_num = new_num;
		      max_den = new_den;
		    }
		}
	      else
		{
		  max_num = new_num;
		  max_den = new_den;
		  trouve_max = 1;
		}
	    }
	  /* si le min est devenu > au max pas la peine de continuer */
	  if (trouve_min && trouve_max)
	    if (max_num * min_den < min_num * max_den)
	      return 1;
	}
      else
	{			/* egalite */
	  /* si on a deja un min ou un max qui empiete on s'arrete la. */
	  if (trouve_min && (new_num * min_den < min_num * new_den))
	    return 1;
	  if (trouve_max && (new_num * max_den > max_num * new_den))
	    return 1;
	  trouve_min = 1;
	  trouve_max = 1;
	  min_num = max_num = new_num;
	  min_den = max_den = new_den;
	}
    }

  if (!trouve_min || !trouve_max)
    {
      fprintf (stderr, "Error in scan polyhedron : non closed polyhedron\n");
      Matrix_Print (stderr, "%3d ", C);

      Warn ("Can't visualize an infinite polyhedron\n");

      return 0;
    }


  g = pgcd (min_den, max_den);
  den = min_den * max_den / g;
  min_num = min_num * max_den / g;
  max_num = max_num * min_den / g;

  /* se place sur le premier "vrai" */
  if (min_num < 0)
    min_num += (-min_num) % den;
  else if ((min_num) % den)
    min_num += den - (min_num) % den;

  /* decrit la boucle */
  if (c == C->NbColumns - 3)
    {
      for (i = min_num; i <= max_num; i += den)
	{
	  pt[c] = i / den;
	  if (!(*f) (pt))
	    return 0;
	}
    }
  else
    {
      for (i = min_num; i <= max_num; i += den)
	{
	  pt[c] = i / den;
	  if (!FORALL_P_at_pt_do (C, c + 1, pt, f))
	    return 0;
	}
    }
  return 1;
}

int FORALL_P_do (Polyhedron * P, Value * pt, int (*f) (Value *))
{
  Matrix *constraints;
  Matrix *o;
  int i;

  if (!P || P->NbRays == 0)
    return 1;

  o = Matrix_Alloc (P->Dimension, P->Dimension + 2);
  memset (&o->p[0][0], 0, o->NbRows * o->NbColumns * sizeof (Value));
  for (i = 0; i < o->NbRows; ++i)
    o->p[i][i + 1] = 1;

  constraints = Scanning_Matrix (P, o, WS);
  Matrix_Free (o);

  if (constraints)
    return (FORALL_P_at_pt_do (constraints, 0, pt, f));
  else
    return 1;


}


static int _CIP_ok (Value * pt)
{
  return 0;
}

int contains_int_point (Polyhedron * P)
{
  int r;
  Value *pt;
  pt = malloc (sizeof (Value) * P->Dimension);
  r = FORALL_P_do (P, pt, &_CIP_ok);
  free (pt);
  return (!r);
}

static int compt;
static int compte_nb_pts_poly (Value * pt)
{
  ++compt;
  return 1;
}

/* compte le nb de pts entiers dans un polyedre */
int nb_pts_poly (Polyhedron * P)
{
  Value *pt;
  pt = malloc (sizeof (Value) * P->Dimension);
  compt = 0;
  FORALL_P_do (P, pt, &compte_nb_pts_poly);
  free (pt);

  return (compt);
}

/* teste si le point de coord. pt est dans P */
int DansPolyedre (Polyhedron * P, Value * pt)
{
  int c, i, r;

  for (c = 0; c < P->NbConstraints; ++c)
    {
      r = 0;
      for (i = 0; i < P->Dimension; ++i)
	{
	  r += pt[i] * P->Constraint[c][i + 1];
	}
      r += P->Constraint[c][P->Dimension + 1];	/* constante */

      if (P->Constraint[c][0])	/* inegalite ou egalite */
	{
	  if (r < 0)
	    return (0);
	}
      else
	{
	  if (r != 0)
	    return (0);
	}
    }
  return (1);
}

/* teste si le point de coord. pt est dans le domaine P (convexe ou non) */
gboolean DansDomaine (Polyhedron * P, Value * pt)
{
	Polyhedron * tmp;

	for (tmp = P; tmp; tmp = tmp->next)
		if( DansPolyedre (tmp, pt) )
			return( TRUE );
	return( FALSE );
}


/************************************************************************/
/* NP_Domain()								*/
/************************************************************************/
/* calcul du polyhedre non parametre correspondant a l'instantiation
			des params dans p. */
static Polyhedron *NP_Domain (Polyhedron * P)
{
  Matrix *CNP;			/* Non-Parameterized constraints */
  Polyhedron *DNP;
  int l, k;

  if (!P)
    return (NULL);

  CNP = Matrix_Alloc (P->NbConstraints, DimNP + 2);
  for (l = 0; l < P->NbConstraints; ++l)
    {
      /* les dnp+1 premieres colonnes */
      for (k = 0; k <= DimNP; ++k)
	CNP->p[l][k] = P->Constraint[l][k];

      /* la constante : */
      CNP->p[l][DimNP + 1] = P->Constraint[l][1 + P->Dimension];
      for (k = 0; k < NbParam; ++k)
	CNP->p[l][DimNP + 1] +=
	  ValeurParam[k] * P->Constraint[l][DimNP + 1 + k];
    }
  DNP = Constraints2Polyhedron (CNP, WS);
  Matrix_Free (CNP);

  DNP = AddPolyToDomain( DNP, NP_Domain(P->next) );
  if( DNP )
  	if( emptyQ(DNP) )
  	{
  		Domain_Free( DNP );
  		return( NULL );
  	}
  return( DNP );


}



void clear_A_domains ()
{
	if (U_NPDomains)
		Domain_Free (U_NPDomains);
	if (A_Union)
		Domain_Free (A_Union);
	if (A_Conv)
		Domain_Free (A_Conv);
	if (U_IntPDomains)
		Domain_Free (U_IntPDomains);

	A_Union = NULL;
	A_Conv = NULL;
	U_NPDomains = NULL;
	U_IntPDomains = NULL;
}

/* Cre l'union des domaines que l'on veut afficher pour pouvoir calculer 
   les extrmas de la figure  visualiser... */
void compute_A_domains ()
{
  Domain *d;

	if (U_NPDomains)
		Domain_Free (U_NPDomains);
  /* Construction de l'union des domaines (instancis)  afficher */
  U_NPDomains = NULL;
  for( d = DomainList ; d ; d = d->next )
  {
		if (d->visual != NULL)
		{
/*			U_NPDomains = DomainUnion (d->visual->NPdomain, U_NPDomains, WS); */
			Polyhedron *p;
			for( p=d->visual->NPdomain ; p ; p=p->next )
				U_NPDomains = AddPolyToDomain(Polyhedron_Copy(p), U_NPDomains);
		}
  }


/* Cre l'enveloppe convexe de l'union des domaines que l'on veut afficher pour
   calculer les extrmas de la figure  visualiser... */
	if (A_Conv)
		Domain_Free (A_Conv);

	A_Conv = DomainConvex( U_NPDomains, WS);

/********************** limite le bord  -10,10 dans toutes les dimensions **************/
/*
	r->NbRows = A_Conv->Dimension*2;
	for( j=0 ; j<A_Conv->Dimension ; j++ )
	{
	 for( i=1 ; i<DimNP+1 ; i++ )
	 {
   	r->p[2*j][i] = 0;
   	r->p[2*j+1][i] = 0;
	 }
	 r->p[2*j][0] = 1;
	 r->p[2*j][j+1] = -1;
	 r->p[2*j][DimNP + 1] = 10;

	 r->p[2*j+1][0] = 1;
	 r->p[2*j+1][j+1] = 1;
	 r->p[2*j+1][DimNP + 1] = 10;
	}
	A_Conv = DomainAddConstraints( A_Conv, r, WS );

	Matrix_Free (r);
*/

  return;
}

/* Cre l'union des domaines dont les points entiers sont visualiss... */
void union_intpoints_domains ()
{
	Domain *d;

	if (U_IntPDomains)
		Domain_Free (U_IntPDomains);

	U_IntPDomains = NULL;
	for( d = DomainList ; d ; d = d->next )
	{
		if (d->visual && d->visual->intpoints )
		{
/*			U_IntPDomains = DomainUnion (d->visual->NPdomain, U_IntPDomains, WS); */
			Polyhedron *p;
			for( p=d->visual->NPdomain ; p ; p=p->next )
				U_IntPDomains = AddPolyToDomain(Polyhedron_Copy(p), U_IntPDomains);
		}
	}

	return;
}

/* Procdure d'affichage de domaines paramtrs */
void Affich_P_domains (Domain * domain, gint row)
{
  Polyhedron *tmp_domain;
  gint Rrow;
  gboolean delete;
  char *Status;

  delete = FALSE;
  if (ContextOK == TRUE)
    {
      /* Les paramtres et le contexte sont OK */

      if (domain->Pdomain->Dimension - NbParam != DimNP)
	{
	 /* La dimension du domaine n'est pas cohrente avec les 
	    initialisations de DimNP et du contexte (Paranoa: a 
	    ne peut pas arriver car dans ce cas le domaine n'est
	    pas dans la liste graphique!) */
	 printf
	   ("Domain %s can't be vizualized with this context!\n",
	    domain->name);
	 delete = TRUE;
	 }
      else
	{
	  /* Le domaine aura la bonne dimension aprs instanciation */
	  tmp_domain = NP_Domain (domain->Pdomain);
	  if( tmp_domain )
	  {
// A CHANGER  0 &
	      if ( value_zero_p
		  (tmp_domain->Ray[0][tmp_domain->Dimension + 1]))
		{
		 /* Le domaine  afficher n'est pas born dans ce contexte */
		 printf("Domain %s is not bounded!\n", domain->name);
       delete = TRUE;
		 }
	      else
		{
		  /* Le domaine instanci est born */
		  /* Modifier la structure du domaine en consquence */
		  if (!domain->visual)
		    {
		     /* Ce domaine n'tait pas affich */
		     domain->visual =
		       (VisualDomain *) malloc (sizeof (VisualDomain));
// A CHANGER		     domain->visual->intpoints = FALSE;
		     domain->visual->intpoints = TRUE;
		     domain->visual->Lpoints = NULL;
		     domain->visual->NPdomain = NULL;
		     /* M--j de la liste graphique */
		     gtk_clist_set_foreground (GTK_CLIST (clist), row,
		                               domain->color);
		     gtk_clist_set_text (GTK_CLIST (clist), row, 1, "Visu");
		     }
		  if (domain->visual->NPdomain)
		     Domain_Free (domain->visual->NPdomain);
		  domain->visual->NPdomain = tmp_domain;
		  if (domain->visual->Lpoints)
		     ListValuePoints_Free (domain->visual->Lpoints);
		  domain->visual->Lpoints = NULL;
		  domain->visual->Lpoints = ListValuePoints_Update (domain);
		}
	  }
	  else
	  {
	    printf ("Empty domain %s!\n", domain->name);
      delete = TRUE;
	  }

	}
    }
  else
    {
      /* ContextOK = FALSE */
/* pas grave !
      Warn ("Bad parameters!");
      delete = TRUE;
*/
    }

  if (delete)
  {
		if (domain->visual)
		{
			/* Ce domaine tait affich avant le changement de paramtres, il
			faut le masquer... */
			Rrow = gtk_clist_find_row_from_data (GTK_CLIST (clist), domain);
			if (Rrow == -1)
			{
				Warn ("Internal problem: to mask a domain!");
				return;
			}
			else
			{
				delete = gtk_clist_get_text (GTK_CLIST (clist), Rrow, 1, &Status);
				if (strcmp (Status, "Visu") == 0)
					gtk_clist_select_row (GTK_CLIST (clist), Rrow, 1);
			}
		}
  }
  compute_A_domains ();

  calc_rep_extr ();
  return;
}

void Affich_NOT_P_domains (Domain * domain, gint row)
{
  if (domain->Pdomain->Dimension != DimNP)
    /* La dimension du domaine n'est pas cohrente avec les 
       initialisations de DimNP et du contexte */
    printf
      ("Domain %s can't be visualized!\n", domain->name);
  else
    {
      /* Le domaine a la bonne dimension */
      if( domain->Pdomain && domain->Pdomain->NbRays )
      {
        if (value_zero_p
  	    (domain->Pdomain->Ray[0][domain->Pdomain->Dimension + 1]))
  	  /* Le domaine  afficher n'est pas born dans ce contexte */
	  printf
	  ("Domain %s is not bounded!\n", domain->name);
        else
	{
	  /* Le domaine est born */

	  /* Modifier la structure du domaine en consquence */
	  domain->visual = (VisualDomain *) malloc (sizeof (VisualDomain));
	  domain->visual->intpoints = TRUE;
	  domain->visual->Lpoints = NULL;
	  domain->visual->NPdomain = Domain_Copy (domain->Pdomain);
	  /* M--j de la liste graphique */
	  gtk_clist_set_foreground (GTK_CLIST (clist), row, domain->color);
	  gtk_clist_set_text (GTK_CLIST (clist), row, 1, "Visu");
	}
      }
      else
        printf ("Empty domain %s!\n", domain->name);

    }
	/* Il y a des domaines  afficher */
	compute_A_domains ();

	calc_rep_extr ();
  return;
}
