/*
 * This file is part of the "nlr" software.
 *
 * Copyright (C) 2009-2021 Alain Ketterlin, Philippe Clauss and
 * Université de Strasbourg
 *
 * Licensing information can be found in the LICENSE file that comes
 * with the source distribution.
 *
 * Contact:
 * - Alain Ketterlin (alain@unistra.fr)
 * - Philippe Clauss (clauss@icps.u-strasbg.fr)
 */

#ifndef _TERMES_H_INCLUDED
#define _TERMES_H_INCLUDED


#include <stdio.h>
#include <unistd.h>
#include <stdbool.h>


typedef long long int value_t;

typedef struct s_array {
    size_t dim;
    size_t len;
    union u_element {
        value_t val;
        struct s_array * arr;
    } * data;
} array_t;

/* A function is:
   1) either F_SYM: a single, raw symbol
   2) or a combination of F_BIN and F_CST/F_ARR, i.e.,
      a) a single F_CST ("constant" model)
      b) a comb of F_BIN/F_CST+F_ARR ("affine" model)
      c) a complete tree of F_BIN/F_CST+F_ARR ("poly" model)
*/
enum e_functype { F_CST, F_BIN, F_SYM, F_ARR };
typedef struct s_func  {
    enum e_functype type;
    union u_funcbody {
        value_t cst;
        struct s_funcbin { /* may be expanded to accomodate higher degrees */
            struct s_func * exp0;
            struct s_func * exp1;
        } bin;
        const char * sym;
        array_t * arr;
    } u;
} func_t;

typedef struct s_dumpfmt {
    /* functions */
    int hexa; /* 0=decimal, 1=valeurs, constantes, 2=tout */
    bool color;
    const char * indice;
    /* terms */
    int delim;
    /* common */
    unsigned indent;
    unsigned hmax; /* per line, in tuples */
    bool aelide; /* whether to elide array values */
    unsigned amax; /* array elements to print if aelide == true*/
    bool asize; /* whether to print the size of the array */
} dumpfmt_t;

#define DUMPFMT_DEFAULTS        \
    {                           \
        INITFIELD(hexa,1),      \
        INITFIELD(color,false), \
        INITFIELD(indice,"i"),  \
        INITFIELD(delim,0),     \
        INITFIELD(indent,2),    \
        INITFIELD(hmax,5),      \
        INITFIELD(aelide,true), \
        INITFIELD(amax,3),      \
        INITFIELD(asize,true),  \
    }

typedef enum { M_POLY, M_LINEAR, M_CONSTANT } model_t;


typedef func_t * (*func_unif3_t)(const func_t * f1, value_t p1,
                                 const func_t * f2, value_t p2,
                                 const func_t * f3, value_t p3);


typedef func_t * (*func_inif3_t)(const func_t * f1,
                                 const func_t * f2,
                                 const func_t * f3, bool irreg);


typedef struct s_tuple {
    unsigned len;
    func_t * funcs [];
} tuple_t;

enum e_termtype { T_TUPLE, T_LOOP };
typedef struct s_term {
    enum e_termtype type;
    union u_termbody {
        tuple_t * tuple;
        struct s_termloop {
            func_t * upper;
            struct s_term * sub;
        } loop;
    } u;
    struct s_term * sib;
} term_t;



func_t *
func_constant(value_t cst);

func_t *
func_binary(func_t * l, func_t * r);

func_t *
func_symbol(const char * s);

void
func_dump(const func_t * f, FILE * fd, unsigned depth,
          const dumpfmt_t * fmt, bool inval);

void
func_dump_smart(const func_t * f, FILE * fd, unsigned depth,
                const dumpfmt_t * fmt, bool inval);

void
func_free(func_t * f);

func_t *
func_unif3_poly(const func_t * f1, value_t p1,
                const func_t * f2, value_t p2,
                const func_t * f3, value_t p3);

func_t *
func_unif3_affine(const func_t * f1, value_t p1,
                  const func_t * f2, value_t p2,
                  const func_t * f3, value_t p3);

func_t *
func_unif3_constant(const func_t * f1, value_t p1,
                    const func_t * f2, value_t p2,
                    const func_t * f3, value_t p3);

tuple_t *
tuple_new(unsigned l);

void
tuple_dump(const tuple_t * t, FILE * fd, unsigned depth,
           void (*fdump)(const func_t * f, FILE * fd, unsigned depth,
                         const dumpfmt_t * fmt, bool inval),
           const dumpfmt_t * fmt);

void
tuple_dump_raw(const tuple_t * t, FILE * fd);

tuple_t *
tuple_parse(const char * s);

void
tuple_free(tuple_t * t);

tuple_t *
tuple_unif3(const tuple_t * t1, value_t p1,
            const tuple_t * t2, value_t p2,
            const tuple_t * t3, value_t p3,
            func_unif3_t unif);


term_t *
term_tuple(tuple_t * tuple, term_t * sib);

term_t *
term_loop(func_t * upper, term_t * sub, term_t * sib);

term_t *
term_text(model_t values, model_t bounds, FILE * fd);

void
term_dump(const term_t * t, model_t values, model_t bounds,
          FILE * fd, const dumpfmt_t * fmt);

void
term_gobble(const term_t * t, model_t values, model_t bounds,
            FILE * fd, const dumpfmt_t * fmt);


void
term_free(term_t * t);

term_t *
term_unif3(const term_t * t1, value_t p1,
           const term_t * t2, value_t p2,
           const term_t * t3, value_t p3,
           func_unif3_t unif_value, func_unif3_t unif_bound);

term_t *
term_triplet(const term_t * f, const term_t * s, const term_t * t,
             func_unif3_t u_value, func_unif3_t u_bound);



void
term_minus_one(term_t * t);

value_t
term_loop_at_expect(const term_t * l, value_t p);

bool
term_follows_soft(const term_t * l, const term_t * t);

bool
term_loop_tail(const term_t * l, const term_t * t);



func_t *
func_inif3_poly(const func_t * f1,
                const func_t * f2,
                const func_t * f3,
                bool irreg);
func_t *
func_inif3_affine(const func_t * f1,
                  const func_t * f2,
                  const func_t * f3,
                  bool irreg);
func_t *
func_inif3_constant(const func_t * f1,
                    const func_t * f2,
                    const func_t * f3,
                    bool irreg);
term_t *
term_triplet_irreg(const term_t * f, const term_t * s, const term_t * t,
                   func_inif3_t i_value, bool irreg_value,
                   func_inif3_t i_bound, bool irreg_bound);
bool
term_follows_irreg_test(const term_t * g, const term_t * t);
void
term_follows_irreg_make(const term_t * g, const term_t * t);


#endif
