/*
 * This file is part of the "nlr" software.
 *
 * Copyright (C) 2009-2021 Alain Ketterlin, Philippe Clauss and
 * Université de Strasbourg
 *
 * Licensing information can be found in the LICENSE file that comes
 * with the source distribution.
 *
 * Contact:
 * - Alain Ketterlin (alain@unistra.fr)
 * - Philippe Clauss (clauss@icps.u-strasbg.fr)
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>

#include "termes.h"
#include "utils.h"


/* a dictionary to hold symbols */
struct dico {
    const char * entry;
    struct dico * next;
};

static struct dico * symbols = NULL;

static const char *
dico_find(const char * s)
{
    struct dico * prev = NULL;
    struct dico * curr = symbols;
    int cmp;
    while ( curr != NULL && (cmp=strcmp(curr->entry,s)) < 0 )
    {
        prev = curr;
        curr = curr->next;
    }
    if ( curr == NULL )
    {   /* not found, insert at the end */
        struct dico * n = xmalloc(sizeof(struct dico));
        n->entry = mystrdup(s);
        n->next = NULL;
        if ( prev != NULL )
            prev->next = n;
        else
            symbols = n;
        return n->entry;
    }
    else if ( cmp > 0 )
    {   /* not found, insert in the middle */
        struct dico * n = xmalloc(sizeof(struct dico));
        n->entry = mystrdup(s);
        n->next = curr;
        if ( prev != NULL )
            prev->next = n;
        else
            symbols = n;
        return n->entry;
    }
    else /* cmp == 0 : found */
    {
        return curr->entry;
    }
}


static func_t * func_repo = NULL;

static func_t * func_repo_get()
{
    if ( func_repo == NULL )
        return xmalloc(sizeof(func_t));
    else
    {
        func_t * r = func_repo;
        func_repo = func_repo->u.bin.exp0;
        return r;
    }
}
static void func_repo_put(func_t * f)
{
    if ( f->type == F_BIN )
    {
        func_repo_put(f->u.bin.exp0);
        func_repo_put(f->u.bin.exp1);
    }
    f->u.bin.exp0 = func_repo;
    func_repo = f;
}


#define ARRAY_CHUNK 8

array_t * array_values(value_t v1, value_t v2, value_t v3)
{
    array_t * n = xmalloc(sizeof(array_t));
    n->dim = 1;
    n->len = 3;
    n->data = xmalloc(sizeof(union u_element)*ARRAY_CHUNK);
    n->data[0].val = v1;
    n->data[1].val = v2;
    n->data[2].val = v3;
    return n;
}
array_t * array_arrays(array_t * a1,
                       array_t * a2,
                       array_t * a3)
{
    array_t * n = xmalloc(sizeof(array_t));
    n->dim = a1->dim+1;
    n->len = 3;
    n->data = xmalloc(sizeof(union u_element)*ARRAY_CHUNK);
    n->data[0].arr = a1;
    n->data[1].arr = a2;
    n->data[2].arr = a3;
    return n;
}
void array_free(array_t * a)
{
    if ( a->dim > 1 )
        for ( size_t i=0 ; i<a->len ; i++ )
            array_free(a->data[i].arr);
    xfree(a->data);
    xfree(a);
}

static void array_check_size(array_t * a)
{
    size_t chunks = (a->len + ARRAY_CHUNK - 1) / ARRAY_CHUNK;
    if ( a->len == chunks*ARRAY_CHUNK )
    {
        size_t sz = (chunks+1)*ARRAY_CHUNK*sizeof(union u_element);
        a->data = xrealloc(a->data,sz);
    }
}
void array_app_value(array_t * a, value_t v)
{
    array_check_size(a);
    a->data[(a->len)++].val = v;
}
void array_app_array(array_t * a, array_t * v)
{
    array_check_size(a);
    a->data[(a->len)++].arr = v;
}

void array_dump(const array_t * a, FILE * fd, unsigned depth,
                const dumpfmt_t * fmt, bool inval)
{
    {depth=depth;} /* indices not printed here */
    bool elide = fmt->aelide && (a->len > fmt->amax);
    if ( fmt->asize )
        fprintf(fd,"[%zu:",a->len);
    else
        fprintf(fd,"[");
    size_t len = (elide ? fmt->amax : a->len);
    if ( a->dim == 1 )
    {
        const char * sep = "";
        for ( size_t i=0 ; i<len ; i++ )
        {
            if (fmt->hexa == 0)
                fprintf(fd,"%s%lld",sep,a->data[i].val);
            else
                fprintf(fd,"%s0x%llx",sep,a->data[i].val);
            sep = ",";
        }
    }
    else
    {
        const char * sep = "";
        for ( size_t i=0 ; i<len ; i++ )
        {
            fprintf(fd,"%s",sep);
            array_dump(a->data[i].arr,fd,depth/*!*/,fmt,inval);
            sep = ",";
        }
    }
    if ( elide )
        fprintf(fd,",...");
    fprintf(fd,"]");
}

func_t *
func_constant(value_t cst)
{
    func_t * n = func_repo_get();
    n->type = F_CST;
    n->u.cst = cst;
    return n;
}

func_t *
func_binary(func_t * l, func_t * r)
{
    func_t * n = func_repo_get();
    n->type = F_BIN;
    n->u.bin.exp0 = l;
    n->u.bin.exp1 = r;
    return n;
}

func_t *
func_symbol(const char * s)
{
    func_t * n = func_repo_get();
    n->type = F_SYM;
    n->u.sym = dico_find(s);
    return n;
}

func_t *
func_array(array_t * a)
{
    func_t * n = func_repo_get();
    n->type = F_ARR;
    n->u.arr = a;
    return n;
}

void
func_free(func_t * f)
{
    /* FIXME: what to do with arrays? */
    func_repo_put(f);
}

void
func_dump(const func_t * f, FILE * fd, unsigned depth,
          const dumpfmt_t * fmt, bool inval)
{
    switch ( f->type )
    {
        case F_CST:
            fprintf(fd,"%lld",f->u.cst);
            break;
        case F_BIN:
            fprintf(fd,"(");
            func_dump(f->u.bin.exp0,fd,depth-1,fmt,inval);
            fprintf(fd,") + i%d*(",depth);
            func_dump(f->u.bin.exp1,fd,depth-1,fmt,inval);
            fprintf(fd,")");
            break;
        case F_SYM:
            fprintf(fd,"%s",f->u.sym);
            break;
        case F_ARR:
            array_dump(f->u.arr,fd,depth-1,fmt,inval);
            for ( size_t i=0 ; i<f->u.arr->dim ; i++ )
                /* FIXME: indices */
                fprintf(fd,"[%s%zu]",fmt->indice,depth-f->u.arr->dim+i);
            break;
    }
}

static struct outputdescr {
    const char * pref;
    const char * suff;
} ansisequences [] = {
    { "\033[31m", "\033[0m" }, /* red */
    { "\033[32m", "\033[0m" }, /* green */
    { "\033[34m", "\033[0m" }, /* blue */
    { "\033[35m", "\033[0m" }, /* magenta */
    { "\033[33m", "\033[0m" }, /* yellow */
    { "\033[36m", "\033[0m" }, /* cyan */
    { "\033[1m", "\033[0m" },  /* bright */
    { NULL , NULL }
};

static void
highlight_indice(unsigned i, bool color, const char * name, FILE * fd)
{
    extern int fileno(FILE *);
    //if ( isatty(fileno(fd)) )
    if ( color )
    {
        unsigned p = 0;
        while ( p < i && ansisequences[p].pref != NULL )
            ++p;
        if ( ansisequences[p].pref == NULL )
            --p;
        fprintf(fd,"%s%s%d%s",
                ansisequences[p].pref,name,i,ansisequences[p].suff);
    }
    else
    {
        fprintf(fd,"%s%d",name,i);
    }
}

static void
func_dump_smart_aux(const func_t * f, FILE * fd, unsigned depth,
                    bool * flags, unsigned len,
                    const dumpfmt_t * fmt, bool inval,
                    bool * pfirst)
{
    switch ( f->type )
    {
        case F_CST:
            if ( f->u.cst != 0 )
            {
                if ( ! *pfirst )
                    fprintf(fd," + ");
                bool hexa = ( fmt->hexa != 0 );
                if ( fmt->hexa == 1 ) /* constants only */
                {
                    hexa = inval;// to avoid hex in bounds
                    for ( unsigned i=depth+1 ; hexa && i<len ; i++ )
                        if ( flags[i] )
                            hexa = false;
                }
                fprintf(fd,(hexa?"0x%llx":"%lld"),f->u.cst);
                for ( unsigned i=depth+1 ; i<len ; i++ )
                    if ( flags[i] )
                    {
                        fprintf(fd,"*");
                        highlight_indice(i,fmt->color,fmt->indice,fd);
                    }
                *pfirst = false;
            }
            break;
        case F_BIN:
            flags[depth] = false;
            func_dump_smart_aux(f->u.bin.exp0,fd,depth-1,flags,len,
                                fmt,inval,pfirst);
            flags[depth] = true;
            func_dump_smart_aux(f->u.bin.exp1,fd,depth-1,flags,len,
                                fmt,inval,pfirst);
            break;
        case F_SYM:
            fatal("dead code");
            break;
        case F_ARR:
            if ( ! *pfirst )
                fprintf(fd," + ");
            array_dump(f->u.arr,fd,depth-1,fmt,inval);
            for ( size_t i=0 ; i<f->u.arr->dim ; i++ )
            {
                fprintf(fd,"[");
                highlight_indice(depth-f->u.arr->dim+i+1,
                                 fmt->color,fmt->indice,fd);
                fprintf(fd,"]");
            }
            for ( unsigned i=depth+1 ; i<len ; i++ )
                if ( flags[i] )
                {
                    fprintf(fd,"*");
                    highlight_indice(i,fmt->color,fmt->indice,fd);
                }
            *pfirst = false;
            break;
    }
}

void
func_dump_smart(const func_t * f, FILE * fd, unsigned depth,
                const dumpfmt_t * fmt, bool inval)
{
    if ( f->type == F_SYM )
    {
        fprintf(fd,(fmt->color ? "\033[1m%s\033[0m" : "%s"),f->u.sym);
    }
    else
    {
        bool flags[depth+1];
        bool first = true;
        func_dump_smart_aux(f,fd,depth,flags,depth+1,fmt,inval,&first);
        if ( first )
            fprintf(fd,"0");
    }
}

static func_t *
func_unif3_interpole(value_t x1, value_t y1,
                     value_t x2, value_t y2,
                     value_t x3, value_t y3)
{
     /* avoid all these / ( __divdi3) and % (__moddi3) because
        gprof tells me that these are where most of the work is done */
    if ( x1 == 0 && x2 == 1 && x3 == 2)
    {
        if ( y2-y1 == y3-y2 )
            return func_binary(func_constant(y1),func_constant(y2-y1));
        else
            return NULL;
    }
    else
    {
        value_t dx = x2 - x1;
        value_t dy = y2 - y1;
        if ( dy % dx != 0 )
            return NULL;
        value_t dz = x2*y1 - x1*y2;
        if ( dz % dx != 0 )
            return NULL;
        if ( dy*x3 + dz == dx*y3 )
            return func_binary(func_constant(dz/dx),func_constant(dy/dx));
        else
            return NULL;
    }
}

func_t *
func_unif3_poly(const func_t * f1, value_t p1,
                const func_t * f2, value_t p2,
                const func_t * f3, value_t p3)
{
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
                return func_unif3_interpole(p1,f1->u.cst,
                                            p2,f2->u.cst,
                                            p3,f3->u.cst);
                break;
            }
            case F_BIN:
            {
                func_t * g0;
                func_t * g1;
                if ( (g0=func_unif3_poly(f1->u.bin.exp0,p1,
                                         f2->u.bin.exp0,p2,
                                         f3->u.bin.exp0,p3)) == NULL )
                {
                    return NULL;
                }
                else if ( (g1=func_unif3_poly(f1->u.bin.exp1,p1,
                                              f2->u.bin.exp1,p2,
                                              f3->u.bin.exp1,p3)) == NULL )
                {
                    func_free(g0);
                    return NULL;
                }
                else
                {
                    return func_binary(g0,g1);
                }
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return NULL;
                break;
            }
            default:
                fatal("dead code");
        }
    }
}

func_t *
func_unif3_affine(const func_t * f1, value_t p1,
                  const func_t * f2, value_t p2,
                  const func_t * f3, value_t p3)
{
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
                return func_unif3_interpole(p1,f1->u.cst,
                                            p2,f2->u.cst,
                                            p3,f3->u.cst);
                break;
            }
            case F_BIN:
            {
                func_t * g0;
                if ( f1->u.bin.exp1->u.cst != f2->u.bin.exp1->u.cst
                     || f1->u.bin.exp1->u.cst != f3->u.bin.exp1->u.cst )
                {
                    return NULL;
                }
                else if ( (g0=func_unif3_affine(f1->u.bin.exp0,p1,
                                                f2->u.bin.exp0,p2,
                                                f3->u.bin.exp0,p3)) == NULL )
                {
                    return NULL;
                }
                else
                {
                    return func_binary(g0,func_constant(f1->u.bin.exp1->u.cst));
                }
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return NULL;
                break;
            }
            default:
                fatal("dead code");
        }
    }
}

func_t *
func_unif3_constant(const func_t * f1, value_t p1,
                    const func_t * f2, value_t p2,
                    const func_t * f3, value_t p3)
{
    {p1=p1;p2=p2;p3=p3;}
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
                if ( f1->u.cst == f2->u.cst && f1->u.cst == f3->u.cst )
                    return func_constant(f1->u.cst);
                else
                    return NULL;
                break;
            }
            case F_BIN:
            {
                return NULL;
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return NULL;
                break;
            }
            default:
                fatal("dead code");
        }
    }
}



static tuple_t * tuple_repos [] = {
    /* 0*/ NULL, NULL, NULL, NULL,
    /* 4*/ NULL, NULL, NULL, NULL,
    /* 8*/ NULL, NULL, NULL, NULL,
    /*12*/ NULL, NULL, NULL, NULL
};

static tuple_t * tuple_repo_get(unsigned l)
{
    if ( l > 16 || tuple_repos[l-1] == NULL )
    {
        tuple_t * n = xmalloc(sizeof(tuple_t)+l*sizeof(func_t *));
        n->len = l;
        for ( unsigned i=0 ; i<n->len ; i++ )
            n->funcs[i] = NULL;
        return n;
    }
    else
    {
        tuple_t * r = tuple_repos[l-1];
        tuple_repos[l-1] = (tuple_t *)tuple_repos[l-1]->funcs[0];
        return r;
    }
}
static void tuple_repo_put(tuple_t * t)
{
    for ( unsigned i=0 ; i<t->len ; i++ )
        if ( t->funcs[i] != NULL )
            func_free(t->funcs[i]);
    if ( t->len > 16 )
        xfree(t);
    else
    {
        t->funcs[0] = (func_t *)tuple_repos[t->len-1];
        tuple_repos[t->len-1] = t;
    }
}



tuple_t *
tuple_new(unsigned l)
{
    return tuple_repo_get(l);
}

void
tuple_free(tuple_t * t)
{
    tuple_repo_put(t);
}

void
tuple_dump(const tuple_t * t, FILE * fd, unsigned depth,
           void (*fdump)(const func_t * f, FILE * fd, unsigned depth,
                         const dumpfmt_t * fmt, bool inval),
           const dumpfmt_t * fmt)
{
    if ( t->len <= fmt->hmax )
    {
        fprintf(fd,"%*s",depth*(fmt->indent),"");
        fprintf(fd,"val ");
        for ( unsigned f=0 ; f<t->len ; f++ )
        {
            if ( f > 0 )
                fprintf(fd," , ");
            fdump(t->funcs[f],fd,depth-1,fmt,true);
        }
        fprintf(fd,"\n");
    }
    else
    {
        for ( unsigned f=0 ; f<t->len ; f++ )
        {
            fprintf(fd,"%*s",depth*(fmt->indent),"");
            fprintf(fd,( f==0 ? "val " : "  , " ));
            fdump(t->funcs[f],fd,depth-1,fmt,true);
            fprintf(fd,"\n");
        }
    }
}

void
tuple_dump_raw(const tuple_t * t, FILE * fd)
{
    const char * prefix = "";
    for ( unsigned f=0 ; f<t->len ; f++ )
    {
        switch ( t->funcs[f]->type    )
        {
            case F_CST:
                fprintf(fd,"%s%lld",prefix,t->funcs[f]->u.cst);
                break;
            case F_SYM:
                fprintf(fd,"%s%s",prefix,t->funcs[f]->u.sym);
                break;
            case F_BIN:
            case F_ARR:
                /* FIXME: this function used for prediction only... */
                /* where t is either observed input or predicted tuple */
                fatal("tuple_dump_raw");
                break;
        }
        prefix = " ";
    }
}


static bool
tuple_parse_aux_number(const char * start, const char ** endptr,
                       value_t * n)
{
    *n = strtoll(start,(char **)endptr/*!*/,0);
    bool err = ( ( n == 0 && start == *endptr ) // no number
                 || ( **endptr != '\0' && ! isspace(**endptr) ) // bad suffix
                 || ( errno != 0 ) ); // under/over-flow 
    return (! err);
}

static bool
tuple_parse_aux_symbol(const char * start, const char ** endptr,
                       char * buffer, unsigned len)
{
    static const char * acc =
        "abcdefghijklmnopqrstuvwxyz"
        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        "0123456789"
        ".-_:/@=";
    const char * p = start;
    char * r = buffer;
    if ( ! isalnum(*p) )
        return false;
    *r++ = *p++;
    while ( ! isspace(*p) && --len > 1 )
    {
        if ( strchr(acc,*p) != NULL )
            *r++ = *p++;
        else
            return false;
    }
    if ( isspace(*p) )
    {
        *endptr = p;
        *r = '\0';
        return true;
    }
    else
        return false;
}

static tuple_t *
tuple_parse_aux(unsigned pos, const char * s)
{
    const char * start = s;

    while ( isspace(*start) )
        ++start;
    if ( *start == '\0' )
        return ( pos > 0 ? tuple_new(pos) : NULL );
    errno = 0;
    const char * endptr = NULL;
    value_t n;
    char buffer[256];
    func_t * f;
    if ( tuple_parse_aux_number(start,&endptr,&n) )
        f = func_constant(n);
    else if ( tuple_parse_aux_symbol(start,&endptr,buffer,256) )
        f = func_symbol(buffer);
    else
        return NULL;
    tuple_t * t = tuple_parse_aux(pos+1,endptr);
    if ( t == NULL )
        return NULL;
    t->funcs[pos] = f;
    return t;
}


tuple_t *
tuple_parse(const char * s)
{
    return tuple_parse_aux(0,s);
}

tuple_t *
tuple_unif3(const tuple_t * t1, value_t p1,
            const tuple_t * t2, value_t p2,
            const tuple_t * t3, value_t p3,
            func_unif3_t unif)
{
    if ( t1->len != t2->len || t1->len != t3->len )
        return NULL;
    else
    {
        func_t * fu [ t1->len ];
        for ( unsigned i=0 ; i<t1->len ; i++ )
        {
            if ( (fu[i]=unif(t1->funcs[i],p1,
                             t2->funcs[i],p2,
                             t3->funcs[i],p3)) == NULL )
            {
                for ( unsigned j=0 ; j<i ; j++ )
                    func_free(fu[j]);
                return NULL;
            }
        }
        tuple_t * u = tuple_new(t1->len);
        for ( unsigned i=0 ; i<t1->len ; i++ )
            u->funcs[i] = fu[i];
        return u;
    }
}


static term_t * term_repo = NULL;

static term_t * term_repo_get()
{
    if ( term_repo == NULL )
        return xmalloc(sizeof(term_t));
    else
    {
        term_t * r = term_repo;
        term_repo = term_repo->sib;
        return r;
    }
}
static void term_repo_put(term_t * t)
{
    if ( t->sib != NULL )
        term_repo_put(t->sib);
    switch ( t->type )
    {
        case T_TUPLE:
            tuple_free(t->u.tuple);
            break;
        case T_LOOP:
            func_free(t->u.loop.upper);
            term_repo_put(t->u.loop.sub);
            break;
    }
    t->sib = term_repo;
    term_repo = t;
}

term_t *
term_tuple(tuple_t * tuple, term_t * sib)
{
    term_t * n = term_repo_get();
    n->type = T_TUPLE;
    n->u.tuple = tuple;
    n->sib = sib;
    return n;
}

term_t *
term_loop(func_t * upper, term_t * sub, term_t * sib)
{
    term_t * n = term_repo_get();
    n->type = T_LOOP;
    n->u.loop.upper = upper;
    n->u.loop.sub = sub;
    n->sib = sib;
    return n;
}

void
term_free(term_t * t)
{
    term_repo_put(t);
}

term_t *
term_text(model_t values, model_t bounds, FILE * fd)
{
    {values=values;bounds=bounds;}
    char buffer[4096];
    while ( 1 )
    {
        char * r = fgets(buffer,sizeof(buffer),fd);
        if ( r != NULL )
        {
            tuple_t * t = tuple_parse(buffer);
            if ( t != NULL )
                return term_tuple(t,NULL);
            else
                fprintf(stderr,"Line '%s' ignored\n",buffer);
        }
        else if ( feof(fd) )
            return NULL;
        else if ( ferror(fd) && errno == EINTR )
            /* silently restart */;
        else
            fatal("I/O error: %s",strerror(errno));
    }
    return NULL;
}


static void
term_desc(const term_t * t, char * str, unsigned len, unsigned * n)
{
    switch ( t->type )
    {
        case T_TUPLE:
            if ( *n < len )
                str[*n] = 'V';
            ++ (*n);
            break;
        case T_LOOP:
            if ( *n < len )
                str[*n] = 'F';
            ++ (*n);
            term_desc(t->u.loop.sub,str,len,n);
            if ( *n < len )
                str[*n] = 'D';
            ++ (*n);
            break;
    }
    if ( t->sib != NULL )
        term_desc(t->sib,str,len,n);
}

/* FIXME */
static void
term_dump_aux(const term_t * t, FILE * fd, unsigned depth,
              const dumpfmt_t * fmt)
{
    void (*fdump)(const func_t * f, FILE * fd, unsigned depth,
                  const dumpfmt_t * fmt, bool inval)
        = func_dump_smart; /* func_dump */
    switch ( t->type )
    {
        case T_TUPLE:
            tuple_dump(t->u.tuple,fd,depth,fdump,fmt);
            break;
        case T_LOOP:
            fprintf(fd,"%*s",depth*(fmt->indent),"");
            fprintf(fd,"for ");
            highlight_indice(depth,fmt->color,fmt->indice,fd);
            fprintf(fd," = 0 to ");
            fdump(t->u.loop.upper,fd,depth-1,fmt,false);
            if ( 0 ) /* FIXME */
            {
                char buffer[1024];
                unsigned n = 0;
                term_desc(t,buffer,1024,&n); /* FIXME */
                buffer[n] = 0;
                fprintf(fd," %*s",n,buffer);
            }
            fprintf(fd,"\n");
            term_dump_aux(t->u.loop.sub,fd,depth+1,fmt);
            if ( fmt->delim )
                fprintf(fd,"%*sdone\n",depth*(fmt->indent),"");
            break;
    }
    if ( t->sib != NULL )
        term_dump_aux(t->sib,fd,depth,fmt);
}

void
term_dump(const term_t * t, model_t values, model_t bounds,
          FILE * fd, const dumpfmt_t * fmt)
{
    {values=values;bounds=bounds;}
    term_dump_aux(t,fd,0,fmt);
}

void
term_gobble(const term_t * t, model_t values, model_t bounds,
            FILE * fd, const dumpfmt_t * fmt)
{
    {t=t;values=values;bounds=bounds;fd=fd;fmt=fmt;}
}

term_t *
term_unif3(const term_t * t1, value_t p1,
           const term_t * t2, value_t p2,
           const term_t * t3, value_t p3,
           func_unif3_t u_value, func_unif3_t u_bound)
{
    term_t * usib = NULL;

    if ( t1->type != t2->type || t1->type != t3->type )
        return NULL;

    if ( ( t1->sib == NULL && t2->sib == NULL && t3->sib == NULL )
         ||
         ( t1->sib != NULL && t2->sib != NULL && t3->sib != NULL
           &&
           (usib=term_unif3(t1->sib,p1,t2->sib,p2,t3->sib,p3,
                            u_value,u_bound)) != NULL ) )
        ; /* OK */
    else
        return NULL;

    switch ( t1->type /* which is the same as t2->type */ )
    {
        case T_TUPLE:
        {
            tuple_t * utuple = tuple_unif3(t1->u.tuple,p1,
                                           t2->u.tuple,p2,
                                           t3->u.tuple,p3,
                                           u_value);
            if ( utuple == NULL )
            {
                if ( usib != NULL ) term_free(usib);
                return NULL;
            }
            else
                return term_tuple(utuple,usib);
            break;
        }
        case T_LOOP:
        {
            func_t * uupper = NULL;
            term_t * usub = NULL;
            if ( (usub=term_unif3(t1->u.loop.sub,p1,
                                  t2->u.loop.sub,p2,
                                  t3->u.loop.sub,p3,
                                  u_value,u_bound)) != NULL
                 &&
                 (uupper=u_bound(t1->u.loop.upper,p1,
                                 t2->u.loop.upper,p2,
                                 t3->u.loop.upper,p3)) != NULL )
            {
                return term_loop(uupper,usub,usib);
            }
            else
            {
                if ( uupper != NULL ) func_free(uupper);
                if ( usub != NULL ) term_free(usub);
                if ( usib != NULL ) term_free(usib);
                return NULL;
            }
            break;
        }
        default:
            fatal("dead code");
    }
}

term_t *
term_triplet(const term_t * f, const term_t * s, const term_t * t,
             func_unif3_t u_value, func_unif3_t u_bound)
{
    term_t * u = term_unif3(f,0,s,1,t,2,u_value,u_bound);
    return ( u == NULL ? NULL : term_loop(func_constant(2),u,NULL) );
}


// Experimental stuff

typedef struct s_cntstack {
    const struct s_cntstack * next;
    value_t val;
    bool fixed;
} cntstack_t;

/* currently unused
static void cntstack_dump(const cntstack_t * s, FILE * fd)
{
    if ( s == NULL )
        fprintf(fd,"[]");
    else
    {
        if ( s->next != NULL )
            cntstack_dump(s->next,fd);
        fprintf(fd,"%s",(s->next==NULL?"":","));
        if ( s->fixed )
            fprintf(fd,"=%lld",s->val);
        else
            fprintf(fd,"*");
    }
}
*/

static value_t
func_bound_cntstack_eval(const func_t * f, const cntstack_t * s)
{
    // assert( f->type != F_SYM )
    // assert( s->fixed == true )
    if ( f->type == F_CST )
        return f->u.cst;
    else
        return func_bound_cntstack_eval(f->u.bin.exp0,s->next)
            + s->val * func_bound_cntstack_eval(f->u.bin.exp1,s->next);
}

static void // FIXME: doesn't work with the "affine" model
func_fma(func_t * f, value_t s, const func_t * g)
{
    switch ( f->type )
    {
        case F_CST:
            f->u.cst += s * g->u.cst;
            break;
        case F_BIN:
            func_fma(f->u.bin.exp0,s,g->u.bin.exp0);
            func_fma(f->u.bin.exp1,s,g->u.bin.exp1);
            break;
        case F_SYM:
        case F_ARR:
            fatal("func_fma");
            break;
    }
}

static func_t *
func_bound_cntstack_partial(const func_t * f, const cntstack_t * s)
{
    switch ( f->type )
    {
        case F_SYM:
            return func_symbol(f->u.sym);
            break;
        case F_CST:
            return func_constant(f->u.cst);
            break;
        case F_BIN:
        {
            func_t * f0 = func_bound_cntstack_partial(f->u.bin.exp0,s->next);
            func_t * f1 = func_bound_cntstack_partial(f->u.bin.exp1,s->next);
            if ( s->fixed )
            {
                func_fma(f0,s->val,f1);
                func_free(f1);
                return f0;
            }
            else
            {
                return func_binary(f0,f1);
            }
            break;
        }
        case F_ARR:
            fatal("array in follows_soft");
            break;
        default:
            fatal("dead code");
    }
}
static bool
func_equal(const func_t * f, const func_t * g)
{
    if ( f->type != g->type )
        return false;
    switch ( f->type )
    {
        case F_SYM:
            return f->u.sym == g->u.sym;
            break;
        case F_CST:
            return f->u.cst == g->u.cst;
            break;
        case F_BIN:
            return func_equal(f->u.bin.exp0,g->u.bin.exp0)
                && func_equal(f->u.bin.exp1,g->u.bin.exp1);
            break;
        case F_ARR:
            fatal("array in follows_soft");
            break;
        default:
            fatal("dead code");
    }
}
static bool
func_is_constant(const func_t * f, value_t * v)
{
    switch ( f->type )
    {
        case F_SYM:
            *v = 0; // shut up compiler
            return true;
            break;
        case F_CST:
            *v = f->u.cst;
            return true;
            break;
        case F_BIN:
        {
            value_t v0;
            value_t v1;
            bool c0 = func_is_constant(f->u.bin.exp0,&v0);
            bool c1 = func_is_constant(f->u.bin.exp1,&v1);
            if ( c0 && c1 && v1 == 0 )
            {
                *v = v0;
                return true;
            }
            else
            {
                *v = 0; // shut up compiler
                return false;
            }
            break;
        }
        case F_ARR:
            fatal("array in follows_soft");
            break;
        default:
            fatal("dead-code");
    }
}

static bool
func_follows_soft(const func_t * f, const cntstack_t * stack, const func_t * g)
{
    // FIXME: this is very expensive and it is used all the time...
    func_t * instance = func_bound_cntstack_partial(g,stack);
    bool r = func_equal(f,instance);
    func_free(instance);
    return r;
}


static value_t
term_loop_expect_aux(const term_t * body, const cntstack_t * stack)
{
    if ( body == NULL )
    {
        return 0;
    }
    else if ( body->type == T_TUPLE )
    {
        return 1 + term_loop_expect_aux(body->sib,stack);
    }
    else // body->type == T_LOOP
    {
        value_t r;
        int trip = func_bound_cntstack_eval(body->u.loop.upper,stack);
        if ( trip < 0 )
        {
            r = 0;
        }
        else if ( trip == 0 )
        {
            cntstack_t pop = { .next=stack, .val=0, .fixed=true };
            r = term_loop_expect_aux(body->u.loop.sub,&pop);
        }
        else if ( trip == 1 )
        {
            const cntstack_t pop0 = { .next=stack, .val=0, .fixed=true };
            const cntstack_t pop1 = { .next=stack, .val=1, .fixed=true };
            r = term_loop_expect_aux(body->u.loop.sub,&pop0) +
                term_loop_expect_aux(body->u.loop.sub,&pop1);
        }
        else // trip > 1
        {
            r = 1;
        }
        return r + term_loop_expect_aux(body->sib,stack);
    }
}

value_t
term_loop_at_expect(const term_t * l, value_t p)
{
    const cntstack_t s = { .next=NULL, .val=p, .fixed=true };
    return term_loop_expect_aux(l->u.loop.sub,&s);
}


//
// matching along a frontier
//

static bool
tuple_follows_soft(const tuple_t * t,
                   const cntstack_t * stack,
                   const tuple_t * g)
{
    if ( t->len != g->len )
        return false;
    for ( unsigned i=0 ; i<t->len ; i++ )
        if ( ! func_follows_soft(t->funcs[i],stack,g->funcs[i]) )
            return false;
    return true;
}

static bool
term_follows_soft_aux(const term_t ** ts,
                      const cntstack_t * stack,
                      const term_t * g)
{
    if ( g == NULL )
    {
        return true;
    }
    else if ( g->type == T_TUPLE )
    {
        if ( *ts == NULL || (*ts)->type != T_TUPLE )
            return false;
        if ( ! tuple_follows_soft((*ts)->u.tuple,stack,g->u.tuple) )
            return false;
        *ts = (*ts)->sib;
        return term_follows_soft_aux(ts,stack,g->sib);
    }
    else // g->type == LOOP
    {
        func_t * part = func_bound_cntstack_partial(g->u.loop.upper,stack);
        value_t trip;
        bool fixtrip = func_is_constant(part,&trip);
        func_free(part);
        if ( fixtrip && trip < 0 )
        {
            /* ok */
        }
        else if ( fixtrip && trip == 0 )
        {
            cntstack_t pop = { .next=stack, .val=0, .fixed=true };
            if ( ! term_follows_soft_aux(ts,&pop,g->u.loop.sub) )
                return false;
        }
        else if ( fixtrip && trip == 1 )
        {
            const cntstack_t pop0 = { .next=stack, .val=0, .fixed=true };
            if ( ! term_follows_soft_aux(ts,&pop0,g->u.loop.sub) )
                return false;
            const cntstack_t pop1 = { .next=stack, .val=1, .fixed=true };
            if ( ! term_follows_soft_aux(ts,&pop1,g->u.loop.sub) )
                return false;
        }
        else // !fix trip or trip > 1
        {
            if ( *ts == NULL || (*ts)->type != T_LOOP )
                return false;
            /* FIXME: use "part" directly here? */
            bool ru = func_follows_soft((*ts)->u.loop.upper,
                                        stack,
                                        g->u.loop.upper);
            if ( ! ru )
                return false;
            const cntstack_t nstack
                = { .next=stack, .val=trip, .fixed=false/*fixtrip*/ };
            const term_t * ch = (*ts)->u.loop.sub;
            bool rs = term_follows_soft_aux(&ch,&nstack,g->u.loop.sub);
            if ( ! rs )
                return false;
            if ( ch != NULL )
                return false;
            // FIXME: check here whether we have matched anything...
            *ts = (*ts)->sib;
        }
        return term_follows_soft_aux(ts,stack,g->sib);
    }
}
bool term_follows_soft(const term_t * l, const term_t * t)
{
    bool r = false;
    if ( l->type == T_LOOP )
    {
        value_t n = l->u.loop.upper->u.cst; /* must be constant here */
        cntstack_t s = { .next=NULL, .val=n+1, .fixed=true };
        const term_t * ts = t;
        r = term_follows_soft_aux(&ts,&s,l->u.loop.sub);
        r = r && ( ts == NULL );
    }
    return r;
}

static void func_minus_one(func_t * f)
{
    switch ( f->type )
    {
        case F_BIN:
            if ( f->u.bin.exp0->type == F_CST
                 && f->u.bin.exp1->type == F_CST )
            {
                f->u.bin.exp0->u.cst -= f->u.bin.exp1->u.cst;
            }
            else
            {
                func_minus_one(f->u.bin.exp0);
                func_minus_one(f->u.bin.exp1);
            }
            break;
        case F_SYM:
            break;
        case F_CST:
        case F_ARR:
            fatal("func_minus_one");
            break;
    }
}
void term_minus_one(term_t * t)
{
    if ( t != NULL )
    {
        switch ( t->type )
        {
            case T_TUPLE:
                for ( unsigned i=0 ; i<t->u.tuple->len ; i++ )
                    func_minus_one(t->u.tuple->funcs[i]);
                break;
            case T_LOOP:
                func_minus_one(t->u.loop.upper);
                term_minus_one(t->u.loop.sub);
                break;
        }
        term_minus_one(t->sib);
    }
}

bool term_loop_tail(const term_t * l, const term_t * t)
{
    cntstack_t s = { .next=NULL, .val=-1, .fixed=true };
    const term_t * ts = t;
    bool r = term_follows_soft_aux(&ts,&s,l->u.loop.sub);
    return r && ( ts == NULL );
}

//
// collecting arrays (inif = irregular unif)
//

#define INIF_DO_INTERPOLATION 1

func_t *
func_inif3_poly(const func_t * f1,
                const func_t * f2,
                const func_t * f3,
                bool irreg)
{
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
#if INIF_DO_INTERPOLATION
                func_t * g = func_unif3_interpole(0,f1->u.cst,
                                                  1,f2->u.cst,
                                                  2,f3->u.cst);
                if ( g != NULL )
                    return g;
#else
                if ( f1->u.cst == f2->u.cst && f1->u.cst == f3->u.cst )
                    return func_binary(func_constant(f1->u.cst),
                                       func_constant(0));
#endif
                else if ( irreg )
                    return func_array(array_values(f1->u.cst,
                                                   f2->u.cst,
                                                   f3->u.cst));
                else
                    return NULL;
                break;
            }
            case F_BIN:
            {
                func_t * g0;
                func_t * g1;
                if ( (g0=func_inif3_poly(f1->u.bin.exp0,
                                         f2->u.bin.exp0,
                                         f3->u.bin.exp0,irreg)) == NULL )
                {
                    return NULL;
                }
                else if ( (g1=func_inif3_poly(f1->u.bin.exp1,
                                              f2->u.bin.exp1,
                                              f3->u.bin.exp1,irreg)) == NULL )
                {
                    func_free(g0);
                    return NULL;
                }
                else
                {
                    return func_binary(g0,g1);
                }
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return func_array(array_arrays(f1->u.arr,
                                               f2->u.arr,
                                               f3->u.arr));
                break;
            }
            default:
                fatal("dead code");
        }
    }
}

func_t *
func_inif3_affine(const func_t * f1,
                  const func_t * f2,
                  const func_t * f3,
                  bool irreg)
{
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
#if INIF_DO_INTERPOLATION
                func_t * g = func_unif3_interpole(0,f1->u.cst,
                                                  1,f2->u.cst,
                                                  2,f3->u.cst);
                if ( g != NULL )
                    return g;
#else
                if ( f1->u.cst == f2->u.cst && f1->u.cst == f3->u.cst )
                    return func_binary(func_constant(f1->u.cst),
                                       func_constant(0));
#endif
                else if ( irreg )
                    return func_array(array_values(f1->u.cst,
                                                   f2->u.cst,
                                                   f3->u.cst));
                else
                    return NULL;
                break;
            }
            case F_BIN:
            {
                func_t * g0;
                if ( f1->u.bin.exp1->u.cst != f2->u.bin.exp1->u.cst
                     || f1->u.bin.exp1->u.cst != f3->u.bin.exp1->u.cst )
                {
                    return NULL;
                }
                else if ( (g0=func_inif3_affine(f1->u.bin.exp0,
                                                f2->u.bin.exp0,
                                                f3->u.bin.exp0,irreg)) == NULL )
                {
                    return NULL;
                }
                else
                {
                    return func_binary(g0,func_constant(f1->u.bin.exp1->u.cst));
                }
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return func_array(array_arrays(f1->u.arr,f2->u.arr,f3->u.arr));
                break;
            }
            default:
                fatal("dead code");
        }
    }
}

func_t *
func_inif3_constant(const func_t * f1,
                    const func_t * f2,
                    const func_t * f3,
                    bool irreg)
{
    {irreg=irreg;}
    if ( f1->type != f2->type || f1->type != f3->type )
        return NULL;
    else
    {
        switch ( f1->type /* which is the same as f2->type */ )
        {
            case F_CST:
            {
                if ( f1->u.cst == f2->u.cst && f1->u.cst == f3->u.cst )
                    return func_constant(f1->u.cst);
                else
                    return NULL;
                break;
            }
            case F_BIN:
            {
                return NULL;
                break;
            }
            case F_SYM:
            {
                if ( f1->u.sym == f2->u.sym && f1->u.sym == f3->u.sym )
                    return func_symbol(f1->u.sym);
                else
                    return NULL;
                break;
            }
            case F_ARR:
            {
                return NULL;
                break;
            }
            default:
                fatal("dead code");
        }
    }
}


tuple_t *
tuple_inif3(const tuple_t * t1,
            const tuple_t * t2,
            const tuple_t * t3,
            func_inif3_t inif, bool irreg)
{
    if ( t1->len != t2->len || t1->len != t3->len )
        return NULL;
    else
    {
        func_t * fu [ t1->len ];
        for ( unsigned i=0 ; i<t1->len ; i++ )
        {
            if ( (fu[i]=inif(t1->funcs[i],
                             t2->funcs[i],
                             t3->funcs[i],irreg)) == NULL )
            {
                for ( unsigned j=0 ; j<i ; j++ )
                    func_free(fu[j]);
                return NULL;
            }
        }
        tuple_t * u = tuple_new(t1->len);
        for ( unsigned i=0 ; i<t1->len ; i++ )
            u->funcs[i] = fu[i];
        return u;
    }
}

term_t *
term_inif3(const term_t * t1,
           const term_t * t2,
           const term_t * t3,
           func_inif3_t i_value, bool irreg_value,
           func_inif3_t i_bound, bool irreg_bound)
{
    term_t * usib = NULL;

    if ( t1->type != t2->type || t1->type != t3->type )
        return NULL;

    if ( ( t1->sib == NULL && t2->sib == NULL && t3->sib == NULL )
         ||
         ( t1->sib != NULL && t2->sib != NULL && t3->sib != NULL
           &&
           (usib=term_inif3(t1->sib,t2->sib,t3->sib,
                            i_value,irreg_value,
                            i_bound,irreg_bound)) != NULL ) )
        ; /* OK */
    else
        return NULL;

    switch ( t1->type /* which is the same as t2->type */ )
    {
        case T_TUPLE:
        {
            tuple_t * utuple = tuple_inif3(t1->u.tuple,
                                           t2->u.tuple,
                                           t3->u.tuple,
                                           i_value,irreg_value);
            if ( utuple == NULL )
            {
                if ( usib != NULL ) term_free(usib);
                return NULL;
            }
            else
                return term_tuple(utuple,usib);
            break;
        }
        case T_LOOP:
        {
            func_t * uupper = NULL;
            term_t * usub = NULL;
            if ( (usub=term_inif3(t1->u.loop.sub,
                                  t2->u.loop.sub,
                                  t3->u.loop.sub,
                                  i_value,irreg_value,
                                  i_bound,irreg_bound)) != NULL
                 &&
                 (uupper=i_bound(t1->u.loop.upper,
                                 t2->u.loop.upper,
                                 t3->u.loop.upper,irreg_bound)) != NULL )
            {
                return term_loop(uupper,usub,usib);
            }
            else
            {
                if ( uupper != NULL ) func_free(uupper);
                if ( usub != NULL ) term_free(usub);
                if ( usib != NULL ) term_free(usib);
                return NULL;
            }
            break;
        }
        default:
            fatal("dead code");
    }
}
term_t *
term_triplet_irreg(const term_t * f, const term_t * s, const term_t * t,
                   func_inif3_t i_value, bool irreg_value,
                   func_inif3_t i_bound, bool irreg_bound)
{
    term_t * u = term_inif3(f,s,t,i_value,irreg_value,i_bound,irreg_bound);
    return ( u == NULL ? NULL : term_loop(func_constant(2),u,NULL) );
}

bool
func_follows_irreg_test(const func_t * g, value_t pos, const func_t * f)
{
    if ( g->type == F_SYM && f->type == F_SYM )
    {
        return g->u.sym == f->u.sym;
    }
    else if ( g->type == F_CST && f->type == F_CST )
    {
        return g->u.cst == f->u.cst;
    }
    else if ( g->type == F_BIN && f->type == F_CST )
    {
        return g->u.bin.exp0->u.cst + pos*g->u.bin.exp1->u.cst == f->u.cst;
    }
    else if ( g->type == F_BIN && f->type == F_BIN )
    {
        return func_follows_irreg_test(g->u.bin.exp0,pos,f->u.bin.exp0)
            && func_follows_irreg_test(g->u.bin.exp1,pos,f->u.bin.exp1);
    }
    else if ( g->type == F_ARR )
    {
        return true;
    }
    else
    {
        return false;
    }
}
void
func_follows_irreg_make(const func_t * g, const func_t * f)
{
    if ( g->type == F_BIN && f->type == F_BIN )
    {
        func_follows_irreg_make(g->u.bin.exp0,f->u.bin.exp0);
        func_follows_irreg_make(g->u.bin.exp1,f->u.bin.exp1);
    }
    else if ( g->type == F_ARR )
    {
        if ( f->type == F_CST )
            array_app_value(g->u.arr,f->u.cst);
        else if ( f->type == F_ARR )
            array_app_array(g->u.arr,f->u.arr);
    }
}

bool
tuple_follows_irreg_test(const tuple_t * g, value_t pos, const tuple_t * t)
{
    if ( g->len != t->len )
        return false;
    else
    {
        for ( unsigned i=0 ; i<t->len ; i++ )
            if ( ! func_follows_irreg_test(g->funcs[i],pos,t->funcs[i]) )
                return false;
        return true;
    }
}
void
tuple_follows_irreg_make(const tuple_t * g, const tuple_t * t)
{
    for ( unsigned i=0 ; i<t->len ; i++ )
        func_follows_irreg_make(g->funcs[i],t->funcs[i]);
}

bool
term_follows_irreg_test_aux(const term_t * g, value_t pos, const term_t * t)
{
    if ( g->type != t->type )
        return false;

    if ( ( g->sib == NULL && t->sib == NULL )
         ||
         ( g->sib != NULL && t->sib != NULL
           && term_follows_irreg_test_aux(g->sib,pos,t->sib) ) )
        ; /* OK */
    else
        return false;

    switch ( t->type /* which is the same as g->type */ )
    {
        case T_TUPLE:
            return tuple_follows_irreg_test(g->u.tuple,pos,t->u.tuple);
            break;
        case T_LOOP:
            return func_follows_irreg_test(g->u.loop.upper,pos,t->u.loop.upper)
                && term_follows_irreg_test_aux(g->u.loop.sub,pos,t->u.loop.sub);
            break;
        default:
            fatal("dead code");
    }
}
bool term_follows_irreg_test(const term_t * l, const term_t * t)
{
    return ( l->type == T_LOOP
             && term_follows_irreg_test_aux(l->u.loop.sub,
                                            l->u.loop.upper->u.cst+1,
                                            t) );
}
void
term_follows_irreg_make_aux(const term_t * g, const term_t * t)
{
    if ( g->sib != NULL /* && t->sib != NULL */ )
        term_follows_irreg_make_aux(g->sib,t->sib);

    switch ( t->type /* which is the same as g->type */ )
    {
        case T_TUPLE:
            tuple_follows_irreg_make(g->u.tuple,t->u.tuple);
            break;
        case T_LOOP:
            func_follows_irreg_make(g->u.loop.upper,t->u.loop.upper);
            term_follows_irreg_make_aux(g->u.loop.sub,t->u.loop.sub);
            break;
        default:
            fatal("dead code");
    }
}
void term_follows_irreg_make(const term_t * l, const term_t * t)
{
    term_follows_irreg_make_aux(l->u.loop.sub,t);
}
