#include <stdio.h>
#include <math.h>
#include <stdlib.h>

/* 
 * PSC2001 sample program (OpenMP version)
 */
/* prototypes */
void psc2001_get_config(int problem_no,
			int * size_p, int * n_steps_p, double * dt_p);
void psc2001_get_data(int problem_no, double * a);
void psc2001_verify(int problem_no, double * a);

/* bug fix 4/13 */
double round(double x){
    if(x < 0.0) return -floor(-x+0.5);
    else return floor(x+0.5);
}

typedef struct my_particle {
    double m;
    double x,y,z;
    double vx,vy,vz;
    double ax,ay,az;
} particle;

double DT;
int n_steps;
int n_particles;
particle *particles;

void do_step();

int main(int argc, char ** argv)
{
    int problem_no,it,i;
    double *a,*ap;
    particle *p;

    if(argc != 2){
	fprintf(stderr,"bad args...\n");
	exit(1);
    }
    problem_no = atoi(argv[1]);
    
    psc2001_get_config(problem_no,&n_particles,&n_steps,&DT);

    printf("problem_no=%d: n_particle=%d, n_steps=%d, DT=%e\n",
	   problem_no, n_particles, n_steps, DT);

    a = (double *)malloc(sizeof(double)*7*n_particles);
    particles = (particle *)malloc(sizeof(particle) * n_particles);
    if(a == NULL || particles == NULL){
	fprintf(stderr,"no memory...\n");
	exit(1);
    }

    psc2001_get_data(problem_no,a);

    ap = a;
    for(i = 0; i < n_particles; i++) {
	p = &particles[i];
	p->m = *ap++;
	p->x = *ap++;
	p->y = *ap++;
	p->z = *ap++;
	p->vx = *ap++;
	p->vy = *ap++;
	p->vz = *ap++;
    }

#pragma omp parallel private(it)
{
    for(it = 0; it < n_steps; it++){
	do_step();
    }
}

    ap = a;
    for(i = 0; i < n_particles; i++) {
	p = &particles[i];
	*ap++ = p->m;
	*ap++ = p->x;
	*ap++ = p->y;
	*ap++ = p->z;
	*ap++ = p->vx;
	*ap++ = p->vy;
	*ap++ = p->vz;
    }

    psc2001_verify(problem_no,(double *)a);

    exit(0);
}

void do_step()
{
    int i,j;
    double a2 = 256.0;
    double b2 = 1024.0;
    double ax,ay,az,dx,dy,dz,X,f;
    particle *p,*q;

#pragma omp for
    for(i = 0; i < n_particles; i++) {
	p = &particles[i];
	ax = 0.0;
	ay = 0.0;
	az = 0.0;
	for(j = 0; j < n_particles; j++){
	    if(i == j) continue;
	    q = &particles[j];
	    dx = p->x - q->x;
	    dy = p->y - q->y;
	    dz = p->z - q->z;

	    dx = round(dx);
	    dy = round(dy);
	    dz = round(dz);

	    X = dx * dx + dy * dy + dz * dz;
	    if (X < b2) {
		f = q->m * (X - a2) * (X - b2);
		ax += f * dx;
		ay += f * dy;
		az += f * dz;
	    } 
	}
	p->ax = ax;
	p->ay = ay;
	p->az = az;
    }

#pragma omp for
    for(i = 0; i < n_particles; i++){
	p = &particles[i];
	p->x += p->vx * DT;
	p->y += p->vy * DT;
	p->z += p->vz * DT;
	p->vx += p->ax * DT;
	p->vy += p->ay * DT;
	p->vz += p->az * DT;
    }
}


