/*
 * Laplace equation with explict method 
 *	(originally written by Taiskue Boku (taisuke@is.tsukuba.ac.jp))
 *      this OpenMP version is written by Shigehisa Satoh (sh-sato@acm.org)
 */
#include <stdio.h>
#include <math.h>

/* square region */
#define XSIZE 2000
#define YSIZE XSIZE
#define NITER 20

#define PI 3.1415927

extern double second(void);
static void init(void);
static void solve(void);
static void verify(void);

static double u[XSIZE+2][YSIZE+2],uu[XSIZE+2][YSIZE+2];

int main() {
    double init_time, solve_time, verify_time;

    printf("laplace equation solver: %d by %d matrix, %d iterations\n",
	   XSIZE, YSIZE, NITER);fflush(stdout);
    init_time = -second();
    init();
    init_time += second();
    printf("initialized\n");fflush(stdout);

    solve_time = -second();
    solve();
    solve_time += second();

    verify_time = -second();
    verify();
    verify_time += second();

    printf("init_time:   %7.3f\n", init_time);
    printf("solve_time:  %7.3f\n", solve_time);
    printf("verify_time: %7.3f\n", verify_time);
}

static void solve(void) {
    int x,y,k;

#pragma omp parallel private(k,x,y)
{
    for(k = 0; k < NITER; k++){
	/* save old values */
#pragma omp for 
	for(x = 1; x <= XSIZE; x++)
	  for(y = 1; y <= YSIZE; y++)
	    uu[x][y] = u[x][y];
	/* update */
#pragma omp for
	for(x = 1; x <= XSIZE; x++)
	  for(y = 1; y <= YSIZE; y++)
	    u[x][y] = (uu[x-1][y] + uu[x+1][y] + uu[x][y-1] + uu[x][y+1])/4.0;
    }
}
}

static void init(void) {
    int x,y;

#pragma omp parallel private(x,y)
{
#pragma omp for
    for(x = 1; x <= XSIZE; x++)
      for(y = 1; y <= YSIZE; y++)
	u[x][y] = sin((double)(x-1)/XSIZE*PI) + cos((double)(y-1)/YSIZE*PI);

#pragma omp for
    for(x = 0; x < (XSIZE+2); x++){
	u[x][0] = 0.0;
	u[x][YSIZE+1] = 0.0;
	uu[x][0] = 0.0;
	uu[x][YSIZE+1] = 0.0;
    }

#pragma omp for
    for(y = 0; y < (YSIZE+2); y++){
	u[0][y] = 0.0;
	u[XSIZE+1][y] = 0.0;
	uu[0][y] = 0.0;
	uu[XSIZE+1][y] = 0.0;
    }
}
}

static void verify(void) {
    int x, y;
    double sum;

    /* check sum */
    sum = 0.0;
#pragma omp parallel for private(y) reduction(+:sum)
    for(x = 1; x <= XSIZE; x++)
	for(y = 1; y <= YSIZE; y++)
	  sum += (uu[x][y]-u[x][y]);

    printf("sum = %g\n",sum);
}
