/** This is Ancre-PILS,
 the non-parametric rational version of PILS.
@author B. Meister, 
@version 1.01, 02 oct 2005 
Uses polylib. */

#include "HLP.h"

/** 
   finds the facet that contains the maximum w.r.t. a linear order.
input: 
@param cons: a (m x n) matrix containing the constraints of a polyhedron
@param order: a set of vectors representing the hierarchically linear order 
@param cur_o: the number of the vector to be considered
@param permanent-NSPs the adress of a permanently allocated NSPs matrix 
   (to avoid multiple allocs)
   -> if NULL, we alloc/free it at each call
@return the rank of the constraint that is maximum wrt the order vector
   (-1 if there is no negative NSP) 
remarks:  
 * we are only interested in negative NSPs
*/
int max_scalar(Matrix * cons, Matrix * order, int cur_o, Matrix * permanent_NSPs) {
  /* the rows of "constraints" are going to be ranked according to 
      their scalar product with the current order vector. */
  /* if there are not enough (<n) negative scalar products with the 
      current vector, use the next vector. */
  unsigned i;
  unsigned nb_vars = cons->NbColumns-2;
  unsigned max_rank=0;
  Value max, com_den, factor;
  Matrix * NSPs;

  /* 0- if the polyhedron has an equality, it must be eliminated first */
  for (i=0; i< cons->NbRows; i++) 
    if (value_zero_p(cons->p[i][0])) return i;

  /* an integer numerator [0] and denominator [1] for each constraint */
  if (!permanent_NSPs) NSPs= Matrix_Alloc(2, cons->NbRows);
  else NSPs = permanent_NSPs;

  /* 1- Compute the NSPs with the current order vector */
  for (i=0; i< cons->NbRows; i++) {
    /* OLD: do not rank a constraint that was already a maximum: do as if the SPs were 0 for them */
    /* if (cons->p[i][0]!=0) { */
    Inner_Product(&(cons->p[i][1]), order->p[cur_o], nb_vars, &(NSPs->p[0][i]) );
    if (value_neg_p(NSPs->p[0][i])) {
      /* compute a f(NSP), where f() is a positive monotone decreasing rational function on Z*- */
      value_multiply(NSPs->p[0][i], NSPs->p[0][i], NSPs->p[0][i]);
      Inner_Product(&(cons->p[i][1]), &(cons->p[i][1]), nb_vars, &(NSPs->p[1][i]));
      /* OLD:  } */
    }

    /* if the current constraint's SP is nonnegative, set its NSP to zero */
    /* as it is not interesting for us */
    else {
      value_set_si(NSPs->p[0][i], 0);
      value_set_si(NSPs->p[1][i], 1);
    }
  }

  /* 2- Put all the rational NSPs to the same denominator */
  value_init(com_den);
  value_set_si(com_den, 1);
  for (i=0; i< cons->NbRows; i++) {
    Lcm3(NSPs->p[1][i], com_den, &com_den);
  }
  value_init(max);
  value_init(factor);
  for (i=0; i< cons->NbRows; i++) {
    value_division(factor, com_den, NSPs->p[1][i]);
    value_multiply(NSPs->p[0][i], NSPs->p[0][i], factor);

  /* 3- get the rank of the maximum */
    if (value_gt(NSPs->p[0][i], max)) {
      value_assign(max, NSPs->p[0][i]);
      max_rank = i;
    }
  }
  value_clear(factor);
  value_clear(com_den);
  if (!permanent_NSPs) Matrix_Free(NSPs);
  if (value_notzero_p(max)) {
    value_clear(max);
    return max_rank;
  }
  /* if no NSP is negative, no constraint is to be selected */
  else {
    value_clear(max);
    return -1;
  }
} /* max_scalar() */

/** tests if the row^th row-vector of matrix M is zero 
  can be replaced by the polylib function Vector_IsZero(Value *, unsigned length) 
  as soon as it is available */
int zero_vector(Matrix *M, unsigned row) {
  int i, last = M->NbColumns-1;
  if (value_notzero_p(M->p[row][last])) return 0;
  else {
    value_set_si(M->p[row][last], 1);
    for (i=0; value_zero_p(M->p[row][i]); i++);
    value_set_si(M->p[row][last], 0);
    return (i==last);
  }
}

/** make a fake matrix by parasiting a polyhedron's "Constraints" field */
void Parasite_Matrix(Polyhedron * P, Matrix * Parasite) {
  Parasite->NbRows=P->NbConstraints;
  Parasite->NbColumns=P->Dimension+2;
  Parasite->p = P->Constraint;
}


/** solves a Hierarchcially Linear Programming (HLP) problem.
 
@param the affine constraints, under Polylib form (a m x (n+2) Matrix)
@param the hierarchically linear order Matrix (row-vectors)
@param the max number of rays to be used by Polylib in Constraints2Polyhedron
 
@return The solution, a matrix of equalities (a polyhedron)
remarks: 
 * might be efficient when m is not too large compared to n
 * assumes that the starting constraints are non-redundant
*/
Matrix * HLP_solve(Matrix * cons, Matrix * order, unsigned MAXRAYS) {
  unsigned cur_o=0, nb_vars = cons->NbColumns-2;
  unsigned max, var_to_elim, i, j, cur_solution=0;
  Matrix cur_face_M, * solution;
  Matrix * degenerate_solution;
  Polyhedron * cur_face_P = NULL;
  Polyhedron * new_face_P=NULL;
  cur_face_M = *cons;
  Value mul_a, mul_b, cur_lcm;
  Matrix * NSPs = Matrix_Alloc(2, cons->NbRows);
  
  value_init(mul_a);
  value_init(mul_b);
  value_init(cur_lcm);

  solution = Matrix_Alloc(nb_vars, cons->NbColumns);

  while ( (cur_o<order->NbRows) && /* no more order vec or no more var to eliminate */
	  (max = max_scalar(&cur_face_M, order, cur_o, NSPs))!=-1) {

    /* 1- the maximum we're looking for is on the max^th facet */
    show_matrix(order);
    printf("Eliminating facet %u for polyhedron \n", max);
    show_matrix((&cur_face_M));

    /*  a- turn the max^th inequality into an equality and copy it into solution */
    value_set_si(cur_face_M.p[max][0], 0);
    Vector_Copy(cur_face_M.p[max], solution->p[cur_solution], nb_vars+2);
 
    /* non-degenerate end */
    if (cur_solution==nb_vars-1) {
      cur_solution++;
      break;
    }

    /*  b- eliminate a variable and the equality */
    for (var_to_elim=1; var_to_elim< cur_face_M.NbColumns && 
	   (value_zero_p(cur_face_M.p[max][var_to_elim])); var_to_elim++);
    var_to_elim--;
    assert(var_to_elim<=nb_vars);

    /*  OPTIONAL:   eliminate this variable in the solution (diagonalize partially)
    for(i=0; i< cur_solution; i++) {
      eliminate_var_with_constr(solution, cur_solution, solution, i, var_to_elim);
    */
      
    /* eliminate the chosen variable in the current facet */
    for(i=0; i< cur_face_M.NbRows; i++) {
      if (i!=max) { /* we still need the equality */
	eliminate_var_with_constr(&cur_face_M, max, &cur_face_M, i, var_to_elim);
      }
    }
    /* now, replace the equality itself by a trivial inequality */
    value_set_si(cur_face_M.p[max][0], 1);
    for (i=var_to_elim+1; i< cur_face_M.NbColumns; i++) 
      value_set_si(cur_face_M.p[max][i], 0);

    /* 2- eliminate redundant constraints */
    new_face_P = Constraints2Polyhedron(&cur_face_M, MAXRAYS);
    Parasite_Matrix(new_face_P, &cur_face_M);
    if (cur_face_P) Polyhedron_Free(cur_face_P);
    cur_face_P = new_face_P;
    
    /* 3- eliminate the chosen variable in the order */

    if (value_notzero_p(order->p[cur_o][var_to_elim])) {
      Lcm3(solution->p[cur_solution][var_to_elim+1], 
	    order->p[cur_o][var_to_elim], &cur_lcm);
      value_division(mul_a, cur_lcm, solution->p[cur_solution][var_to_elim+1]);
      value_division(mul_b, cur_lcm, order->p[cur_o][var_to_elim]);
      /* mul_b must be positive */
      if (value_pos_p(mul_b))
	value_oppose(mul_a, mul_a);
      else 
	value_oppose(mul_b, mul_b);
      for(i=var_to_elim; i<nb_vars; i++) {
	value_multiply(order->p[cur_o][i], order->p[cur_o][i], mul_b);
	value_addmul(order->p[cur_o][i], solution->p[cur_solution][i+1], mul_a); 
      }
    }

    /* 4- if the current order vector leads to a degenerate solution */
    /* i.e., it is nul in the optimal facet */
    /* select the next non-nul order vector */
    while ( (cur_o<order->NbRows) && zero_vector(order, cur_o) ) {
      cur_o++;
      while( (cur_o<order->NbRows) && zero_vector(order, cur_o) ) cur_o++;
      if (cur_o==order->NbRows) break;
      /* eliminate the variables already defined in the solution */
      for (j=0; j<= cur_solution; j++) {
	/* determine the variable to eliminate (first non-zero in the eq) */
	for (var_to_elim=1; value_zero_p(solution->p[j][var_to_elim]); var_to_elim++);
	var_to_elim--;
	if (value_notzero_p(order->p[cur_o][var_to_elim])) {
	  Lcm3(solution->p[j][var_to_elim+1], 
		order->p[cur_o][var_to_elim], &cur_lcm);
	  value_division(mul_a, cur_lcm, solution->p[j][var_to_elim+1]);
	  value_division(mul_b, cur_lcm, order->p[cur_o][var_to_elim]);
	  if (value_pos_p(mul_b)) 
	    value_oppose(mul_a, mul_a);
	  else 
	    value_oppose(mul_b, mul_b);
	  for(i=var_to_elim; i<nb_vars+1; i++) {
	    value_multiply(order->p[cur_o][i], order->p[cur_o][i], mul_b);
	    value_addmul(order->p[cur_o][i], solution->p[cur_o][i], mul_a);
	  }
	}
      }
    }
    cur_solution++;
  }
  /* 5- a- case of an infinite solution */
  if (max==-1) {
    /* 
       keep the constraints aI>=0 for which a.v=0, 
       where v is the current order vector
       (they are neither upper nor lower bounds for v.I).
    */
    /* hum we are cheap so let us reuse:
       - max to count the number of such constraints 
       - mul_a to store scalar products
       - degenerate solution as solution matrix */
    max=0;
    for (i=0; i<cur_face_M.NbRows; i++) {
      Inner_Product(&(cur_face_M.p[i][1]), order->p[cur_o], nb_vars, &mul_a);
      if (value_zero_p(mul_a)&&
	  (!Vector_IsZero(&cur_face_M.p[i][1], nb_vars))) max++;
    }
    printf("nb of parallel constraints: %d\n",max);
 
    /* do sthg only if there are such constraints */
    if (max) {
      show_matrix(solution);
      show_matrix(&cur_face_M);
      degenerate_solution = Matrix_Alloc(cur_solution+max+1, 
					 solution->NbColumns);
      for (i=0; i< cur_solution; i++) {
	Vector_Copy(solution->p[i], degenerate_solution->p[i], 
		    solution->NbColumns);
      }
      for (i=0, j=0; i<cur_face_M.NbRows; i++) {
	Inner_Product(&(cur_face_M.p[i][1]), 
		      order->p[cur_o], nb_vars, &mul_a);
	if (value_zero_p(mul_a) && 
	    (!Vector_IsZero(&cur_face_M.p[i][1], nb_vars))) {
	  Vector_Copy(cur_face_M.p[i], degenerate_solution->p[j+cur_solution], 
		      solution->NbColumns);
	  j++;
	}
      }
      Matrix_Free(solution);
      solution= degenerate_solution;
      cur_solution+=max;
    }
    /* the solution still lies on the current facet, and */
    /* the current order function tends to +inf */
    /* a "2" is put at the first column to indicate that */
    value_set_si(solution->p[cur_solution][0], 2);
    Vector_Copy(order->p[cur_o], &(solution->p[cur_solution][1]), nb_vars);
  }
  

  /*   b- case of a finite degenerate solution, */
  else if (cur_solution<nb_vars) {
    /* there are still inequalities that define the solution */
    /* concatenate them to the equalities */
    degenerate_solution=Matrix_Alloc(cur_solution+cur_face_M.NbRows, 
				     solution->NbColumns);
    for (i=0; i< cur_solution; i++) {
      Vector_Copy(solution->p[i], degenerate_solution->p[i], 
		  solution->NbColumns);
    }
    for (i=0; i< cur_face_M.NbRows; i++) {
      Vector_Copy(cur_face_M.p[i], degenerate_solution->p[i+cur_solution], 
		  solution->NbColumns);
    }
    Matrix_Free(solution);
    solution = degenerate_solution;
  }
  Polyhedron_Free(cur_face_P);
  /* as cur_face_M is a parasite Matrix, do not free it */
  Matrix_Free(NSPs);
  value_clear(mul_a);
  value_clear(mul_b);
  value_clear(cur_lcm);
  return solution;
}
