/********************************************************************
 * Name    : test.2.c
 * Author  : Paolo D'Alberto ( al secolo Aleppe)
 *
 * Note : we read from stdin a basic set of inequalities represented
 * as matrixes. Then we are going to add constraints. The first time
 * one constraint, then two, ..... and for every polyhedron we are
 * going to enumerate the number of points so determined:
 *
 * The code is generated looking at the main() procedure in ehrhart.c
 * file.
 *
 * We need to generate polyhedra from some basic ones, this is a test
 * to learn how to deal with polylib as library
 *
 * */


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>

#include "polylib/polylib64.h"
#define FIRST_VARIABLE_NAME 'I'
#define ICHAR (FIRST_VARIABLE_NAME-1)

/***********************************8
 * 
 * A domain is a pair of polyhedra, P and C 
 *
 */ 

typedef struct domain Domain;

struct domain { 
  Matrix *C1,*P1;     /* temporary matrix pointers */
  Polyhedron *C,*P;   /* P system of inequalities, C constraints on Parameters*/
  Enumeration *en;    /* Ehrhart polynomial */
  
  char **param_name;  /* name of parameters, for visualization */
  char **var_name;    /* name of variables, for visualization */
};
  

/****************************
 *
 * Constructor: given two matrixes, it produce the domain
 *
 * Note: it is not used
 *
Domain *newDomain(Matrix *PPP,Matrix *CCC) {
  
  Domain *temp = (Domain *) malloc(sizeof(Domain));
  
  printf(" Matrixes are not read \n");
  temp->P1 = PPP;
  temp->C1 = CCC;
  temp->P = Constraints2Polyhedron(temp->P1, 1024);
  if (!temp->P) 
    printf("ERROR IN GENERATING POLYHEDRON VARIABLES \n");
  temp->C = Constraints2Polyhedron(temp->C1, 1024);   
  if (!temp->C) 
    printf("ERROR IN GENERATING POLYHEDRON PARAMETERS \n");
  temp->param_name = Read_ParamNames(stdin, temp->C->Dimension);
  printf("end  Matrixes are not read \n");
  
  temp->en=0;
  temp->var_name = 0;
  temp->P1=0;
  temp->C1=0;
  
  return temp;
}

*/

/****************************
 *
 * Constructor: it reads from the STDIN the two matrixes representing
 * the system of inequalities.
 * */

Domain *newDomainFromFile() {     
  Domain *temp = (Domain *) malloc(sizeof(Domain));
  
  printf(" Matrix Read \n");
  temp->P1 = Matrix_Read();
  temp->C1 = Matrix_Read();
  if( temp->C1->NbColumns < 2 ) {	
    fprintf( stderr, "Not enough parameters !\n" );
    exit(0);
  }
  temp->P = Constraints2Polyhedron(temp->P1, 1024);
  if (!temp->P) 
    printf("ERROR IN GENERATING POLYHEDRON VARIABLES \n");
  temp->C = Constraints2Polyhedron(temp->C1, 1024);   
  if (!temp->C) 
    printf("ERROR IN GENERATING POLYHEDRON PARAMETERS \n");
  temp->param_name = Read_ParamNames(stdin, temp->C->Dimension);
  
  
  Matrix_Free(temp->P1);
  Matrix_Free(temp->C1);
  temp->P1=0;
  temp->C1=0;
  temp->var_name = 0;
  temp->en=0; 
  return temp;

}



/****************************
 * Destructor 
 */

void destroyDomain(Domain *s) { 
  Polyhedron_Free(s->C );
  Polyhedron_Free(s->P );
  free(s->var_name);
  free(s->param_name);
  printf(" Matrix free \n");
}


/****************************
 *
 * Given a domain source it make a duplicate of the attribute P, the
 * system of inequalities but it does not duplicate C;
 * */

Domain*  copyButSameConstrain(Domain* s) { 
  
  Domain *temp = (Domain *)malloc(sizeof(Domain*));

  temp->C1 = s->C1;
  temp->C  = s->C;
  
  temp->P = Domain_Copy(s->P);
  
  temp->P1 = s->P1;
  temp->param_name= s->param_name;
  temp->var_name=s->var_name;
  
  return temp;

}

/*********************
 * 
 * This masks the routine from ehrhart.c module 
 **/

void  polyhedronEnumeration(Domain* s) {

    s->en = Polyhedron_Enumerate(s->P,s->C,1024);

  }


/*********************
 * 
 * This masks the routine from ehrhart.c module 
 **/

int computePolynomial(Domain* s,Value *p) {
  
  return compute_poly( s->en, p );
}


/*********************
 * 
 * It generates the names for the variables starting from I
 * then J, K, L, M (5 nested loops are more than enough )
 **/

char **defaultFreeVariable( int m) {
  char **param_name;
  int i;
  
  param_name = (char **)malloc( m*sizeof(char *) );
    
  printf(" !in \n");
  for(i=0 ; i<m ; ++i )
    {
      param_name[i] = (char *) malloc(2*sizeof(char));
      sprintf( param_name[i], "%c", ICHAR+i+1 );
      printf(" %c \n",ICHAR+i+1);
    }
  
  return( param_name );
  
}


/**********************************
 *
 * Given a domain it prints on STDOUT the inequalities associated with
 * Polyhedron P and C, and double check the format of the matrix of
 * constraints associated with P.
 *
 * */

  
void printDomain(Domain* s) { 
  unsigned int i;
  Matrix *justChecking;
  char **names;

  if (s->var_name==0x0) 
    s->var_name = defaultFreeVariable(s->P->Dimension-s->C->Dimension);
  names = (char **) malloc(s->P->Dimension*sizeof(char *));
  
  printf("\n VARAIBLES %d \n",s->P->Dimension-s->C->Dimension);
  
  for (i=0;i<s->P->Dimension-s->C->Dimension;i++) {
    names[i] = s->var_name[i];
    printf(" %s \n", names[i]);
  }
  printf("\n PARAMATERS %d\n",s->C->Dimension);
  for (i=s->P->Dimension-s->C->Dimension;i<s->P->Dimension;i++) {
    names[i] = s->param_name[i-s->P->Dimension+s->C->Dimension];
    printf(" %s \n", names[i]);
    
  }
  
  printf("\n PARAMETER DOMAIN \n");    
  Print_Domain(stdout,s->C,s->param_name);
  printf("\n VARIABLE DOMAIN \n");    
  
  Print_Domain(stdout,s->P,names);
  
  justChecking = Polyhedron2Constraints(s->P);
  printf(" Double Check the matrix  \n");
  Matrix_Print(stdout,VALUE_FMT,justChecking);
  Matrix_Free(justChecking);
  free(names);
}

/*********************
 *
 * it prints in reverse order the pair Validity Domain - Ehrhart
 * Polynomial, obtained by Loecher-Clauss algorithm.
 * */

void printEhrhartPolinomial(Enumeration *res, char **param_name) { 
  
  if (res) {
    
    if (res->next) 
      printEhrhartPolinomial(res->next,param_name);
    
    
    fprintf(stdout, "---------------------------------------\n" );
    fprintf(stdout, "Domain:\n");
    Print_Domain(stdout, res->ValidityDomain, param_name );
    printf("\nEhrhart polynomial  \n");
    print_evalue(stdout,&(res->EP),param_name);
    
    
  }
}

/************************
 * This routine generates a Matrix of size row x col, driven by the
 * vectors which[] and what[]. which[i] specifies which variable in
 * row i must be set and what[i] specifies the constant value to
 * introduce into the last column.
 *
 *
 * Note: only the last row introduce and equation as a < k the otherd
 * introduce equation a >= k
 *
 * */

Matrix *addPositiveConstrains(unsigned int row, unsigned int col,
			      int *which, int *what) { 
 
  Value *p;
  unsigned int i,j;

  Matrix *temp =  Matrix_Alloc(row,col);

  
  printf("Adding  extra rows \n");
  p = temp->p[0];
  
  for (i=0;i<row;i++)
    for (j=0;j<col;j++)
      *p++=0;

  p = temp->p[0];

  for (i=1;i<row;i++) { 
    
    p[0] = 1;
    p[which[i-1]+1]= 1;
    p[i-1] = what[i-1];
    p += temp->NbColumns;
  }

    
  p[0] = 1;
  p[which[i-1]+1]= -1;
  p[col-1] = what[i-1]-1;
 

  printf("Added  extra rows \n");
  
  Matrix_Print(stdout,VALUE_FMT,temp);
  return temp;

}

/****************************************
 *
 * Main ()
 *
 *
 */

int main() {

  Domain *d = newDomainFromFile();  /* reading from STDIN*/
  
  Value *p;
  Matrix *t2, *t3;
  unsigned int k;
  
  unsigned int depthLoopNest = d->P->Dimension-d->C->Dimension;

  Matrix *lp[depthLoopNest];
  Domain *ld[depthLoopNest];
  int which[depthLoopNest], what[depthLoopNest] ;

  /* let's compute the domain and print out the result */

  printf(" loop nest depth %d\n",depthLoopNest); 
  printDomain(d);
  polyhedronEnumeration(d);
  printEhrhartPolinomial(d->en,d->param_name);
 
  printf(" d->polyhedronEnumeration(); \n");

  /**************************
   *
   * Producing positive constraints 
   **/

  for( k=0 ; k< depthLoopNest; ++k )	{
    printf(" which[%d] = %d; \n",k,k);
    which[k] = k;
    what[k]= 11;
  }

  /**************************
   *
   * let's double check the structure reproducing the generating
   * matrix
   **/
  
  printf(" Print matrix t2 \n");
  t2  = Polyhedron2Constraints(d->P);
  Matrix_Print(stdout,VALUE_FMT,t2);

  /************
   * for each loop level we produce different polyhedra by adding
   * constraints to the basic polyhedron
   */
  
  for( k=0 ; k<depthLoopNest; ++k )	{
    printf(" addPositiveConstrains(d.P1,%d,which,what ); \n",k+1);
 
    /****
     * New constraint in shape of Matrix 
     */
    
    t3 =  addPositiveConstrains(k+1,d->P->Dimension+2,which,what);


    /****
     * Let's watch the result
     **/
    printf(" Print matrix t3 \n");
    Matrix_Print(stdout,VALUE_FMT,t3);


    /****
     * Let's add to the previous matrix and generate a new polyhedron
     **/
    lp[k] = Matrix_Alloc(t2->NbRows+t3->NbRows,t2->NbColumns);
    memcpy(lp[k]->p_Init, t2->p_Init,(t2->NbRows-1)*t2->NbColumns*sizeof(Value));

    memcpy(lp[k]->p[t2->NbRows-1], t3->p_Init,t3->NbRows*t3->NbColumns*sizeof(Value));
    memcpy(lp[k]->p[lp[k]->NbRows-1], t2->p[t2->NbRows-1],
	   t2->NbColumns*sizeof(Value));
    

    /****
     * Let's watch the result
     **/

    printf(" Print matrix lp[k] \n");
    Matrix_Print(stdout,VALUE_FMT,lp[k]);
    
    /****
     * Let's make a copy of the basic polyhedron
     */
    
    ld[k] = copyButSameConstrain(d);
    
    printf(" d.copy  ---->  %d \n",  lp[k]->NbColumns);
    /**********
     * this is commented but the other way is to add the constrain to
     * the polyhedron ...... 
     
     * t1 = DomainAddConstraints(ld[k]->P ,t3,1024);
     * Domain_Free(ld[k]->P);
     * ld[k]->P = t1; 
     */
    
    /****
     * ... but let's modify the system of inequalyties
     */ 

    Domain_Free(ld[k]->P);

    // this is the place where it crashes !!!!!!!!!!

    ld[k]->P = Constraints2Polyhedron(lp[k], 1024);

    /****
     * Let's watch the result
     **/

    printDomain(ld[k]);
    polyhedronEnumeration(ld[k]);
    printEhrhartPolinomial(ld[k]->en,ld[k]->param_name);
  }

	
  printf( "Evaluation of the Ehrhart polynomial :\n" );
  p = (Value *)malloc( sizeof(Value) * (d->P->Dimension) );

  for( k=0 ; k<d->P->Dimension ; ++k )	{
    p[k]=(k+1)*20;
  }

  printf( "EP( "VALUE_FMT, p[0] );
  for( k=1 ; k<d->P->Dimension ; ++k )
    printf( ", "VALUE_FMT, p[k] );
  printf( " ) = %d\n",	computePolynomial(d,p) );

  destroyDomain(d);
 
  return 0;
}
