/* polytest.c */
#include <stdio.h>

#include <polylib/polylib.h>

/* alpha.c
     COPYRIGHT
          Both this software and its documentation are

              Copyright 1993 by IRISA /Universite de Rennes I - France,
              Copyright 1995,1996 by BYU, Provo, Utah
                         all rights reserved.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define bexchange(a, b, t, l)\
{\
  memcpy((char *)(t), (char *)(a), (int)(l));\
  memcpy((char *)(a), (char *)(b), (int)(l));\
  memcpy((char *)(b), (char *)(t), (int)(l));\
}

/*---------------------------------------------------------------------*/
/* int exist_points(pos,P,context)                                     */
/*    pos : index position of current loop index (0..hdim-1)           */
/*    P: loop domain                                                   */
/*    context : context values for fixed indices                       */
/* recursive procedure, recurs for each imbriquation                   */
/* returns 1 if there exists any integer points in this polyhedron     */
/* returns 0 if there are none                                         */
/*---------------------------------------------------------------------*/
static int exist_points (pos, P, context)
int pos;
Polyhedron *P;
Value *context;
{
   Value LB, UB, k;

	value_assign(LB, VALUE_ZERO );
	value_assign(UB, VALUE_ZERO );

   if ( lower_upper_bounds(pos, P, context, &LB, &UB) !=0 )
   /* Problem if UB or LB is INFINITY */
   {
      errormsg1("exist_points", "infdom", "infinite domain");
      return -1;
   }

   value_assign( context[pos], VALUE_ZERO );
   if ( value_lt(UB,LB) )
      return 0;
   if (!P->next)
     return (value_pos_p(value_plus(value_minus(UB,LB),VALUE_ONE)));

   for ( value_assign(k,LB); value_le(k,UB) ; value_increment(k) )
   {
      /* insert k in context */
      value_assign( context[pos], k );
      if ( exist_points( pos+1, P->next, context) > 0 ) return 1;
   }

   /* reset context */
   value_assign( context[pos], VALUE_ZERO );
   return 0;
}


/*--------------------------------------------------------------*/
/* Test to see if there are any integral points in a polyhedron */
/*--------------------------------------------------------------*/
int Polyhedron_Not_Empty(P, C, MAXRAYS)
Polyhedron *P;  /* Polyhedron to count */
Polyhedron *C;  /* parameter Context domain */
int MAXRAYS;    /* workspace size */
{
   Polyhedron *L;
   Value *context;
   int res;

   /* Create a context vector size dim+2 and set it to all zeros */
   context = (Value *) malloc ( (P->Dimension+2)*sizeof(Value) );
   memset ((char *)context, 0, (P->Dimension+2)*sizeof(Value) );

   /* Set context[P->Dimension+1] = 1  (the constant) */
   value_assign( context[P->Dimension+1], VALUE_ONE );
 
   L = Polyhedron_Scan(P, C, MAXRAYS);
   res = exist_points( 1, L, context );
   Domain_Free(L);
   free(context);
   return res;
}

/* PolyhedronLTQ ( P1, P2 ) */
/* P1 and P2 must be simple polyhedra */
/* result =  0 --> not comparable */
/* result = -1 --> P1 < P2        */
/* result =  1 --> P1 > P2        */
int PolyhedronLTQ ( P1, P2, MAXRAYS )
Polyhedron *P1, *P2;
int MAXRAYS;
{   int res, dim, i, j, k;
    Polyhedron *Q1, *Q2, *Q3, *Q;
    Matrix *M;
#define INDEX 1

    if ( P1->next || P2->next )
    {   errormsg1("PolyhedronLTQ", "compoly", "Can only compare polyhedra");
        return 0;
    }
    if ( P1->Dimension != P2->Dimension )
    {   errormsg1("PolyhedronLTQ","diffdim","Polyhedra are not same dimension");
        return 0;
    }
    dim = P1->Dimension+2;

#ifdef DEBUG
    fprintf(stdout, "P1\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, P1);
    fprintf(stdout, "P2\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, P2);
#endif

    /* Create the Line to add */
    M = Matrix_Alloc(1, dim);
    Vector_Init(M->p_Init, dim);
    value_assign( M->p[0][INDEX], VALUE_ONE );  /* line in INDEX dimension */

    Q1 = AddRays(M->p[0], 1, P1, MAXRAYS);
    Q2 = AddRays(M->p[0], 1, P2, MAXRAYS);

#ifdef DEBUG
    fprintf(stdout, "Q1\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, Q1);
    fprintf(stdout, "Q2\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, Q2);
#endif

    Matrix_Free(M);
    Q  = DomainIntersection(Q1, Q2, MAXRAYS);

#ifdef DEBUG
    fprintf(stdout, "Q\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, Q);
#endif

    Domain_Free(Q1);
    Domain_Free(Q2);

  if (emptyQ(Q)) res = 0;	/* not comparable */
  else
  {
	Q1 = DomainIntersection(P1, Q, MAXRAYS);
	Q2 = DomainIntersection(P2, Q, MAXRAYS);

#ifdef DEBUG
    fprintf(stdout, "Q1\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, Q1);
    fprintf(stdout, "Q2\n");
    Polyhedron_Print(stdout,P_VALUE_FMT, Q2);
#endif

	/* Test if Q1 < Q2 */      
	/* Build a constraint array for >= Q1 and <= Q2 */
	M = Matrix_Alloc(2, dim);
	Vector_Init(M->p_Init, 2*dim);

	/* choose a contraint from Q1 */
	for (i=0; i<Q1->NbConstraints; i++)
	{
		if ( value_zero_p(Q1->Constraint[i][0]) )
		{	/* equality */
			if ( value_zero_p(Q1->Constraint[i][INDEX]) )
			{   /* ignore side constraint (they are in Q) */
				continue;
			}
			else if ( value_neg_p(Q1->Constraint[i][INDEX]) )
			{   /* copy -constraint to M */
				value_assign( M->p[0][0], VALUE_ONE );
				for (k=1; k<dim; k++)
					value_assign(M->p[0][k], value_uminus(Q1->Constraint[i][k]) );
			}
			else
			{   /* copy constraint to M */
				value_assign(M->p[0][0], VALUE_ONE );
				for (k=1; k<dim; k++)
					value_assign( M->p[0][k], Q1->Constraint[i][k] );
			}
		}
		else if (value_neg_p(Q1->Constraint[i][INDEX]) )
		{   /* upper bound -- make a lower bound from it */
			value_assign( M->p[0][0], VALUE_ONE );
			for (k=1; k<dim; k++)
				value_assign( M->p[0][k], value_uminus(Q1->Constraint[i][k]) );
		}
		else
		{	/* lower or side bound -- ignore it */
			continue;
		}

		/* choose a constraint from Q2 */
		for (j=0; j<Q2->NbConstraints; j++)
		{
			if ( value_zero_p(Q2->Constraint[j][0]) )
			{   /* equality */
				if ( value_zero_p(Q2->Constraint[j][INDEX]) )
				{   /* ignore side constraint (they are in Q) */
					continue;
				}
				else if ( value_pos_p(Q2->Constraint[j][INDEX]) )
				{   /* copy -constraint to M */
					value_assign( M->p[1][0], VALUE_ONE );
					for (k=1; k<dim; k++)
						value_assign(M->p[1][k], value_uminus(Q2->Constraint[j][k]));
				}
				else
				{   /* copy constraint to M */
					value_assign( M->p[1][0], VALUE_ONE );
					for (k=1; k<dim; k++)
						value_assign(M->p[1][k], Q2->Constraint[j][k]);
				};
			}
			else if ( value_pos_p(Q2->Constraint[j][INDEX]) )
			{   /* lower bound -- make an upper bound from it */
				value_assign( M->p[1][0], VALUE_ONE );
				for (k=1; k<dim; k++)
					value_assign(M->p[1][k], value_uminus(Q2->Constraint[j][k]) );
			}
			else
			{   /* upper or side bound -- ignore it */
				continue;
			};

#ifdef DEBUG
		fprintf(stdout, "i=%d j=%d M=\n", i+1, j+1);
		Matrix_Print(stdout,P_VALUE_FMT, M);
#endif

			/* add M to Q and see if anything is made */
			Q3 = AddConstraints(M->p[0], 2, Q, MAXRAYS);

#ifdef DEBUG
		fprintf(stdout, "Q3\n");
		Polyhedron_Print(stdout, P_VALUE_FMT, Q3);
#endif

			if (!emptyQ(Q3))
			{   Domain_Free(Q3);
#ifdef DEBUG
		    fprintf(stdout, "not empty\n");
#endif
				res = -1;
				goto LTQdone;
			}
#ifdef DEBUG
		fprintf(stdout, "empty\n");
#endif
			Domain_Free(Q3);
		} /* end for j */
	} /* end for i */
	res = 1;
LTQdone:
	Matrix_Free(M);
	Domain_Free(Q1);
	Domain_Free(Q2);
  }
  Domain_Free(Q);

#ifdef DEBUG
  fprintf(stdout, "res = %d\n", res);
#endif

  return res;
}


/* GaussSimplify --
   Given Mat, a matrix of equalities, performs Gaussian elimination.
   Find a minimum basis, Returns the rank.
   M is context, M2 is reduced in context of M
 */
int GaussSimplify(M, M2)
Matrix *M, *M2;
{ int NbRows = M->NbRows;
  int NbCols = M->NbColumns;
  int RowSize = NbCols * sizeof(Value);
  int *col_of_rank;
  Value *temp;
  int i, j, k, n, pivot, Rank;
  Value gcd, *cp;

  col_of_rank=(int *)malloc(NbCols * sizeof(int) );
  temp       =(Value *)malloc(RowSize);
  if (!col_of_rank || !temp)
  {  errormsg1("GaussSimplify", "outofmem", "out of memory space\n");
     Pol_status = 1;
     return 0;
  }
  Rank=0;
  for (j=0; j<NbCols; j++)		/* for each column (except status) */ 
  { for (i=Rank; i<NbRows; i++)		/* starting at diagonal, look down */
      if ( value_notzero_p(M->p[i][j])) break;	/* to find the first non zero */
    if (i!=NbRows)			/* was one found ? */
    { if (i!=Rank)			/* was it found below the diagonal? */
          bexchange(M->p[Rank], M->p[i], temp, RowSize);

      /* Normalize the pivot row */
		value_assign(gcd,Vector_Gcd(M->p[Rank], temp, NbCols));
      if( value_gt(gcd,int_to_value(2)) )
      {	value_assign( cp, M->p[Rank] );
        for (k=0; k<NbCols; k++,cp++)
          value_division( *cp, gcd );
			 /* *cp/=gcd; */
      }
      if ( value_neg_p(M->p[Rank][j]) )
      {	value_assign(cp, M->p[Rank]);
        for (k=0; k<NbCols; k++,cp++)
		   value_oppose(*cp);
      }
      /* end normalize */

      pivot=i;
      for (i=0; i<NbRows; i++)	/* zero out the rest of the column */
      if (i!=Rank)
      { if ( value_notzero_p(M->p[i][j]) )
        {  Value a, a1, a2;
           value_assign( a1, M->p[i][j] );
			  value_assign( a2, M->p[Rank][j] );
           value_assign( a, Gcd(value_abs(a1), value_abs(a2)) );
           value_division( a1, a );
           value_division( a2, a );
           Vector_Combine( M->p[i], M->p[Rank], M->p[i], a2, 
                      value_uminus(a1), NbCols );
           Vector_Normalize(M->p[i], temp, NbCols);
           /* CombineRows(M->p[i], M->p[Rank], M->p[i], j, temp, NbCols); */
        }
      }
      col_of_rank[Rank]=j;
      Rank++;
    }
  } /* end of Gauss elimination */

  if (M2)  /* M2 is a transformation matrix  (i,j->f(i,j))....
              can't scale it because can't scale both sides of -> */
  { /* normalizes an affine transformation        */
    /* priority of forms                          */
    /*    1. i' -> i                (identity)    */
    /*    2. i' -> i + constant     (uniform)     */
    /*    3. i' -> constant         (broadcast)   */
    /*    4. i' -> j                (permutation) */
    /*    5. i' -> j + constant     (      )      */
    /*    6. i' -> i + j + constant (non-uniform) */
    for (k=0; k<Rank; k++)
    { j = col_of_rank[k];
      for (i=0; i<(M2->NbRows-1); i++)      /* all but the last row 0...0 1 */
      { if ( (i!=j) && value_notzero_p(M2->p[i][j]) )
                /* remove dependency of i' on j */
        {  Value a, a1, a2;
           value_assign( a1, M2->p[i][j] );
           value_assign( a2,  M->p[k][j] );
           value_assign( a, Gcd(value_abs(a1), value_abs(a2)) );
           value_division( a1, a );
           value_division( a2, a );
           if ( value_one_p(a2) )
           {  Vector_Combine( M2->p[i],  M->p[k], M2->p[i], a2,
                  value_uminus(a1), NbCols );
              /* Vector_Normalize(M2->p[i], temp, NbCols); -- can't do T */
           } /* otherwise, can't do it without mult lhs prod (2i,3j->...) */
        }
        else if ( (i==j) && value_zero_p(M2->p[i][j]) ) 
                  /* i' does not depend on j */
        {  for (n=j+1; n < (NbCols-1); n++)
           {  if (value_notzero_p(M2->p[i][n]) )  /* i' depends on some n */
              {  Vector_Combine(M2->p[i],  M->p[k], M2->p[i], VALUE_ONE,
                            VALUE_ONE, NbCols);
                 break;
              }  /* if i' depends on just a constant, then leave it alone.*/
           }
        }
      }
    }
    /* check last row of transformation M2 */
	for (j=0; j<(NbCols-1); j++)
		if ( value_notzero_p(M2->p[M2->NbRows-1][j]) )
		{  errormsg1("GaussSimplify", "corrtrans", "Corrupted transformation\n");
			break;
		}

	if ( value_notone_p(M2->p[M2->NbRows-1][NbCols-1]) )
	{  errormsg1("GaussSimplify", "corrtrans", "Corrupted transformation\n");
	}
  }

  free(temp);
  free(col_of_rank);
  return Rank;
}

char s[128];

int main()
{   Matrix *a=NULL, *b=NULL, *c, *d, *e, *f;
    Polyhedron *A, *B, *C, *D, *last, *tmp;
    int i, nbPol, nbMat, func;

    fgets(s, 128, stdin);
    nbPol = nbMat = 0;
    while ( (*s=='#') ||
        ((sscanf(s, "D %d", &nbPol)<1) && (sscanf(s, "M %d", &nbMat)<1)) )
        fgets(s, 128, stdin);

    for (i=0, A=last=(Polyhedron *)0; i<nbPol; i++)
    {  a = Matrix_Read();
       tmp = Constraints2Polyhedron(a, 600);
       Matrix_Free(a);
       if (!last) A = last = tmp;
       else
       { last->next = tmp;
         last = tmp;
       }
    }

    if (nbMat)
    {  a = Matrix_Read(); }

    fgets(s, 128, stdin);
    nbPol = nbMat = 0;
    while ( (*s=='#') ||
        ((sscanf(s, "D %d", &nbPol)<1) && (sscanf(s, "M %d", &nbMat)<1)) )
        fgets(s, 128, stdin);

    for (i=0, B=last=(Polyhedron *)0; i<nbPol; i++)
    {  b = Matrix_Read();
       tmp = Constraints2Polyhedron(b, 200);
       Matrix_Free(b);
       if (!last) B = last = tmp;
       else
       { last->next = tmp;
         last = tmp;
       }
    }

    if (nbMat)
    {  b = Matrix_Read(); }

    fgets(s, 128, stdin);
    while ( (*s=='#') || (sscanf(s, "F %d", &func)<1) ) fgets(s, 128, stdin);

    switch (func)
    {
        case 1:
            C = DomainUnion(A, B, 200);
            D = DomainConvex(C, 200);
            d = Polyhedron2Constraints(D);
            Matrix_Print(stdout,P_VALUE_FMT, d);
            Matrix_Free(d);
            Domain_Free(C);
            Domain_Free(D);
            break;
        case 2:
            D = DomainSimplify(A, B, 200);
            d = Polyhedron2Constraints(D);
            Matrix_Print(stdout,P_VALUE_FMT, d);
            Matrix_Free(d);
            Domain_Free(D);
            break;
        case 3:
            a = Polyhedron2Constraints(A);
            Matrix_Print(stdout,P_VALUE_FMT, a);
            b = Polyhedron2Constraints(B);
            Matrix_Print(stdout,P_VALUE_FMT, b);
            break;
        case 4:
            a = Polyhedron2Rays(A);
            Matrix_Print(stdout,P_VALUE_FMT, a);
            break;
        case 5:
            /* a = ec , da = c , ed = 1 */
            right_hermite(a,&c,&d,&e);
            Matrix_Print(stdout,P_VALUE_FMT, c);
            Matrix_Print(stdout,P_VALUE_FMT, d);
            Matrix_Print(stdout,P_VALUE_FMT, e);
            f = Matrix_Alloc(e->NbRows, c->NbColumns);
            Matrix_Product(e,c,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            Matrix_Free(f);
            f = Matrix_Alloc(d->NbRows, a->NbColumns);
            Matrix_Product(d,a,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            Matrix_Free(f);
            f = Matrix_Alloc(e->NbRows, d->NbColumns);
            Matrix_Product(e,d,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            break;
        case 6:
            /* a = ce , ad = c , de = 1 */
            left_hermite(a,&c,&d,&e);
            Matrix_Print(stdout,P_VALUE_FMT, c);
            Matrix_Print(stdout,P_VALUE_FMT, d);
            Matrix_Print(stdout,P_VALUE_FMT, e);
            f = Matrix_Alloc(c->NbRows, e->NbColumns);
            Matrix_Product(c,e,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            Matrix_Free(f);
            f = Matrix_Alloc(a->NbRows, d->NbColumns);
            Matrix_Product(a,d,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            Matrix_Free(f);
            f = Matrix_Alloc(d->NbRows, e->NbColumns);
            Matrix_Product(d,e,f);
            Matrix_Print(stdout,P_VALUE_FMT, f);
            break;
	case 7:
/*	    Polyhedron_Print(stdout,"%5d", A); */
/*             Matrix_Print(stdout,"%4d", b); */
            C = Polyhedron_Image(A, b, 400);
            Polyhedron_Print(stdout,P_VALUE_FMT, C);
            break;
        case 8:
            printf("%s\n",
                Polyhedron_Not_Empty(A,B,600) ? "Not Empty" : "Empty");
            break;
        case 9:
            i = PolyhedronLTQ(A,B,600);
            printf("%s\n",
                i==-1 ? "A<B" : i==1 ? "A>B" : i==0 ? "A><B" : "error");
            i = PolyhedronLTQ(B,A,600);
            printf("%s\n",
                i==-1 ? "A<B" : i==1 ? "A>B" : i==0 ? "A><B" : "error");
            break;
        default:
            printf("? unknown function\n");
    }

    Domain_Free(A);
    Domain_Free(B);

    return 0;
}
