/***********************************************************************/
/*                Parametrized polyhedra V4.00                         */
/*                copyright 1995, 1997, 1998, Vincent Loechner         */
/*                copyright 1996, 1997, Doran Wilde                    */
/*       Permission is granted to copy, use, and distribute            */
/*       for any commercial or noncommercial purpose under the terms   */
/*       of the GNU General Public license, version 2, June 1991       */
/*       (see file : LICENSING).                                       */
/***********************************************************************/

/* #define DEBUGPP */
/* #define DEBUGPP3 */	/* initialization of domain, context, ... */
/* #define DEBUGPP31 */ /* even more init-domains */
/* #define DEBUGPP4 */  /* m-faces scan */
/* #define DEBUGPP41 */ /* inverse Di in scan */
/* #define DEBUGPP5 */  /* Compute_PDomains */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#ifdef DEBUGPP
#include <time.h>
#endif

#include <polylib/polylib.h>

#define Vector_Init(p1, length) \
  memset((char *)(p1), 0, (int)((length)*sizeof(Value)))


#ifdef __STDC__
static void traite_m_face(Polyhedron *, int *);
static void scan_m_face( int, int, Polyhedron *, int * );
Param_Polyhedron *Find_m_faces( Polyhedron *, Polyhedron *, int, int);
#else
static void traite_m_face();
static void scan_m_face();
Param_Polyhedron *Find_m_faces();
#endif /* __STDC__ */

/* Modified DomainIntersection that discards degenerate results */
/* (i.e. intersections that are of smaller dimension than the operands) */
Polyhedron *PDomainIntersection(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned   NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;

	if (!Pol1 || !Pol2) return (Polyhedron*) 0;
	if( (Pol1->Dimension != Pol2->Dimension) || (Pol1->NbEq != Pol2->NbEq) )
	{
		fprintf( stderr,
				"? PDomainIntersection: operation on different dimensions\n" );
		return (Polyhedron*) 0;
	}
 
	d = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		for (p2=Pol2; p2; p2=p2->next)
		{
			p3 = AddConstraints(p2->Constraint[0],
								p2->NbConstraints, p1, NbMaxRays);
			if (!p3) continue;
			if (p3->NbEq!=Pol1->NbEq)
				/* Polyhedron_Free(p3) */;  /* no new equations */
			else
				d = AddPolyToDomain (p3, d);
		}
	}
	return d;
} /* PDomainIntersection */


/* Modified DomainDifference which does not put a 1 unit space around cut */
Polyhedron *PDomainDifference(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;
	int i;

	if (!Pol1 || !Pol2)
		return (Polyhedron*) 0;
	if( (Pol1->Dimension != Pol2->Dimension) || (Pol1->NbEq != Pol2->NbEq) )
	{
		fprintf( stderr,
				"? PDomainDifference: operation on different dimensions\n" );
		return (Polyhedron*) 0;
	}
 
	d = (Polyhedron *)0;
	for (p2=Pol2; p2; p2=p2->next)
	{
		for (p1=Pol1; p1; p1=p1->next)
		{
			for (i=0; i<p2->NbConstraints; i++)
			{
				p3 = SubConstraint(p2->Constraint[i], p1, NbMaxRays,2);
				if (!p3) continue;
				if (emptyQ(p3) || p3->NbEq!=Pol1->NbEq)
					Polyhedron_Free(p3);
				else
					d = AddPolyToDomain (p3, d);
			}
		}
		Pol1 = d;
		d = (Polyhedron *)0;
	}
	return Pol1;
} /* PDomainDifference */



/*******************************************/
/* TestRank : test if matrix M is full row */
/* WARNING. M is modified by the algo.     */
/*******************************************/
static int TestRank( Matrix *M )
{
	int i,j,k;
	Value m1,m2,m3;

	for( k=0 ; k<M->NbColumns ; ++k )
	{
		/* si pivot nul, on en cherche un !=0 en dessous. */
		/* et on echange les 2 lignes */
		/* s'il n'y en a pas, c'est perdu. */
		if( value_zero_p(M->p[k][k]) )
		{
			for( j=k+1 ; j<M->NbRows ; ++j )
			{
				if( value_notzero_p(M->p[j][k]) )
				{	/* ok on en a trouve un */
					/* on echange les lignes j et k */
					for( i=k ; i<M->NbColumns ; ++i )
					{	Value t;
						value_assign( t, M->p[j][i] );
						value_assign( M->p[j][i], M->p[k][i] );
						value_assign( M->p[k][i], t );
					}
					break;
				}
			}
			if( j>=M->NbRows )
				return 0;
		}
		/* M[k][k]=pivot */
		for( j=k+1 ; j<M->NbRows ; ++j )
		{	/* pour toutes les lignes j > k */
			/* calculer gcd des pivots et multiplier les lignes par
				l'autre pivot / gcd avant de faire la difference */
			Value g;
			g = Gcd( M->p[j][k], M->p[k][k] );
			for( i=k+1 ; i<M->NbColumns ; ++i )
			{	/* pour tous les indices i > k */
				value_assign( m1, value_mult( M->p[j][i], M->p[k][k] ) );
				value_assign( m2, value_mult( M->p[j][k], M->p[k][i]) );
				value_assign( m3, value_minus(m1,m2 ) );
				value_assign( M->p[j][i],value_div( m3,g) );
			}
		}

	}

	return 1;
	/* ok, c'est full rank */
}





/*
 * Saturation Matrix : integer
 *
 */

typedef struct
{
  unsigned int NbRows;
  unsigned int NbColumns;
  int **p;
  int *p_init;
} SatMatrix; 




static SatMatrix *SMAlloc (rows, cols)
int rows, cols;
{
  int **q, *p, i;

  SatMatrix *result = (SatMatrix *) malloc (sizeof (SatMatrix));
  assert (result != NULL);

  result->NbRows = rows;
  result->NbColumns = cols;
  result->p = q = (int **)malloc (rows * sizeof(int *));
  assert (result->p != NULL);  
  result->p_init = p = (int *)malloc (rows * cols * sizeof (int));
  assert (result->p_init != NULL);  

  for (i=0; i < rows; i++)
  {
    *q++ = p;
    p += cols;
  }

  return result;
}


static void SMFree (m)
SatMatrix *m;
{
  free ((char *) m->p_init);
  free ((char *) m->p);
  free ((char *) m);
}



/*----------------------------------------------------------------------*/
/* Shared Global Variables                                              */
/*        Used by procedures: Find_m_face, scan_m_face, Poly2Sat,       */
/*                            traite_m_face, count_sat                  */
/*----------------------------------------------------------------------*/
static int m;			/* # of parameters */
static int m_dim;		/* dim of m-face */
static int n;			/* dimension (not including parameters) */
static int ws;			/* Working Space size */
static int nr;			/* (NbRays-1)/32 + 1 */

static Polyhedron *CEqualities;	/* Equalities in the context */
/* the saturation matrix is an INTEGER matrix */
static SatMatrix   *Sat;	/* Saturation Matrix (row=constraint, col=ray)*/
static int *egalite;		/* bool vector marking constraints in m-face */

static Matrix *Xi, *Pi;		/* Xi and Pi */
static Matrix *PiTest;		/* test Pi full row rank ? */
static Matrix *PiInv;		/* matrix inverse Pi, with the last col of */
				/* each line = denominator of the line */
static Matrix *RaysDi;		/* Constraint matrix for computing Di */

static int KD;			/* flag : keep the full domains in mem ? */
				/* 1 = yes; 0 = no, keep constraints only */

static int nbPV;		/* the number of parameterized vertices */
static Param_Vertices *PV_Result; /* list of parameterized vertices */
static Param_Domain *PDomains;  /* list of domains. */

#ifdef DEBUGPP
static int nbfaces;
#endif


static Polyhedron *Add_CEqualities( Polyhedron *D )
{
	Polyhedron *d, *r, *tmp;
	if( !CEqualities )
		return D;
	else
	{
		if( !D || emptyQ(D) )
		{
			if( D )
				Domain_Free( D );
			return Polyhedron_Copy(CEqualities);
		}

		r = AddConstraints(D->Constraint[0], D->NbConstraints,
				                    CEqualities, ws );
		tmp = r;
		for( d=D->next ; d ; d=d->next )
		{
			tmp->next = AddConstraints(d->Constraint[0], d->NbConstraints,
			                    CEqualities, ws );
			tmp = tmp->next;
		}
		Domain_Free( D );
		return( r );
	}
}

/*----------------------------------------------------------------------*/
/* traite_m_face                                                        */
/*       Given an m-face, compute the parameterized vertex              */
/*----------------------------------------------------------------------*/
static void traite_m_face( D, mf )
Polyhedron *D;	/* The entire domain */
int *mf;	/* Bit vector marking the lines/rays in the m-face */
{
   Matrix *Si;				/* Solution */
   Polyhedron *PDi;		/* polyhedron Di */
   Matrix *Di;				/* Definition Domain */
   Param_Vertices *PV;
   int j,k,c;
   unsigned kx, bx;

#ifdef DEBUGPP
	++nbfaces;
#endif
   /* extract  Xi, Pi, and RaysDi from D */
	RaysDi->NbRows = 0;
	for( k=0,c=0,kx=0,bx=MSB; k<D->NbRays; ++k )
	{
		if( mf[kx]&bx )      /* this ray is in the current m-face */
		{
			if (c<m+1)
			{
				int i;
				/* tester si cette nouvelle colonne est lin. indep. des autres */
				/* i.e. si gauss ne donne pas de '0' sur la colonne Pi
					jusqu'a l'indice 'c' */
				/* construit PiTest */
				for( j=0 ; j<m+1 ; ++j )
				{
					for( i=0 ; i<c ; ++i )
						/* les c premieres colonnes */
						value_assign( PiTest->p[j][i], Pi->p[j][i] );
					/* la nouvelle */
					value_assign( PiTest->p[j][c], D->Ray[k][j+1+n] );
				}
				PiTest->NbColumns = c+1;
				if( TestRank( PiTest ) )
				{
					/* ok, c'est lin. indep. */
					for (j=0;j<n;j++)
						value_assign( Xi->p[j][c], D->Ray[k][j+1] );	/* Xi */
					for (j=0;j<m;j++)
						value_assign( Pi->p[j][c], D->Ray[k][j+1+n] );	/* Pi */
					value_assign( Xi->p[n][c], D->Ray[k][n+m+1] );	/* const */
					value_assign( Pi->p[m][c], D->Ray[k][n+m+1] );	/* const */
					c++;
				}
			}
			/* status bit */
			value_assign( RaysDi->p[RaysDi->NbRows][0], D->Ray[k][0] );

			memcpy(&RaysDi->p[RaysDi->NbRows][1], &D->Ray[k][n+1],
						(m+1)*sizeof(Value));
			++RaysDi->NbRows;
		}
		NEXT(kx,bx);
	}

#ifdef DEBUGPP41
	printf( "\nRaysDi=\n" );
	Matrix_Print( stdout, P_VALUE_FMT, RaysDi );
	if( c < m+1 )
	 printf( "Invalid ");
	printf( "Pi=\n" );
	Matrix_Print( stdout, P_VALUE_FMT, Pi );
#endif

#ifdef DEBUGPP4
	if( c < m+1 )
		fprintf(stderr, "Eliminated because of no vertex\n");
#endif

	if( c < m+1 )	return;

   /* RaysDi->numColumns = m+2; */  /* stays the same */

	/*	Xi->NbColumns = m+1;*/	/* VIN100: stays the same. was 'c'! */
	/*	Xi->NbRows = n+1; */ 	/* stays the same */
	/*	Pi->NbColumns = m+1;*/	/* VIN100: stays the same. was 'c'! */
	/*	Pi->NbRows = m+1; */		/* stays the same */

#ifdef DEBUGPP4
	fprintf(stderr, "Xi = ");
	Matrix_Print(stderr, P_VALUE_FMT, Xi );
	fprintf(stderr, "Pi = ");
	Matrix_Print(stderr, P_VALUE_FMT, Pi );
#endif
   /* (Right) invert Pi if POSSIBLE, if not then next m-face */
   /* Pi is destroyed */
   if( !MatInverse( Pi, PiInv ) )
   {
#ifdef DEBUGPP4
		fprintf(stderr, "Eliminated because of no inverse Pi\n");
#endif
      return;
   }

#ifdef DEBUGPP4
	fprintf(stderr, "FACE GENERATED!\n");
	fprintf(stderr, "PiInv = ");
	Matrix_Print(stderr, P_VALUE_FMT, PiInv);
#endif
   /* Compute  Si (now called Ti in the paper) */
   Si = Matrix_Alloc( Xi->NbRows, PiInv->NbColumns );
   rat_prodmat( Si, Xi, PiInv );
#ifdef DEBUGPP4
	fprintf(stderr, "Si = ");
	Matrix_Print(stderr, P_VALUE_FMT, Si);
#endif

	Si->NbRows--;      /* throw out the last row = 0 ... 0 1 */

   /* Copy all of that into the PV structure */
   PV = (Param_Vertices *) malloc( sizeof(Param_Vertices) );
   PV->next = PV_Result;
   PV->Vertex = Si;
   PV->Domain = NULL;
   PV_Result = PV;
   nbPV++;         /* increment vertex count */

   /* ok... now compute the parameter domain */
   PDi = Rays2Polyhedron( RaysDi, ws );

#ifdef DEBUGPP3
	fprintf( stderr, "RaysDi = " );
	Matrix_Print(stderr, P_VALUE_FMT, RaysDi );
	fprintf( stderr, "PDi = " );
	Polyhedron_Print(stderr, P_VALUE_FMT, PDi );
#endif
   if( KD==0 )
   {
		/* add the equalities again to the domain */
		if( CEqualities )
			 PDi = Add_CEqualities( PDi );

      Di = Matrix_Alloc( PDi->NbConstraints, m+2 );
      memcpy( &Di->p[0][0], &PDi->Constraint[0][0],
         PDi->NbConstraints*(m+2)*sizeof(Value) );
      Polyhedron_Free( PDi );
      PV->Domain = Di;
   }
   else
   {  Param_Domain *PD;

      PD = (Param_Domain *) malloc ( sizeof(Param_Domain) );
      PD->Domain = PDi;
      PD->F = NULL;
      PD->next = PDomains;
      PDomains = PD;
   }
   return;
}

/*----------------------------------------------------------------------*/
/* count_sat                                                            */
/*      count the number of saturated rays in the bit vector mf         */
/*      Uses nr from global area                                        */
/*----------------------------------------------------------------------*/
int cntbit[256] = {				/* counts for 8 bits */
0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,

1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,

1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,

2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8 };

static int count_sat (mf)
int *mf;
{	register int i, tmp, cnt=0;
	for (i=0; i<nr; i++)
	{	tmp = mf[i];
		cnt = cnt
			+ cntbit[ tmp & 0xff ]
			+ cntbit[ (tmp>>8) & 0xff ]
			+ cntbit[ (tmp>>16) & 0xff ]
			+ cntbit[ (tmp>>24) & 0xff ]
			;
	}
	return cnt;
}


/*----------------------------------------------------------------------*/
/* let D + E + L be the dimension of the polyhedron                     */
/* D = Dimension of polytope (ray space)                                */
/* L = Dimension of Lineality space (number of lines, bid)              */
/* E = Dimension of Affine hull (number of equations)                   */
/* n = number of data indices                                           */
/* m = number of parameters                                             */
/* full domain:                                                         */
/*     n + m = D + E + L                                                */
/* projected domains:                                                   */
/*     m = D_m + E_m + L_m                                              */
/*     n = D_n + E_n + L_n                                              */
/* What dimension M-face, when projected onto parameter space,          */
/* will give an L_m-face?                                               */
/* What are the conditions?                                             */
/*   - at least one vertex                                              */
/*   - number of rays >= D_m+1 after removal of redundants              */
/*                                                                      */
/* dim of face    nb saturated constraints   nb saturated lines,rays    */
/* -----------    ------------------------   -----------------------    */
/* (0+L)-face     all E eqns + >=D ineq      all L lines + 1 ray        */
/* (M+L)-face     all E eqns + >=(D-M) ineq  all L lines + >=(M+1) rays */
/* (D+L)-face     all E eqns + 0 ineq        all L lines + >=(D+1) rays */
/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/
/* scan_m_face                                                          */
/*      pos : the next candidate constraint position                    */
/*    nb-un : number of saturated constraints needed to finish a face   */
/*        D : the source polyhedron (context included )                 */
/*       mf : bit-array marking rays which are satuarated so far        */
/* From Global area:                                                    */
/* ----------------                                                     */
/*        n : number of data indices                                    */
/*        m : number of parameters                                      */
/*  egalite : boolean vector marking saturated constraints in m-face    */
/*      Sat : Saturation Matrix (row=constraints, col=rays)             */
/*       ws : working space size                                        */
/*       nr : (NbRays-1)/32 + 1                                         */
/*                                                                      */
/* Recursive function to find the rays and vertices of each m-face      */
/*----------------------------------------------------------------------*/
static void scan_m_face( pos, nb_un, D, mf )
int pos;	/* the next candidate constraint position */
int nb_un;	/* the number of constraints needed to finish a face */
Polyhedron *D;	/* the source polyhedron */
int *mf;	/* (bit vector) marks rays that are saturated so far */
{
	int *new_mf;

#ifdef DEBUGPP4
fprintf(stderr,"Start scan_m_face(pos=%d, nb_un=%d, n=%d, m=%d\n",
pos, nb_un,n,m);
fprintf(stderr,"mf = ");
{	int i;
	for (i=0; i<nr; i++)
		fprintf(stderr,"%08X", mf[i]);
	fprintf(stderr, "\negalite = [");
	for (i=0; i<D->NbConstraints; i++)
		fprintf(stderr," %1d", egalite[i]);
	fprintf(stderr,"]\n");
}
#endif
	if( nb_un == 0 )	/* base case */
	{
		int i,j;

		/*********** ELIMINATION OF REDUNDANT FACES ***********/
		/* if all these vertices also verify a previous constraint */
		/* which is NOT already selected, we eliminate this face */
		/* This keeps the lexicographically greatest selection */
		for( i=0 ; i<pos ; i++ )
		{	if( egalite[i] )
				continue;  /* already selected */
			/* if Sat[i] & mf == mf then its redundant */
			for (j=0; j<nr; j++)
			{
#ifdef DEBUGPP4
fprintf(stderr,"mf=%08X Sat[%d]=%08X &=%08X\n",mf[j],i,Sat->p[i][j],
(mf[j] & Sat->p[i][j]) );
#endif
				if ((mf[j] & Sat->p[i][j]) != mf[j])
					break;	/* it's not redundant */
			}
#ifdef DEBUGPP4
if (j==nr) fprintf(stderr, "Redundant with constraint %d\n", i);
#endif
			if (j==nr) return;	/* it is redundant */
		}
		/********* END OF ELIMINATION OF DEGENERATE FACES *********/

		/* if we haven't found a constraint verified by all */
		/* the rays, its OK, it's a new face. */
		traite_m_face( D, mf );
		return;
	}

	/* see if there enough constraints left to finish */
	if( (pos+nb_un)>D->NbConstraints ) return;

	/* recurring part of the procedure */
	/* add the pos'th constraint, compute new saturation vector */
	{	int k;
		new_mf  = (int *)malloc(nr*sizeof(int));
		for (k=0; k<nr; k++)
			new_mf[k] = mf[k] & Sat->p[pos][k];
	}
#ifdef DEBUGPP4
fprintf(stderr,"new_mf = ");
{  int i;
   for (i=0; i<nr; i++)
   {  fprintf(stderr,"%08X", new_mf[i]);
   }
   fprintf(stderr, "\ncount(new_mf) = %d\n", count_sat(new_mf) );
}
#endif

	/* at least m_dim+1 rays must be saturated to add this constraint */
	if (count_sat(new_mf)>m_dim)
	{	egalite[pos]=1;		/* Try it with the pos-th constraint */
		scan_m_face(pos+1, nb_un-1, D, new_mf);
	}
	free(new_mf);

	egalite[pos]=0;		/* Try it without the pos-th constraint */
        if ((pos+nb_un)>=D->NbConstraints) return;
	scan_m_face( pos+1, nb_un, D, mf );
	return;
}


/*----------------------------------------------------------------------*/
/* Poly2Sat                                                             */
/*       Creates a saturation matrix (row=constraint, col=rays)         */
/*       Writes global vars: nr                                         */
/*----------------------------------------------------------------------*/
static SatMatrix *Poly2Sat(P,L)
Polyhedron *P;
int **L;
{   SatMatrix *Sat;
    int i, j, k, kx, *Tmp;
	 Value *p1, *p2, p3;
    unsigned Dimension, NbRay, NbCon, bx;
 
    NbRay = P->NbRays;
    NbCon = P->NbConstraints;
    Dimension = P->Dimension+1;       /* homogeneous Dimension */
 
    /* build the Sat matrix */
    nr      = (NbRay - 1) / (sizeof(int)*8) + 1;	/* set globally */
    Sat     = SMAlloc(NbCon, nr);
    Tmp     = (int *)malloc(nr*sizeof(int));
    memset(Sat->p_init, 0, nr*NbCon*sizeof(int));
    memset(Tmp, 0, nr*sizeof(int));
    kx=0; bx=MSB;
    for (k=0; k<NbRay; k++)
    {   for (i=0; i<NbCon; i++)
        {   p1 = &P->Constraint[i][1];
            p2 = &P->Ray[k][1];
            value_assign( p3, VALUE_ZERO );
            for (j=0; j<Dimension; j++)
				{
					value_addto( p3, value_mult( *p1, *p2 ) );
					p1++; p2++;
				}
            if ( value_zero_p(p3) )
					Sat->p[i][kx]|=bx;
        }
        Tmp[kx] |= bx;
        NEXT(kx, bx);
    }
    *L = Tmp;
    return Sat;
}

/*----------------------------------------------------------------------*/
/* GenParamPolyhedron                                                   */
/*       Creates a parameterized polyhedron with 0 parameters           */
/* written by Xavier Redon, modified by Vincent Loechner                */
/*----------------------------------------------------------------------*/
Param_Polyhedron *GenParamPolyhedron(Polyhedron *poly)
{
	Param_Polyhedron *result;
	Matrix *rays;
	int nbRows, nbColumns;
	int i, size;

	rays=Polyhedron2Rays(poly);
	nbRows=rays->NbRows;
	nbColumns=rays->NbColumns;

	/* Check that the polyhedron is bounded */
	for(i=0;i<nbRows;i++)
		if( value_notone_p(rays->p[i][0]) ||
				value_notone_p(rays->p[i][nbColumns-1]) )
			return NULL;

	/* Initialize the result */
	result=(Param_Polyhedron *)malloc(sizeof(Param_Polyhedron));
	result->nbV=nbRows;
	result->V=NULL;

	/* Build the parametric vertices */
	for(i=0;i<nbRows;i++)
	{
		Matrix *vertex;
		Param_Vertices *paramVertex;
		int j;

		vertex=Matrix_Alloc(nbColumns-2,2);
		for(j=1;j<nbColumns-1;j++)
		{
			value_assign( vertex->p[j-1][0], rays->p[i][j] );
			value_assign( vertex->p[j-1][1], VALUE_ONE );
		}
		paramVertex=(Param_Vertices *)malloc(sizeof(Param_Vertices));
		paramVertex->Vertex=vertex;

		/* there is one validity domain : universe of dimension 0 */
		paramVertex->Domain=Matrix_Alloc(1,2);
		value_assign( paramVertex->Domain->p[0][0], VALUE_ONE );
		value_assign( paramVertex->Domain->p[0][1], VALUE_ONE );

		paramVertex->next=result->V;
		result->V=paramVertex;
	}

	/* Build the parametric domains (only one here) */
	size=(nbRows-1)/(8*sizeof(int))+1;
	result->D=(Param_Domain *)malloc(sizeof(Param_Domain));
	result->D->next=NULL;
	result->D->Domain=Universe_Polyhedron(0);
	result->D->F=(int *)malloc(size*sizeof(int));
	memset( &result->D->F[0], 0xFF, size*sizeof(int) );

	/* return result */
	return result;
}

/*----------------------------------------------------------------------*/
/* Find_m_faces                                                         */
/*       Creates a list of the parameterized vertices                   */
/*       Writes global vars: m,n,ws,Sat,egalite,mf,Xi,Pi,PiInv,RaysDi   */
/*----------------------------------------------------------------------*/
Param_Polyhedron *Find_m_faces( D, C, keep_dom, working_space )
Polyhedron *D, *C;
int keep_dom, working_space;
{	int *mf, i, j;
	Polyhedron *C1, *C2, *D1;	/* the polyedron,simplified equations */
	Matrix *M;
	Param_Polyhedron *res;

	if( !D || !C ) return (Param_Polyhedron *) 0;

	m = C->Dimension;
	n = D->Dimension - m;
	ws = working_space;

	if( n<0 )
	{	fprintf( stderr,
		"Find_m_faces: ?%d parameters of a %d-polyhedron !\n", m, n );
		return (Param_Polyhedron *) 0;
	}
	if( m==0 )
	{
		Param_Polyhedron *result=GenParamPolyhedron(D);
		if(result)
			return result;
		else
		{
			fprintf( stderr, "Find_m_faces: polyhedron is not bounded!\n" );
			return (Param_Polyhedron *) 0;
		}
	}

	/* extract out the true context C1 and find its true dimension */
	C1 = align_context(C, D->Dimension, ws);

#ifdef DEBUGPP31
	fprintf(stderr,"m = %d\n", m);
	fprintf( stderr, "D = " );
	Polyhedron_Print(stderr, P_VALUE_FMT, D );
	fprintf( stderr, "C1 = " );
	Polyhedron_Print(stderr, P_VALUE_FMT, C1 );
#endif

	D1 = DomainIntersection(D, C1, ws);
#ifdef DEBUGPP31
	fprintf( stderr, "D1 = " );
	Polyhedron_Print(stderr, P_VALUE_FMT, D1 );
#endif
	Domain_Free(C1);

	/* M : lines in the direction of the first n indices (index space) */
	M   = Matrix_Alloc(n, D->Dimension+2);
	Vector_Init( M->p[0], n*(D->Dimension+2) );
	for (i=0; i<n; i++)
		M->p[i][i+1] = 1;
	C1 = DomainAddRays(D1, M, ws);
	Matrix_Free(M);

	/* Leave constraints in D1, do not simplify... */
	/*	D1 = DomainSimplify(D1, C1, ws);  */

/******************  VIN100, jan-97, mar-97 ***********************/
/* simplify D1 with the _EQUALITIES_ in the context */
	if( C1->NbEq != 0 )
	{
		/* compute CEqualities = matrix of equalities in C1, projected in
			the parameter space */
		M = Matrix_Alloc(C1->NbEq, m+2);
		/* equalities are at the beginning of matrix C1->Constraint */
	/* modified vin100 14/may/99 */
		for( j=0,i=0 ; i<C1->NbEq ; ++i,++j )
		{
			while( value_notzero_p(C1->Constraint[j][0]) )
				++j;
			value_assign( M->p[i][0], C1->Constraint[j][0] );
			memcpy( &M->p[i][1], &C1->Constraint[j][D->Dimension-m+1],
				sizeof(Value)*(m+1) );
		}
		CEqualities = Constraints2Polyhedron( M, ws );
		C2 = align_context( CEqualities, D->Dimension, ws );
		D1 = DomainSimplify(D1, C2, ws);
		Polyhedron_Free( C2 );
		Matrix_Free( M );
	}
	else
		CEqualities = NULL;
/**********************************************************/

#ifdef DEBUGPP3
	fprintf(stderr, "D1\t Dim = %3d\tNbEq = %3d\tLines = %3d\n", D1->Dimension,
                D1->NbEq, D1->NbBid);
	fprintf(stderr, "C1\t Dim = %3d\tNbEq = %3d\tLines = %3d\n", C1->Dimension,
                C1->NbEq, C1->NbBid);

	fprintf(stderr, "Polyhedron D1 (D AND C) = ");
	Polyhedron_Print(stderr,P_VALUE_FMT, D1);
	fprintf(stderr, "Polyhedron CEq (equalities) = ");
	if(CEqualities) Polyhedron_Print(stderr,P_VALUE_FMT, CEqualities);
	else fprintf(stderr, "NULL\n");
#endif

	KD = keep_dom;
	PDomains = NULL;
	PV_Result = NULL;
	nbPV = 0;

	if (D1->NbRays==0) return 0;

	/* mf : a bit array indicating which rays are part of the m-face */
	/* Poly2Sat initializes mf to all ones */
	/* set global variable nr to size (number of words) of mf */
	Sat = Poly2Sat(D1, &mf);
#ifdef DEBUGPP4
/* fprintf(stderr,"Sat = ");  Matrix_Print(stderr,"%08X",Sat); */
	fprintf(stderr,"mf = ");
	for (i=0; i<nr; i++)
		fprintf(stderr,"%08X", mf[i]);
	fprintf(stderr, "\n");
#endif
	/* a boolean array saying which constraints are part of the m-face */
	egalite = (int *)malloc( sizeof(int)*D1->NbConstraints );
	memset( egalite, 0, sizeof(int)*D1->NbConstraints );
	/* Equalities are automatically included */

	for (i=0; i<D1->NbEq; i++)
		egalite[i] = 1;

	Xi     = Matrix_Alloc( n+1, m+1 );
	Pi     = Matrix_Alloc( m+1, m+1 );
	PiTest = Matrix_Alloc( m+1, m+1 );
	PiInv  = Matrix_Alloc( m+1, m+2 );
	RaysDi = Matrix_Alloc( D1->NbRays, m+2 );

	/* Alg : find the rays verifying (n - NbEqualities) constraints */

	/***************  change by VIN100, jan-97 ****************/
	m_dim = C->Dimension;
	/* WAS:	m_dim = C->Dimension - C1->NbEq; */
	/**********************************************************/

#ifdef DEBUGPP
nbfaces=0;
#endif
#ifdef DEBUGPP3
fprintf(stderr,
"Target: find faces that saturate %d constraints and %d rays/lines\n",
D1->Dimension - m_dim, m_dim+1);
#endif
	/* D->NbEq constraints already saturated ! */
	scan_m_face( D1->NbEq, (D1->Dimension - m_dim - D1->NbEq), D1, mf );
	/*           pos,       number of constraints needed */

#ifdef DEBUGPP
fprintf( stderr, "Number of m-faces: %d\n", nbfaces );
#endif
	Matrix_Free( RaysDi );
	Matrix_Free( PiInv );
	Matrix_Free( PiTest );
	Matrix_Free( Pi );
	Matrix_Free( Xi );
	free( egalite );
	free( mf );
	SMFree( Sat );
	if( CEqualities && keep_dom==0 )
		Domain_Free( CEqualities );
	Domain_Free( D1 );
	Domain_Free( C1 );

	res = (Param_Polyhedron *) malloc (sizeof(Param_Polyhedron));
	res->nbV = nbPV;
        res->V = PV_Result;
	res->D = PDomains;
	return( res );
}

/*----------------------------------------------------------------------*/
/* Compute_PDomains                                                     */
/*      Finds the vertices that belong to distinct subdomains           */
/*----------------------------------------------------------------------*/
void Compute_PDomains(PD, nb_domains, working_space)
Param_Domain *PD;
int nb_domains;
int working_space;
{
   Polyhedron *dx, *d1, *d2;
   unsigned bx;
   int i, ix, nv;
   Param_Domain *p1, *p2, *p2prev, *PDNew;

   if (nb_domains==0)
	{
#ifdef DEBUGPP5
		fprintf( stderr, "No domains\n");
#endif
		return;
	}

   /*initialization */
   nv = (nb_domains - 1)/(8*sizeof(int)) + 1;
#ifdef DEBUGPP5
		fprintf( stderr, "nv = %d\n", nv);
#endif

   for( p1=PD, i=0, ix=0, bx=MSB ; p1 ; p1=p1->next, i++ )
   {
      p1->F = (unsigned *) malloc (nv * sizeof(unsigned));
					/* bit array */
      memset( p1->F, 0, nv * sizeof(unsigned) );
      p1->F[ix] |= bx;      /* set i'th bit to one */
      NEXT(ix, bx);
   }
#ifdef DEBUGPP5
		fprintf( stderr, "nb of vertices=%d\n", i);
#endif

   /* walk the PD list with two pointers */
   ix = 0; bx=MSB;
   for ( p1=PD; p1; p1=p1->next )
   {
      for ( p2prev=p1, p2=p1->next ; p2 ; p2prev=p2, p2=p2->next )
      {
         /* find intersection */
         dx = PDomainIntersection ( p1->Domain, p2->Domain, working_space);

         if ( !dx || emptyQ(dx) )
         {
#ifdef DEBUGPP5
				fprintf( stderr, "Empty dx (p1 inter p2). Continuing\n");
#endif
            if( dx )
            	Domain_Free( dx );
            continue;
         }
#ifdef DEBUGPP5
			fprintf( stderr, "Begin PDomainDifference\n");
			fprintf( stderr, "p1=");
			Polyhedron_Print( stderr, P_VALUE_FMT, p1->Domain );
			fprintf( stderr, "p2=");
			Polyhedron_Print( stderr, P_VALUE_FMT, p2->Domain );
#endif
         d1 = PDomainDifference( p1->Domain, p2->Domain, working_space);
         d2 = PDomainDifference( p2->Domain, p1->Domain, working_space);

#ifdef DEBUGPP5
			fprintf( stderr, "p1\\p2=");
			Polyhedron_Print( stderr, P_VALUE_FMT, d1 );
			fprintf( stderr, "p2\\p1=");
			Polyhedron_Print( stderr, P_VALUE_FMT, d2 );
			fprintf( stderr, "END PDomainDifference\n\n");
#endif
         if ( !d1 || emptyQ(d1) || d1->NbEq!=0 )
         {
#ifdef DEBUGPP5
				fprintf( stderr, "Empty d1\n");
#endif
            if (d1) Domain_Free( d1 );
            Domain_Free( dx );

            if ( !d2 || emptyQ(d2) || d2->NbEq!=0 )
            {
#ifdef DEBUGPP5
					fprintf( stderr, "Empty d2 (deleting)\n");
#endif
               /* dx = p1->Domain = p2->Domain */
               if (d2) Domain_Free( d2 );

               /* update p1 */
               for ( i=0; i<nv; i++ )
                  p1->F[i] |= p2->F[i];

               /* delete p2 */
               p2prev->next = p2->next;
               Domain_Free( p2->Domain );
               free( p2->F );
               free( p2 );
               p2 = p2prev;
            }
            else   /* d2 is not empty --> dx==p1->domain */
            {
#ifdef DEBUGPP5
					fprintf( stderr, "p2 replaced by d2\n");
#endif
               /* update p1 */
               for ( i=0; i<nv; i++ )
                  p1->F[i] |= p2->F[i];

               /* replace p2 with d2 */
               Domain_Free( p2->Domain );
               p2->Domain = d2;
            }
         }
         else /* d1 is not empty */
         {
            if ( !d2 || emptyQ(d2) || d2->NbEq!=0 )
            {
#ifdef DEBUGPP5
					fprintf( stderr, "p1 replaced by d1\n");
#endif
               if (d2) Domain_Free(d2);
               /* dx = p2->domain */
               Domain_Free(dx);

               /* update p2 */
               for ( i=0; i<nv; i++ )
                  p2->F[i] |= p1->F[i];

               /* replace p1 with d1 */
               Domain_Free( p1->Domain );
               p1->Domain = d1;
            }
            else /*d2 is not empty-->d1,d2,dx are distinct*/
            {
#ifdef DEBUGPP5
					fprintf( stderr, "Non-empty d1 and d2\nNew node created\n");
#endif
               /* create a new node for dx */
               PDNew = (Param_Domain *) malloc( sizeof(Param_Domain) );
               PDNew->F = (int *)malloc( nv*sizeof(int) );
               memset( PDNew->F, 0, nv*sizeof(int) );
               PDNew->Domain = dx;

               for ( i=0; i<nv; i++ )
                  PDNew->F[i] = p1->F[i] | p2->F[i];

               /* replace p1 with d1 */
               Domain_Free( p1->Domain );
               p1->Domain = d1;

               /* replace p2 with d2 */
               Domain_Free( p2->Domain );
               p2->Domain = d2;

               /* insert new node after p1 */
               PDNew->next = p1->next;
               p1->next = PDNew;
            }
         }
      }  /* end of p2 scan */
   } /* end of p1 scan */
}
					

/*----------------------------------------------------------------------*/
/* Polyhedron2Param_Vertices                                            */
/*      Computes the parameterized vertices and their def. domains      */
/*                                                                      */
/*      Inputs: Polyhedron * parametrized domain                        */
/*              int          number of parameters                       */
/*              int          Working Space (see polyhedron library)     */
/*	Output: Param_Vertices *   list of parametrized vertices        */
/*----------------------------------------------------------------------*/
Param_Polyhedron *Polyhedron2Param_Vertices( Din, Cin, working_space )
Polyhedron *Din;			/* polyhedron */
Polyhedron *Cin;			/* context polyhedron */
int working_space;			/* as usual... */
{
	Param_Polyhedron *res;
#ifdef DEBUGPP
fprintf(stderr,"Polyhedron2Param_Vertices algorithm starting at : %.2fs\n",
					(float)clock()/CLOCKS_PER_SEC);
#endif

	/********** Scan the m-faces      ***********/
	res = Find_m_faces( Din, Cin, 0, working_space );

#ifdef DEBUGPP
fprintf(stderr, "nb of points : %d\n",res->nbV);
#endif

#ifdef DEBUGPP
fprintf(stderr, "end main loop : %.2fs\n", (float)clock()/CLOCKS_PER_SEC);
#endif

	return( res );
}

/*----------------------------------------------------------------------*/
/* Param_Vertices_Free                                                  */
/*      Frees the memory allocated by a call to                         */
/*                                          Polyhedron2Param_Vertices() */
/*      Input :	list of parametrized vertices                           */
/*      Output : -                                                      */
/*----------------------------------------------------------------------*/
void Param_Vertices_Free( PV )
Param_Vertices *PV;
{
	Param_Vertices *n;

	while( PV )
	{
		n = PV->next;
		if (PV->Vertex) Matrix_Free( PV->Vertex );
		if (PV->Domain) Matrix_Free( PV->Domain );
		free( PV );

		PV = n;
	}
}


/*----------------------------------------------------------------------*/
/* Print_Vertex                                                         */
/*----------------------------------------------------------------------*/
void Print_Vertex( DST, V )
FILE *DST;
Matrix *V;
{
   int l, v;
   int first;
   Value g,t;

   fprintf(DST, "[" );
   for( l=0 ; l<V->NbRows ; ++l )
   {
      /* variables */
      first=1;
      fprintf(DST, " " );
      for( v=0 ; v<V->NbColumns-2 ; ++v )
      {
         if( value_notzero_p(V->p[l][v]) )
         {
				value_assign( t, Gcd( V->p[l][v], V->p[l][V->NbColumns-1] ));
            value_assign( g, value_abs(t) );

				if( value_posz_p( value_div( V->p[l][v], g ) ) )
				{
					if( !first ) fprintf(DST, "+" );
					if( value_notone_p(value_div( V->p[l][v], g )) )
						fprintf(DST, VALUE_FMT, value_div(V->p[l][v],g) );
				}
				else /* V->p[l][v]/g<0 */
				{
					if( value_mone_p(value_div(V->p[l][v],g)) )
						fprintf(DST, "-" );
					else
						fprintf(DST, VALUE_FMT, value_div(V->p[l][v],g) );
				}
				if( value_notone_p( value_div(V->p[l][V->NbColumns-1],g) ) )
				{
					fprintf(DST, "%c/", PCHAR+v+1 );
					fprintf(DST, VALUE_FMT,  value_div(V->p[l][V->NbColumns-1],g) );
				}
				else
					fprintf(DST, "%c", PCHAR+v+1 );
				first=0;
			}
		}

		/* constant */
		if( value_notzero_p(V->p[l][v]) || first )
		{
			if( value_posz_p(V->p[l][v]) && !first )
				fprintf(DST, "+" );
			value_assign( t, Gcd( V->p[l][v], V->p[l][V->NbColumns-1] ));
			value_assign( g, value_abs(t) );
			fprintf(DST, VALUE_FMT, value_div(V->p[l][v],g) );
			if( value_notone_p( value_div(V->p[l][V->NbColumns-1],g) ) )
			{
				fprintf(DST, "/"VALUE_FMT" ",
						value_div(V->p[l][V->NbColumns-1],g) );
			}
		}
		if (l<V->NbRows-1) fprintf(DST, ", ");
	}
	fprintf(DST, " ]");
}

/*----------------------------------------------------------------------*/
/* Print_Domain                                                         */
/*----------------------------------------------------------------------*/
void Print_Domain( DST, D )
FILE *DST;
Polyhedron *D;
{
   int l, v;
   int first;

   for( l=0 ; l<D->NbConstraints ; ++l )
   {
      fprintf(DST, "         " );
      first = 1;
      for( v=1 ; v<=D->Dimension ; ++v )
      {
         if( value_notzero_p(D->Constraint[l][v]) )
         {
            if( value_one_p(D->Constraint[l][v]) )
            {
               if( first ) fprintf(DST, "%c ", PCHAR+v );
               else fprintf(DST, "+ %c ", PCHAR+v );
            }
            else if( value_mone_p(D->Constraint[l][v]) )
					fprintf(DST, "- %c ", PCHAR+v );
            else
            {
               if( value_pos_p(D->Constraint[l][v]) && !first )
						fprintf(DST, "+ " );
               fprintf(DST, VALUE_FMT"%c ", D->Constraint[l][v], PCHAR+v );
            }
            first = 0;
         }
      }
      if( value_notzero_p(D->Constraint[l][v]) )
      {
         if( value_pos_p(D->Constraint[l][v]) && !first )
				fprintf(DST, "+" );
         fprintf(DST, " "VALUE_FMT, D->Constraint[l][v] );
      }
      fprintf(DST, (value_notzero_p(D->Constraint[l][0]))?" >= 0":" = 0" );
      fprintf(DST, "\n" );
   }
   fprintf(DST, "\n");
}

/*----------------------------------------------------------------------*/
/* Param_Vertices_Print                                                 */
/*      prints a list of parametrized vertices in a comprehendible      */
/*               format on stdout. Parameters are named M, N, ...       */
/*      Input :	list of parametrized vertices                           */
/*      Output : <prints to stdout>                                     */
/*----------------------------------------------------------------------*/
void Param_Vertices_Print( DST, PV )
FILE *DST;
Param_Vertices *PV;
{
   Polyhedron *a;

   while( PV )
   {
      fprintf(DST, "Vertex :\n" );
      Print_Vertex(DST, PV->Vertex );

      /* pour le domaine : */
      fprintf(DST, "   If :\n" );
      a = Constraints2Polyhedron(PV->Domain, 200);
      Print_Domain(DST, a );
      Domain_Free( a );

      PV = PV->next;
   }
}

/*----------------------------------------------------------------------*/
/* Polyhedron2Param_Domain                                              */
/*      Computes the parameterized vertices and their def. domains      */
/*      Input : Polyhedron *       parametrized domain                  */
/*              int                number of parameters                 */
/*              int                Working Space(see polyhedron library)*/
/*      Output : Param_Domain *    list of parametrized polyhedra       */
/*----------------------------------------------------------------------*/
Param_Polyhedron *Polyhedron2Param_Domain( Din, Cin, working_space )
Polyhedron *Din;			/* polyhedron */
Polyhedron *Cin;			/* polyhedron */
int working_space;			/* as usual... */
{
	Param_Polyhedron *res;
	Param_Domain *D;

#ifdef DEBUGPP
fprintf(stderr,"Polyhedron2Param_Polyhedron algorithm starting at : %.2fs\n",
			(float)clock()/CLOCKS_PER_SEC);
#endif

	/* Find the m-faces, keeping the corresponding domains */
	/* in the linked list PDomains */
	res = Find_m_faces( Din, Cin, 1, working_space );

#ifdef DEBUGPP
if(res) fprintf(stderr, "Number of vertices : %d\n",res->nbV);
fprintf(stderr, "Vertices found at : %.2fs\n", (float)clock()/CLOCKS_PER_SEC);
#endif

	/* processing of PVResult and PDomains */
	if( res && Cin->Dimension>0 )		/* at least 1 parameter */
		Compute_PDomains(res->D, res->nbV, working_space);

	if( CEqualities )
		for( D=res->D ; D ; D=D->next )
			D->Domain = Add_CEqualities( D->Domain );
#ifdef DEBUGPP
fprintf(stderr, "domains found at : %.2fs\n", (float)clock()/CLOCKS_PER_SEC);
#endif

	return( res );
}

/*----------------------------------------------------------------------*/
/* Param_Domain_Free                                                    */
/*      Frees the memory allocated by a call to                         */
/*                                          Polyhedron2Param_Domain     */
/*      Input :	parametrized polyhedron                                 */
/*      Output : -                                                      */
/*----------------------------------------------------------------------*/
void Param_Domain_Free( PP )
Param_Domain *PP;
{
	Param_Domain *n;

	while( PP )
	{
		free( PP->F );
		Domain_Free( PP->Domain );

		n = PP->next;
		free( PP );
		PP = n;
	}
}

/*----------------------------------------------------------------------*/
/* Param_Polyhedron_Free                                                */
/*      Frees the memory allocated by a call to                         */
/*         Polyhedron2Param_Vertices or Polyhedron2Param_Domain         */
/*      Input :	parametrized polyhedron                                 */
/*      Output : -                                                      */
/*----------------------------------------------------------------------*/
void Param_Polyhedron_Free( P )
Param_Polyhedron *P;
{
	if (!P) return;
	Param_Vertices_Free( P->V );
	Param_Domain_Free( P->D );
	free(P);
}

