
#include <stdlib.h>
#include "polylib.h"

mpz_t *Vector_Mp_Alloc(int length)
{
  mpz_t *vect;
  int i;
  
  vect = (mpz_t *) malloc(sizeof(mpz_t) * length);
  for (i = 0; i < length; i++)
    mpz_init(vect[i]);
  return vect;
}

void Vector_Mp_Free(mpz_t *vect, int length)
{
  int i;
  
  for (i = 0; i < length; i++)
    mpz_clear(vect[i]);
  free(vect);
}

Matrix_Mp *Matrix_Mp_Alloc(int rows, int cols)
{
  Matrix_Mp *mat;
  int i;
  
  mat = (Matrix_Mp *) malloc(sizeof(Matrix_Mp));
  mat->NbRows = rows;
  mat->NbColumns = cols;
  mat->p = (mpz_t **) malloc(sizeof(mpz_t *) * rows);
  for (i = 0; i < rows; i++)
    mat->p[i] = Vector_Mp_Alloc(cols);

  return mat;
}

void Matrix_Mp_Free(Matrix_Mp *mat)
{
  int i;
  
  for (i = 0; i < mat->NbRows; i++)
    Vector_Mp_Free(mat->p[i], mat->NbColumns);
}

void Matrix_2_Matrix_Mp(Matrix_Mp *dest, Matrix *src)
{
  int i, j;
  
  for (i = 0; i < dest->NbRows; i++)
    for (j = 0; j < dest->NbColumns; j++)
      mpz_set_si(dest->p[i][j], src->p[i][j]);
}

void Matrix_Mp_2_Matrix(Matrix *dest, Matrix_Mp *src)
{
  int i, j;
  
  for (i = 0; i < dest->NbRows; i++)
    for (j = 0; j < dest->NbColumns; j++)
      dest->p[i][j] = mpz_get_si(src->p[i][j]);
}

void Vector_Gcd_Mp(mpz_t gcd, mpz_t *p, mpz_t *q, int length)
{
  mpz_t *cq, *cp;
  int Index_Min = 0, Not_Zero, i;
  
  if (!length)
    {
      mpz_set_si(gcd, 1);
      return;
    }
  
  /* copy abs values fron cp to working vector cq */
  for (cq=q, cp=p, i=0 ; i<length ; i++)
     mpz_abs(*cq++, *cp++);

  do
    {
      /* Compute min of all entrys != 0 in vector */
      mpz_set_si(gcd, 1);
      Index_Min = -1;
      for (i=0, cq=q; i<length; i++, cq++)
	if (mpz_cmp_si(*cq, 0) != 0 && /* not a zero value */
	    (Index_Min == -1 || mpz_cmp(*cq, gcd) < 0) )
	     /* smaller than min or min not yet initialized */
	  { mpz_set(gcd, *cq); Index_Min = i; }

      if (mpz_cmp_si(gcd, 1) != 0)
	{ /* min greater than 1 */
	  Not_Zero = 0;
	  for (i=0, cq=q ; i<length; i++, cq++)
	    if (i != Index_Min)
	      {
		mpz_mod(*cq, *cq, gcd);
		if (mpz_cmp_si(*cq, 0))
		  Not_Zero = 1;  /* not dividable by minimum */
	      }
	} 
      else 
	  break;
    } 
  while (Not_Zero);
} /* Vector_Gcd_Mp */

void Vector_Combine_Mp(mpz_t *p1, mpz_t *p2, mpz_t *p3, mpz_t lambda, mpz_t mu,
		       int length)
{
  mpz_t *cp1, *cp2, *cp3, l, m;
  int i;

  mpz_init(l);
  mpz_init(m);
  
  cp1=p1;
  cp2=p2;
  cp3=p3;
  for (i=0;i<length;i++)
    {
      mpz_mul(l, lambda, *cp1);
      mpz_mul(m, mu, *cp2);
      mpz_add(*cp3, l, m);
      cp1++; cp2++; cp3++;
    }

  mpz_clear(l);
  mpz_clear(m);
} /* Vector_Combine */

/* gcd reduce a vector */
/* making the last element positive is *not* OK for equalities */
void Vector_Normalize_Mp(mpz_t *p, mpz_t *tmp, int length)
{
  mpz_t g;
  int i;
  
  mpz_init(g);
  Vector_Gcd_Mp(g, p, tmp, length);	/* g is always positive */
  if (mpz_cmp_si(g,1) != 0)
    for (i=0; i<length; i++)
      mpz_div(p[i], p[i], g);		/* normalize */
  mpz_clear(g);
} /* Vector_Normalize */

/* Combine_Mp --
     (p3) = linear combination of (p1) and (p2)
     such that (p3[pos]) is zero.
     First element of each vector is status word and is not
     changed in (p3).  (pos) may be 0 however.
     (length) does not include status word.1
*/
static void Combine_Mp(mpz_t *p1, mpz_t *p2, mpz_t *p3, int pos, 
		       mpz_t *temp, int length)
{
  mpz_t a, a1, a2;

  mpz_init(a);
  mpz_init_set(a1, p1[pos]);
  mpz_init_set(a2, p2[pos]);
  mpz_gcd(a, a1, a2);
  mpz_abs(a, a);
  mpz_div(a1, a1, a);
  mpz_div(a2, a2, a);
  mpz_neg(a1, a1);
  Vector_Combine_Mp(p1+1, p2+1, p3+1, a2, a1, length);
  Vector_Normalize_Mp(p3+1, temp, length);
  mpz_clear(a);
  mpz_clear(a1);
  mpz_clear(a2);
} /* Combine_Mp */

/* Gauss_Mp --
   Computes a minimal system of equations.
   performs Gaussian elimination, detects contridiction.
   Given Mat, a matrix of constraints in which the first NbEq
      constraints are equations,
   Given NbEq, the number of equations in Mat
   Given Dimension, the dimension of the homonogenous system,
   Produce, col_of_rank, the vector of independent variables,
   Uses temp, a temporary vector,
   Returns the rank, if Rank=Dimension ... is there a contradiction
 */
int Gauss_Mp(Matrix_Mp *Mat, int NbEq, int Dimension, mpz_t *temp)
{
  mpz_t gcd, *cp;
  int i, j, k, pivot, Rank, *col_of_rank;
  mpz_t *temp_ptr;
  
  mpz_init(gcd);
  col_of_rank = malloc(sizeof(int) * Dimension);
  Rank=0;
  for (j=1; j<=Dimension; j++)         	/* for each column (except status) */ 
  { for (i=Rank; i<NbEq; i++)           /* starting at diagonal, look down */
      if (mpz_cmp_si(Mat->p[i][j],0)) break;  /* to find the first non zero */
    if (i!=NbEq)                        /* was one found ? */
    { 
      if (i!=Rank)                      /* was it found below the diagonal? */
	{ /* exchange row rank with row i, except the first entry (col 0) */
	  temp_ptr = Mat->p[Rank]; 
	  Mat->p[Rank] = Mat->p[i]; 
	  Mat->p[i] = temp_ptr;
	  mpz_set(temp[0], Mat->p[Rank][0]);
	  mpz_set(Mat->p[Rank][0], Mat->p[i][0]);
	  mpz_set(Mat->p[i][0], temp[0]);
	}

      /* Normalize the pivot row */
      Vector_Gcd_Mp(gcd, Mat->p[Rank]+1, temp, Dimension);
      if (mpz_cmp_si(gcd, 2) >= 0)
      { cp = &Mat->p[Rank][1];
        for (k=0; k<Dimension; k++, cp++) 
	  mpz_div(*cp, *cp, gcd);
      }
      if (mpz_cmp_si(Mat->p[Rank][j],0)<0)
      { cp = Mat->p[Rank]+1;
        for (k=0; k<Dimension; k++, cp++) 
	  mpz_neg(*cp, *cp);
      }
      /* end normalize */

      pivot=i;
      for (i=pivot+1; i<NbEq; i++)      /* zero out the rest of the column */
      { if (mpz_cmp_si(Mat->p[i][j],0)!=0)
          Combine_Mp(Mat->p[i], Mat->p[Rank], Mat->p[i], j, temp, Dimension);
      }
      col_of_rank[Rank]=j;
      Rank++;
    }
  } /* end of Gauss elimination */

  /* Back Substitution -- normalizes the system of equations */
  for (k=Rank-1; k>=0; k--)
  { j = col_of_rank[k];
    /* Normalize the equations */
    for (i=0; i<k; i++)
    { if (mpz_cmp_si(Mat->p[i][j],0) !=0 )
        Combine_Mp(Mat->p[i], Mat->p[k], Mat->p[i], j, temp, Dimension);
    }
    /* Normalize the inequalities */
    for (i=NbEq; i<Mat->NbRows; i++)
    { if (mpz_cmp_si(Mat->p[i][j],0)!=0 )
        Combine_Mp(Mat->p[i], Mat->p[k], Mat->p[i], j, temp, Dimension);
    }
  }
  free(col_of_rank);
  return Rank;
}
