/* matrix.c 
     COPYRIGHT
          Both this software and its documentation are

              Copyright 1993 by IRISA /Universite de Rennes I - France,
              Copyright 1995,1996 by BYU, Provo, Utah
                         all rights reserved.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef MAC_OS
  #define abs __abs
#endif

#include "arithmetique.h"
#include "polylib.h"

/* #define DEBUG 1 */



Matrix *Matrix_Alloc(NbRows, NbColumns)
unsigned NbRows, NbColumns;
{
	Matrix *Mat;
	Value **q, *p;
	int i;

	Mat=(Matrix *)malloc(sizeof(Matrix));
	if( !Mat )
	{	errormsg1("Matrix_Alloc", "outofmem", "out of memory space");
		return 0;
	}

	Mat->NbRows=NbRows;
	Mat->NbColumns=NbColumns;
	if( NbRows==0 )
	{
		Mat->p=0;
		Mat->p_Init=0;
	}
	else
	{
		Mat->p=q=(Value **)malloc(NbRows * sizeof(Value *));
		if( !Mat->p )
		{	errormsg1("Matrix_Alloc", "outofmem", "out of memory space");
			return 0;
		}
		if( NbColumns==0 )
		{
			Mat->p_Init=0;
		}
		else
		{
			Mat->p_Init=p=(Value *)malloc(NbRows * NbColumns * sizeof(Value));
			if( !Mat )
			{	errormsg1("Matrix_Alloc", "outofmem", "out of memory space");
				return 0;
			}
			for (i=0;i<NbRows;i++) 
			{
			  *q++=p;
			  p+=NbColumns;
			}
		}
	}
	return Mat;
} /* Matrix_Alloc */



void  Matrix_Free(Mat)
Matrix *Mat;
{
	free((char *) Mat->p_Init);
	free((char *) Mat->p);
	free((char *) Mat);
} /* Matrix_Free */



void  Matrix_Print(Dst, Format, Mat)
FILE* Dst;
char *Format;
Matrix *Mat;
{
	Value *p;
	int i, j;
	unsigned NbRows, NbColumns;

	fprintf(Dst,"%d %d\n", NbRows=Mat->NbRows, NbColumns=Mat->NbColumns);
	for (i=0;i<NbRows;i++) 
	{
		p=*(Mat->p+i);
		for (j=0;j<NbColumns;j++)
		{
			if( !Format )
				fprintf(Dst, " "P_VALUE_FMT" ", *p++);
			else
				fprintf(Dst, Format, *p++);
		}
		fprintf(Dst, "\n");
	}
} /* Matrix_Print */




/* a '#' in the first column is a comment line */
Matrix *Matrix_Read( void )
{
  Matrix *Mat;
  Value *p;
  int i, j, n;
  unsigned NbRows, NbColumns;
  char *c, s[128];

	while( fgets(s, 128, stdin)==0 )
		;
  while ( (*s=='#' || *s=='\n') || (sscanf(s, "%d %d", &NbRows, &NbColumns)<2) )
    fgets(s, 128, stdin);

  Mat = Matrix_Alloc(NbRows, NbColumns);
	if(!Mat)
	{	errormsg1("Matrix_Read", "outofmem", "out of memory space");
		return 0;
	}
  p = Mat->p_Init;
  for (i=0;i<NbRows;i++)
  {
    do
	 	c = fgets(s, 128, stdin);
    while(c && (*c=='#'  || *c=='\n') );
    if (!c)
    {  errormsg1( "Matrix_Read", "baddim", "not enough rows" );
       break;
    }
    for (j=0;j<NbColumns;j++)
    { if (sscanf(c, VALUE_FMT"%n", p++, &n)==0)
      {  errormsg1( "Matrix_Read", "baddim", "not enough columns" );
         break;
      }
      c += n;
    }
  }
  return Mat;
} /* Matrix_Read */





/* basic hermite engine */
static int hermite(H, U, Q)
Matrix *H, *U, *Q;
{
	int nc, nr, i, j, k, rank, reduced, pivotrow;
	Value pivot, x;
	Value *tmp, *tmp2;

   /*                     T                     -1   T */
   /* Computes form: A = Q H  and U A = H  and U  = Q  */

   if (!H)
   {  errormsg1("Domlib", "nullH", "hermite: ? Null H");
      return -1;
   }
   nc = H->NbColumns;
   nr = H->NbRows;
   tmp = (Value *) malloc(nc * sizeof(Value));
   tmp2 = (Value *) malloc(nr * sizeof(Value));
   if (!tmp ||!tmp2)
   {  errormsg1("Domlib", "outofmem", "out of memory space");
      return -1;
   }

#ifdef DEBUG
  fprintf(stderr,
  "Start  -----------\n");
  Matrix_Print(stderr,0, H);
#endif

   for (k=0, rank=0; k<nc && rank<nr; k=k+1)
   {  reduced = 1;	/* go through loop the first time */
#ifdef DEBUG
  fprintf(stderr, "Working on col %d.  Rank=%d ----------\n", k+1, rank+1);
#endif
      while (reduced)
      {
			reduced=0;

			/* 1. find pivot row */
			value_assign( pivot, value_abs(H->p[rank][k]) );
										/* the kth-diagonal element */
         pivotrow = rank;
         /* find the row i>rank with smallest nonzero element in col k */
         for (i=rank+1; i<nr; i++)
         {
				value_assign( x,value_abs(H->p[i][k]) );
            if ( value_notzero_p(x) &&
						( value_lt(x,pivot) || value_zero_p(pivot) ) )
				{
					value_assign( pivot, x );
					pivotrow = i;
				}
         }

         /* 2. bring pivot to diagonal (exchange rows pivotrow and rank) */
         if (pivotrow != rank)
         {  Vector_Exchange(H->p[pivotrow], H->p[rank], tmp, nc );
            if (U)
              Vector_Exchange(U->p[pivotrow], U->p[rank], tmp2, nr );
            if (Q)
              Vector_Exchange(Q->p[pivotrow], Q->p[rank], tmp2, nr );
#ifdef DEBUG
fprintf(stderr,"Exchange rows %d and %d  -----------\n", rank+1, pivotrow+1);
Matrix_Print(stderr,0, H);
#endif
         }
         value_assign( pivot, H->p[rank][k] );	/* actual ( no abs() ) pivot */

         /* 3. Invert the row 'rank' if pivot is negative */
         if ( value_neg_p(pivot) )
         {  value_oppose( pivot ); /* pivot = -pivot */
            for (j=0; j<nc; j++)
					value_oppose( H->p[rank][j] );
					 /* H->p[rank][j] = -(H->p[rank][j]); */
            if (U)
					for (j=0; j<nr; j++)
						value_oppose( U->p[rank][j] );
						/* U->p[rank][j] = -(U->p[rank][j]); */
            if (Q)
					for (j=0; j<nr; j++)
						value_oppose( Q->p[rank][j] );
						/* Q->p[rank][j] = -(Q->p[rank][j]); */
#ifdef DEBUG
  fprintf(stderr,
  "Negate row %d  -----------\n", rank+1);
  Matrix_Print(stderr,0, H);
#endif
         }

         if ( value_notzero_p(pivot) )
         {  /* 4. Reduce the column modulo the pivot */
            /*    This eventually zeros out everything below the */
            /*    diagonal and produces an upper triangular matrix */
            for (i=rank+1; i<nr; i++)
            {  value_assign( x, H->p[i][k] );
					if ( value_notzero_p(x) )
					{  /* floor[integer division] (corrected for neg x) */
                  if (value_neg_p(x) && value_notzero_p(value_mod(x,pivot)))
						{
							/* x=(x/pivot)-1; */
							value_division( x, pivot );
							value_decrement( x );
						}	
						else
						{
							value_division( x, pivot );
							/* x/=pivot; */
						}
                  for (j=0; j<nc; j++)
							value_substract( H->p[i][j], value_mult(x,H->p[rank][j]) );
							/* H->p[i][j] -= (x * H->p[rank][j]); */
                  if (U)
							for (j=0; j<nr; j++)
								value_substract( U->p[i][j],
										value_mult(x, U->p[rank][j]) );
								/* U->p[i][j] -= (x * U->p[rank][j]); */
                  if (Q)
							for (j=0; j<nr; j++)
								value_addto( Q->p[rank][j], value_mult(x,Q->p[i][j]) );
								/* Q->p[rank][j] += (x * Q->p[i][j]); */
                  reduced = 1;
#ifdef DEBUG
  fprintf(stderr,
  "row %d = row %d - %d row %d -----------\n", i+1, i+1, x, rank+1);
  Matrix_Print(stderr,0, H);
#endif
               } /* if (x) */
            } /* for (i) */
         } /* if (pivot != 0) */
      } /* while (reduced) */

      /* Last finish up this column */
      /* 5. Make pivot column positive (above pivot row) */
      /*    x should be zero for i>k */
      if ( value_notzero_p(pivot) )
      {  for (i=0; i<rank; i++)
         {  value_assign( x, H->p[i][k] );
            if ( value_notzero_p(x) )
            {  /* floor[integer division] (corrected for neg x) */
               if ( value_neg_p(x) && value_notzero_p(value_mod(x,pivot)) )
					{
						value_division( x, pivot );
						value_decrement( x );
						/* x=(x/pivot)-1; */
					}
					else
						value_division( x, pivot );
						/* x/=pivot; */

               for (j=0; j<nc; j++)
						value_substract( H->p[i][j], value_mult(x,H->p[rank][j]) );
						/* H->p[i][j] -= x * H->p[rank][j]; */
               if (U)
                 for (j=0; j<nr; j++)
							value_substract( U->p[i][j], value_mult(x,U->p[rank][j]) );
							/* U->p[i][j] -= x * U->p[rank][j]; */
               if (Q)
                 for (j=0; j<nr; j++)
							value_addto( Q->p[rank][j], value_mult(x,Q->p[i][j]) );
							/* Q->p[rank][j] += x * Q->p[i][j]; */
#ifdef DEBUG
  fprintf(stderr,
  "row %d = row %d - %d row %d -----------\n", i+1, i+1, x, rank+1);
  Matrix_Print(stderr,0, H);
#endif
            } /* if (x) */
         } /* for (i) */
         rank++;
      } /* if (pivot!=0) */
   } /* for (k) */
   free(tmp2);
   free(tmp);
   return rank;
}

void right_hermite(A, Hp, Up, Qp)
Matrix *A, **Up, **Qp, **Hp;
{
	Matrix *H, *Q, *U;
	int i, j, nr, nc, rank;
	Value tmp;

   /* Computes form: A = QH , UA = H */

   nc = A->NbColumns;
   nr = A->NbRows;

   /* H = A */
   *Hp = H = Matrix_Alloc(nr, nc);
   if (!H)
   {  errormsg1("DomRightHermite", "outofmem", "out of memory space");
      return;
   }

   Vector_Copy(A->p_Init, H->p_Init, nr*nc );

   /* U = I */
   if (Up)
   {  *Up = U = Matrix_Alloc(nr, nr);
      if (!U)
      {  errormsg1("DomRightHermite", "outofmem", "out of memory space");
         return;
      }
      Vector_Init(U->p_Init, nr*nr);             /* zero's */
      for (i=0; i<nr; i++)         /* with diagonal of 1's */
			value_assign( U->p[i][i], VALUE_ONE );
   }
   else
		U = (Matrix *)0;

   /* Q = I */
   /* Actually I compute Q transpose... its easier */
   if (Qp)
   {  *Qp = Q = Matrix_Alloc(nr, nr);
      if (!Q)
      {  errormsg1("DomRightHermite", "outofmem", "out of memory space");
         return;
      }
      Vector_Init(Q->p_Init, nr*nr);            /* zero's */
      for (i=0; i<nr; i++)        /* with diagonal of 1's */
			value_assign( Q->p[i][i], VALUE_ONE );
   }
   else
		Q = (Matrix *)0;

   rank = hermite(H,U,Q);
   /* Q is returned transposed */

   /* Transpose Q */
   if (Q)
   {  for (i=0; i<nr; i++)
      {  for (j=i+1; j<nr; j++)
         {
				value_assign( tmp, Q->p[i][j] );
				value_assign( Q->p[i][j], Q->p[j][i] );
				value_assign( Q->p[j][i], tmp );
         }
      }
   }
}

void left_hermite(A, Hp, Up, Qp)
Matrix *A, **Hp, **Qp, **Up;
{
	Matrix *H, *HT, *Q, *U;
	int i, j, nc, nr, rank;
	Value tmp;

   /* Computes left form: A = HQ , AU = H , 
                           T    T T    T T   T
        using right form  A  = Q H  , U A = H */

   nr = A->NbRows;
   nc = A->NbColumns;

   /* HT = A transpose */
   HT = Matrix_Alloc(nc, nr);
   if (!HT)
   {  errormsg1("DomLeftHermite", "outofmem", "out of memory space");
      return;
   }
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         value_assign( HT->p[j][i], A->p[i][j] );

   /* U = I */
   if (Up)
   {  *Up = U = Matrix_Alloc(nc, nc);
      if (!U)
      {  errormsg1("DomLeftHermite", "outofmem", "out of memory space");
         return;
      }
      Vector_Init(U->p_Init, nc*nc);            /* zero's */
      for (i=0; i<nc; i++)        /* with diagonal of 1's */
			value_assign( U->p[i][i], VALUE_ONE );
   }
   else U=(Matrix *)0;

   /* Q = I */
   if (Qp)
   {  *Qp = Q = Matrix_Alloc(nc, nc);
      if (!Q)
      {  errormsg1("DomLeftHermite", "outofmem", "out of memory space");
         return;
      }
      Vector_Init(Q->p_Init, nc*nc);            /* zero's */
      for (i=0; i<nc; i++)        /* with diagonal of 1's */
			value_assign( Q->p[i][i], VALUE_ONE );
   }
   else Q=(Matrix *)0;

   rank = hermite(HT, U, Q);

   /* H = HT transpose */
   *Hp = H = Matrix_Alloc(nr, nc);
   if (!H)
   {  errormsg1("DomLeftHermite", "outofmem", "out of memory space");
      return;
   }
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         value_assign( H->p[i][j], HT->p[j][i] );
   Matrix_Free(HT);

   /* Transpose U */
   if (U)
   {  for (i=0; i<nc; i++)
      {  for (j=i+1; j<nc; j++)
         {  value_assign( tmp, U->p[i][j] );
            value_assign( U->p[i][j], U->p[j][i] );
            value_assign( U->p[j][i], tmp );
         }
      }
   }
}






/*----------------------------------------------------------------------*/
/* MatInverse                                                           */
/*      Matrix Inversion (integer -> rational).                         */
/*      Input : Matrix M  k*k >>WARNING: This matrix is modified        */
/*	Output : Matrix MInv  k*(k+1)  >>Must be preallocated           */
/*          The last column is the common denominator for each line     */
/*      Returns : 1 if ok, 0 if M not invertible.                       */
/*----------------------------------------------------------------------*/
int MatInverse( M, MInv )
Matrix *M, *MInv;
{
	int p, k, li, c;
	Value *tmp;
	Value x, g, piv;
	Value m1,m2;

	if( M->NbRows != M->NbColumns )
	{
		fprintf( stderr, "Trying to invert a non-square matrix !\n");
		return 0;
	}
	k = M->NbRows;
	tmp = (Value *) malloc( (k+1)*sizeof(Value) );

	/* initialise MInv */
	Vector_Init( MInv->p[0], k*(k+1) );
	for( p=0 ; p<k ; ++p )
	{
		value_assign( MInv->p[p][p], VALUE_ONE);	/* diagonal */
		value_assign( MInv->p[p][k], VALUE_ONE);	/* denum */
	}

	/* gauss-jordan on the 2 matrices in parallel. */
	for( p=0 ; p<k ; ++p )
	{
		if ( value_zero_p(M->p[p][p]))	/* need a new pivot */
		{	for( li=p ; li<k ; ++li )	/* search for one */
				if( value_notzero_p(M->p[li][p]) )
					break;
			if( li==k )		/* no pivot found... */
				return 0;

			/* exchange of the 2 lines (in the 2 matrices...)*/
			for( c=0 ; c<k ; ++c )
			{
				value_assign( x, M->p[li][c] );
				value_assign( M->p[li][c], M->p[p][c] );
				value_assign( M->p[p][c], x );

				value_assign( x, MInv->p[li][c] );
				value_assign( MInv->p[li][c], MInv->p[p][c] );
				value_assign( MInv->p[p][c], x );
			}
		}

		/* zero-out everything but the pivot */
		for( li=0 ; li<k ; ++li )
		{
			if (li==p) continue;	/* skip the pivot */
			value_assign( x, M->p[li][p] );
			if( value_notzero_p(x) )
			{
				value_assign( piv, M->p[p][p] );
				value_assign( g, Gcd( x, piv ) );
				if ( value_notone_p(g) )
				{
					value_division( x, g );
					value_division( piv, g );
				}
				for( c=((li>p)?p:0) ; c<k ; ++c )
				{
					value_assign( m1, value_mult( piv,M->p[li][c] ) );
					value_assign( m2, value_mult( x, M->p[p][c] ) );
					value_assign( M->p[li][c], value_minus( m1, m2 ) );
				}
				for( c=0 ; c<k ; ++c )
				{
					value_assign( m1, value_mult( piv, MInv->p[li][c] ) );
					value_assign( m2, value_mult( x, MInv->p[p][c] ) );
					value_assign( MInv->p[li][c], value_minus( m1, m2 ) );
				}

				/* simplification if possible */
				value_assign( m1, Vector_Gcd(&MInv->p[li][0],tmp,k) );
				value_assign( m2, Vector_Gcd(&M->p[li][0],tmp,k) );
				value_assign( g,Gcd( m1,m2 ) );
				if( value_notone_p(g) )
				{
					for( c=0 ; c<k ; ++c )
					{
						value_division( M->p[li][c], g );
						value_division( MInv->p[li][c], g );
					}
				}
			}
		}
	}

	/* ok... Now simplify each line (common denom) */
	for( li=0 ; li<k ; ++li )
	{
		value_assign( MInv->p[li][k], M->p[li][li] );
		Vector_Normalize_Positive( &MInv->p[li][0], tmp, k+1, k); /*denom>0*/
	}
	free(tmp);

	return 1;
}

/*----------------------------------------------------------------------*/
/* rat_prodmat                                                          */
/*      Matrix Product (integer X rational -> rational)                 */
/*      Outputs: S result (rational matrix) >> must be preallocated     */
/*      Inputs:  X (integer matrix)                                     */
/*               P (rational matrix)                                    */
/*----------------------------------------------------------------------*/
void rat_prodmat( S, X, P )
Matrix *S, *X, *P;
{
	int i,j,k;
	int denom = P->NbColumns - 1;
	Value dcom, olddc, g, d, s, *tmp;
	Value m1,m2;

	tmp = (Value *) malloc ( P->NbColumns * sizeof(Value) );

	/* compute the lcm of the denominators */
	value_assign( dcom, P->p[0][denom] );	/* denominateur commun */
	for( k=1 ; k<P->NbRows ; ++k )
	{
		value_assign( olddc, dcom );
		value_assign( d, P->p[k][denom] );
		value_assign( g, Gcd( dcom, d ) );
		value_assign( m1, value_div(d,g) );
		value_product( dcom, m1 );

/* test inutile : effectue dans value_product */
/*		if( value_ne( value_mult(value_div(dcom,d),g), olddc ) )
		{
			fprintf( stderr, "Multiplication overflow in the computing of Si!\n" );
			value_assign( dcom, VALUE_ZERO );
			break;
		}
*/
	}

	for( i=0 ; i<X->NbRows ; ++i )
		for( j=0 ; j<P->NbColumns-1 ; ++j )
		{
			value_assign( s, VALUE_ZERO );
			for( k=0 ; k<P->NbRows ; ++k )
				if( value_zero_p(dcom) )
					value_addto( s, value_mult(X->p[i][k], P->p[k][j]) );
				else
				{
					value_assign( m1, value_mult( X->p[i][k], P->p[k][j] ) );
					value_assign( m2, value_div(dcom, P->p[k][denom]) );
					value_addto( s, value_mult(m1,m2) );
				}

			value_assign( S->p[i][j], s );
		}

	for( i=0 ; i<S->NbRows ; ++i )
	{
		value_assign( S->p[i][denom], dcom );
		/* normalize so that last element >=0 */
		Vector_Normalize_Positive( &S->p[i][0], tmp, S->NbColumns,
				S->NbColumns-1 );
	}
	free(tmp);
}






void Matrix_Vector_Product(mat, p1, p2)
Matrix *mat;
Value *p1, *p2;
{
	int NbRows, NbColumns, i, j;
	Value **cm, *q, *cp1, *cp2;

	NbRows=mat->NbRows;
	NbColumns=mat->NbColumns;

	cm = mat->p;
	cp2 = p2;
	for (i=0;i<NbRows;i++)
	{
		q = *cm++;
		cp1 = p1;
		value_assign( *cp2, value_mult(*q,*cp1) );
		q++;
		cp1++;
		/* *cp2=*q++ * *cp1++; */
		for (j=1;j<NbColumns;j++)
		{
			value_addto( *cp2, value_mult(*q, *cp1) );
			q++;
			cp1++;
			/* *cp2+= *q++ * *cp1++; */
		}
		cp2++;
	}
} /* Matrix_Vector_Product */


void Vector_Matrix_Product(p1, mat, p2)
Matrix *mat;
Value *p1, *p2;
{
	int NbRows, NbColumns, i, j;
	Value **cm, *cp1, *cp2;

	NbRows=mat->NbRows;
	NbColumns=mat->NbColumns;
	cp2 = p2;
	cm  = mat->p;
	for (j=0;j<NbColumns;j++)
	{
		cp1 = p1;
		value_assign( *cp2, value_mult(*(*cm+j), *cp1) );
		cp1++;
		/* *cp2= *(*cm+j) * *cp1++; */
		for (i=1;i<NbRows;i++)
		{
			value_addto( *cp2, value_mult(*(*(cm+i)+j), *cp1) );
			cp1++;
			/* *cp2+=*(*(cm+i)+j) * *cp1++; */
		}
		cp2++;
	}
} /* Vector_Matrix_Product */


/* mat3 must be allocated. */
void Matrix_Product(mat1, mat2, mat3)
Matrix *mat1, *mat2, *mat3;
{
	int Size, i, j, k;
	unsigned NbRows, NbColumns;
	Value **q1, **q2, *p1, *p3, sum;

	NbRows    = mat1->NbRows;
	NbColumns = mat2->NbColumns;
	/* mat3 = Matrix_Alloc(NbRows, NbColumns); */
	Size      = mat1->NbColumns;
	if (mat2->NbRows!=Size || mat3->NbRows!=NbRows || mat3->NbColumns!=NbColumns)
	{
		fprintf(stderr, "? Matrix_Product : incompatable matrix dimension\n");
		return;
	}
  
	p3 = mat3->p_Init;
	q1 = mat1->p;
	q2 = mat2->p;
	for (i=0;i<NbRows;i++)
	{
		for (j=0;j<NbColumns;j++)
		{
			p1 = *(q1+i);
			value_assign( sum, VALUE_ZERO );
			for (k=0;k<Size;k++)
			{
				value_addto( sum, value_mult(*p1, *(*(q2+k)+j)) );
				p1++;
				/* sum+=*p1++ * *(*(q2+k)+j); */
			}
			value_assign( *p3, sum );
			p3++;
			/* *p3++=sum; */
		}
	}
} /* Matrix_Product */

