/***********************************************************************/
/*                Parametrized polyhedra V3.08                         */
/*                copyright 1995, 1997, Vincent Loechner               */
/*                copyright 1996, 1997, Doran Wilde                    */
/*       Permission is granted to copy, use, and distribute            */
/*       for any commercial or noncommercial purpose under the terms   */
/*       of the GNU General Public license, version 2, June 1991       */
/*       (see file : LICENSING).                                       */
/***********************************************************************/

#include <stdio.h>
#include "stdansi.h"

#define PCHAR 'O'

#ifdef DEBUGPP4
#define DEBUGPP3
#endif
#ifdef DEBUGPP3
#define DEBUGPP2
#endif
#ifdef DEBUGPP2
#define DEBUGPP
#endif
#ifdef DEBUGPP
#include <time.h>
#endif

#include "types.h"
#include "vector.h"
#include "polyhedron.h"
#include "partition.h"

/* number of bytes per word */
#define BYTES sizeof(int)


/* Modified DomainIntersection that discards degenerate results */
/* (i.e. intersections that are of smaller dimension than the operands) */
Polyhedron *PDomainIntersection(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned   NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;

	if (!Pol1 || !Pol2) return (Polyhedron*) 0;
	if( (Pol1->Dimension != Pol2->Dimension) || (Pol1->NbEq != Pol2->NbEq) )
	{
		fprintf( stderr,
				"? PDomainIntersection: operation on different dimensions\n" );
		return (Polyhedron*) 0;
	}
 
	d = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		for (p2=Pol2; p2; p2=p2->next)
		{
			p3 = AddConstraints(p2->Constraint[0],
								p2->NbConstraints, p1, NbMaxRays);
			if (!p3) continue;
			if (p3->NbEq!=Pol1->NbEq)
				Polyhedron_Free(p3);  /* no new equations */
			else
				d = AddPolyToDomain (p3, d);
		}
	}
	return d;
} /* PDomainIntersection */

/* Modified DomainDifference which does not put a 1 unit space around cut */
Polyhedron *PDomainDifference(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d, *ptmp;
	int i;

	if (!Pol1 || !Pol2)
		return (Polyhedron*) 0;
	if( (Pol1->Dimension != Pol2->Dimension) || (Pol1->NbEq != Pol2->NbEq) )
	{
		fprintf( stderr,
				"? PDomainDifference: operation on different dimensions\n" );
		return (Polyhedron*) 0;
	}
 
	d = (Polyhedron *)0;
	for (p2=Pol2; p2; p2=p2->next)
	{
		for (p1=Pol1; p1; p1=p1->next)
		{
			for (i=0; i<p2->NbConstraints; i++)
			{
				p3 = SubConstraint(p2->Constraint[i], p1, NbMaxRays,2);
				if (!p3) continue;
				if (emptyQ(p3) || p3->NbEq!=Pol1->NbEq)
					Polyhedron_Free(p3);
				else
					d = AddPolyToDomain (p3, d);

				ptmp = SubConstraint(p2->Constraint[i], p1, NbMaxRays,3);

			}
		}
		Pol1 = d;
		d = (Polyhedron *)0;
	}
	return Pol1;
} /* PDomainDifference */


/*----------------------------------------------------------------------*/
/* Compute_PDomains                                                     */
/*      Finds the vertices that belong to distinct subdomains           */
/*----------------------------------------------------------------------*/
void Compute_PDomains(PD, nb_domains, working_space)
Param_Domain *PD;
int nb_domains;
int working_space;
{
   Polyhedron *dx, *d1, *d2;
   unsigned bx;
   int i, ix, nv;
   Param_Domain *p1, *p2, *p2prev, *PDNew;


   if (nb_domains==0) return;

   /*initialization */
   nv = (nb_domains - 1)/(8*sizeof(int)) + 1;

   for( p1=PD, i=1, ix=0, bx=MSB ; p1 ; p1=p1->next, i++ )
   {
      p1->F = (int *) malloc (nv * sizeof(int));   /* bit array */
      memset( p1->F, 0, nv * sizeof(int) );
      p1->F[ix] |= bx;      /* set i'th bit to one */
      NEXT(ix, bx);
   }

   /* walk the PD list with two pointers */
   ix = 0; bx=MSB;
   for ( p1=PD; p1; p1=p1->next )
   {
      for ( p2prev=p1, p2=p1->next ; p2 ; p2prev=p2, p2=p2->next )
      {
         /* find intersection */
         dx = PDomainIntersection ( p1->Domain, p2->Domain, working_space);

         if ( !dx ) continue;
         if ( emptyQ(dx) )
         {
            Domain_Free( dx );
            continue;
         }
         d1 = PDomainDifference( p1->Domain, p2->Domain, working_space);
         d2 = PDomainDifference( p2->Domain, p1->Domain, working_space);

         if ( !d1 || emptyQ(d1) || d1->NbEq!=0 )
         {
            if (d1) Domain_Free( d1 );
            Domain_Free( dx );

            if ( !d2 || emptyQ(d2) || d2->NbEq!=0 )
            {
               /* dx = p1->Domain = p2->Domain */
               if (d2) Domain_Free( d2 );

               /* update p1 */
               for ( i=0; i<nv; i++ )
                  p1->F[i] |= p2->F[i];

               /* delete p2 */
               p2prev->next = p2->next;
               Domain_Free( p2->Domain );
               free( p2->F );
               free( p2 );
               p2 = p2prev;
            }
            else   /* d2 is not empty --> dx==p1->domain */
            {
               /* update p1 */
               for ( i=0; i<nv; i++ )
                  p1->F[i] |= p2->F[i];

               /* replace p2 with d2 */
               Domain_Free( p2->Domain );
               p2->Domain = d2;
            }
         }
         else /* d1 is not empty */
         {
            if ( !d2 || emptyQ(d2) || d2->NbEq!=0 )
            {
               if (d2) Domain_Free(d2);
               /* dx = p2->domain */
               Domain_Free(dx);

               /* update p2 */
               for ( i=0; i<nv; i++ )
                  p2->F[i] |= p1->F[i];

               /* replace p1 with d1 */
               Domain_Free( p1->Domain );
               p1->Domain = d1;
            }
            else /*d2 is not empty-->d1,d2,dx are distinct*/
            {
               /* create a new node for dx */
               PDNew = (Param_Domain *) malloc
                  ( sizeof(Param_Domain) );
               PDNew->F = (int *)malloc( nv*sizeof(int) );
               memset( PDNew->F, 0, nv*sizeof(int) );
               PDNew->Domain = dx;

               for ( i=0; i<nv; i++ )
                  PDNew->F[i] = p1->F[i] | p2->F[i];

               /* replace p1 with d1 */
               Domain_Free( p1->Domain );
               p1->Domain = d1;

               /* replace p2 with d2 */
               Domain_Free( p2->Domain );
               p2->Domain = d2;

               /* insert new node after p1 */
               PDNew->next = p1->next;
               p1->next = PDNew;
            }
         }
      }  /* end of p2 scan */
   } /* end of p1 scan */
}
					

/*----------------------------------------------------------------------*/
/* Print_Domain                                                         */
/*----------------------------------------------------------------------*/
void Print_Domain( DST, D )
FILE *DST;
Polyhedron *D;
{
   int l, v;
   int first;

   for( l=0 ; l<D->NbConstraints ; ++l )
   {
      fprintf(DST, "         " );
      first = 1;
      for( v=1 ; v<=D->Dimension ; ++v )
      {
         if(D->Constraint[l][v] )
         {
            if( D->Constraint[l][v]==1 )
            {
               if( first ) fprintf(DST, "%c ", PCHAR+v );
               else fprintf(DST, "+ %c ", PCHAR+v );
            }
            else if( D->Constraint[l][v]==-1 ) fprintf(DST, "- %c ", PCHAR+v );
            else
            {
               if( D->Constraint[l][v] > 0 && !first ) fprintf(DST, "+ " );
               fprintf(DST, "%d%c ", D->Constraint[l][v], PCHAR+v );
            }
            first = 0;
         }
      }
      if( D->Constraint[l][v] )
      {
         if( D->Constraint[l][v] > 0 && !first ) fprintf(DST, "+" );
         fprintf(DST, " %d", D->Constraint[l][v] );
      }
      fprintf(DST, D->Constraint[l][0]?" >= 0":" = 0" );
      fprintf(DST, "\n" );
   }
   fprintf(DST, "\n");
}


/*----------------------------------------------------------------------*/
/* Param_Domain_Free                                                    */
/*      Frees the memory allocated by a call to                         */
/*                                          Polyhedron2Param_Domain     */
/*      Input :	parametrized polyhedron                                 */
/*      Output : -                                                      */
/*----------------------------------------------------------------------*/
void Param_Domain_Free( PP )
Param_Domain *PP;
{
	Param_Domain *n;

	while( PP )
	{
		free( PP->F );
		Domain_Free( PP->Domain );

		n = PP->next;
		free( PP );
		PP = n;
	}
}

