/* University of Strasbourg - Master ILC-ISI-RISE
 * Compilation Lab - Compiler for the arith language
 * Written by Cedric Bastoul cedric.bastoul@unistra.fr
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "symbol.h"
#include "ast.h"

// Allocate the memory for a new symbol and set its fields to default values.
struct symbol* symbol_alloc() {
  struct symbol* symbol = malloc(sizeof(struct symbol));
  symbol->identifier = NULL;
  symbol->isconstant = false;
  symbol->value = 0;
  symbol->next = NULL;
  return symbol;
}

// Free the memory allocated for a symbol list.
void symbol_free(struct symbol* symbol) {
  struct symbol* next;
  while (symbol != NULL) {
    next = symbol->next;
    free(symbol->identifier);
    free(symbol);
    symbol = next;
  }
}

// Create a temporary symbol, add it to the table and return a pointer to it.
struct symbol* symbol_new_temp(struct symbol** table) {
  static int temporary_number = 0;
  char temporary_name[SYMBOL_MAX_STRING];
  snprintf(temporary_name, SYMBOL_MAX_STRING, "temp_%d", temporary_number);
  temporary_number++;
  return symbol_add(table, strdup(temporary_name));
}

// Create, add to the table and return a temporary symbol for a constant.
struct symbol* symbol_new_constant(struct symbol** table, int value) {
  struct symbol* new = symbol_new_temp(table);
  new->isconstant = 1;
  new->value = value;
  return new;
}

// Return a pointer to the symbol corresponding to the identifier in the table.
struct symbol* symbol_lookup(struct symbol* table, char* identifier) {
  while (table != NULL) {
    if (strcmp(table->identifier, identifier) == 0)
      return table;
    table = table->next;
  }
  return NULL;
}

// Create a named symbol, add it to the table and return a pointer to it.
struct symbol* symbol_add(struct symbol** table, char* name) {
  if (*table == NULL) {
    *table = symbol_alloc();
    (*table)->identifier = name;
    return *table;
  } else {
    struct symbol* scan = *table;
    while (scan->next != NULL)
      scan = scan->next;
    scan->next = symbol_alloc();
    scan->next->identifier = name;
    return scan->next;
  }
}

// Scan the ast to build then return the symbol table
void symbol_build_table(struct ast* ast, struct symbol** symbol_table) {
  struct symbol* identifier;
  
  do {
    switch (ast->type) {
      case ast_type_number:
        break;
      case ast_type_symbol:
        break;
      case ast_type_identifier:
        identifier = symbol_lookup(*symbol_table, ast->u.identifier);
        if (identifier == NULL) {
	  fprintf(stderr, "[S] SEMANTIC ERROR: %s is not initialized\n",
                  ast->u.identifier);
	  exit(1);
        }
        free(ast->u.identifier);
        ast->type = ast_type_symbol;
        ast->u.symbol = identifier;
        break;
      case ast_type_statement:
        if (ast->u.statement.identifier != NULL) {
          identifier = symbol_lookup(*symbol_table, ast->u.identifier);
          if (identifier == NULL)
	    identifier = symbol_add(symbol_table, ast->u.identifier);
          ast->u.statement.symbol = identifier;
          ast->u.statement.identifier = NULL;
        }
        symbol_build_table(ast->u.statement.expression, symbol_table);
        break;
      case ast_type_add:
        symbol_build_table(ast->u.operation.left,  symbol_table);
        symbol_build_table(ast->u.operation.right, symbol_table);
        break;
      case ast_type_mul:
        symbol_build_table(ast->u.operation.left,  symbol_table);
        symbol_build_table(ast->u.operation.right, symbol_table);
        break;
      default:
        fprintf(stderr, "[S] ERROR: unknown AST node type\n");
        exit(1);
    }
    if (ast->type == ast_type_statement)
      ast = ast->u.statement.next;
  } while ((ast != NULL) && (ast->type == ast_type_statement));
}

// Print a symbol list.
void symbol_print(struct symbol* symbol) {
  while (symbol != NULL) {
    printf("identifier: %7s, isconstant: ", symbol->identifier);
    if (symbol->isconstant)
      printf("true,  value: %d\n", symbol->value);
    else
      printf("false, value: N/A\n");
    symbol = symbol->next;
  }
}
