/* University of Strasbourg - Master ILC-ISI-RISE
 * Compilation Lab - Compiler for the arith language
 * Written by Cedric Bastoul cedric.bastoul@unistra.fr
 */

#include <stdio.h>
#include <stdlib.h>
#include "symbol.h"
#include "quad.h"
#include "final.h"

// Create a new instruction with fields set to corresponding parameters.
struct instruction* instruction_gen(enum instruction_type type,
                                    struct symbol* address) {
  struct instruction* new = malloc(sizeof(struct instruction));
  new->type = type;
  new->address = address;
  new->next = NULL;
  return new;
}

// Free the memory allocated for an instruction list (not the linked symbols).
void instruction_free(struct instruction* instruction) {
  struct instruction* next;
  while (instruction != NULL) {
    next = instruction->next;
    free(instruction);
    instruction = next;
  }
}

// Append the src instruction list to the dest instruction list.
void instruction_add(struct instruction** dest, struct instruction* src) {
  if (*dest == NULL) {
    *dest = src;
  } else {
    struct instruction* scan = *dest;
    while (scan->next != NULL)
      scan = scan->next;
    scan->next = src;
  }
}

// Generate the instruction sequence corresponding to an add quad.
struct instruction* instruction_gen_add(struct quad* quad) {
  struct instruction* list = NULL;
  instruction_add(&list, instruction_gen(instruction_type_push, quad->arg1));
  instruction_add(&list, instruction_gen(instruction_type_push, quad->arg2));
  instruction_add(&list, instruction_gen(instruction_type_add, NULL));
  instruction_add(&list, instruction_gen(instruction_type_pop, quad->res));
  return list;
}

// Generate the instruction sequence corresponding to a mul quad.
struct instruction* instruction_gen_mul(struct quad* quad) {
  struct instruction* list = NULL;
  instruction_add(&list, instruction_gen(instruction_type_push, quad->arg1));
  instruction_add(&list, instruction_gen(instruction_type_push, quad->arg2));
  instruction_add(&list, instruction_gen(instruction_type_mul, NULL));
  instruction_add(&list, instruction_gen(instruction_type_pop, quad->res));
  return list;
}

// Generate the instruction sequence corresponding to a print quad.
struct instruction* instruction_gen_print(struct quad* quad) {
  struct instruction* list = NULL;
  instruction_add(&list, instruction_gen(instruction_type_push, quad->res));
  instruction_add(&list, instruction_gen(instruction_type_print, NULL));
  instruction_add(&list, instruction_gen(instruction_type_pop, quad->res));
  return list;
}

// Generate the instruction sequence corresponding to an assignment quad.
struct instruction* instruction_gen_assignment(struct quad* quad) {
  struct instruction* list = NULL;
  instruction_add(&list, instruction_gen(instruction_type_push, quad->arg1));
  instruction_add(&list, instruction_gen(instruction_type_pop, quad->res));
  return list;
}

// Generate the instruction sequence corresponding to a quad list.
struct instruction* instruction_generate(struct quad* quad) {
  struct instruction* program = NULL;
  
  while (quad != NULL) {
    switch (quad->op) {
      case '+':
        instruction_add(&program, instruction_gen_add(quad));
        break;
      case '*':
        instruction_add(&program, instruction_gen_mul(quad));
        break;
      case 'P':
        instruction_add(&program, instruction_gen_print(quad));
        break;
      case '=':
        instruction_add(&program, instruction_gen_assignment(quad));
        break;
      default:
        printf("unknown quad type\n");
        exit(1);
    }
    quad = quad->next;
  }
  return program;
}

// Print an instruction sequence.
void instruction_print(struct instruction* instruction) {
  while (instruction != NULL) {
    switch (instruction->type) {
      case instruction_type_push:
        printf("PUSH  %s\n", instruction->address->identifier);
        break;
      case instruction_type_pop:
        printf("POP   %s\n", instruction->address->identifier);
        break;
      case instruction_type_add:
        printf("ADD\n");
        break;
      case instruction_type_mul:
        printf("MUL\n");
        break;
      case instruction_type_print:
        printf("PRINT\n");
        break;
      default:
        printf("unknown instruction type\n");
        exit(1);
    }
    instruction = instruction->next;
  }
}

// Print the memory state.
void instruction_print_memory_state(struct symbol* symbol) {
  while (symbol != NULL) {
    printf("Address: %7s, ", symbol->identifier);
    if (symbol->isconstant)
      printf("initial value: %d\n", symbol->value);
    else
      printf("initial value: N/A\n");
    symbol = symbol->next;
  }
}
