/* University of Strasbourg - Master ILC-ISI-RISE
 * Compilation Lab - Compiler for the arith language
 * Written by Cedric Bastoul cedric.bastoul@unistra.fr
 */

%{
  #include <stdio.h>
  #include <stdlib.h>
  #include "ast.h"
  #include "symbol.h"

  void yyerror(char*);
  int  yylex();
  void lex_free();
  extern FILE* yyin;

  struct ast* parser_ast = NULL;
%}

%union {
  char* string;
  int value;
  struct ast* ast;
}

%token PRINT
%token <string> IDENTIFIER
%token <value>  NUMBER
%type  <ast>    statement_list
%type  <ast>    statement
%type  <ast>    expression

%left '+'
%left '*'

%%

axiom:
    statement_list               { parser_ast = $1; return 0; }
  ;

statement_list:
    statement statement_list     { $$ = ast_concat($1, $2); }
  | statement                    { $$ = $1; }
  ;

statement:
    IDENTIFIER '=' expression ';'{ $$ = ast_new_statement($1, $3); }
  | PRINT expression ';'         { $$ = ast_new_statement(NULL, $2); }
  ;

expression:
    expression '+' expression    { $$ = ast_new_operation(ast_type_add,$1,$3);}
  | expression '*' expression    { $$ = ast_new_operation(ast_type_mul,$1,$3);}
  | '(' expression ')'           { $$ = $2; }
  | IDENTIFIER                   { $$ = ast_new_identifier($1); }
  | NUMBER                       { $$ = ast_new_number($1); }
  ;

%%

void yyerror (char *s) {
    printf("[Y] ERROR: %s\n", s);
}

int main(int argc, char* argv[]) {
  struct symbol* symbol_table = NULL;

  if (argc > 2) {
    fprintf(stderr, "usage: %s [file]\n", argv[0]);
    exit(1);
  }

  if (argc == 2) {
    yyin = fopen(argv[1], "r");
    if (yyin == NULL) {
      fprintf(stderr, "unable to open %s\n", argv[1]);
      exit(1);
    }
  }

  yyparse();
  
  printf("-----------------\nAbstract syntax tree:\n");
  ast_print(parser_ast, 0);
  
  printf("\n-----------------\nSymbol table:\n");
  symbol_build_table(parser_ast, &symbol_table);
  symbol_print(symbol_table);

  // Be clean.
  lex_free();
  ast_free(parser_ast);
  symbol_free(symbol_table);
  fclose(yyin);
  return 0;
}
