/************************************************/
/*  eval_ehrhart.c                              */
/* functions to evaluate an Ehrhart polynomial. */
/* written by Emmanuel Jeannot (c) 1997.        */
/*  Emmanuel.Jeannot@ens-lyon.fr                */
/*  http://www.ens-lyon.fr/~ejeannot            */
/*                                              */
/* modified (ArithmetiqueLib), Vincent Loechner */
/*  Mar/4/1998                                  */
/************************************************/

#include <stdio.h>
#include <string.h>

#include "arithmetique.h"
#include "polylib.h"


/* #define EVAL_EHRHART_DEBUG */

/********************************************************/
/* function in domain                                   */
/*    check if the parameters in list_args              */
/*    verifies the constraints of Polyhedron P          */
/********************************************************/
static int in_domain(Polyhedron *P, Value *list_args)
{
  int col,row;
  Value v; /* value of the constraint of a row when
	       parameters are instanciated*/

  /*P->Constraint constraint matrice of polyhedron P*/

	for(row=0;row<P->NbConstraints;row++)
	{
		value_assign( v, P->Constraint[row][P->Dimension+1] ); /*constant part*/
		for(col=1;col<P->Dimension+1;col++)
			value_addto( v, value_mult(P->Constraint[row][col],list_args[col-1]) );

		if ( value_notzero_p(P->Constraint[row][0]) )
		{
			/*if v is not >=0 then this constraint is not respected */
			if ( value_neg_p(v) ) 
				return 0;
		}
		else
		{
			/*if v is not =0 then this constraint is not respected */
			if ( value_notzero_p(v) )
				return 0;
		}
	}
  
  /*if not retrun before this point => all 
    the constraints are respected*/
  return 1;
}

/****************************************************/
/* function compute enode                           */
/*     compute the value of enode p with parameters */
/*     list "list_args                              */
/*     compute the polynomial or the periodic       */
/****************************************************/

static double compute_enode(enode *p, Value *list_args)
{
	int i;
	Value m, param;
	double res=0.0;

	if (!p)
		return( 0. );

	if (p->type == polynomial)
	{
		value_assign( param, list_args[p->pos-1] );
		/*compute the polynomial using Horner's rule*/
		for (i=p->size-1; i>0; i--)
		{
			res+=compute_evalue( &p->arr[i],list_args);
			res*=VALUE_TO_DOUBLE(param);
		}
		res+=compute_evalue( &p->arr[0],list_args);
	}
	else if (p->type == periodic)
	{
		value_assign( param, list_args[p->pos-1] );
		/*choose the right element of the periodic*/
		value_assign( m, value_mod(param,int_to_value(p->size)) );
		res=compute_evalue(&p->arr[VALUE_TO_INT(m)],list_args);
	}
	return res;
}

/*************************************************/
/* return the value of Ehrhart Polynomial        */
/* It returns a double, because since it is      */
/* a recursive function, some intermediate value */
/* might not be integral                         */
/*************************************************/


double compute_evalue(evalue *e,Value *list_args)
{
	double res;
  
	if ( value_notzero_p(e->d) )
	{
		if ( value_notone_p(e->d) ) 
			res = VALUE_TO_DOUBLE(e->x.n) / VALUE_TO_DOUBLE(e->d);
		else 
			res = VALUE_TO_DOUBLE(e->x.n);
	}
	else 
		res = compute_enode(e->x.p,list_args);
  return res;
}


/****************************************************/
/* function compute_poly :                          */
/* Check for the good validity domain               */
/* return the number of point in the Polyhedron     */
/* Using the Ehrhart pseudo-polynomial              */
/****************************************************/
Value compute_poly(Enumeration *en, Value *list_args)
{
	if( !en )
		return( VALUE_ZERO );	/* no ehrhart polynomial */
	if(en->ValidityDomain)
	{
		if(!en->ValidityDomain->Dimension)
			return( VALUE_ZERO ); /* no parameters, empty polyhedron */
	}
	else 
		return( VALUE_ZERO ); /* no Validity Domain */
  
  
	while( en )
	{
		if( in_domain(en->ValidityDomain,list_args) )
		{
#ifdef EVAL_EHRHART_DEBUG
			Print_Domain(stdout,en->ValidityDomain);
			print_evalue(stdout,&en->EP);
#endif
			return( double_to_value(compute_evalue(&en->EP,list_args)) );
		}
		else
			en=en->next;
	}

  return( VALUE_ZERO ); /* no compatible domain with the arguments */

}
/****************** End Compute poly *******************************/

