// function doing the (basic) matrix decomposition described in 
// "On a matrix decomposition" by B. Meister 3/2002 : http://icps.u-strasbg.fr/~meister/decomp.ps.gz

#include<polylib/polylib32.h>

int debug=0;

/* gcd function */
int gcd(Value num, Value den) {

  // take the absolute values of num and den
  Value a = (num>0)?num:-num;
  Value b = (den>0)?den:-den;
  
  while ((a>0)&&(b>0)) {
    if (a>b) a-=b;
    else if(b>a) b-=a;
    else return a;
  }
  if (a>0) return a;
  if (b>0) return b;
  if ((a==0)&&(b==0)) return 0;
}

/* utility function: gcd of a matrix M's i-th column */
int col_gcd(Matrix * M, int i) {
  int j, cur_gcd;
  cur_gcd = M->p[0][i];
  for(j=0 ; j<M->NbRows; j++) {
    cur_gcd = gcd(cur_gcd,M->p[j][i]);
    if (debug) printf(" cur_gcd(H[%i][%i])=%i ", j, i, cur_gcd);   
  }
  if (debug) printf(" col_gcd(H[%i])=%i ",i, cur_gcd); 
  return cur_gcd;
}
			       
/* Input: A full row rank matrix M                                          */
/* Output: An array of ordered matrices M, constituting the decomposition:  */
/* M = N_m . U_m . N_{m-1} ... N_1 . U_1 . N_0 . U_0                        */
/* the parameter "multi_comp" (boolean) tells whether we can have more than */ 
/* one compressing dimension at a time in a compression matrix or not       */
/* Matrix_List[2*i] contains U_i, whereas Matrix_List[2*i+1] contains N_i   */
/* If the list contains less that 2*dim matrices                            */
/* (where dim is the dimension of M), the rest of the List is NULL          */ 

Matrix ** Decomposition(Matrix * M, int multi_comp) {
  Matrix ** Matrix_List;
  Matrix * H, *Q, *U, *N, *K;
  int dim = H->NbRows;
  int bdim = H->NbColumns;
  int i,j;
  int b, rho;
  int cur_index;

  left_hermite(M, &H, &Q, &U); /* does the decomposition :  M = H.Q */
  Matrix_Free(U);
  dim = H->NbRows;
  bdim = H->NbColumns;

  /* here, we treat the case where M is not a full rank square matrix (but just a full row rank matrix) */
  if (H->NbColumns > H->NbRows) {
    Matrix * Hp = Matrix_Alloc(H->NbRows, H->NbRows);
    for (i=0 ; i< H->NbRows-1 ; i++) {
      for (j=0 ; j< H->NbRows-1 ; j++) {
	Hp->p[i][j] = H->p[i][j];
      }
    }
    Matrix_Free(H);
    H = Hp;
  }

  /* create a list of matrices where we will put all the components */
  Matrix_List = (Matrix **) malloc(2*dim*sizeof(Matrix*));
  for(i=0; i<2*dim; i++) Matrix_List[i] = NULL;
  Matrix_List[2*0]= Q;
  if (debug) {
    printf("\nU_0: \n"); 
    Matrix_Print(stdout,"%4d",Q); 
  } 
  Q=NULL;
  cur_index = 0;

  /* compute b */
  for(b = H->NbRows-1; (b>0)&&(H->p[b][b]==1); b--);

  /* main decomposition process: N_i then U_{i+1} ... U_n */
  while (b>0) {

    /* 1- LINEAR COMPRESSION
    /* determine the b^{th} compression factor and put it in the right compression matrix N_i*/
    N = Matrix_Alloc(dim, dim);
    for (i=0; i<dim; i++) {
      for (j=0; j<dim; j++) {
	N->p[i][j]=(i==j)?1:0;
      }
    }

    /* A- case where only one-dimensional compresison is allowed */
    if (multi_comp==0) {
      N->p[b][b] = H->p[b][b];
      
      /* DO the compression */
      H->p[b][b] = 1; 
    }

    /* B- case where multi-dimensional compression is allowed    */ 
    else {
      for (i = 0; i<dim; i++) { 
	rho = col_gcd(H,i);
	if (rho > 1) {
	  N->p[i][i] = rho;
	  if (debug) printf("\nrho(%i)=%i\n", rho);
	  /* DO the compression */
	  for (j=0; j<dim; j++) H->p[j][i] = H->p[j][i] / rho;
	}
      }
    }
    if (debug) {
      printf("\nH_%i: \n",cur_index); 
      Matrix_Print(stdout,"%4d",H); 
    }  

    /* Attach N_i to the Matrix list */
    Matrix_List[2*cur_index+1] = N;
    if (debug) {
      printf("\nN_%i: \n",cur_index); 
      Matrix_Print(stdout,"%4d",N); 
    }
    N=NULL;
    
    /* 2- left Hermite Normal Form DECOMPOSITION */
    left_hermite(H, &K, &Q, &U); /* does the decomposition :  H = K.Q */
    Matrix_Free(U);
    
    /* Attach U_{i+1} (i.e. Q) to the matrix list */
    cur_index++;
    Matrix_List[2*cur_index] = Q;
    if (debug) {
      printf("\nU_%i: \n",cur_index); 
      Matrix_Print(stdout,"%4d",Q); 
    } 
    Q=NULL;
    Matrix_Free(H);
    
    /* Prepare the next couple of decompositions: */
    H = K;

    /* compute b for the new H */
    for(b = H->NbRows-1 ; (b>0)&&(H->p[b][b]==1) ; b--);
    if (debug) printf("\nb=%i, H->p[b][b]=%i\n",b, H->p[b][b]); 
  }

  /* here, b is supposed to be = 1 , so H = N_m. Thus, we attach N_m to the matrix list. */
  Matrix_List[2*cur_index+1] = H;
  if (debug) {
    printf("\nN_%i: \n",cur_index); 
    Matrix_Print(stdout,"%4d",H); 
  } 
  return Matrix_List;
}

main() {
  int i,j,k, method;
  Matrix * M;
  Matrix ** List;
  Matrix * test, * test2;

  printf("\nPlease enter your matrix.\n");
  M = Matrix_Read();

  printf("\nWhich method would be nice to you ?");
  scanf("%i",&method);
  List = Decomposition(M,method);

  /* check if the product of all matrices is actually a decomposition of the original matrix */
  test = Matrix_Alloc(List[0]->NbRows, List[0]->NbColumns);
  for (j=0; j<List[0]->NbRows; j++) {
    for (k=0; k<List[0]->NbColumns; k++) {
      test->p[j][k] = List[0]->p[j][k];
    }
  }
  for (i=1; i<2*M->NbRows; i++) if (List[i]) {
    test2 = Matrix_Alloc(List[i]->NbRows, test->NbColumns);
    Matrix_Product(List[i], test, test2);
    free(test);
    test = test2;
    test2 = NULL;
  }
  printf("\n Validity check : test = \n");
  Matrix_Print(stdout, "%4d", test);
  free(test);
  for(i=0; i<2*M->NbRows; i++) if (List[i]) free(List[i]);
  printf("\n done \n");
}
