#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "gmp.h"

/* #define EDEBUG		*/ 	/* normal debug */
/* #define EDEBUG1	*/ 	/* prints enumeration points */
/* #define EDEBUG2	*/ 	/* prints domains */
/* #define EDEBUG3	*/ 	/* prints systems of equations that are solved */
/* #define EDEBUG4	*/ 	/* prints message for degree reduction */
/* #define EDEBUG5	*/ 	/* prints result before simplification */

#include "polylib.h"

#define REDUCE_DEGREE

static void deinit_evalue_mp(evalue_mp *e);
static void init_evalue_mp(evalue_mp *e);

/*-------------------------------------------------------------------*/
/* EHRHART POLYNOMIAL SYMBOLIC ALGEBRA SYSTEM                        */
/*-------------------------------------------------------------------*/
/*------------------------------------------------------------*/
/* enode_mp *new_enode_mp(type, size, pos)                    */
/*      type : enode_mp type                                  */
/*      size : degree+1 for polynomial, period for periodic   */
/*      pos  : 1..nb_param, position of parameter             */
/* returns a newly allocated enode_mp                         */
/* can be freed with a simple free(x)                         */
/*------------------------------------------------------------*/
static enode_mp *new_enode_mp(enode_type_mp type, int size, int pos)
{
  enode_mp *res;
  int i;

  res = (enode_mp *) malloc( sizeof(enode_mp) + sizeof(evalue_mp) * (size-1) );
  res->type = type;
  res->size = size;
  res->pos = pos;
  for (i = 0; i < size; i++)
    init_evalue_mp(&(res->arr[i]));
  return res;
}

static void free_enode_mp(enode_mp *p)
{
  int i;

  if (!p)
    return;
  for (i = 0; i < p->size; i++)
    deinit_evalue_mp(&(p->arr[i]));
  free(p);
}

static void init_evalue_mp(evalue_mp *e)
{
  mpz_init_set_si(e->d, 0);
  mpz_init_set_si(e->n, 0);
  e->p=NULL;
}

static void deinit_evalue_mp(evalue_mp *e)
{
	if( e->p )
		free_enode_mp(e->p);
	mpz_clear(e->n);
	mpz_clear(e->d);
}

static void set_enode_mp_size(enode_mp *p, int size)
{
  int i;

  if (size < p->size && size >= 0)
    {
      for (i = size; i < p->size; i++)
	deinit_evalue_mp(&(p->arr[i]));
      p->size = size;
    }
  else
    { /* assertion */
      fprintf(stderr, "set_enode_mp_size: wrong size!\n");
      exit(1);
    }
}

/*------------------------------------------------------------*/
/* evalue_mp_copy(e)                                          */
/*     e : pointer to an evalue_mp                            */
/*------------------------------------------------------------*/
static enode_mp *ecopy_mp(enode_mp *e)
{
  enode_mp *res;
  int i;

  if (!e) /* empty value */
    return 0;

  res = new_enode_mp(e->type, e->size, e->pos);
  for (i = 0; i < e->size; i++)
    {
		mpz_set(res->arr[i].d, e->arr[i].d);
		if( mpz_cmp_si(e->arr[i].d, 0) == 0 )
			res->arr[i].p = ecopy_mp(e->arr[i].p);
		else
			mpz_set(res->arr[i].n, e->arr[i].n);
    }
  return res;
}

/*------------------------------------------------------------*/
/* print_evalue_mp(DST, e)                                    */
/*    DST : destination file                                  */
/*    e : pointer to evalue_mp to be printed                  */
/* Returns nothing, prints the evalue_mp to DST               */
/*------------------------------------------------------------*/
static void print_enode_mp(FILE *DST, enode_mp *p); /* forward declaration */

static void print_evalue_mp(FILE *DST, evalue_mp *e)
{
  if (mpz_cmp_si(e->d, 0) == 0)
    print_enode_mp(DST, e->p);
  else
    {
      mpz_out_str(DST, 10, e->n);
      if (mpz_cmp_si(e->d, 1) != 0)
	{
	  fprintf(DST, "/");
	  mpz_out_str(DST, 10, e->d);
	}
    }
}

/*------------------------------------------------------------*/
/* print_enode_mp(DST, p )                                    */
/*    DST : destination file                                  */
/*    p : pointer to enode_mp  to be printed                  */
/* Returns nothing, prints the enode_mp  to DST               */
/*------------------------------------------------------------*/
static void print_enode_mp(FILE *DST, enode_mp *p)
{
  int i;

  if (!p)
    {
      fprintf(DST, "NULL");
      return;
    }
  if (p->type == evector)
    {
      fprintf(DST, "{ ");
      for (i=0; i<p->size; i++)
        {
	  print_evalue_mp(DST, &p->arr[i]);
	  if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
      fprintf(DST, " }\n");
    }
  else if (p->type == polynomial)
    {
      fprintf(DST, "( ");
      for (i=p->size-1; i>=0; i--)
        {
	  print_evalue_mp(DST, &p->arr[i]);
	  if (i==1) fprintf(DST, " * %c + ", PCHAR+p->pos);
	  else if ( i>1 ) fprintf(DST, " * %c^%d + ", PCHAR+p->pos, i);
        }
      fprintf(DST, " )\n");
    }
  else if (p->type == periodic)
    {
      fprintf(DST, "[ ");
      for (i=0; i<p->size; i++)
        {
	  print_evalue_mp(DST, &p->arr[i]);
	  if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
      fprintf(DST, " ]_%c", PCHAR+p->pos);
    }
}

/*------------------------------------------------------------*/
/* enode enode_mp2value( enode_mp e )                         */
/*      e : pointer to evnode                                 */
/* converts an enode_mp to an enode                           */
/*------------------------------------------------------------*/
static enode *enode_mp2value( enode_mp *e )
{
	enode *res;
	int i;

	if (!e) /* empty value */
    return NULL;

	res = new_enode(e->type, e->size, e->pos);
	for (i = 0; i < e->size; i++)
	{
		res->arr[i].d = mpz_get_si(e->arr[i].d);
		if( value_zero_p(res->arr[i].d) )
			res->arr[i].x.p = enode_mp2value(e->arr[i].p);
		else
			res->arr[i].x.n = mpz_get_si(e->arr[i].n);
	}
	return res;
}


/*------------------------------------------------------------*/
/* int eequal (e1, e2)                                        */
/*      e1, e2 : pointers to evalue_mps                       */
/* returns 1 (true) if they are equal, 0 (false) if not       */
/*------------------------------------------------------------*/
static int eequal(evalue_mp *e1, evalue_mp *e2)
{  int i;
   enode_mp *p1, *p2;

   if (mpz_cmp(e1->d, e2->d) != 0) return 0;
   /* e1->d == e2->d */
   if (mpz_cmp_si(e1->d, 0) != 0)
   {  if (mpz_cmp(e1->n, e2->n) != 0) return 0;
      /* e1->d == e2->d != 0  AND e1->n == e2->n */
      return 1;
   }
   /* e1->d == e2->d == 0 */
   p1 = e1->p;
   p2 = e2->p;
   if (p1->type != p2->type) return 0;
   if (p1->size != p2->size) return 0;
   if (p1->pos  != p2->pos ) return 0;
   for (i=0; i<p1->size; i++)
      if (!eequal(&p1->arr[i], &p2->arr[i]) ) return 0;
   return 1;
}

/*------------------------------------------------------------*/
/* void reduce_evalue_mp ( e )                                */
/*       e : pointer to an evalue_mp                          */
/*------------------------------------------------------------*/
static void reduce_evalue_mp (evalue_mp *e)
{  enode_mp *p;
   int i, j, k;

   if (mpz_cmp_si(e->d, 0) != 0)
     return;	/* a rational number, its already reduced */
   p = e->p;

   /* first reduce the components of p */
   for (i=0; i<p->size; i++) reduce_evalue_mp(&p->arr[i]);

   if (p->type==periodic)
   {  /* try to reduce the period */
      for (i=1; i<=(p->size)/2; i++)
      {  if ( (p->size % i)==0 )
         {  /* can we reduce the size to i ? */
            for (j=0; j<i; j++)
               for (k=j+i; k < e->p->size; k+=i)
                  if (!eequal(&p->arr[j], &p->arr[k]) ) goto you_lose;
            /* OK, lets do it */
	    set_enode_mp_size(p, i);
            break;
you_lose:   /* OK, lets not do it */
            continue;
         }
      }
   }
   else if (p->type==polynomial)
   {  /* try to reduce the degree */
      for (i=p->size-1; i>=0; i--)
      {  if (mpz_cmp_si(p->arr[i].d,1) == 0 &&
	     mpz_cmp_si(p->arr[i].n,0) == 0)  /* zero coefficient */
            continue;
         else
            break;
      }
      if (i == -1)
	set_enode_mp_size(p, 1);
      else if (i+1<p->size)
	set_enode_mp_size(p, i+1);
   }

   /* try to reduce its strength */
   if (p->size == 1)
     {
       mpz_set(e->d, p->arr[0].d);
       mpz_set(e->n, p->arr[0].n);
       e->p = p->arr[0].p;
       p->arr[0].p = 0;
       free_enode_mp(p);
     }
}

/*------------------------------------------------------------*/
/* void emul_mp (e1, e2, res)                                 */
/*       e1 : pointer to an evalue_mp                         */
/*       e2 : pointer to a constant evalue_mp                 */
/*       res : pointer to result evalue_mp = e1 * e2          */
/* multiplies two evalue_mps and puts the result in res       */
/*------------------------------------------------------------*/
static void emul_mp (evalue_mp *e1, evalue_mp *e2, evalue_mp *res)
{  enode_mp *p;
   int i;
   mpz_t g;

   if (mpz_cmp_si(e2->d, 0) == 0)
   {  fprintf(stderr, "emul_mp: ?expecting constant value\n");
      return;
   }
   if (mpz_cmp_si(e1->d, 0) != 0)
   {  /* product of two rational numbers */
      mpz_init(g);
      mpz_mul(res->d, e1->d, e2->d);
      mpz_mul(res->n, e1->n, e2->n);
      mpz_gcd(g, res->n, res->d);
      if (mpz_cmp_si(g,1) != 0)
      {  mpz_div(res->d, res->d, g);
         mpz_div(res->n, res->n, g);
      }
      mpz_clear(g);
   }
   else /* e1 is an expression */
   {  mpz_set_si(res->d, 0);
      p = e1->p;
      res->p = new_enode_mp(p->type, p->size, p->pos);
      for (i=0; i < p->size; i++)
      {  emul_mp(&p->arr[i], e2, &res->p->arr[i]); }
   }
}

/*------------------------------------------------------------*/
/* void eadd_mp (e1, res)                                     */
/*       e1 : an evalue_mp                                    */
/*       res : result = res + e1                              */
/* adds one evalue_mp to evalue_mp 'res'                      */
/*------------------------------------------------------------*/
static void eadd_mp(evalue_mp *e1, evalue_mp *res)
{  int i;
   mpz_t g, tmp1;

   if (mpz_cmp_si(e1->d, 0) != 0 && mpz_cmp_si(res->d, 0) != 0)
   {  /* add two rational numbers*/
      mpz_init(g); mpz_init(tmp1);

      /* nominator: d*n' + d'*n */
      mpz_mul(tmp1, res->n, e1->d);
      mpz_mul(res->n, res->d, e1->n);
      mpz_add(res->n, tmp1, res->n);

      /* denominator: d*d' */
      mpz_mul(res->d, e1->d, res->d);

      mpz_gcd(g, res->n, res->d);
      if (mpz_cmp_si(g, 1) != 0)
	{
	  mpz_div(res->d, res->d, g);
	  mpz_div(res->n, res->n, g);
	}
      mpz_clear(g); mpz_clear(tmp1);
      return;
   }
   else if (mpz_cmp_si(e1->d, 0) != 0 && mpz_cmp_si(res->d, 0) == 0)
   {  if (res->p->type == polynomial)
      {  /* add the constant to the constant term */
         eadd_mp(e1, &res->p->arr[0]);
         return;
      }
      else if (res->p->type == periodic)
      {  /* add the constant to all elements of periodic number */
         for (i=0; i < res->p->size; i++)
         {  eadd_mp(e1, &res->p->arr[i]);
         }
         return;
      }
      else
      {  fprintf(stderr, "eadd_mp: cannot add const with vector\n");
         return;
      }
   }
   else if (mpz_cmp_si(e1->d, 0) == 0 && mpz_cmp_si(res->d, 0) != 0)
   {
       fprintf(stderr, "eadd_mp: cannot add evalue_mp to const\n");
       return;
   }
   else /* ((e1->d==0) && (res->d==0)) */
   {  if ( (e1->p->type != res->p->type) ||
           (e1->p->pos  != res->p->pos ) )
      {  fprintf(stderr, "eadd_mp: ?cannot add, incompatible types\n");
         return;
      }
      if (e1->p->size == res->p->size)
      {  for (i=0; i < res->p->size; i++)
         {  eadd_mp(&e1->p->arr[i], &res->p->arr[i]);
         }
         return;
      }
      /* sizes are different*/
      if (res->p->type == polynomial)
      {
	  /* VIN100: if e1-size > res-size you have to copy e1 in a
	     new enode_mp and add res to that new node. If you do not do
	     that, you lose the the upper weight part of e1 ! */
	  if( e1->p->size > res->p->size )
	    {
	      enode_mp *tmp;
	      tmp = ecopy_mp( e1->p );
	      for( i=0 ; i < res->p->size ; ++i )
		{
		  eadd_mp(&res->p->arr[i], &tmp->arr[i]);
		  deinit_evalue_mp(&res->p->arr[i]);
		}
	      res->p = tmp;
	    }
	  else
	    {
	      for (i=0; i < e1->p->size ; i++)
		{
		  eadd_mp(&e1->p->arr[i], &res->p->arr[i]);
		}
	      return;
	    }
      }
      else if (res->p->type == periodic)
      {  fprintf(stderr, "eadd_mp: ?addition of different sized periodic nos\n");
         return;
      }
      else /* evector */
      {  fprintf(stderr, "eadd_mp: ?cannot add vectors of different length\n");
         return;
      }
   }
}

/*------------------------------------------------------------*/
/* void edot_mp (v1, v2, res)                                 */
/*       v1 : an enode_mp (vector)                            */
/*       v2 : an enode_mp (vector of constants)               */
/*       res : result (evalue_mp) = v1.v2 (dot product)       */
/* computes the inner product of two vectors. Result = res    */
/*------------------------------------------------------------*/
static void edot_mp(enode_mp *v1, enode_mp *v2, evalue_mp *res)
{
   int i;
   evalue_mp tmp;

   if ((v1->type != evector) || (v2->type != evector))
   {  fprintf(stderr, "edot_mp: ?expecting vectors\n");
      return;
   }
   if (v1->size != v2->size)
   {  fprintf(stderr, "edot_mp: ? vector lengths do not agree\n");
      return;
   }
   if (v1->size <= 0)
   {  mpz_set_si(res->d, 1);	/* set result to 0/1 */
      mpz_set_si(res->n, 0);
      return;
   }

   /* vector v2 is expected to have only rational numbers in */
   /* the array.  No pointers. */

   init_evalue_mp(&tmp);
   emul_mp(&v1->arr[0], &v2->arr[0], res);
   for (i=1; i < v1->size; i++)
   {  /* res = res + v1[i]*v2[i] */
      emul_mp(&v1->arr[i], &v2->arr[i], &tmp);
      eadd_mp(&tmp, res);
   }
   deinit_evalue_mp(&tmp);
}

/*---------------------------------------------------------------------*/
/* PROCEDURES TO COMPUTE ENUMERATION                                   */
/*---------------------------------------------------------------------*/


/*-------------------------------------------------------------*/
/* enode_mp *P_Enum_mp(L, LQ, context, pos, nb_param, dim, lcm)*/
/*     L : list of polyhedra for the loop nest                 */
/*     LQ : list of polyhedra for the parameter loop nest      */
/*     pos : 1..nb_param, position of the parameter            */
/*     nb_param : number of parameters total                   */
/*     dim : total dimension of the polyhedron, param incl.    */
/*     lcm : the denominator of the polyhedron                 */
/* Returns an enode_mp tree representing the pseudo polynomial */
/*   expression for the enumeration of the polyhedron.         */
/* A recursive procedure.                                      */
/*-------------------------------------------------------------*/
static enode_mp *PE_mp2(Polyhedron *L, Polyhedron *LQ,
		Value *context,int pos, int nb_param, int dim, Value lcm );

enode *P_Enum_mp(L, LQ, context, pos, nb_param, dim, lcm)
Polyhedron *L, *LQ;
Value *context;
int pos, nb_param, dim;
Value lcm;
{
	enode_mp *e;
	enode *res;

	e = PE_mp2(L, LQ, context, pos, nb_param, dim, lcm);
	res = enode_mp2value( e );
	free_enode_mp( e );
	return( res );
}
static enode_mp *PE_mp2(L, LQ, context, pos, nb_param, dim, lcm)
Polyhedron *L, *LQ;
Value *context;
int pos, nb_param, dim;
Value lcm;
{  enode_mp *res, *B, *C;
   int hdim, i, j, rank, flag;
	Value n, k, nlcm, noff, k1;
	Value  nLB, nUB;
   mpz_t *temp;
   Matrix_Mp *A;
   mpz_t nexp, g, nn;

#ifdef EDEBUG
   printf("P_Enum: pos=%d, nb_param=%d, dim=%d, lcm=%d\n", pos, nb_param, dim, lcm);
   Polyhedron_Print(stdout, "%3d", L);
   Polyhedron_Print(stdout, "%3d", LQ);
#endif

   mpz_init(nexp);
   mpz_init(g);
   mpz_init(nn);

   /* Xavier Redon modification: handle the case when there is no parameter */
   if (nb_param == 0)
     {
       res = new_enode_mp(polynomial, 1, 0);
       mpz_set_si(res->arr[0].d, 1);
       mpz_set_si(res->arr[0].n, count_points(1,L,context));
       return res;
     }

   /* hdim is the degree of the polynomial + 1 */
   hdim = dim-nb_param+1;		/* homogenous dim w/o parameters */

   /* code to limit generation of equations to valid parameters only */
   /*----------------------------------------------------------------*/
   flag = lower_upper_bounds(pos, LQ, &context[dim-nb_param], &nLB, &nUB);
   if (flag & LB_INFINITY)
   {  if (!(flag & UB_INFINITY))
      {  /* compute nLB such that (nUB-nLB+1) >= (hdim*lcm) */
         nLB = nUB - hdim*lcm + 1;  /* only an upper limit: set lower limit */
			if( nLB > 0 )
				nLB = 0;
		}
      else
			nLB = 0;	/* no upper nor lower limit: set lower limit to 0 */
   }

   /* if (nUB-nLB+1) < (hdim*lcm) then we have more unknowns than equations */
   /* We can: 1. Find more equations by changing the context parameters, or */
   /* 2. Assign extra unknowns values in such a way as to simplify result. */
   /* Think about ways to scan parameter space to get as much info out
      of it as possible */

#ifdef REDUCE_DEGREE
   if ( pos == 1 && (flag & UB_INFINITY) == 0 )
   /* only for finite upper bound on first parameter */
   /* NOTE: only first parameter because subsequent parameters may
            be artificially limited by the choice of the first parameter */
   {
		n = nUB-nLB+1;				/* total number of samples>0 */
		if( n < 0 )
			i = 0;
		else
			i = (n%lcm != 0) ? (n/lcm)+1 : n/lcm;	/* ceiling of n/lcm */
		/* reduce degree of polynomial based on number of sample points */
		if (i < hdim)
			hdim=i;
   }
#endif
   /* hdim is now set */

   /* allocate result structure */
   res = new_enode_mp(polynomial, hdim, pos);
   for (i=0; i < hdim; i++)
   {  res->arr[i].p = new_enode_mp(periodic, lcm, pos);
   }

   /* utility arrays */
   temp = Vector_Mp_Alloc(2*hdim+1);		/* used for Gauss */
   A = Matrix_Mp_Alloc(hdim, 2*hdim+1);		/* used for Gauss */
   B = new_enode_mp(evector, hdim, 0);
   C = new_enode_mp(evector, hdim, 0);

   /*----------------------------------------------------------------*/
   /*                                                                */
   /*                                0<-----+---k--------->          */
   /* |---------noff----------------->-nlcm->-------lcm---->         */
   /* |--- . . . -----|--------------|------+-------|------+-------|-*/
   /* 0          (q-1)*lcm         q*lcm    |  (q+1)*lcm   |         */
   /*                                      nLB          nLB+lcm      */
   /*                                                                */
   /*----------------------------------------------------------------*/
   nlcm = (nLB<0) ? (nLB%lcm+lcm) : (nLB%lcm);
   noff = nLB - nlcm;			/* noff is a multiple of lcm */
   for (k=nlcm ; k < lcm+nlcm; k++)
   {
      for (n=k+noff, i=0; (n < hdim*lcm+nLB); n=n+lcm, i++)
      {
			/* n == i*lcm + k + noff;   */
			/* nlcm <= k < nlcm+lcm     */
			/* n mod lcm == k1 */

#ifdef ALL_OVERFLOW_WARNINGS
         if ( ((flag & UB_INFINITY)==0) && value_gt(n,nUB))
         {
				fprintf(stdout, "Domain Overflow: Parameter #%d:", pos );
				fprintf(stdout, "nLB="VALUE_FMT" n="VALUE_FMT" nUB="VALUE_FMT"\n",
						nLB, n, nUB);
			}
#else
			if ( overflow_warning_flag && ((flag & UB_INFINITY)==0) &&
					value_gt(n,nUB) )
			{
				fprintf(stdout, "\nWARNING: Parameter Domain Overflow." );
				fprintf(stdout, " Result may be incorrect on this domain.\n" );
				overflow_warning_flag = 0;
			}
#endif

			/* set parameter to n */
			value_assign( context[dim-nb_param+pos], n );

			/* setup B vector */
			if (pos == nb_param)
			{  /* call count */
            /* count can only be called when the context is fully specified */
            mpz_set_si(B->arr[i].d, 1);
            mpz_set_si(B->arr[i].n, count_points(1, L, context));
			}
			else /* count is a function of other parameters */
			{  /* call P_Enum recursively */
				mpz_set_si(B->arr[i].d, 0);
				B->arr[i].p =
					PE_mp2(L,LQ->next,context,pos+1,nb_param,dim,lcm);
			}

         /* create i-th equation*/
         /* K_0 + K_1 * n**1 + ... + K_dim * n**dim | Identity Matrix */
         mpz_set_si(A->p[i][0], 0);  /* status bit = equation */
         mpz_set_si(nexp, 1);
			mpz_set_si(nn,n);
         for (j=1; j <= hdim; j++)
         {  mpz_set(A->p[i][j], nexp);
            mpz_set_si(A->p[i][j+hdim], 0);
            mpz_mul(nexp, nexp, nn);
         }
         mpz_set_si(A->p[i][i+1+hdim], 1);
      }
      /* assertion check */
      if (i!=hdim)
         fprintf(stderr, "P_Enum: ?expecting i==hdim\n");

      /* solve hdim (=dim+1) equations with hdim unknowns, result in CNT */
      rank = Gauss_Mp(A, hdim, 2*hdim, temp);

      /* assertion check */
      if (rank != hdim)
      {  fprintf(stderr, "P_Enum: ?expecting rank==hdim\n");
      }

      /* if (rank < hdim) then set the last hdim-rank coefficients to ? */
      /* if (rank == 0) then set all coefficients to 0 */

      /* copy result as k1-th element of periodic numbers */
      k1 = (k < lcm) ? k : k-lcm;

      for (i = 0; i < rank; i++)
      {  /* set up coefficient vector C from i-th row of inverted matrix */
         for (j=0; j<rank; j++)
         {  mpz_gcd(g, A->p[i][i+1], A->p[i][j+1+hdim]);
            mpz_div(C->arr[j].d, A->p[i][i+1], g);
            mpz_div(C->arr[j].n, A->p[i][j+1+hdim], g);
         }

         /* the i-th enode_mp is the lcm-periodic coefficient of term n**i */
         edot_mp(B, C, &(res->arr[i].p->arr[k1]));
      }
   }

   /* reset context */
   value_assign( context[dim-nb_param+pos], VALUE_ZERO );

   /* release memory */
   Vector_Mp_Free(temp,2*hdim+1);
   Matrix_Mp_Free(A);
   free(B);
   free(C);

   return res;
}

/*----------------------------------------------------------------*/
/* Polyhedron_Enumerate(P, C, MAXRAYS)                            */
/*    P : Polyhedron to enumerate                                 */
/*    C : Context Domain                                          */
/*    MAXRAYS : size of workspace                                 */
/* Procedure to count points in a (fixed-parameter) polytope.     */
/* Returns a list of validity domains + evalue_mps EP             */
/*----------------------------------------------------------------*/
Enumeration_mp *PE_mp(P, C, MAXRAYS)
Polyhedron *P;	/* Polyhedron to count */
Polyhedron *C;	/* parameter Context domain */
unsigned MAXRAYS;	/* workspace size */
{
   Polyhedron *L, *CQ, *CQ2, *LQ, *U;
   Param_Polyhedron *PP;
   Param_Domain   *Q;
   Param_Vertices *V;
   int i, j, ix, hdim, dim, nb_param, nbPV;
	Value k, lcm, m1, hdv;
   unsigned bx;
   Value *context;
	Enumeration_mp *en, *res;

	res = NULL;

#ifdef EDEBUG2
fprintf(stderr, "C = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, C);
#endif

   hdim		= P->Dimension + 1;
   dim		= P->Dimension;
   nb_param	= C->Dimension;

   /* Create a context vector size dim+2 */
   context = (Value *) malloc ( (hdim+1)*sizeof(Value) );
   /* Set it to all zeros */
   memset ((char *)context, 0, (hdim+1)*sizeof(Value) );
   /* Set context[hdim] = 1  (the constant) */
   value_assign( context[hdim], VALUE_ONE );
 
#ifdef EDEBUG2
fprintf(stderr, "P = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, P);
#endif
	
	/* don't call Polyhedron2Param_Domain if there are no parameters */
	if( nb_param == 0 )
	{
		L = Polyhedron_Scan(P, C, MAXRAYS);
		res = (Enumeration_mp *)malloc( sizeof(Enumeration_mp) );
		res->next = NULL;
		res->ValidityDomain = Universe_Polyhedron(0); /* no parameters */
       mpz_set_si(res->EP.d, 0);
		res->EP.p = PE_mp2( L, NULL, context, 1, 0, dim, 1);
		Domain_Free( L );
#ifdef EPRINT
fprintf(stdout, "\nEhrhart Polynomial:\n");
print_evalue_mp(stdout, &res->EP);
fprintf( stdout, "\n" );
/* sometimes the final \n lacks (when a single constant is printed) */
#endif
		return( res );
	}
	/* continue with the classical procedure */


   PP = Polyhedron2Param_Domain( P, C, MAXRAYS );
	if( !PP )
	{
#ifdef EPRINT
		fprintf(stdout, "\nEhrhart Polynomial:\n0\n");
#endif
		return( NULL );
   }
   nbPV = PP->nbV;

   for( Q=PP->D ; Q ; Q=Q->next )
   {
   	en = (Enumeration_mp *)malloc( sizeof(Enumeration_mp) );
		en->next = res;
		res = en;

		res->ValidityDomain = Q->Domain;

#ifdef EPRINT
fprintf(stdout, "---------------------------------------\n" );
fprintf(stdout, "Domain :\n");
Print_Domain(stdout, Q->Domain);
#endif

#ifndef ALL_OVERFLOW_WARNINGS
		overflow_warning_flag = 1;
#endif

      /* compute "denominator of P" */
      /* lcm = Least Common Multiple of the denominators of the vertices of P */
      value_assign( lcm, VALUE_ONE );
#ifdef EPRINT
fprintf(stdout, "Vertices :\n");
#endif
 
      for( i=0, ix=0, bx=MSB, V=PP->V ; V && i<nbPV ; i++, V=V->next )
      {
         if (Q->F[ix] & bx)
         {
#ifdef EPRINT
Print_Vertex(stdout, V->Vertex );
fprintf(stdout, "\n" );
#endif
            for( j=0; j<V->Vertex->NbRows; j++)
            {
					 /* a matrix */
				   value_assign(k, V->Vertex->p[j][V->Vertex->NbColumns-1] );
               if ( value_notzero_p(k) && value_notone_p(k) )
					{
						value_assign( m1, Gcd(lcm, k) );
						value_multiply( lcm, k );
						value_division( lcm, m1 );
					}
            }
         }
         NEXT(ix,bx);
      }
#ifdef EDEBUG
fprintf(stderr, "Denominator = "VALUE_FMT"\n", lcm);
#endif

      CQ = DomainIntersection(C, Q->Domain, MAXRAYS);
#ifdef EDEBUG2
fprintf(stderr, "CQ = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, CQ);
#endif

      /* before scanning, add constraints to ensure at least hdim*lcm */
      /* points in every dimension */
		value_assign( hdv, int_to_value((int)hdim) );
		value_assign( m1, value_mult(hdv,lcm) );
		value_substract( m1, int_to_value(2) );
      CQ2 = Polyhedron_Preprocess(CQ, m1, MAXRAYS);
      if (emptyQ(CQ2))
      {
			fprintf(stdout, "Degenerate Domain. Can not continue.\n");
			mpz_set_si(res->EP.d, 1);
			mpz_set_si(res->EP.n, 1 );
		}
		else
		{
#ifdef EDEBUG2
fprintf(stderr, "CQ2 = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, CQ2);
#endif

	      /* L is used in counting the number of points in the base cases */
	      L = Polyhedron_Scan(P, CQ, MAXRAYS);
	      U = Universe_Polyhedron(0);
	      /* LQ is used to scan the parameter space */
	      LQ = Polyhedron_Scan(CQ2, U, MAXRAYS); /* bounds on parameters */
	      Domain_Free(U);
	      Domain_Free(CQ);
	      Domain_Free(CQ2);

#ifdef EDEBUG2
fprintf(stderr, "L = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, L);
fprintf(stderr, "LQ = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, LQ);
#endif
#ifdef EDEBUG3
fprintf(stdout, "\nSystem of Equations:\n");
#endif
			mpz_set_si(res->EP.d, 0);
			res->EP.p = PE_mp2(L, LQ, context, 1, nb_param, dim, lcm);

			Domain_Free( L );
         Domain_Free( LQ );
#ifdef EDEBUG5
fprintf(stdout, "\nEhrhart Polynomial (before simplification):\n");
print_evalue_mp(stdout, &res->EP);
#endif
	      reduce_evalue_mp(&res->EP);
#ifdef EPRINT
fprintf(stdout, "\nEhrhart Polynomial:\n");
print_evalue_mp(stdout, &res->EP);
fprintf( stdout, "\n" );
/* sometimes the final \n lacks (when a single constant is printed) */
#endif
		}
   }
   free( context );
	return res;
}


/******************* TO DO !!! **************/
Enumeration *mp2Enumeration( Enumeration_mp *enmp )
{
	Enumeration *en;

	en = NULL;
	return( en );
}

