/************************************************/
/*  eval_ehrhart.c                              */
/* functions to evaluate an Ehrhart polynomial. */
/* written by Emmanuel Jeannot (c) 1997.        */
/*  Emmanuel.Jeannot@ens-lyon.fr                */
/*  http://www.ens-lyon.fr/~ejeannot            */
/************************************************/

#include "polylib.h"
#include "gmp.h"
#include "eval_ehrhart.h"

#undef EVAL_EHRHART_DEBUG

/********************************************************/
/* function in domain                                   */
/*    check if the parameters in list_args              */
/*    verifies the constraints of Polyhedron P          */
/********************************************************/
int in_domain(Polyhedron *P,int *list_args){
  int col,row;
  int value; /*value of the constraint of a row when
	       parameters are instanciated*/

  /*P->Constraint constraint matrice of polyhedron P*/

  for(row=0;row<P->NbConstraints;row++){
    value=P->Constraint[row][P->Dimension+1]; /*constant part*/
    for(col=1;col<P->Dimension+1;col++)
      value+=P->Constraint[row][col]*list_args[col-1];

    if (P->Constraint[row][0]){
      if (value<0) 
	return 0; /*if value is not >=0 then this
		    constraint is not respected*/
    } else {
      if (value!=0)
	return 0; /*if value is not =0 then this
		    constraint is not respected*/
    }
  }
  
  /*if not retrun before this point => all 
    the constraints are respected*/
  return 1;
}

/****************************************************/
/* function compute enode                           */
/*     compute the value of enode p with parameters */
/*     list  list_args                              */
/*     compute the polynomial or the periodic       */
/****************************************************/

double compute_enode(enode *p, int *list_args)
{
  int i, param;
  double res = 0.0;

  if (!p)
    return 0;

  if (p->type == polynomial)
    {
      param = list_args[p->pos-1];
      /*compute the polynomial using Horner's rule*/
      for (i = p->size-1; i > 0; i--)
	{
	  res += compute_evalue(&p->arr[i], list_args);
	  res *= param;
	}
      res += compute_evalue(&p->arr[0], list_args);
    }
  else if (p->type == periodic)
    {
      param = list_args[p->pos-1];
      /*choose the right element of the periodic*/
      i = (param)%(p->size);/* not the Ehrhart convention but
			       the Clauss/Wilde/Loechner one*/
      res = compute_evalue(&p->arr[i], list_args);
    }
  return res;
}

/*************************************************/
/* return the value of Ehrhart Polynomial        */
/* It returns a double, because since it is      */
/* a recursive function, some intermediate value */
/* might not be integral                         */
/*************************************************/

double compute_evalue(evalue *e, int *list_args)
{
  double res;
  
  if (mpz_cmp_si(e->d, 0) != 0)
    {   
      if (mpz_cmp_si(e->d, 1) != 0) 
	res = mpz_get_d(e->n) / mpz_get_d(e->d);
      else 
	res = mpz_get_d(e->n);
    } 
  else 
    res = compute_enode(e->p, list_args);
  return res;
}

/****************************************************/
/* function compute_poly :                          */
/* Check for the good validity domain               */
/* return the number of point in the Polyhedron     */
/* Using the Ehrhart pseudo-polynomial              */
/****************************************************/
int compute_poly(Enumeration *en, int *list_args)
{
  if (!en)
    return 0;	/* no ehrhart polynomial */
  if (en->ValidityDomain)
    {
      if(!en->ValidityDomain->Dimension)
	return 0; /* no parameters, empty polyhedron */
    }
  else 
    return 0; /* no Validity Domain */
  
  while(en)
    {
      if(in_domain(en->ValidityDomain,list_args))
	return((int)compute_evalue(&en->EP, list_args));
      else
	en = en->next;
    }
  
  return 0; /* no compatible domain with the arguments */
}

/****************** End Compute poly *******************************/

