/* Bibliotheque pour le raycasting basee sur le tutoriel de Lode Vandevenne
 * http://student.kuleuven.be/~m0216922/CG/raycasting.html
 * Lode Vandevenne et Cedric Bastoul
 */

#include <stdlib.h>
#include <math.h>
#include <SDL/SDL.h>
#include "moteur.h"
#include "sdl.h"

#define VITESSE_DEPLACEMENT 5   /* En nombre de carres par seconde */
#define VITESSE_ROTATION 3      /* En radians par seconde */


extern int carte[LARGEUR_CARTE][HAUTEUR_CARTE]; /* Carte de la scene 3D */
extern int ralentissement;      /* Pour ralentir l'application */

double posX = 22, posY = 12;    /* Position de depart en x et y */
double dirX = -1, dirY = 0;     /* Direction initiale en x et y */
double planX = 0, planY = 0.66; /* Plan de la camera initial en x et y */
double accumulation;            /* Sert juste pour ralentir l'application */


/*----------------     Fonctionnalites liees aux couleurs     ----------------*/

/* Structure de couleur RGB */
struct couleur {
  int rouge;
  int vert;
  int bleu;
};

/* Constantes couleur */
const struct couleur noir    = {  0,  0,  0};
const struct couleur rouge   = {255,  0,  0};
const struct couleur vert    = {  0,255,  0};
const struct couleur bleu    = {  0,  0,255};
const struct couleur blanc   = {255,255,255};
const struct couleur jaune   = {255,255,  0};
const struct couleur cyan    = {  0,255,255};
const struct couleur magenta = {255,  0,255};
const struct couleur gris    = {128,128,128};
const struct couleur marron  = {128,  0,  0};
const struct couleur violet  = {128,  0,128};

/* Fonction rendant une couleur attenuee de moitie par rapport au parametre */
struct couleur moteur_demi_couleur(struct couleur ancienne) {
  struct couleur nouvelle;
  nouvelle.rouge = ancienne.rouge / 2;
  nouvelle.vert  = ancienne.vert  / 2;
  nouvelle.bleu  = ancienne.bleu  / 2;
  return nouvelle;
}


/*---------------     Fonctionnalites liees aux raycasting     ---------------*/

/* Fonction dessiner_colonne :
 * Cette fonction dessine la colonne de pixels de coordonnee x en appliquant
 * l'algorithme de raycasting, sachant que la fenetre a une largeur
 * "largeur" et une hauteur "hauteur"
 */
void moteur_dessiner_colonne(int x, int largeur, int hauteur) {
  int carteX,           /* Position du carre dans lequel on est en x */
      carteY,           /* Position du carre dans lequel on est en y */
      pasX,             /* direction vers laquelle on va en x (-1 ou 1) */
      pasY,             /* direction vers laquelle on va en y (-1 ou 1) */
      touche = 0,       /* 1 si on a touche un mur, 0 sinon */
      type_cote,        /* 1 pour mur nord-sud, 0 pour un est-ouest */
      hauteur_ligne,    /* Hauteur de la partie de mur a dessiner */
      pixel_haut,       /* Pixel du bas de la partie de mur a dessiner */
      pixel_bas;        /* Pixel du haut de la partie de mur a dessiner */
  double cameraX,       /* Coordonnee x dans l'espace de la camera */
         rayPosX,       /* Position du rayon en x */
	 rayPosY,       /* Position du rayon en y */
	 rayDirX,       /* Direction du rayon en x */
	 rayDirY,       /* Direction du rayon en y */
         bordDistX,     /* Longueur du rayon jusqu'au prochain bord en x */
         bordDistY,     /* Longueur du rayon jusqu'au prochain bord en y */
         deltaDistX,    /* Longueur du rayon d'un cote x a l'autre cote x */
	 deltaDistY,    /* Longueur du rayon d'un cote y a l'autre cote y */
         perpWallDist;  /* Distance projetee sur la direction de la camera */
  struct couleur color; /* Couleur de la partie de mur a dessiner */
  
  /* On calcule la position et la direction du rayon */
  rayPosX = posX;
  rayPosY = posY;
  cameraX = 2 * x / (double)(largeur) - 1;
  rayDirX = dirX + planX * cameraX;
  rayDirY = dirY + planY * cameraX;
  
  /* On calcule le carre de la carte dans lequel on est */
  carteX = (int)(rayPosX);
  carteY = (int)(rayPosY);
       
  /* On calcule la longueur du rayon d'un cote x ou y a l'autre cote x ou y */
  deltaDistX = sqrt(1 + (rayDirY * rayDirY) / (rayDirX * rayDirX));
  deltaDistY = sqrt(1 + (rayDirX * rayDirX) / (rayDirY * rayDirY));
       
  /* On calcule la direction vers laquelle on va et la longueur jusqu'au bord */
  if (rayDirX < 0) {
    pasX = -1;
    bordDistX = (rayPosX - carteX) * deltaDistX;
  }
  else {
    pasX = 1;
    bordDistX = (carteX + 1.0 - rayPosX) * deltaDistX;
  }
  if (rayDirY < 0) {
    pasY = -1;
    bordDistY = (rayPosY - carteY) * deltaDistY;
  }
  else {
    pasY = 1;
    bordDistY = (carteY + 1.0 - rayPosY) * deltaDistY;
  }

  /* Coeur du raycasting : Digital Differential Analysis ou DDA */
  while (touche == 0) {
    /* On saute au prochain carre ou en direction x OU en direction y */
    if (bordDistX < bordDistY) {
      bordDistX += deltaDistX;
      carteX += pasX;
      type_cote = 0;
    }
    else {
      bordDistY += deltaDistY;
      carteY += pasY;
      type_cote = 1;
    }
    /* On regarde si on a frappe dans un mur */ 
    if (carte[carteX][carteY] > 0)
      touche = 1;
  } 
  
  /* On calcule la distance projetee dans la direction de la camera */
  if (type_cote == 0)
    perpWallDist = fabs((carteX - rayPosX + (1 - pasX) / 2) / rayDirX);
  else
    perpWallDist = fabs((carteY - rayPosY + (1 - pasY) / 2) / rayDirY);
      
  /* On calcule la hauteur de la colonne de mur a dessiner a l'ecran */
  hauteur_ligne = abs((int)(hauteur / perpWallDist));
       
  /* On calcule le y du pixel du haut et du bas de la ligne de mur */
  pixel_haut = -hauteur_ligne / 2 + hauteur / 2;
  if(pixel_haut < 0)
    pixel_haut = 0;
  pixel_bas = hauteur_ligne / 2 + hauteur / 2;
  if(pixel_bas >= hauteur)
    pixel_bas = hauteur - 1;
        
  /* On choisit la bonne couleur */
  switch(carte[carteX][carteY]) {
    case 1:  color = rouge;   break;
    case 2:  color = vert;    break;
    case 3:  color = bleu;    break;
    case 4:  color = blanc;   break;
    case 5:  color = jaune;   break;
    case 6:  color = cyan;    break;
    case 7:  color = magenta; break;
    case 8:  color = gris;    break;
    case 9:  color = marron;  break;
    default: color = violet;  break;
  }
      
  /* En fonction du cote x ou y, on ajuste la couleur */
  if (type_cote == 1) {
    color = moteur_demi_couleur(color);
  }

  /* Juste pour ralentir un peu et voir les effets du multithread. La variable
   * accumulation doit etre en global pour ne pas que le compilateur ne
   * retire ce bout de code qui ne sert a rien (par "dead code elimination").
   */
  int k;
  for (k=0; k < ralentissement; k++)
    accumulation += 3.14;
       
  /* Enfin on dessine la colonne de mur a l'ecran */
  sdl_ligne_verticale(x,pixel_haut,pixel_bas,color.rouge,color.vert,color.bleu);
}


/* Fonction gestion_actions :
 * Cette fonction se charge de la gestion des touches du clavier, en
 * particulier le pave directionnel pour le deplacement et les touches Esc et Q
 * pour quitter. Elle se charge aussi du reglage de la vitesse de deplacement.
 */
void moteur_gestion_actions(double temps_frame) {
  double old_dirX, old_planX;
  double vitesse_dep;            /* Vitesse de deplacement en ligne */
  double vitesse_rot;            /* Vitesse de deplacement en rotation */
  
  /* Ajustement de la vitesse de deplacement et de rotation */
  vitesse_dep = temps_frame * VITESSE_DEPLACEMENT;
  vitesse_rot = temps_frame * VITESSE_ROTATION;
  
  sdl_touches_lire();
  
  /* Touche up : on se deplace vers l'avant si aucun mur devant soi */
  if (sdl_touches_appuyee(SDLK_UP)) {
    if(carte[(int)(posX + dirX * vitesse_dep)][(int)(posY)] == 0)
      posX += dirX * vitesse_dep;
    if(carte[(int)(posX)][(int)(posY + dirY * vitesse_dep)] == 0)
      posY += dirY * vitesse_dep;
  }

  /* Touche down : on se deplace vers l'arriere si aucun mur devant soi */
  if (sdl_touches_appuyee(SDLK_DOWN)) {
    if(carte[(int)(posX - dirX * vitesse_dep)][(int)(posY)] == 0)
      posX -= dirX * vitesse_dep;
    if(carte[(int)(posX)][(int)(posY - dirY * vitesse_dep)] == 0)
      posY -= dirY * vitesse_dep;
  }
  
  /* Touche droite : on se deplace vers la droite si aucun mur devant soi */
  if (sdl_touches_appuyee(SDLK_RIGHT)) {
    /* La direction ET le plan de la camera doivent effectuer la rotation */
    old_dirX = dirX;
    dirX = dirX * cos(-vitesse_rot) - dirY * sin(-vitesse_rot);
    dirY = old_dirX * sin(-vitesse_rot) + dirY * cos(-vitesse_rot);
    old_planX = planX;
    planX = planX * cos(-vitesse_rot) - planY * sin(-vitesse_rot);
    planY = old_planX * sin(-vitesse_rot) + planY * cos(-vitesse_rot);
  }
  
  /* Touche gauche : on se deplace vers la gauche si aucun mur devant soi */
  if (sdl_touches_appuyee(SDLK_LEFT)) {
    /* La direction ET le plan de la camera doivent effectuer la rotation */
    old_dirX = dirX;
    dirX = dirX * cos(vitesse_rot) - dirY * sin(vitesse_rot);
    dirY = old_dirX * sin(vitesse_rot) + dirY * cos(vitesse_rot);
    old_planX = planX;
    planX = planX * cos(vitesse_rot) - planY * sin(vitesse_rot);
    planY = old_planX * sin(vitesse_rot) + planY * cos(vitesse_rot);
  }
  
  /* Touche 'Esc' ou 'q', on arrete l'application */
  if (sdl_touches_appuyee(SDLK_ESCAPE) || sdl_touches_appuyee(SDLK_q)) {
    sdl_quitter();
    exit(0);
  }
}
