package p2pmpi.mpi;

import p2pmpi.mpi.dev.*;
import p2pmpi.message.*;
import p2pmpi.common.*;

import java.net.*;
import java.io.*;
import java.util.*;
import java.nio.*;
import java.nio.channels.*;

/**
 * Point-to-Point communication class
 */
public class Comm {
	private static short globalCommID = 0;
	protected Device myDevice;
	protected short commID;
	protected Group myGroup;
	protected MapRankTable mapRankTable;
	//protected Properties midLog = new Properties();
	protected HashMap<String, Integer> midLog = new HashMap<String, Integer>();

	private int numReplica;
	private int[] myReplica;
	private int[] myReplicaRIL;
	private static int MAX_REPLICA = 256;
	private int numProc;
	private int waitingTime;

	// Comm world construct
	/**
	 * Internal use
	 */
	public Comm(Device myDevice, RankTable rankTable, int rank, 
		    int rankInList, int numRank, MapRankTable mapRankTable) {
		this.myDevice = myDevice;
		myGroup = new Group(myDevice, rankTable, rank, rankInList, numRank, mapRankTable);

		commID = globalCommID;
		globalCommID++;

		// find my replica
		myReplica = new int[MAX_REPLICA];
		RankTable tmpTable = myGroup.__getCommTable();

		MapRankTable tmpMapTable = myGroup.__getMapCommTable();
		//int numProc = tmpTable.size();
		numProc = tmpMapTable.size();
		int myRank  = myGroup.Rank();

		for(int i = 0; i < numProc; i++) {
			int realRIL = tmpMapTable.getRankInList(i);
			if(tmpTable.isAlive(realRIL)) {
				if(myRank == tmpMapTable.getCurrentRank(i)) {
					myReplica[numReplica] = i;
					numReplica++;
				}
			}
		}
	
		// find master
		int myRankInList = myGroup.RankInList();
		boolean master = true;
		int myMaster = myRankInList;
		for(int i = 0; i < numReplica; i++) {
			int realRIL = tmpMapTable.getRankInList(myReplica[i]);
			// Recheck isAlive again to make sure
			if(tmpTable.isAlive(realRIL)) {
				if(myRankInList > myReplica[i]) {
					master = false;
				} 
				if(myMaster > myReplica[i]) {
					myMaster = myReplica[i];
				}
			}
		}
		//TODO: about the master process
		myDevice.setMaster(master);
		myDevice.setMyMaster(myMaster);
		String usedProtocol = myDevice.getGossipProtocol();
		if(usedProtocol.equalsIgnoreCase("DBRR")) {
			waitingTime = (((int)(Math.log(numProc)/Math.log(2)) * 3) * myDevice.getTGossip() * 2) + myDevice.getTMargin(); 
		} else {
			waitingTime = (((int)(Math.log(numProc)/Math.log(2)) * 2) * myDevice.getTGossip() * 2) + myDevice.getTMargin(); 
		}
		int repRIL;
		myReplicaRIL = new int[myReplica.length];
		for(int i = 0; i < numReplica; i++) {
			repRIL = myReplica[i];
			myReplicaRIL[i] = tmpMapTable.getRankInList(myReplica[i]);
		}
	
		myDevice.registerMap(commID, myGroup.__getMapCommTable());
		myDevice.registerCOMMWORLD(rankTable);
	}

	// construct from Group class
	/**
	 * Create a new communicator by group
	 *
	 * @param group Group
	 */
	public Comm(Group group) {
		myGroup      = group;
		this.myDevice = group.__getDevice();

		commID = globalCommID;
		globalCommID++;



		// find my replica
		myReplica = new int[MAX_REPLICA];
		RankTable tmpTable = myGroup.__getCommTable();
		MapRankTable tmpMapTable = myGroup.__getMapCommTable();
		//int numProc = tmpTable.size();
		numProc = tmpMapTable.size();
		int myRank  = myGroup.Rank();

		for(int i = 0; i < numProc; i++) {
			int realRIL = tmpMapTable.getRankInList(i);
			if(tmpTable.isAlive(realRIL)) {
				if(myRank == tmpMapTable.getCurrentRank(i)) {
					myReplica[numReplica] = i;
					numReplica++;
				}
			}
		}
	
		// find master
		int myRankInList = myGroup.RankInList();
		boolean master = true;
		int myMaster = myRankInList;
		for(int i = 0; i < numReplica; i++) {
			int realRIL = tmpMapTable.getRankInList(myReplica[i]);
			// Recheck isAlive again to make sure
			if(tmpTable.isAlive(realRIL)) {
				if(myRankInList > myReplica[i]) {
					master = false;
				} 
				if(myMaster > myReplica[i]) {
					myMaster = myReplica[i];
				}
			}
		}

		myDevice.setMaster(master);
		myDevice.setMyMaster(myMaster);
		String usedProtocol = myDevice.getGossipProtocol();
		//sendLog = myGroup.__getLog();
		//backupMessage = myGroup.__getBackupMessage();

		if(usedProtocol.equalsIgnoreCase("DBRR")) {
			waitingTime = (((int)(Math.log(numProc)/Math.log(2)) * 3) * myDevice.getTGossip() * 2) + myDevice.getTMargin(); 
		} else {
			waitingTime = (((int)(Math.log(numProc)/Math.log(2)) * 2) * myDevice.getTGossip() * 2) + myDevice.getTMargin(); 
		}
		//System.out.println("Waiting Time for Sending = " + waitingTime);

		int repRIL;
		myReplicaRIL = new int[myReplica.length];
		for(int i = 0; i < numReplica; i++) {
			repRIL = myReplica[i];
			myReplicaRIL[i] = tmpMapTable.getRankInList(myReplica[i]);
		}

		myDevice.registerMap(commID, myGroup.__getMapCommTable());
	}

      	/**
      	 * Returns group of this communicator
	 *
	 * @return group
	 **/
	public Group Group() {
		//return (new Group(myGroup));
		return myGroup;
	}

      	/**
       	* Returns the number of processes in communicator.
	*
	* @return number of processes in co
	 **/
	public int Size() {
		return myGroup.Size();
	}

      	/**
       	* Internal use.
	**/
	public int SizeTotal() {
		return myGroup.__sizetotal();
	}

      /**
       * Returns my rank number in communicator.
       *
       * @return MPI rank
	 **/
	public int Rank() {
		return myGroup.Rank();
	}

      /**
       * Basic send operation.
	 * 
	 * @param sendBuffer send buffer array
       * @param offset initial offset in send buffer
       * @param count number of items to send
       * @param datatype datatype of send buffer items
       * @param dest rank of destination
       * @param tag message tag
	 **/
	public int Ssend(Object sendBuffer, int offset, int count,
			Datatype datatype, int dest, int tag) {

		if(Rank() == MPI.UNDEFINED) {
			return -1;
		}
		if(datatype.getBaseType() == Datatype.NULL) {
			return 0;
		}

		int numsend = increaseAndGetNumSend(Rank(), dest, tag);
		ByteBuffer sendMessage = datatype.getByteBuffer(sendBuffer, offset, count, commID, Rank(), dest, tag, numsend);

		RankTable commTable = myGroup.__getCommTable();
		MapRankTable mapCommTable = myGroup.__getMapCommTable();
		Vector<Integer> dstRIL = mapCommTable.getRankInListByCurrentRank(dest);

		if (myDevice.isMaster()) {	// Master of replica
			// Get IP:Port of Destination
			myDevice.ssend(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); //Master sends messages
		} else {
			myDevice.sendToBackup(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); 
		}
		return count;	//return value
	}


	public int Send(Object sendBuffer, int offset, int count,
			Datatype datatype, int dest, int tag) {

		if(Rank() == MPI.UNDEFINED) {
			return -1;
		}
		if(datatype.getBaseType() == Datatype.NULL) {
			return 0;
		}

		int numsend = increaseAndGetNumSend(Rank(), dest, tag);
		
		//long t1 = System.currentTimeMillis();
		ByteBuffer sendMessage = datatype.getByteBuffer(sendBuffer, offset, count, commID, Rank(), dest, tag, numsend);
		RankTable commTable = myGroup.__getCommTable();
		MapRankTable mapCommTable = myGroup.__getMapCommTable();
		Vector<Integer> dstRIL = mapCommTable.getRankInListByCurrentRank(dest);

		if (myDevice.isMaster()) {	// Master of replica
			// Get IP:Port of Destination
			myDevice.send(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); //Master sends messages

		} else {
			myDevice.sendToBackup(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); 
		}
		return count;	//return value
	}

	public Request Isend(Object sendBuffer, int offset, int count,
			Datatype datatype, int dest, int tag) {

		NIORequest req = null;
		if(Rank() == MPI.UNDEFINED) {
			return null;
		}
		if(datatype.getBaseType() == Datatype.NULL) {
			return null;
		}

		int numsend = increaseAndGetNumSend(Rank(), dest, tag);
		ByteBuffer sendMessage = datatype.getByteBuffer(sendBuffer, offset, count, commID, Rank(), dest, tag, numsend);
		RankTable commTable = myGroup.__getCommTable();
		MapRankTable mapCommTable = myGroup.__getMapCommTable();

		Vector<Integer> dstRIL = mapCommTable.getRankInListByCurrentRank(dest);

		if (myDevice.isMaster()) {	// Master of replica
			//Master sends messages
			req = myDevice.isend(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); 

		} else {
			req = myDevice.sendToBackup(commTable, sendMessage, dstRIL, commID, Rank(), dest, tag, numsend, myReplicaRIL); 
		}
		return new Request(req);	//return value
	}



	/*
	public int requestCheckStatus(int threadID) 
	{
		ThreadStatus ts;
		Enumeration e = threadTable.elements();
		while(e.hasMoreElements()) {

			ts = (ThreadStatus)e.nextElement();
			if(ts.getID() == threadID) {
				return ts.getStatus();
			}

		}
		return -1;
	}
	*/


     

	public Status Sendrecv(Object sendBuffer, int sendOffset, int sendCount,
			       Datatype sendType, int dest, int sendTag,
			       Object recvBuffer, int recvOffset, int recvCount,
			       Datatype recvType, int source, int recvTag)
	{
		Request req = Irecv(recvBuffer, recvOffset, recvCount, recvType, source, recvTag);
		Send(sendBuffer, sendOffset, sendCount, sendType, dest, sendTag);
		return req.Wait(); //return status
	}
	/**
	 * Basic (blocking) receive.
	 * 
	 * @param recvBuffer receive buffer array
	 * @param offset initial offset in receive buffer
	 * @param count number of items to receive
	 * @param datatype datatype of received items
	 * @param src rank of source
	 * @param tag message tag
	 **/

	public Status Recv(Object recvBuffer, int offset, int count,
			Datatype datatype, int src, int tag) 
	{
		if(Rank() == MPI.UNDEFINED) {
			return null;
		}
		if(datatype.getBaseType() == Datatype.NULL) {
			return null;
		}

		int numsend = increaseAndGetNumSend(src, Rank(), tag);
		IStatus istatus = myDevice.recv(recvBuffer, offset, count, commID, src, Rank(), tag, numsend, datatype.getBuffer());
		return new Status(istatus);
	}
	/**
	 * Begins a non-blocking receive.
	 * 
	 * @param recvBuffer receive buffer array
	 * @param offset initial offset in receive buffer
	 * @param count number of items to receive
	 * @param datatype datatype of received items
	 * @param src rank of source
	 * @param tag message tag
	 **/
	public Request Irecv(	Object recvBuffer, int offset, int count, 
			Datatype datatype, int src, int tag) 
	{

		if(Rank() == MPI.UNDEFINED) {
			return null;
		}
		if(datatype.getBaseType() == Datatype.NULL) {
			return null;
		}
		int numsend = increaseAndGetNumSend(src, Rank(), tag);
		NIORequest req = myDevice.irecv(recvBuffer, offset, count, commID, src, Rank(), tag, numsend, datatype.getBuffer());
		Request reqret = new Request(req);
		return reqret;	
	}


	private synchronized int increaseAndGetNumSend(int src, int dest, int tag) {
		int iNumSent;
		String midKey = src + "_" + dest + "_" + tag;
		Integer i = midLog.get(midKey);
		if(i == null) {
			iNumSent = 0;
			midLog.put(midKey, new Integer(1));
		} else {
			iNumSent = i.intValue();
			midLog.put(midKey, new Integer(iNumSent+1));
		}
		return iNumSent;
	}
}
