/**
 * $Id:
 * PeerCache.java
 *
 * Created on jul 19 23:22:45 CEST 2007
 *
 * documentation used : http://java.sun.com/j2se/1.5.0/docs/api/org/w3c/dom/Document.html
 * 
 * @author  Stephane Genaud
 * 
 **/
package p2pmpi.mpd;

import java.io.File;
import java.io.IOException;

import org.w3c.dom.*; 
import org.xml.sax.*; 

import javax.xml.parsers.*;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;



public class PeerCache {

	private Document desc=null; // the DOM document maintaining information
	private static DocumentBuilder docBuilder = null; // the builder instance used
	private String errorStatus = null;

	//----------------------- C o n s t r u c t o r   M e t h o d s ---------------------------------
	/**
	 * Constructor 1
	 **/
	public PeerCache() {
		// first, launch a builder
		try {
			docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch(ParserConfigurationException e) {
			System.err.println("[PeerCache ] Error : cannot create an XML DocumentBuilder for generating MPD advertisement. Exiting.");
			System.exit(1);
		}
		// initialize the DOM document
		desc = docBuilder.newDocument();
		desc.setXmlVersion("1.0");
 		// root element
		desc.appendChild( desc.createElement("peercache"));
		// all subsequent properties will be children of root element
	}


	/**
	 * Constructor 2
	 *
	 * @param filename name of the file from which to build the cache in memory
	 **/
	public PeerCache ( String filename ) {
		try {
			docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch(ParserConfigurationException e) {
			System.err.println("[PeerCache] Error : cannot create an XML DocumentBuilder for generating MPD advertisement. Exiting.");
			System.exit(1);
		}

		desc = readDocument( filename );
		if (desc != null ) {
			//System.out.println("reading " + filename + " ...");
			// set the id so that we can search hosts entries
			NodeList nl = desc.getDocumentElement().getElementsByTagName( "peer" );
			for (int i=0;i<nl.getLength();i++)
				((Element)nl.item(i)).setIdAttribute("id",true);
		}
		else {
		      // initialize the DOM document
	            desc = docBuilder.newDocument();
            	desc.setXmlVersion("1.0");
            	// root element
            	desc.appendChild( desc.createElement("peercache") );
            	// all subsequent properties will be children of root element
		}

	} 

	//----------------------- G e t / S e t / R e m o v e   M e t h o d s ---------------------------------

	/**
	 * Tell which last error message was set.
	 **/
	 public String getErrorStatus( ) {
		if (this.errorStatus == null)
			return("No error reported");
		else
			return( this.errorStatus );
	 }

	/**
	 * Tell if a given ID attribute present in the document.
	 * @param id the id we seach for.
	 * @return true if at least one tag has this id, false otherwise 
	 **/
	 public boolean hasId( String id ) {
		return(  desc.getElementById( id ) != null );
	}

	/**
	 * Insert in the DOM document a property (tag) under a peer identifed by 'id'.
	 *
	 * @param id the parent to which we want to add the tag tagname 
	 * @param tagname the tag name of the property (e.g. <cpu>)
	 * @param text the contents inside the tag (e.g. 512MB)
	 * @return true if the element could be inserted, false otherwise
	 *         i.e., the parent searched for (id) could not be found, 
	 *         or the tag could not be created in the document.
	 */
	private boolean addProperty( String id, String tagname, String text) {
		Element e;
		try {
			e = desc.createElement( tagname );
		} catch (DOMException excpt) { 
			return (false); 
		}
		//e.addAttribute("id",text);
		if (text == null)
			e.setTextContent( "n/a" );
		else		
			e.setTextContent( text );
		// attach e to peerElmt
		Element peerElmt = desc.getElementById( id );
		if (peerElmt == null) {
			//System.out.println("Node creation node <peer id=" + id + ">");
			peerElmt = desc.createElement("peer");
			peerElmt.setAttribute( "id", id );
			peerElmt.setIdAttribute( "id", true );
			desc.getFirstChild().appendChild( peerElmt );
		}
		peerElmt.appendChild( e );
		return (true);
	}


	/** Set methods **/
	public boolean setMpdVer  ( String id,String s ) { return(addProperty( id, "mpdver", s ));}
	public boolean setCpuName ( String id,String s ) { return(addProperty( id, "cpu", s ));}
	public boolean setCpuFreq ( String id,String s ) { return(addProperty( id, "cpufreq", s ));}
	public boolean setArch    ( String id,String s ) { return(addProperty( id, "arch", s ));}
	public boolean setOsName  ( String id,String s ) { return(addProperty( id, "osname", s ));}
	public boolean setMem     ( String id,String s ) { return(addProperty( id, "mem", s ));}
	public boolean setFdPort  ( String id,String s ) { return(addProperty( id, "fdport", s ));}
	public boolean setFtPort  ( String id,String s ) { return(addProperty( id, "ftport", s ));}
	public boolean setRsPort  ( String id,String s ) { return(addProperty( id, "rsport", s ));}
	public boolean setMpdPort ( String id,String s ) { return(addProperty( id, "mpdport", s ));}

	/**
	 * Get the value associated with an ID and a tag in the DOM doc. 
	 * 
	 * for : <peer id="foohost"><cpu>Intel Xeon</cpu></peer>, 
       * getProperty("foohost","cpu") returns "intel Xeon"
       *
	 * @param tagname the selected tag
	 * @param id the tag's id attribute to search tagname into.
	 * @return the string associated with the tag tagname, inside the tag with ID=id, or null if id does not exist
	 */
	private String getProperty( String id, String tagname ) {
		Element peerElmt = desc.getElementById( id );
            if (peerElmt == null) {
			System.err.println("Invalid id in getProperty("+id+","+tagname+")");
			return( null );
		}
		Node tag = peerElmt.getElementsByTagName( tagname ).item( 0 );
		// to check tag, use  tag.getNodeName()
		Node data = tag.getFirstChild();
 		// we should find concrete data (text) under this tag
		if ( data.getNodeType() != Node.TEXT_NODE)
			return( new String("undefined"));
		return ( data.getNodeValue());			    
	}


	/** Get methods **/
	public String getMpdVer  ( String id ) { return( getProperty( id, "mpdver" ));}
	public String getCpuName ( String id ) { return( getProperty( id, "cpu" )); } 
	public String getCpuFreq ( String id ) { return( getProperty( id, "cpufreq" )); } 
	public String getArch    ( String id ) { return( getProperty( id, "arch" )); } 
	public String getOsName  ( String id ) { return( getProperty( id, "osname" )); } 
	public String getMem     ( String id ) { return( getProperty( id, "mem" )); } 
	public String getFdPort  ( String id ) { return( getProperty( id, "fdport" )); } 
	public String getFtPort  ( String id ) { return( getProperty( id, "ftport" )); } 
	public String getRsPort  ( String id ) { return( getProperty( id, "rsport" )); } 
	public String getMpdPort ( String id ) { return( getProperty( id, "mpdport" )); } 


	/**
	 * remove a peer entry
	 * 
	 * @param id the string that identifies the peer (<peer id="..">)
	 * @return true if the removal succeeded, false if id could not be found
	 **/
	public boolean remove( String id ) {
		Element peerElmt = desc.getElementById( id );
		
		if (peerElmt == null)
			return (false);
		Node parent = peerElmt.getParentNode();
		Node save  = parent.removeChild( peerElmt );
		return (true);
	}	


	//----------------------- I / O   M e t h o d s ---------------------------------

	/**
	 * Write the DOM document to a file under given filename.
	 * 
	 * @param doc the DOM document to write to a file
	 * @param filename name of the output file 
	 * @return true if document is saved, false in case of failure.
	 */
	public boolean saveDocument( String filename ) {

		/* Filename should be something like  
		 * PEER_CACHE=c:/p2pmpi-current/tmp/cache.xml 
		 * on Windows 
		 **/		
		  Source source = new DOMSource( desc );

		  // The transformation output goes to a file
		  Result sres = new StreamResult( new File( filename ) );

		  // get an XML transformer instance
		  Transformer transfo = null;
		  try {
			    transfo = TransformerFactory.newInstance().newTransformer();
		  } catch(TransformerConfigurationException e) {
			    this.errorStatus = "Warning : can not create an XML transformer during file saving: "+e.getCause();
			    System.err.println( this.errorStatus );
			    return( false );
		  }
		  // output, XML format
		  transfo.setOutputProperty(OutputKeys.METHOD, "xml");
		  // include XML declaration
		  transfo.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "no");
		  // UTF-8 charset. could be ISO-8859-1 ...
		  transfo.setOutputProperty(OutputKeys.ENCODING, "utf-8");
		  // set indent on 
		  transfo.setOutputProperty(OutputKeys.INDENT, "yes");

		  try {
			    transfo.transform( source, sres);

		  } catch(TransformerException e) {
			    this.errorStatus = "[PeerCache] Could not perform saving. Reason :"+e.getCause();
			    System.err.println( this.errorStatus );
			    return( false );
		  }
		  return( true );
	}

	/**
	 * Read an XML document from a file and build the corresponding DOM.
	 * 
	 * @param filename name of the input file 
	 * @return doc the DOM document 
	 */
	private static Document readDocument(String filename) {
		try {
			return docBuilder.parse(new File(filename));
		} catch(SAXException e) {
			System.err.println("[PeerCache] Warning : cannot parse " + filename);
		} catch (IOException e) {
			System.err.println("[PeerCache] Warning : no cache file " + filename);
		}
		return null;
	}


	/**
	 * display cache based on the DOM document in memeory
       **/
	public void display() {
		displayDocument( desc );
	}
	/**
	 * Display a DOM document
	 * 
	 * @param doc the document to display
	 */
	private static void displayDocument( Document doc ) {
		Element e = doc.getDocumentElement();
		displayElement( e );
	}
	
	/**
	 * Display a DOM element along with its attributes (recursively)
	 * 
	 * @param e the element to display 
	 */
	private static void displayElement(Element e) {
		System.out.print("<" + e.getNodeName() + " ");
		
		NamedNodeMap attr = e.getAttributes();
		for(int i=0; i<attr.getLength(); i++) {
			Attr a = (Attr)attr.item(i);
			System.out.print(a.getName() + "=\"" + a.getNodeValue() + "\" ");
		}
		System.out.println(">");
		
		for(Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
			switch(n.getNodeType()) {
			case Node.ELEMENT_NODE:
				displayElement((Element)n);
				break;
			case Node.TEXT_NODE:
				String data = ((Text)n).getData();
				System.out.println(data);
				break;
			}
		}
		System.out.println("</" + e.getNodeName() + ">");
	}


	/**
	 * main to test class
	 *
       **/
	public static void main( String args[]) {
		PeerCache a = new PeerCache();
		String id = new String("130.79.192.166");
		String id2 = new String("130.79.192.167");
		System.out.println("----------adding properties for peer id="+id+"----------------------------");	
		if (a.setCpuName(id,"Intel xeon")) System.out.println("\t* setCpuName("+ id + ",intel xeon) : ok");
		if (a.setCpuFreq(id,"2345 MHz")) System.out.println("\t* setCpuFreq(" + id + ",2345 MHz) : ok");
		if (a.setFtPort(id,"9801")) System.out.println("\t* setFtPort("+id+", 9801) : ok");
		if (a.setMem(id,"512 MB")) System.out.println("\t* setMem("+id+", 512 MB) : ok");

		System.out.println("----------saving peer id="+id+" under a.xml----------------------------");	
		a.saveDocument("a.xml");

		System.out.println("----------reading a.xml to a new Cache----------------------------");	
		PeerCache b = new PeerCache("a.xml");
		System.out.println("\t* getCpuName("+ id + ") : "+ b.getCpuName( id ));
		System.out.println("----------adding a property for a new peer id=" + id2 + "-----------------------------");
		if (b.setOsName(id2,"linux")) System.out.println("setOsName("+id2+",linux) : ok");
		System.out.println("----------remove peer id=" + id + "-----------------------------");
		b.remove( id );
		System.out.println("----------displaying Cache (DOM document in memory)-----------------------------");
		b.display();
		System.out.println("----------saving Cache under b.xml----------------------------");	
		b.saveDocument("b.xml");

	}

}
