/**
 * @author choopan
 *
 */


package p2pmpi.mpd;

import p2pmpi.common.*;
import p2pmpi.common.ProcessTable;
import p2pmpi.message.*;
import java.net.*;
import java.util.*;
import java.io.*;
import p2pmpi.p2p.*;
import org.apache.log4j.*;


public class P2PMPI_MPD extends P2PMPI_MPDPeer {
	  int appMinPort;
	  int appMaxPort;
	  int appCurrentPort;

	  int numCPU;

	  Properties confProps;

	  long startBOOT, stopBOOT;
	  protected ProcessTable procTab = null;
	  URI superNode;
	  URI myURI;
	  
	  // Constructor of P2P_MPD
        public P2PMPI_MPD(int mpdPort, int fdPort, int ftPort, 
                        int jxtaMinPort, 
                        int jxtaMaxPort, 
                        String tmpDir, 
                        Properties confProps,
                        String hostDenyList, 
                        String hostAllow,
                        int numProcPerJob , int rsPort, int numCPU, 
                        URI superNode,
                        String peerCacheFile, 
                        String diskCacheDir,
                        String defaultIface,
				String externalIp) {

                startBOOT	= System.currentTimeMillis();
                log           = Logger.getLogger("MPD");
		    this.mpdPort  = mpdPort;
		    this.fdPort   = fdPort;
		    this.ftPort	= ftPort;
		    this.rsPort   = rsPort;
		    this.runPath 	= new File(tmpDir, getUniqueDirName());
		    this.appMinPort	= jxtaMinPort;
		    this.appMaxPort	= jxtaMaxPort;
		    this.confProps	= confProps;
		    this.numCPU		= numCPU;
		    this.superNode	= superNode;

		    log.info("[MPD] ** Starting MPD "+ Version.MAINVERSION + "." + Version.SUBVERSION + " **");
		    // set peer cache file as specified in config., default otherwise.
		    if ( peerCacheFile != null )
				this.peerCacheFile = peerCacheFile;
		    else
				this.peerCacheFile = new String(this.tmpDir + System.getProperty("path.separator") + "cache.xml");

		    // set our IP, but mind NAT configurations. In that case, use 'externalIP' specified in conf.
		    NetIface extIf = new NetIface( defaultIface, externalIp );
		    this.usingIP = extIf.getExternalIfIpv4( );
		    if (usingIP == null ) {
				log.error("Could not determine a working network interface. Exiting."); 
				System.exit(1); 
		    }
		    log.info("[MPD] Network: requested IFACE="+defaultIface+". Using "+extIf.getIfName( )+" ip="+ this.usingIP);
		    try {
				myURI = new URI("tcp://" + usingIP + ":" + mpdPort);
		    } catch(Exception e) {}

		    this.numProcPerJob	= numProcPerJob;
		    procTab = new ProcessTable();
		    appCurrentPort = appMinPort;

                // save my info in my peer cache
		    createCacheEntry( this.usingIP );

                // instantiate a disk cache to access it. Mind that we do not mind about its max size
                diskCache = new DiskCache( tmpDir, diskCacheDir );
		    myHost = new HostEntry(usingIP, mpdPort, fdPort, ftPort, rsPort, numProcPerJob );
	  }

	  /**
	   * initialize the PeerCache system. Renew our own Peer entry at each startup.
	   * @param myIp a String serving as id for cache entry.
	   * @author genaud
	   *
	   * This is a complete rewrite of createMPDAdv() created by GHaZI, which originally used JXTA Advertisments. 
	   */
	  protected void createCacheEntry( String myIP ) {

		    PeerCache cache = new PeerCache( this.peerCacheFile ); 

		    // first remove my entry from the cache. Let us regenerate
		    String id = new String( myIP + ":" + this.mpdPort );
		    cache.remove( id );

		    cache.setMpdVer ( id, Version.MAINVERSION + "." + Version.SUBVERSION ); 
		    cache.setOsName ( id, System.getProperty("os.name") );
		    cache.setArch   ( id, System.getProperty("os.arch") );
		    cache.setMpdPort( id, Integer.toString( this.mpdPort ) );
		    cache.setFdPort ( id, Integer.toString( this.fdPort ) );
		    cache.setFtPort ( id, Integer.toString( this.ftPort ) );

		    // Put in the add as much system information as we can
		    String cpuinfo [] = OsInfo.cpuInfo();
		    String meminfo [] = OsInfo.memInfo();
		    if (cpuinfo == null) {
				cache.setCpuName( id, "N/A" );
				cache.setCpuFreq( id, "N/A" );
		    }
		    else {
				cache.setCpuName( id, cpuinfo[0] );
				cache.setCpuFreq( id, cpuinfo[1] );
		    }
		    if (meminfo == null)
				cache.setMem( id, "N/A");
		    else
				cache.setMem( id, meminfo[0] );

		    if ( cache.saveDocument( this.peerCacheFile ) ) {
		    		log.info("[MPD] Entry created in cache " + this.peerCacheFile + " with id=" + id );
		    }
		    else {
		    		log.info("[MPD] Cache entry not created. Reason: "+ cache.getErrorStatus());
		    }
	  }


	  // Return a unique name used as a temp dir
	  protected synchronized String getUniqueDirName() {
		    return new String(NameGenerator.getUniqueName());
	  }


	  /**
         * Assign a rank to a peer which wants to participate
         * @param IP the IP of the peer as a string
         * @param mdpPort the port to send the message to
         * @param msg the message, typically containing a MPD_REQPEER command
         */
	  protected synchronized void requestPeer(String IP, int mpdPort, MPDMessage msg) {
		    OutputStream os;
		    ObjectOutputStream oos;
		    try {
				Socket s = new Socket(IP, mpdPort);
				log.info("Assign Rank " + msg.getRank() + ": to IP:" + IP);
				os = s.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(msg);
				oos.flush();
				oos.close();
				os.close();
				s.close();
		    } catch (Exception e) {
				log.info("[MPD] Error: failed to assign a rank to" + IP+ ": "+e.toString());
		    }
	  }

	  /**
	   * Cancel a reservation previously posed on some remote Peers through RS servers
	   * @author choopan
	   **/
	  protected void cancelReservation(String hashID, Vector<HostEntry> list) {
		    if(list.size() == 0) 
				return;
		    try {
				RSMessage rsMsg = new RSMessage(RSMessage.MPD_CANCEL_RESERVATION_TO_RS, hashID, usingIP,
						 Version.MAINVERSION + "." + Version.SUBVERSION);
				rsMsg.setList(list);

				Socket socket = new Socket("127.0.0.1", rsPort);
				OutputStream out = socket.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject(rsMsg);
				oos.flush();
				oos.close();
				out.close();
				socket.close();
		    } catch(Exception e) {
				log.info("[MPD] Error: could not contact RS to cancel reservation "+ hashID);
		    }
	  }

	  protected boolean isMyHost(HostEntry host) {
		    return host.getIp().equals(myHost.getIp());
	  }
	  
	  /**
	   * Checks for matching md5sums between remote datacache and the files to be sent.
	   * And calculates the total data present at remote node in bytes.
	   * @param remoteCacheMd5sums the string vector of md5sums of files present in remote datacache
	   * @param fileCharacteristicsVector the FileCharacteristics vector of files to be sent to remote node
	   *         in which the characteristics of files is stored
	   * @return the size in Kbytes of data present at remote datacache
	   */
	  private static long calculateDataPresent(Vector<String> remoteCacheMd5sums, Vector<FileCharacteristics> fileCharacteristicsVector) {
		  
		  Vector<FileCharacteristics> f = fileCharacteristicsVector;
		  long sizeinKBytes = 0;
		  for(int i=0; i<remoteCacheMd5sums.size(); i++) {
			  for(int j=0; j<f.size(); j++) {
				  if(remoteCacheMd5sums.elementAt(i).equals(f.elementAt(j).getMd5sum())) {
					  sizeinKBytes += (int)(f.elementAt(j).getFileSize()/1024);
				  }
			  }
		  }
		  
		  return sizeinKBytes;
	  }

        /**
         * Build a list of what remote hosts have in their cache.
         * @param hostCache the vector of pairs (id_strings, remote_hosts) in which remote_host is the host as known in host cache
         * @param reqFiles the files to be transfered for exec, and their characteristics (md5sum, length, ...)
         * @return a vector of (x->y) pairs, where x is a hostEntry and y the amount of bytes also present in reqFiles
         * this host x owns in its cache.
         **/
        private HashMap<HostEntry,Long> buildDataPresence(HashMap<String,HostEntry> hostCache, Vector<FileCharacteristics> reqFiles) {

                HashMap<HostEntry,Long> v = new HashMap<HostEntry,Long>();
                Iterator iter = hostCache.entrySet().iterator();
                while(iter.hasNext()) {
                        HostEntry rhost = (HostEntry)((Map.Entry)iter.next()).getValue();
                        Vector<String> remoteCacheMd5sums = rhost.getCacheMd5sums();
                        long bytesPresent = calculateDataPresent(remoteCacheMd5sums, reqFiles);
                        v.put( rhost, bytesPresent );
                }	
                return( v );
        }


	  /**
	   * Extract a list of known peers to satisfy the number of hosts required, 
	   * and extend this list by overbooking (3 log_2(n)). Query supernodes for
         * more peers if the locally known peers are not numerous enough.
         * The list is sorted by ascending RTT
         *
	   * @author choopan, genaud
	   * @param maxRequire
	   * @param minRequire
	   * @param maxWaitTime number of seconds
	   * @param allocationmode- strategy of allocation of nodes
	   * @param reqFiles- vector of characteristics of files which are to be sent to remote nodes
         * @return a vector of known hosts 
	   **/
	  private Vector<HostEntry> buildPeerList ( int maxRequire, int minRequire, int maxWaitTime, 
                                                  String allocationMode, Vector<FileCharacteristics> reqFiles ) {

                //get number of reservation 
                int numReserve = maxRequire + (int)(3 * (Math.log(maxRequire)/Math.log(2))); // overbooking
                Vector<HostEntry> v = new Vector<HostEntry>();

                long startTime = System.currentTimeMillis();
                int counter;
                if(allocationMode.startsWith("data")) {

                        updateHostCache(superNode);
                        HashMap<HostEntry,Long> rhostData = buildDataPresence(hostCache, reqFiles);	
                        // Sort updated hostcache by data present at remote node

                        while(true) {
                                ArrayList as = new ArrayList( rhostData.entrySet() );
                                Collections.sort(as, new Comparator() {
                                                // o1 and o2 have type (HostEntry,Integer)
                                                public int compare(Object o1, Object o2) {
                                                Map.Entry e1 = (Map.Entry)o1;
                                                Map.Entry e2 = (Map.Entry)o2;
                                                return ( (int)((Long)e2.getValue() - (Long)e1.getValue()));
                                                }
                                                });

                                //form a message to RS
                                counter = 0;
                                Iterator iter = as.iterator();
                                while(iter.hasNext()) {
                                        HostEntry host = (HostEntry)((Map.Entry)iter.next()).getKey();
						  if(host.isAlive()) {
							    v.add(host);
							    counter++;
							    if(counter >= numReserve)  // we're done: reached the overbooking number
									return( v );
						  }
					}	

					if(counter < maxRequire) {
						  //If present peers are not enough, try to discover more
						  int deadlineTime = (int)(System.currentTimeMillis() - startTime);
						  if ( deadlineTime < (maxWaitTime*1000) ) {
							    v.clear();
							    updateHostCache(superNode);
						  } else {
							    break;
						  }
					} else {
						  break;
					}
					try {
						  Thread.sleep(1000);
					} catch(Exception e) {}
			    }
		    	
		    	
		    	
		    } else {
		    	
			    //Sort HostCache by RTT
			    while(true) {
					ArrayList as = new ArrayList(hostCache.entrySet());
					Collections.sort(as, new Comparator() {
							    public int compare(Object o1, Object o2) {
							    Map.Entry e1 = (Map.Entry)o1;
							    Map.Entry e2 = (Map.Entry)o2;
							    int first  = ((HostEntry)e1.getValue()).getRtt();
							    int second = ((HostEntry)e2.getValue()).getRtt();
							    return first - second;
							    }
							    });

					//form a message to RS
					counter = 0;
					Iterator iter = as.iterator();
					while(iter.hasNext()) {
						  HostEntry host = (HostEntry)((Map.Entry)iter.next()).getValue();
						  if(host.isAlive()) {
							    v.add(host);
							    counter++;
							    if(counter >= numReserve)  // we're done: reached the overbooking number
									return( v );
						  }
					}	

					if(counter < maxRequire) {
						  //If present peers are not enough, try to discover more
						  int deadlineTime = (int)(System.currentTimeMillis() - startTime);
						  if ( deadlineTime < (maxWaitTime*1000) ) {
							    v.clear();
							    updateHostCache(superNode);
						  } else {
							    break;
						  }
					} else {
						  break;
					}
					try {
						  Thread.sleep(1000);
					} catch(Exception e) {}
			    }
		    	
		    }
		    return( v );
	  }

	  /**
	   * Reserve a number of peers, with a given replication degree, using a
	   * a given allocation strategy. This reservation process is bounded in 
	   * time, and declared failed after a certain timeout.
	   *
	   * @author choopan
	   * @param hashID a unique string used for reservation
         * @param optionN the number of peers to reserve (corresponds to p2pmpirun -n)
         * @param optionR the number of replica peers to reserve (corresponds to p2pmpirun -r)
         * @param maxWaitTime the timeout after which we give up the process
         * @return a object containing the hosts reserved
	   **/
	  protected ReservationResult searchMPD(String hashID, int optionN, int optionR, 
			  			int maxWaitTime, String allocationMode, Vector<FileCharacteristics> fileCharacteristicsVector)  {
		    //allocationMode -> spread : distribute process on as many machines as possible (default)
		    //                  concentrate : use as many slots a host provides as soon it as been selected
		    ReservationResult rsResult;

		    //get number of reservation 
		    int maxRequire = optionN * optionR;	//remove rank 0
		    int minRequire = optionR;
		    Vector<HostEntry> v; 
		    int counter;

		    log.info( "[MPD] entering searchMPD (n="+optionN+",r="+optionR+")");
                v = buildPeerList( maxRequire, minRequire, maxWaitTime, allocationMode, fileCharacteristicsVector );   
                counter = v.size(); 
		    log.info("[MPD] number of hosts required: min=" + minRequire + ",max=" + maxRequire + ",found =" + counter);

		    //////////////////////////////////////////////////////////////
		    //Constraint : number of nodes is less than minimum required
		    //////////////////////////////////////////////////////////////
		    if(counter < minRequire - 1) {
			    rsResult = new ReservationResult(null, ReservationResult.NOT_ENOUGH_NODE, counter+1, 0);
			    rsResult.setMyHost(myHost);
			    return rsResult;
		    }

		    RSMessage rsMsg = new RSMessage(RSMessage.MPD_REQUEST_RESERVATION_TO_RS, hashID, usingIP, Version.MAINVERSION + "." + Version.SUBVERSION);
		    rsMsg.setReservePeriod(30000 + (optionN*optionR*1000)); //30s + 1s per node
		    rsMsg.setList(v);
		    Vector<HostEntry> reservedList = null;
		    Vector<HostEntry> deadList = null;

		    //Send message to RS to reserve node
		    try {
			    Socket socket = new Socket("127.0.0.1", rsPort);
			    InputStream in = socket.getInputStream();
			    OutputStream out = socket.getOutputStream();
			    ObjectOutputStream oos = new ObjectOutputStream(out);
			    oos.writeObject(rsMsg);
			    oos.flush();

			    ObjectInputStream ois = new ObjectInputStream(in);
			    Object oMsg = ois.readObject();
			    RSMessage replyMsg = (RSMessage)oMsg;

			    reservedList = replyMsg.getList();
			    deadList     = replyMsg.getDeadList();

			    ois.close();
			    oos.close();
			    in.close();
			    out.close();
			    socket.close();
		    } catch(Exception e) {
			    log.error("[MPD] Error connecting to local RS (127.0.0.1:"+rsPort+") : " + e.toString());
			    //RS dead.. must do something
		    }

		    //Update status of dead node
		    int deadNodeSize = deadList.size();
		    String dhl="";
		    for(int i = 0; i < deadNodeSize; i++) {
			    HostEntry host = deadList.elementAt(i);
			    String key = host.toString();
			    dhl += host.getIp() + " ";
			    (hostCache.get(key)).setAlive(false);
		    }
		    log.info("[MPD] result from RS reservation request: "+ reservedList.size() +
				    " reserved hosts , "+ deadList.size() +" unreachable RS.");
		    for(int i = 0; i < reservedList.size(); i++) {
			    HostEntry host = reservedList.elementAt(i);
			    log.info("[MPD] reserved : " + host.getIp());
		    }

		    if (deadList.size() > 0) 
			    log.info("[MPD] unreachable RS: " + dhl);

		    int reservedSize = reservedList.size();

		    reservedList.add(0, myHost);
		    reservedSize++; 

		    //Check if we can reserve more than minimum require
		    log.info("[MPD] Total of " + reservedSize + " reserved nodes");
		    //////////////////////////////////////////////////////////////
		    //Constraint : number of nodes is less than minimum required
		    //////////////////////////////////////////////////////////////
		    if(reservedSize < minRequire) {
			    //Cancel the reservation
			    reservedList.remove(0);
			    cancelReservation(hashID, reservedList);
			    rsResult = new ReservationResult(null, ReservationResult.NOT_ENOUGH_NODE, reservedSize, 0);
			    rsResult.setMyHost(myHost);
			    return rsResult;
		    }


		    //Check if summation of available slot is more than max require (new term : capacity)
		    int processAvailable = 0;
		    for(int i = 0; i < reservedSize; i++) {
			    HostEntry host = reservedList.elementAt(i);
			    int numProcProvide = host.getNumProcPerJob();
                      log.debug("[MPD] host "+host+" provides "+numProcProvide+" process slot(s)");
			    if(isMyHost(host)) {
				    //myHost already consume 1 slot because of rank 0
				    numProcProvide--;
			    }
			    if(numProcProvide > optionN) {
				    processAvailable += optionN;
			    } else {
				    processAvailable += numProcProvide;
			    }
		    }
                log.debug("[MPD] total process slot(s) provided: "+processAvailable);
		    if(processAvailable < maxRequire) {
			    //Cancel Previous Reservation
			    reservedList.remove(0);
			    cancelReservation(hashID, reservedList);
			    rsResult = new ReservationResult(null, ReservationResult.NOT_ENOUGH_SLOT, reservedSize, processAvailable);
			    rsResult.setMyHost(myHost);
			    return rsResult;
		    }



		    /////////////////////////////////////////////////////////
		    // Calculate process on each machine
		    // pick allocation policy mode :
		    // Scatter or Gather
		    /////////////////////////////////////////////////////////

                // rank0 is always a submitter and needs it share of work.
                maxRequire++;
                optionN++;

		    if(allocationMode.startsWith("data")) {
		    	allocationMode = allocationMode.replaceFirst("data","");
		    }

		    if(allocationMode.equalsIgnoreCase("spread")) {
			    ///////////////////////////////////////////////
			    // Scatter mode
			    ///////////////////////////////////////////////
			    Vector<ReservedHost> usedHost = new Vector<ReservedHost>();
			    Vector<HostEntry> cancelList = new Vector<HostEntry>();
			    ReservedHost rh;

			    int used = 0;
			    int staticReservedSize = reservedSize;
			    int numProcPerJob;

			    int[] reservedSlotTable = new int[reservedSize];

			    boolean fillRankDone = false;
			    while(true) {
				    for(int i = 0; i < staticReservedSize; i++) {
					    HostEntry host = reservedList.elementAt(i);
					    if(used >= maxRequire){
						    fillRankDone = true;
						    break;
					    } else {
						    numProcPerJob = host.getNumProcPerJob();
						    if((reservedSlotTable[i] < numProcPerJob) &&
						       (reservedSlotTable[i] < optionN)) {
							    used++;
							    reservedSlotTable[i]++;
						    }
					    }
				    }
				    if(fillRankDone)
					    break;
			    }

			    for(int i = 0; i < staticReservedSize; i++) {
				    HostEntry host = reservedList.elementAt(i);
				    if(reservedSlotTable[i] == 0) {
					    cancelList.add(host);
					    log.info("[MPD] 'spread' alloc. Cancel reservation " + host.getIp());
				    } else {
					    log.info("[MPD] 'spread' alloc. Reserve " + host.getIp());
					    rh = new ReservedHost(host, reservedSlotTable[i]);
					    usedHost.add(rh);
				    }
			    }
			    //Cancel the rest of node that we don't use
			    cancelReservation(hashID, cancelList);
			    int usedHostSize = usedHost.size();
                      // rank0 is always a submitter and needs it share of work.
			    usedHostSize++;
			    log.info("[MPD] spread alloc: final number of hosts reserved (including rank0): " + usedHostSize);
			    rsResult = new ReservationResult(usedHost, ReservationResult.SUCCESS, usedHostSize, 0);
			    rsResult.setMyHost(myHost);
			    return rsResult;	


		    } else {
			    /////////////////////////////////////////////////////////////////////
			    // Concentrate Mode
			    /////////////////////////////////////////////////////////////////////
			    //comment here : change to sort by RTT instead of MAX_PROCESS_PER_JOB
			    Vector<ReservedHost> usedHost = new Vector<ReservedHost>();
			    Vector<HostEntry> cancelList = new Vector<HostEntry>();
			    ReservedHost rh;

			    int used = 0;
			    int staticReservedSize = reservedSize;
			    int reserveSlot;
			    int numProcPerJob;

			    //Assign to machine that execute rank 0 first
			    numProcPerJob = myHost.getNumProcPerJob();
			    if(numProcPerJob < optionN) {
				    if(numProcPerJob > maxRequire) {
					    reserveSlot = maxRequire;
				    } else {
					    reserveSlot = numProcPerJob;
				    }
			    } else {
				    reserveSlot = optionN;
			    }
			    rh = new ReservedHost(myHost, reserveSlot);
			    usedHost.add(rh);
			    used += reserveSlot;

			    //Assign to the rest of machines
			    for(int i = 0; i < staticReservedSize; i++) {
				    HostEntry host = reservedList.elementAt(i);
				    log.info("[MPD] 'concentrate' alloc. Reserve:" + host.getIp());

				    if(isMyHost(host)) {
					    continue;
				    }
				    if(used >= maxRequire) {
					    cancelList.add(host);
				    } else {
					    numProcPerJob = host.getNumProcPerJob();
					    if(numProcPerJob < optionN) {
						    if(used + numProcPerJob > maxRequire) {
							    reserveSlot = maxRequire - used;
						    } else {
							    reserveSlot = numProcPerJob;
						    }
					    } else {
						    if(used + optionN  > maxRequire) {
							    reserveSlot = maxRequire - used;
						    } else {
							    reserveSlot = optionN;
						    }
					    }
					    rh = new ReservedHost(host, reserveSlot);
					    usedHost.add(rh);
					    used += reserveSlot;
				    }
			    }	

			    //Cancel the rest of node that we don't use
			    cancelReservation(hashID, cancelList);
			    int usedHostSize = usedHost.size();

                      // rank0 is always a submitter and needs it share of work.
			    usedHostSize++;
			    log.info("[MPD] concentrate alloc: final number of hosts reserved (including rank0): " + usedHostSize);
			    rsResult = new ReservationResult(usedHost, ReservationResult.SUCCESS, usedHostSize, 0);
			    rsResult.setMyHost(myHost);
			    return rsResult;	
		    }
	  }


	  public void run() {
		  if(!joinMpiGroup(superNode)) {
			  System.err.println("** [MPD Error] : could not contact SuperNode at " +superNode.toString() + ". Exiting." );
			  log.info("[MPD Error]: could not contact SuperNode at " +superNode.toString() + ". Exiting." );
			  System.exit(1);
		  }

		  updateHostCache(superNode);

		  MPD_Interface myInterface = new MPD_Interface(this);
		  myInterface.start();

		  Timer peerKeepAlive =  new Timer();
		  peerKeepAlive.schedule(new PeerKeepAlive(superNode,myHost,diskCache), 0, 180000);
		  Timer peerUpdateList =  new Timer();
		  peerUpdateList.schedule(new PeerUpdateList(superNode,myHost,hostCache), 0, 300000);

	  }

	  public static void main(String[] args) {

		    File tempDir = new File(System.getProperty("P2PMPI_HOME"), "tmp");

		    P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( log );
		    P2PMPI_MPD mpiDaemon = new P2PMPI_MPD(
					  ppConf.getMPDPort(),  //MPD Port number
					  ppConf.getFDPort(),
					  ppConf.getFTPort(),
					  ppConf.getMinPort(),        // Jxta MinPort number
					  ppConf.getMaxPort(),        // Jxta MaxPort number
					  tempDir.toString(),
					  ppConf.getProperties(),
					  ppConf.getHostDeny(),
					  ppConf.getHostAllow(),
					  ppConf.getNumProcPerJob(),
					  ppConf.getRSPort(),
					  ppConf.getNumCPU(),
					  ppConf.getSuperNode(),
					  ppConf.getPeerCacheFile(), // this cache is to store info about remote peers 
                                ppConf.getDiskCacheDir(),  // this is my own cache for files
					  ppConf.getIface(),
					  ppConf.getExternalIP()
					  );

		    mpiDaemon.start();
	  }
}
