/**
 * File Transfer Server: obeys MPD instructions to send files or URLs to other peers
 *
 * Note to debuggers.
 * this class is normally launched as a daemon, typically with the runFT script, and
 * its output are typically routed to /dev/null. Therefore, for all the System.out.println()
 * inside this code to be effective, you should check that the output of runFT is not
 * thrown away. At the time of writing, the unix script runFT accepts one argument. 
 * If present this is the name of a file where stdout is redirected.
 *
 * @author: choopan
 * @author: genaud
 *
 */

package p2pmpi.ft;

import p2pmpi.common.*;
import p2pmpi.message.*;
import java.net.*;
import java.io.*;
import java.util.*;
import java.lang.Number;


public class FileTransferServer {
	/* XFER_PROTOCOL_SWITCH_LIMIT : the data size under which the data of each file will be sent
	 * in a single message. Above that limit, the file will be transfered piece by piece with
	 * after the FTs have negociated a tcp stream opening. File transfers above the limit
	 * need one extra two-ways message but have a very low memory footprint. */

	private static int XFER_PROTOCOL_SWITCH_LIMIT = 20*1024*1024; 
	private int serverPort;
	private int mpdPort;
	private String baseDir;
	private DiskCache diskCache = null; 
	private static String hostIP = null; // my host IP, i.e the one other can use to contact me
	private static String osname=OsInfo.getName();
	private URI rank0URI;
	private Vector<FileCharacteristics> fileCharacteristicsVector; // the characteristics of the files to be transmitted


	/**
	 * Constructor
	 */
	public FileTransferServer(int serverPort, int mpdPort, String dir, String diskCacheDir, String diskCacheSize) {
		this.serverPort = serverPort;
		this.mpdPort = mpdPort;
		this.baseDir = dir;
		this.diskCache = new DiskCache( dir, diskCacheDir, diskCacheSize );  // create a cache object to handle cache disk space
		System.out.println("[FT] Service starting (port="+ serverPort+
				", cache size="+ this.diskCache.getSize() +
				", dir="+this.diskCache.getDir()+")");
	}

	/**
	 * find which ip this host has, and stores it.
	 */
	private static void setLocalhostIP() {
		// Get Local IP address

		// -- first see if a fixed address has been set
		P2PMPI_ConfigFile conf = new P2PMPI_ConfigFile();

            // if Iface or ExternalIP are null , no effect, i.e ip selection will be automatic
		NetIface extIf = new NetIface( conf.getIface(), conf.getExternalIP());
		hostIP = extIf.getExternalIfIpv4();
	}

	/**
	 * returns IP set.
	 */
	protected static String getLocalhostIP() {
		return (hostIP);
	}

	/**
	 * send a serializable object to destination URI
	 */
	private void sendMessage(Object message, URI destination) {
		OutputStream out = null;
		ObjectOutputStream oos = null;
		Socket soc = null;
		try {
			soc = new Socket(destination.getHost(), destination.getPort());
			out = soc.getOutputStream();
			oos = new ObjectOutputStream(out);
			oos.writeObject(message);
			oos.flush();
			oos.close();
			out.close();
			soc.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}


	/**
	 * Read from a file and send its bytes to a socket.
	 *
	 * @param filename the name of the file to read 
	 * @param filesize the number of bytes to read
	 * @param os the outputstream to write to (the asociated socket must be have been established)
	 */
	private long writeFileToSocket(String filename, long filesize, OutputStream os) throws Exception {

		int  buffersize = 10*1024*1024;
		byte buffer [] = new byte[ buffersize ];
		long totalBytesRead=0;
		int  bytesRead=0;

		System.out.println("[FT] sending file "+filename+", size="+filesize);
		File myFile = new File(filename);
		FileInputStream fis = null;
		//BufferedInputStream bis = null;// We not using this because we are buffering here in large size
		try {
			fis = new FileInputStream(myFile);
			//bis = new BufferedInputStream(fis);
		} 
		catch (Exception e) {
			System.out.println("** [FT] Error trying to access file "+ filename);
			e.printStackTrace();
			throw new Exception( e );
		}

		do {
			try {
				bytesRead = fis.read( buffer, 0 , buffersize);
				//System.out.println("[FT] read "+ bytesRead + "from file");
				if (-1 != bytesRead ) {
					totalBytesRead += bytesRead;
					//System.out.println("[FT] totalRead "+totalBytesRead);
					os.write( buffer, 0, bytesRead );
					//os.flush(); // for outputstream class, flush does nothing.

				}
			}
			catch (Exception e) {
				e.printStackTrace();
			}
		} while (( totalBytesRead < filesize ) && (-1 != bytesRead));
		// done for this file
		System.out.println("[FT] done sending file "+filename+", total sent"+totalBytesRead);
		try {
			fis.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return( totalBytesRead );
	}


	/**
	 * Start
	 */
	public void start() {

		setLocalhostIP();

		ServerSocket servSocket = null;
		Socket socket = null;
		try {
			servSocket = new ServerSocket(serverPort);
		} catch (Exception e) {
			System.err.println("** [Error] FT: Can't bind port.");
			e.printStackTrace();
			System.exit(1);
		}

		while (true) {
			try {
				socket = servSocket.accept();
				FileTransfer ft = new FileTransfer(socket);
				ft.start();
			} catch (Exception e) {
				e.printStackTrace();
				System.exit(1);
			}
		}
	}

	public class FileTransfer extends Thread {
		Socket socket;

		public FileTransfer(Socket socket) {
			this.socket = socket;
		}

		// --------- main thread ------------------------------
		public void run() {
			URI localMPDURI = null;
			InputStream in = null;
			ObjectInputStream ois = null;
			Object oMsg = null;
			URI replyURI = null;
			FTMessage ftMsg = null;
			MPDMessage doneMsg=null;
			String[] onlyFileNameList = null;
			String jarDep = "";
			File appFile;
			File appDir;
			int myRank=0;

			try {
				localMPDURI = new URI("tcp://127.0.0.1:" + mpdPort);
			}
			catch (URISyntaxException e) {
				System.out.println("[FT] internal error: malformed URI: "+ localMPDURI.toString());
			}

			P2PMPI_ConfigFile conf = new P2PMPI_ConfigFile();

			try {
				in = socket.getInputStream();
				ois = new ObjectInputStream(in);
				oMsg = ois.readObject();
			} 
			catch (Exception e) {
				e.printStackTrace();
				return;
			}

			if (oMsg instanceof RequestQuitMessage) {
				System.exit(0);
			} else if (oMsg instanceof FTMessage) {
				ftMsg = (FTMessage) oMsg;

				switch (ftMsg.getCmd()) {

				//-----------------------------------------------------------------
				//  FT_0 receives a message from Rank 0 to prepare data for sending
				//  or to request FT_i(i!=0) to open a new channel to transfer data
				//  if total size of files is greater than a limit or if there is 
				//  any file which is to be cached.
				//-----------------------------------------------------------------

				case MessageCmd.FT_REGISTER:
					System.out.println("[FT] received FTMessage with FT_REGISTER");

					try {
						replyURI = new URI("tcp://" + getLocalhostIP() + ":"+ conf.getFTPort());
					} 
					catch (Exception e) {
						e.printStackTrace();
					}

					int numPeers = ftMsg.size();
					rank0URI = ftMsg.getRank0URI();

					fileCharacteristicsVector = ftMsg.getFileCharacteristicsVector();
					int numFile = fileCharacteristicsVector.size();
					String[] fullFileNameList = new String[numFile];
					onlyFileNameList = new String[numFile];
					for(int i=0; i<numFile; i++) {
						fullFileNameList[i] = fileCharacteristicsVector.elementAt(i).getFileName();
						onlyFileNameList[i] = fileCharacteristicsVector.elementAt(i).getOnlyFileName();	
					}

					// determine which transfer protocol to use
					boolean DataTransferThroughNewChannel=false;
					for(int i=0; i<numFile; i++) {
						if(fileCharacteristicsVector.elementAt(i).isToBeCached()) {
							// if at least one file is 'cached', use new channel whatever the size
							DataTransferThroughNewChannel=true;
							break;

						}
					}
					// We should  choose to transfer through a new channel if the totalsize
					// of the files to transfer exceeds XFER_PROTOCOL_SWITCH_LIMIT²
					if (!DataTransferThroughNewChannel) {
						// compute the size
						long totalFileSize = 0;
						for(int i=0; i<numFile; i++) {

							if(!fileCharacteristicsVector.elementAt(i).getFileName().startsWith("http://")) {
								totalFileSize += fileCharacteristicsVector.elementAt(i).getFileSize();

								if(totalFileSize > XFER_PROTOCOL_SWITCH_LIMIT) {
									DataTransferThroughNewChannel=true;
									break;
								}
							}
						}
					}
					// Now, transfer depending on the method chosen
					if(!DataTransferThroughNewChannel) {
						boolean okRemoteTransfer = true;

						// In this case, create new FTMessage
						FTMessage ftMsgDataContained = new FTMessage(MessageCmd.FT_TRANSFER, ftMsg.getID());
						ftMsgDataContained.setOnlyFileNameList(onlyFileNameList);

						//-- load file contents into the message
						try {
							ftMsgDataContained.setFileContent(fullFileNameList);
						} 
						catch (Exception e) {
							// notify local MPD that an error occured ...
							System.out.println("[FT] failed loading files for transfer. Send a failure notice to MPD."); 
							MPDMessage failMsg = new MPDMessage(MessageCmd.FT_FAILED, ftMsg.getID());
							failMsg.setURI( getLocalhostIP() , serverPort ); 
							failMsg.setErrorMessage( e.getMessage() );

							// send to MPD_INTERFACE
							sendMessage( failMsg, localMPDURI );

							// send to MPI submitter process
							System.out.println("[FT] failed loading files for transfer. "+
									"Send a failure notice to rank0:" + ftMsg.getRank0URI().toString()); 
							sendMessage( failMsg, ftMsg.getRank0URI() );
							okRemoteTransfer = false;
						}

						// if no error encoutered while loading data ...
						if ( okRemoteTransfer ) {
							// ... modify the contents depending on each peer

							for(int i = 0; i < numPeers; i++) {
								ftMsgDataContained.setKey(ftMsg.getKey(i));
								URI destination = ftMsg.getURI(i);
								System.out.println("[FT] constructing message for peer "+i+
										": rank "+ftMsg.getKey(i)+" URI:"+ftMsg.getURI(i));
								sendMessage( ftMsgDataContained, destination );
							}
						}

					} else {
						// Transfer through a new socket

						ftMsg.setCmd(MessageCmd.FT_REQUESTPORT);
						ftMsg.setReplyURI(replyURI);

						for (int i = 0; i < numPeers; i++) {
							ftMsg.setKey(ftMsg.getKey(i));
							URI destination = ftMsg.getURI(i);
							System.out.println("[FT] constructing Request Port message for peer "+ i+ ": rank "+ ftMsg.getKey(i)+ " URI:" + ftMsg.getURI(i));
							sendMessage(ftMsg, destination);
						}
					}
					break;


					//-------------------------------------------------------
					//  FT_i, i !=0 receives a message from FT_0 with data 
					//  already contained in the message
					//  when (totalsizeoffiles<TRANSFER_LIMIT) and no file is
					//  requested to be cached
					//-------------------------------------------------------

				case MessageCmd.FT_TRANSFER :

					appDir = new File(baseDir, (ftMsg.getID() + "--" + ftMsg.getKey()));
					appDir.mkdirs();
					FileOutputStream fileOut;

					//-- get short filenames to write out
					onlyFileNameList = ftMsg.getOnlyFileNameList();
					for (int i = 0; i < onlyFileNameList.length; i++) {
						if(onlyFileNameList[i].endsWith(".jar")) {
							//-- pathSeparator is OS variable
								jarDep = jarDep + File.pathSeparator + onlyFileNameList[i];
						}
						//-- if filename is an http:// URL, download file 
						if (onlyFileNameList[i].startsWith("http://")) {
							String parts [] = onlyFileNameList[i].split(File.separator);

							//-- write remote contents directly to file
							try {
								RemoteData remoteFile = new RemoteData(onlyFileNameList[i]);
								if (remoteFile.setContentandWriteToFile(appDir+ File.separator+ parts[parts.length - 1])) {
									//System.out.println("md5sum :"+ parts[parts.length - 1] + "\t"+ remoteFile.getMD5Sum()); // Will
									// will see later the use of this line
								} else {
									System.err.println("** [Error] could not fetch contents hosted at "+ onlyFileNameList[i]);
								}

							} 
							catch (Exception e) {
								e.printStackTrace();
								MPDMessage failMsg = new MPDMessage(MessageCmd.FT_FAILED, ftMsg.getID());
								failMsg.setURI( getLocalhostIP() , serverPort ); 
								failMsg.setErrorMessage( e.getMessage() );
								sendMessage( failMsg, ftMsg.getRank0URI() );
							}
						}

						//-- otherwise, it is a file whose contents are contained in the message
						else {	
							appFile = new File(appDir, onlyFileNameList[i]);

							try {
								fileOut = new FileOutputStream(appFile);
								fileOut.write(ftMsg.getFileContent(i));
								fileOut.flush();
								fileOut.close();
							} 
							catch (Exception e) {
								e.printStackTrace();
							}
						}
					}
					
					if (jarDep.equals("")) {
						jarDep = "_"; // jarDep can't be empty string because it will be passed
						// as the argument of the p2pclient script
					}

					System.out.println("[FT] Done transfer file for ID = " + ftMsg.getID()+". Going to send a MPDMessage FT_DONE to localhost (myRank="+myRank);

					// Construct message to tell local MPD to spawn an MPI application
					doneMsg = new MPDMessage(MessageCmd.FT_DONE, ftMsg.getID());
					myRank = ftMsg.getKey();
					doneMsg.setKey( myRank );
					doneMsg.setRunDir(appDir);
					//set dependance .jar for CLASSPATH
					doneMsg.setJars(jarDep);
					sendMessage( doneMsg, localMPDURI );

					System.out.println("[FT] Done notify my local MPD " + localMPDURI.toString());
					break;

					//-------------------------------------------------------
					//  FT_i, i !=0 receives a message from FT_0 with a request
					//  to open the new port to send the data through which and
					//  makes one serverPort from one of the available ports and
					//  waits for the data to receive from FT_0
					//-------------------------------------------------------

				case MessageCmd.FT_REQUESTPORT:
					System.out.println("[FT] received FTMessage with FT_REQUESTPORT");
					
					boolean cacheIsChanged = false;// boolean taken to update the md5sumlist of
					// cache files to supernode in case of change in the cache files only

					fileCharacteristicsVector = ftMsg.getFileCharacteristicsVector();
					numFile = fileCharacteristicsVector.size();

					int minPort = conf.getMinPort();
					int maxPort = conf.getMaxPort();

					ReservedPort portToTransferData = new ReservedPort(minPort,maxPort);
					ServerSocket servsock = portToTransferData.getPort();

					try {
						replyURI = new URI("tcp://" + getLocalhostIP() + ":"+ portToTransferData.getPortNumber());
					} 
					catch (Exception e) {
						e.printStackTrace();
					}

					System.out.print("[FT] status of files marked cached [");
					// compare for each file in cache, if its md5sum matches any of 
					// the md5sums contained in the message.

					// build the array which will tell FT_0 which files need to be sent (not in cache)
					boolean[ ] notToBeSent = new boolean[numFile];
					java.util.Arrays.fill( notToBeSent,false);
					
					long sizeReqInCache = 0;

					for (int j=0; j< numFile; j++) { // for each file to be transfered
						if  (fileCharacteristicsVector.elementAt(j).isToBeCached()) {
							System.out.print(fileCharacteristicsVector.elementAt(j).getOnlyFileName());
							sizeReqInCache += fileCharacteristicsVector.elementAt(j).getFileSize();
							// try to find the same file in cache
							String currentMd5 = fileCharacteristicsVector.elementAt(j).getMd5sum();
							Vector<String> matchingFiles = diskCache.searchByMd5( currentMd5 );
							if ( !matchingFiles.isEmpty() ) {
								notToBeSent[j] = true;
								System.out.print("(^) ");
							}
							else { // no files in cache with the same md5sum
								System.out.print("(_) ");
							}
						}
					}
					
					System.out.println("] ^:already present, _:not yet cached");
					
					boolean cacheFit = false;
					if(sizeReqInCache <= diskCache.getSize()) {
						cacheFit = true;
					} else {
						System.out.println("[FT] All files can't be cached so copied only " +
								"to execution directory for the successful execution." +
								"Size Required in Cache is "+ sizeReqInCache +" bytes But the cache " +
										"size is only "+diskCache.getSize()+" bytes");
					}

					appDir = new File(baseDir,(ftMsg.getID() + "--" + ftMsg.getKey()));
					appDir.mkdirs();


					FTMessage ftMsgWithDataTransferPort = new FTMessage(MessageCmd.FT_ASSIGNEDPORT, ftMsg.getID());
					ftMsgWithDataTransferPort.setKey(0);
					ftMsgWithDataTransferPort.setReplyURI(replyURI);
					ftMsgWithDataTransferPort.setBooleanSent(notToBeSent);

					URI destination = null;
					try {
						destination = ftMsg.getReplyURI();
					} catch (Exception e) {
						e.printStackTrace();
					}

					System.out.println("[FT] Sending FT_0 to which socket data should be transferred");
					sendMessage(ftMsgWithDataTransferPort, destination);

					//----- Now, wait for incoming files---------------------------

					Socket soc = null;
					InputStream is = null;
					try {
						soc = servsock.accept();
						is = soc.getInputStream();
					} 
					catch (Exception e) {
						e.printStackTrace();
					}
					System.out.println("[FT] Accepted connection : " + soc);

					BufferedOutputStream bos = null;
					FileOutputStream fos = null;

					long bytesRead;

					// for each file, read incoming bytes from the socket and write to file

					for (int i = 0; i < numFile; i++) {
						FileCharacteristics f = fileCharacteristicsVector.elementAt(i); // a shortcut to designate the file charac. under exam
						if ( f.getOnlyFileName().startsWith("http://")) {
							String parts [] = f.getOnlyFileName().split(File.separator);

							//-- write remote contents directly to file
							try {
								RemoteData remoteFile = new RemoteData( f.getOnlyFileName() );
								if (remoteFile.setContentandWriteToFile(appDir+ File.separator+ parts[parts.length - 1])) {
									//System.out.println("md5sum :"+ parts[parts.length - 1] + "\t"+ remoteFile.getMD5Sum()); // Will
									// will see later the use of this line
								} else {
									System.err.println("** [Error] could not fetch contents hosted at "+ f.getOnlyFileName());
								}

							} 
							catch (Exception e) {
								e.printStackTrace();
								MPDMessage failMsg = new MPDMessage(MessageCmd.FT_FAILED, ftMsg.getID());
								failMsg.setURI( getLocalhostIP() , serverPort ); 
								failMsg.setErrorMessage( e.getMessage() );
								sendMessage( failMsg, ftMsg.getRank0URI() );
							}
						}
						else {  // it is a real file (not a URL)

							if (cacheFit) {
								if (!notToBeSent[i]) { // if file is already present in remote cache, don't send it.
									// check if we must cache this file, and then if we could write it to cache
									boolean cached = f.isToBeCached() && 
									diskCache.writeToLRU( is, f.getOnlyFileName(), f.getFileSize() );
									 
									if(cached==true) {
										cacheIsChanged = true;
									}
									// ... otherwise save it to temp dir
									if ( !cached ) {
										StreamSaver sv= new StreamSaver();
										bytesRead = sv.saveNetworkInputStream( appDir, f.getOnlyFileName(), f.getFileSize(), is);
										if (bytesRead != f.getFileSize()) {
											System.out.println("[FT] internal error for file "+f.getOnlyFileName()+": advertized="+
													f.getFileSize() +", read="+bytesRead);
										}
									}
								}
							}
							else {
								StreamSaver sv= new StreamSaver();
								bytesRead = sv.saveNetworkInputStream( appDir, f.getOnlyFileName(), f.getFileSize(), is);
								if (bytesRead != f.getFileSize()) {
									System.out.println("[FT] internal error for file "+f.getOnlyFileName()+": advertized="+
											f.getFileSize() +", read="+bytesRead);
								}
							}
						}
						// update jar dependencies    
						if ( f.getOnlyFileName().endsWith(".jar") ) {
							// -- pathSeparator is OS variable
								jarDep = jarDep + File.pathSeparator + f.getOnlyFileName();
							
						}
					}

					if ( cacheFit ) {
						  int returnvalue = diskCache.publishFiles(appDir, fileCharacteristicsVector);
						  if (returnvalue == -1) {
							    // there is some problem in publishing the data.
							    System.out.println("[Error] Data in Disk Cache could not be reflected (linked or copied, depending on the OS) to working directory.");
						  }
					}


					try {
						soc.close();
					} 
					catch (Exception e) {
						e.printStackTrace();
					}
					
					if (jarDep.equals("")) {
						jarDep = "_"; // jarDep can't be empty string because it will be passed
						// as the argument of the p2pclient script
					}

					System.out.println("[FT] Done transfer file for ID = "+ ftMsg.getID()+ ". Going to send a MPDMessage FT_DONE to localhost (myRank="+ myRank);

					// Construct message to tell local MPD to spawn an MPI
					// application
					doneMsg = new MPDMessage(MessageCmd.FT_DONE,ftMsg.getID());
					myRank = ftMsg.getKey();

					doneMsg.setKey(myRank);
					doneMsg.setRunDir(appDir);
					doneMsg.setJars(jarDep);
					
					doneMsg.setCacheIsChanged(cacheIsChanged);

					sendMessage(doneMsg, localMPDURI);
					System.out.println("[FT] Done notify my local MPD "+ localMPDURI.toString());
					break;





					//-------------------------------------------------------
					//  FT_0 receives a message from FT_i(i!=0) with the URI
					//  to which data should be sent and FT_0 sends the data
					//-------------------------------------------------------

				case MessageCmd.FT_ASSIGNEDPORT:


					Socket sock = null;
					File myFile;
					OutputStream os = null;

					URI uri = ftMsg.getReplyURI();
					String ip = uri.getHost();
					int port = uri.getPort();

					numFile = fileCharacteristicsVector.size();
					notToBeSent = ftMsg.getBooleanSent();

					System.out.println("[FT] received FTMessage with FT_ASSIGNEDPORT. Start sending to "+uri);

					try {
						sock = new Socket(ip, port);
						os = sock.getOutputStream();
					} 
					catch (Exception e) {
						e.printStackTrace();
					}

					//---- now, send each file in turn -----------------------------------

					for (int i = 0; i < numFile; i++) {
						if(!fileCharacteristicsVector.elementAt(i).getFileName().startsWith("http://")) {
							try {
								if(!notToBeSent[i]) {
									writeFileToSocket( fileCharacteristicsVector.elementAt(i).getFileName(), fileCharacteristicsVector.elementAt(i).getFileSize(), os );
								}

							} catch (Exception e) {

								MPDMessage failMsg = new MPDMessage(MessageCmd.FT_FAILED, ftMsg.getID());
								failMsg.setURI( getLocalhostIP() , serverPort ); 
								failMsg.setErrorMessage( e.getMessage() );
								// send to MPD_INTERFACE
								sendMessage( failMsg, localMPDURI );
								// send to MPI submitter process
								System.out.println("[FT] failed loading files for transfer. "+
										"Send a failure notice to rank0:" + ftMsg.getRank0URI().toString()); 
								sendMessage( failMsg, rank0URI );
							}
						}
					}


					System.out.println("[FT] all files sent throught socket");
					try {
						sock.close();
						os.close();
					} 
					catch (Exception e) {
						e.printStackTrace();
					}
					break;

					//-----------------------------------------------------------------------------------------------

				default:
					System.err.println("** [Error] FTMessage: Unknown Command.");
				break;

				}
			}
		}
	}


	/**
	 * main
	 **/

	static public void main(String[] args) {
		File tmpDir = new File(System.getProperty("P2PMPI_HOME"), "tmp");
		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile();
		FileTransferServer ftServer = new FileTransferServer(
				ppConf.getFTPort(), // FT port
				ppConf.getMPDPort(), // MPD port
				tmpDir.toString(),
				ppConf.getDiskCacheDir(),
				ppConf.getDiskCacheSize()
				// ppConf.getTempDirectory() //tmp dir
		);
		ftServer.start();
		System.out.println("Starting FT ...");
	}
}
