package p2pmpi.fd;

import p2pmpi.common.*;
import p2pmpi.message.*;
import java.net.*;
import java.io.*;
import java.util.*;
import org.apache.log4j.*;

public class GossipDetectorServer { 
	protected static Logger logGos;
	private int serverPort; 
	private int mpdPort;
	protected Timer timer;
	Hashtable<String, Vector<GossipDetector>> hashtable;

	public GossipDetectorServer(int serverPort, int mpdPort)
	{
		this.serverPort = serverPort;
		this.mpdPort	= mpdPort;
		hashtable = new Hashtable<String, Vector<GossipDetector>>();
		timer = new Timer();
		logGos = Logger.getLogger("MPD");
		logGos.info("[GossipDetectorServer] is online ...");
	}

	// Running Fault-Detection Server
	public void start() {
		timer.schedule(new CheckService(), 0, GossipConfig.t_checkalive);
		ServerSocket servSocket = null;
		Socket socket = null;
		Object oMsg = null;
		GossipDetector gos;
		String key;
		Vector <GossipDetector>threadList = null;
		int rankInList;
		try {
			servSocket = new ServerSocket(serverPort);
		} catch (Exception e) {
			System.out.println("FD : Can't bind port ("+serverPort+")");
			e.printStackTrace();
			System.exit(1);
		}

		while(true) {
			try {
				int toRank;

				socket = servSocket.accept();
				InputStream in = socket.getInputStream();
				ObjectInputStream ois = new ObjectInputStream(in);
				oMsg = ois.readObject();
				if(oMsg instanceof RequestQuitMessage) {
					logGos.info("[GossipServer]: shutdown");
					System.exit(0);
				}

				GossipMessage gosMsg = (GossipMessage)oMsg;
				//logGos.debug("-------Got FD Message--------");
				switch(gosMsg.getCommand()) {
					case MessageCmd.GOSSIP_REGISTER :
						key = gosMsg.getGossipID();
						threadList = hashtable.get(key);
						String protocol = gosMsg.getGossipProtocol();
						logGos.debug("[GossipServer] register detector hashID=" + key 
								+ " (rank="+ gosMsg.getMyRank() 
								+ ",protocol=" + protocol +")");

						if(protocol.equalsIgnoreCase("BRR")) {
							gos = new BRR (
									gosMsg.getMyRank(),
									key,
									gosMsg.getMPIPort(),
									gosMsg.getURIList(),
									mpdPort,
									gosMsg.getMPIRank(),
									gosMsg.getTGossip(),
									gosMsg.getTMargin(),
									gosMsg.getTDiff(),
									gosMsg.getTStamp(),
									gosMsg.getTHang()
						  		);
						} else {
							gos = new DBRR(
									gosMsg.getMyRank(),
									key,
									gosMsg.getMPIPort(),
									gosMsg.getURIList(),
									mpdPort,
									gosMsg.getMPIRank(),
									gosMsg.getTGossip(),
									gosMsg.getTMargin(),
									gosMsg.getTDiff(),
									gosMsg.getTStamp(),
									gosMsg.getTHang()
						  		);
						}
	
						if(threadList == null) {
							//System.out.println("Register first time");
							// first process to register to fd service
							// of this key
							threadList = new Vector<GossipDetector>();
							threadList.addElement(gos);
							hashtable.put(key, threadList);
						} else {
							logGos.debug("[GossipServer] register a detector same hashID second time");
							// other processes which use the same key
							// just add element to vector
							threadList.addElement(gos);
						}

						new Thread(gos).start();
						break;

					case MessageCmd.GOSSIP :
						key = gosMsg.getGossipID();
						toRank = gosMsg.getToRank();
						//logGos.debug("[GossipServer]: got gossip message hashID = " + key + " from rank = " + gosMsg.getMyRank() + " to rank = " + toRank);
						threadList = hashtable.get(key);
						if(threadList != null) {
							int numElement = threadList.size();
							for(int i = 0; i < numElement; i++) {
								gos = threadList.elementAt(i);
								if(gos.getMyRank() == toRank) {
									gos.merge(gosMsg.getHeartBeatList());
									break;
								}
							}
						}
						break;

					case MessageCmd.GOSSIP_PING :
						long currentHeartBeat = -1;
						key = gosMsg.getGossipID();
						toRank = gosMsg.getToRank();

						threadList = hashtable.get(key);
						if(threadList != null) {
							int numElement = threadList.size();
							for(int i = 0; i < numElement; i++) {
								gos = threadList.elementAt(i);
								if(gos.getMyRank() == toRank) {
									currentHeartBeat = gos.getCurrentHeartBeat();
									break;
								}
							}
						}

						OutputStream out = socket.getOutputStream();
						ObjectOutputStream oos = new ObjectOutputStream(out);
						GossipPingReplyMessage gReply = new GossipPingReplyMessage(currentHeartBeat);
						oos.writeObject(gReply);
						oos.flush();

						logGos.info("[GossipServer] reply to suspicion from "+toRank+ 
								" by sending " + currentHeartBeat);
						break;

					case MessageCmd.GOSSIP_UNREGISTER :
						key = gosMsg.getGossipID();
						rankInList = gosMsg.getMyRank();
						threadList = hashtable.get(key);
						logGos.info("[GossipServer] unregister request from app. hashID=" 
								    + key + " (rank=" + rankInList+")");
						if(threadList != null) {
							int numElement = threadList.size();
							for(int i = 0; i < numElement; i++) {
								gos = threadList.elementAt(i);
								if(gos.getMyRank() == rankInList) {
									gos.stop();
									threadList.removeElementAt(i);
									break;
								}
							}
						}
						if(threadList.size() == 0) {
							hashtable.remove(key);
						}
						break;

					default :
						logGos.debug("[GossipServer] received an unknown message");
						break;

				}
			} catch (Exception e) {
				//e.printStackTrace();
				//System.exit(1);
			}
		}

	}


	public class CheckService extends TimerTask {
		public void run() {
			//logGos.debug("************** Check Service RUNS ********************");
			//logGos.debug("[GossipServer] check detector process");
			Enumeration keys = hashtable.keys();
			while(keys.hasMoreElements()) {
				String key = (String)keys.nextElement();
				Vector<GossipDetector> threadList = hashtable.get(key);
				if(threadList != null) {
					int numElement = threadList.size();
					//logGos.debug("[GossipServer] There are " + numElement + " instances of GossipDetector");
					for(int i = 0; i < numElement; i++) {
						GossipDetector gos = threadList.elementAt(i);
						if(!gos.isAlive()) {
							gos.stop();
							threadList.removeElementAt(i);
							numElement = threadList.size();
							i--;
						}
					}
					if(threadList.size() == 0) {
						hashtable.remove(key);
					}
				}
			}
			//logGos.debug("[GossipServer] done check detector process");
		}
	}

	static public void main(String[] args) {
		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( );
		GossipDetectorServer gossipServer = new GossipDetectorServer(ppConf.getFDPort(), 
									     ppConf.getMPDPort());
		gossipServer.start();
	}
}

