package p2pmpi.fd;

import p2pmpi.common.*;
import p2pmpi.message.*;
import java.net.*;
import java.io.*;
import java.util.*;
import org.apache.log4j.*;

public abstract class GossipDetector implements Runnable {
	protected int myMPIRank;
	protected int myRank;
	protected int mpdPort;
	protected String hashID;
	protected int mpiPort;
	protected URI[] uriList;
	protected boolean[] aliveList;
	protected long[] heartbeatList;

	protected int numPeers;
	protected long T_GOSSIP;
	protected long T_MARGIN;	//T_MAX_STARTUP
	protected long T_MAX_HANG;
	protected long T_CLEANUP;		
	protected long T_MIN_CLEANUP;		
	protected Timer timer;
	protected Timer aliveTimer;
	protected volatile boolean alive = true;

	protected static Logger log;
	protected long t_stamp;
	protected int t_diff;

	public GossipDetector(int myRank, String hashID, int mpiPort, URI[] uriList, 
			      int mpdPort, int mpiRank,
			      int t_gossip, int t_margin, 
			      int t_diff, long t_stamp, int t_hang) {
		myMPIRank   	= mpiRank;
		this.myRank	= myRank;
		this.hashID	= hashID;
		this.mpiPort	= mpiPort;
		this.uriList	= uriList;
		this.mpdPort	= mpdPort;
		this.t_stamp    = t_stamp;
		this.t_diff     = t_diff;

		log =  Logger.getLogger("MPD");
		T_GOSSIP 	= (long)t_gossip;
		T_MAX_HANG	= (long)t_hang;
		numPeers = uriList.length;

		T_MARGIN  = (long)Math.ceil((double)t_margin/(double)t_gossip); //transform t_margin to number of round;

		aliveList = new boolean[numPeers];
		heartbeatList = new long[numPeers];
		for(int i=0; i < numPeers; i++) {
			aliveList[i] = true;
			heartbeatList[i] = 0;
		}

		timer = new Timer();
		aliveTimer = new Timer();
	}

	public void stop() {
		timer.cancel();
		aliveTimer.cancel();
		log.info("[GossipDetector] *** Stop gossip for hashID="+hashID+" (rank="+myRank+")");
	}

	public boolean isAlive() {
		return alive;
	}

	/**
	 * The Gossip detector starts
	 */
	public void run() {

	    long startHeartBeat = 
	    	(long)Math.ceil((double)(System.currentTimeMillis() - t_stamp + t_diff)/(double)T_GOSSIP); //transform t_diff to number of round
		heartbeatList[myRank] = startHeartBeat;

		log.debug("[GossipDetector] *** Start gossip for hashID ="+hashID+" and myRank=" + myRank);
		log.debug("[GossipDetector] parameters: T_CLEANUP=" + T_CLEANUP + " rounds, T_MARGIN=" + T_MARGIN + " rounds");
		log.debug("[GossipDetector] myrank=" + myRank + ", startHeartBeat=" + startHeartBeat);
		log.debug("[GossipDetector] gossip start time="+ System.currentTimeMillis() + 
							    ", t_stamp=" + t_stamp + ", t_diff="+t_diff);

		timer.scheduleAtFixedRate(new DoGossip(), 0, T_GOSSIP);
		aliveTimer.scheduleAtFixedRate(new CheckAlive(), 0, GossipConfig.t_appalive);
	}

	public int getMyRank() {
		return myRank;
	}

	public long getCurrentHeartBeat() {
		return heartbeatList[myRank];
	}

	public void mergeDestination(int rank, long heartbeat) {
		if(heartbeatList[rank] < heartbeat) 
			heartbeatList[rank] = heartbeat;
	}

	public void merge(long[] extHBList) {
		if(extHBList == null)
			return;

		//log.debug("[GossipDetector: Rank ->"+myRank+" merge heartbeat");
		for(int i = 0; i < numPeers; i++) {
			//log.debug("--------------------");
			if(heartbeatList[i] < extHBList[i]) {
				heartbeatList[i] = extHBList[i];
			}
			//log.debug("Node " + i + " : HB = "+heartbeatList[i]);
		}
	}

	public boolean ping(int destination) {
			//Connect and send gossip message
			String ip = uriList[destination].getHost();
			int port = uriList[destination].getPort();
			GossipMessage gosMsg = new GossipMessage(MessageCmd.GOSSIP_PING, hashID);
			gosMsg.setToRank(destination);
			gosMsg.setHeartBeatList(heartbeatList);
			log.info("[GossipDetector] suspected node " + destination + ", sending ping message to " + ip);
			try {
				  Socket socket = new Socket();
				  OutputStream out;
				  ObjectOutputStream oos;
				  ObjectInputStream ois; 
				  InputStream in;
				  Object o=null;
				  try {
					    InetSocketAddress addr = new InetSocketAddress(ip, port);
					    socket.connect(addr, 500);
					    out = socket.getOutputStream();
					    oos = new ObjectOutputStream(out);
					    oos.writeObject(gosMsg);
					    oos.flush();
					    in = socket.getInputStream();
					    ois = new ObjectInputStream(in);
					    o = ois.readObject();
					    ois.close();
					    in.close();
					    oos.close();
					    out.close();
					    socket.close();
				  } catch (Exception e) { 
					    log.info("[GossipDetector] cannot create socket to "+ip+":"+port); 
					    System.err.println("[GossipDetector] cannot create TCP socket to "+ip+":"+port);
				  }

				  if(o instanceof GossipPingReplyMessage) {
					    long heartbeat = ((GossipPingReplyMessage)o).getCurrentHeartBeat();
					    if (heartbeat < 0) 
						    return false;
					  mergeDestination(destination, heartbeat);
					  log.info("[GossipDetector] suspected  node:" + destination + "[" + ip + "] reply with heartbeat->" +
								heartbeat);
					  return true;
				} else {
					return false;
				}
			} catch (Exception ignored) {
				log.info("[GossipDetector] cannot connect to suspected node:" + destination);
				return false;
			} 
	}



	public abstract int getNextDestination(int round);


	public class CheckAlive extends TimerTask {
		private void warnMPD() {
			try {
				log.info("[GossipDetector:"+hashID+":"+myRank+"] detect application failure. Unregister it from MPD (port:" + mpdPort+")");
				AppUnregisterMessage unregMsg = new AppUnregisterMessage(hashID + "--" + myMPIRank);
				Socket s = new Socket("127.0.0.1", mpdPort);
				OutputStream out = s.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject(unregMsg);
				oos.flush();
				oos.close();
				out.close();
				s.close();
			}catch (Exception e) {
				log.info("[GossipDetector] Error during unregistration from MPD ("+e.toString()+")");
			}
		}

		public void run() {
			try {
				//log.debug("[GossipDetector:"+hashID+":"+myRank+"] send check alive message to application");
				IsAliveMessage isAliveMsg = new IsAliveMessage(hashID);
				Socket s = new Socket("127.0.0.1", mpiPort);
				OutputStream out = s.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject(isAliveMsg);
				InputStream in = s.getInputStream();
				ObjectInputStream ois = new ObjectInputStream(in);
				Object msg = ois.readObject();

				if(msg instanceof AliveMessage) {
					AliveMessage aliveMsg = (AliveMessage)msg;
					if(!hashID.equals(aliveMsg.getKey())) {
						// DEAD	
						alive = false;
						aliveTimer.cancel();
						timer.cancel();
						//Notify MPD to remove app from ProcessInfo
						warnMPD();		
					}
				} else if (msg instanceof NoAliveMessage) {
					// DEAD
					alive = false;
					aliveTimer.cancel();
					timer.cancel();
					//Notify MPD to remove app from ProcessInfo
					warnMPD();		
				} else {
					//System.out.println(">>CheckAlive: Got Unknown message");
				}
			} catch (Exception e) {
				log.info("[GossipDetector]: *** MPI application crashed ***");
				//e.printStackTrace();
				alive = false;
				aliveTimer.cancel();
				timer.cancel();
				//Notify MPD to remove app from ProcessInfo
				warnMPD();		
			}
		}
	}

	/**
	 * To send its gossip message
	 * and compare to other gossip messages 
	 */
	public class DoGossip extends TimerTask {
		int currentStep;

		DoGossip() {
			currentStep = 0;
		}

		public void run() {
			//log.debug("[GossipDetector] myrank = " + myRank + " do gossip");
			long heartbeatDiff;
			//Increase my heartbeat
			heartbeatList[myRank]++;
			// the following debug line will fill up your log quicly !!
			//log.debug("[GossipDetector] pulse++ HB["+myRank+"]="+heartbeatList[myRank]);

			//Check if we find suspect peers
			for(int i = 0; i < numPeers; i++) {
				if(aliveList[i]) {
					heartbeatDiff = heartbeatList[myRank] - heartbeatList[i];
					if(heartbeatDiff > T_CLEANUP) {
						//if this node has never been sent any gossip message
						//we will wait by using T_CLEANUP + T_MARGIN
						if(heartbeatList[i] == 0) {
							if(heartbeatDiff < (T_CLEANUP + T_MARGIN)) 
								continue;
						}

						//Ping node, if ping is success, ignore this false alarm
						if(ping(i)) {
							continue;
						}

						//Set dead communication
						aliveList[i] = false;
						/*
						Calendar rightNow = Calendar.getInstance();
						int hour = rightNow.HOUR;
						int min  = rightNow.MINUTE;
						int sec  = rightNow.SECOND;
						int milli = rightNow.MILLISECOND;
						System.out.println(hour+":"+min+":"+sec+":"+milli+"-> **** [GossipDetector] Detection failure of " + i + "th node in list");
						*/
						log.info("[GossipDetector] Detection failure for node #" + i + 
							" in list ( HB["+myRank+ "]="+heartbeatList[myRank]+
							" vs. remote HB["+i+"]="+heartbeatList[i]+")");

						//if detect another mpi app died, notify MPI
						NotifyMessage noMsg = new NotifyMessage(i);
						try {
							Socket s = new Socket("127.0.0.1", mpiPort);
							OutputStream out = s.getOutputStream();
							ObjectOutputStream oos = new ObjectOutputStream(out);
							oos.writeObject(noMsg);
							oos.close();
							out.close();
							s.close();
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			}

			int destination = getNextDestination(currentStep);
			int slide = 0;
			while(true) {
				destination = (destination + slide)%numPeers;
				if((aliveList[destination]) && (destination != myRank)) {
					break;
				}
				slide++;
				//prevent from all nodes dead inside this loop
				if(slide > numPeers-1)
					break;
			}

			//Connect and send gossip message
			String ip = uriList[destination].getHost();
			int port = uriList[destination].getPort();
			GossipMessage gosMsg = new GossipMessage(MessageCmd.GOSSIP, hashID);
			gosMsg.setMyRank(myRank);
			gosMsg.setToRank(destination);
			gosMsg.setHeartBeatList(heartbeatList);
			//log.debug("[GossipDetector:"+hashID+":"+myRank+"] sending gossip message to " + ip);
			try {
				Socket socket = new Socket();
				InetSocketAddress addr = new InetSocketAddress(ip, port);
				socket.connect(addr, 500);
				OutputStream out = socket.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject(gosMsg);
				oos.flush();
				oos.close();
				out.close();
				socket.close();
			} catch (Exception ignored) {} 

			currentStep++;
			if(currentStep >= T_MIN_CLEANUP) {
				currentStep = 0;
			}
			//log.debug("[GossipDetector] myrank = " + myRank + " end do gossip");
		}

	}
}
