/* $Id: NetIface.java 6539 2009-08-01 08:19:02Z genaud $tag */

/* NetIface handles the logic of network interfaces to use.
 * 
 * - The constructor Iface() discovers all interfaces and  associates to each 
 *   interface name a type (ipv4,linklocal,...) : HashMap ifaceMapNameType.
 *   It sets as default interface the first one it finds usable in this list.
 *
 * - The constructor Iface( String prefIface, String prefAddress ) allows to 
 *   - choose a given interface among those discovered,
 *   - assign a given IP address (typically used to overide a private IP address.
 *
 *
 *
 * Part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package p2pmpi.common;

import java.net.*;
import java.io.*;
import java.util.*;
import p2pmpi.common.*;

public class NetIface {

        private byte LINKLOCAL=1;
        private byte LOOPBACK =2;
        private byte SITELOCAL=4;
        private byte IPV4=8;
        private byte IPV6=16;

        private String INVALID_TAG="invalid";

        private String ifname=null;
        private String canonicalHostName=null;
        // associative list (ifname, type)
        private HashMap<String,Byte> ifaceMapNameType = new HashMap<String,Byte>();
        // associative list (ifname, InetAddress)
        private HashMap<String,InetAddress> ifaceMapNameAddress = new HashMap<String,InetAddress>();

        private String selectedIface=null; // the name of the interface eventually selected
        private String errIfaceMsg=null;   // status msg after operation 
        private String errAddrMsg=null;    // status msg after operation 
        private String extIfIpv4Address=null; // my IPv4 as it must be reported to outside my private net
        private String extIfIpv6Address=null; // my IPv6 as it must be reported to outside my private net
        private String intIfIpv4Address=null; // my IPv4 as it must be reported to local services
        private String intIfIpv6Address=null; // my IPv6 as it must be reported to local services
 

        /**
         * Main constructor
         * Discovers all interfaces and  associates to each interface name a type 
         * (ipv4,linklocal,...) : HashMap ifaceMapNameType. It sets as default interface 
         * a random one it finds usable in this list.
         *
         * sets -selectedIface
         *      -ifaceMapNameType
         *      -ifaceMapNameAddress 
         **/
        public NetIface( ) {
        NetworkInterface iface=null;
                try {
                        for( Enumeration ifaces = NetworkInterface.getNetworkInterfaces();ifaces.hasMoreElements();) {
                                iface = (NetworkInterface)ifaces.nextElement();
                                InetAddress ia = null;
                                byte type=0; 

                                // scans all IPs assigned to this interfaces.
                                for (Enumeration ips = iface.getInetAddresses();ips.hasMoreElements();){
                                        ia = (InetAddress)ips.nextElement();

                                        if (isIPv4Address( ia.getAddress() )) type |= IPV4;
                                        if (isIPv6Address( ia.getAddress() )) type |= IPV6;
                                        if (ia.isLinkLocalAddress()) type |= LINKLOCAL;
                                        if (ia.isSiteLocalAddress()) type |= SITELOCAL;
                                        if (ia.isLoopbackAddress()) type |= LOOPBACK;
                                        // choose any suitable interface
                                        // (calling a common function to determine usability is to avoid
                                        // dupplicate code, on the other hand be careful not to overide this
                                        // function in a subclass. -s.g )
                                        if (isUsable( type )) {
                                                selectedIface = iface.getDisplayName();
                                                //System.out.println("** selected="+ selectedIface);
						    }
						    // will store last IPv4 found for that interface 
						    // (since a put with an existing key will overwrite the value)
						    if (isIPv4Address( ia.getAddress()) ) {
								ifaceMapNameType.put((String) iface.getDisplayName(), new Byte(type) );
								ifaceMapNameAddress.put((String) iface.getDisplayName(), (InetAddress)ia );
						    }
					  }
				}
                        //System.out.println("discover: " + ifaceMapNameType.toString());
                }
                catch (SocketException e) { 
                        System.err.println("[Error] cannot determine network interfaces.");
                }
        }

       /**
         * Constructor specifying a prefered interface and/or a prefered IP.
         * If prefIface is found in the discovered interfaces, then it is selected.
         * Otherwise it is simply ignored.
         * If predAddress exists and is valid, it is selected.
         * sets -selectedIface
         *      -ifaceMapNameType
         *      -ifaceMapNameAddress 
         *
         * @param prefIface the name of the interface to select 
         * @param prefAddress the address (IP as a string) to store. If null, nothing stored.
         **/
        public NetIface( String prefIface, String prefAddress ) {

                  this(); // call basic constructor for interface discovery

                  //------ registers requested iface if any ------------
                  
                  // if it is one of the discovered interfaces
                  if ( ifaceMapNameType.containsKey( prefIface )) {
                        //System.out.println("* prefIface '"+prefIface +"' exists...");
                        //check it is an ipv4 and not a loopback
                        Byte ifTypeValue = (Byte) ifaceMapNameType.get( prefIface );
                        byte ifType = ifTypeValue.byteValue();
                        // (calling a common function to determine usability is to avoid
                        // dupplicate code, on the other hand be careful not to overide this
                        // function in a subclass. -s.g)²
                        if  (isUsable( ifType )) {
                                selectedIface = prefIface;
                                //System.out.println("** selected="+ prefIface);
                        }
                        else {
                                errIfaceMsg = "the specified network interface '"+prefIface+"' is not usable.";
                                //System.out.println("** selected= avoided");
                        }
                  }
                  else {
                          // if prefIface is consistent but not found, ignore but set the err msg to warn user.
                          if ( prefIface != null && prefIface.length()>0 ) {
                                  errIfaceMsg = "the specified network interface '"+prefIface+"' does not exist.";
                          }
                  }
                  //------ register requested address if any ------------
                  if (prefAddress != null ) {
                       // do not trust the user: check it is a valid address
                       InetAddress ia;
                       try {
                           ia = InetAddress.getByName( prefAddress ); 
                           this.extIfIpv4Address =  splitStringAddress( ia.toString() );
                       } catch( UnknownHostException e) {
                              errAddrMsg = "the specified address '"+prefAddress +"' represents an unknown host.";
                              this.extIfIpv4Address = INVALID_TAG; 
                       }
                  }
        }

        /**
         * helper function
         **/
        private boolean isUsable( byte mask ) {
                return (((mask & IPV4) == IPV4) && ((mask & LOOPBACK) != LOOPBACK)); 
        }

        public String getErrorIfaceMsg( )  {
                return( errIfaceMsg );
        }
        public String getErrorAddrMsg( )  {
                return( errAddrMsg );
        }
        /**
         * Return the type of an interface 
         * @param ifName the identifier of the interface
         * @return null if the identifier is not among the discovered interfaces, 
         *         or a string describing the type of the interface. 
         **/
        public String getIfaceType( String ifName )  {

                if (!ifaceMapNameType.containsKey( ifName )) 
                        return( null );
                String typeLine="";
                Byte ifTypeValue = (Byte) ifaceMapNameType.get( ifName );
                byte ifType = ifTypeValue.byteValue();
                if ( (ifType & IPV4) == IPV4 ) typeLine+="ipv4 ";
                if ( (ifType & LINKLOCAL) == LINKLOCAL ) typeLine+="link-local ";
                if ( (ifType & SITELOCAL) == SITELOCAL ) typeLine+="site-local ";
                if ( (ifType & LOOPBACK) == LOOPBACK ) typeLine+="loopback ";
                return( typeLine );
        }	

        /**
         * Return a list interfaces 
         * @param withType when true, add to each iface a complementary string describing its type 
         * @return empty string if no interface, otherwise a string of interface names
         **/
        public String getAllIface( boolean withType )  {

                String typeTab="";
                Iterator iter = ifaceMapNameType.entrySet().iterator();
                while ( iter.hasNext() ) { 
                        Map.Entry me = (Map.Entry)iter.next(); 
                        typeTab += "["+me.getKey()+"] ";
                        if ( withType ) 
                            typeTab += getIfaceType((String) me.getKey())+"\n" ;
                }	
                return( typeTab );
        }
 	  /**
	   * Return the name or the IP contained in the result of InetAddress.toString().
	   * The result parsed should be of the form "<hostname>/<ip>". After split,
	   * first string is the hostname if resolved, 2nd string is the IP.
	   * Return preferably hostname if non empy.
	   **/
	  private String splitStringAddress( String s ) {
		    String [] ret = s.split("/",2);
		    if (ret[0].length() > 0)
				return( ret[0] );
		    return ( ret[1] );
	  }

	  public String getIfName( ) {
		    return( this.selectedIface );
	  }


        /**
         * Set the Ipv4 address (as a String) to be used with the selected interface.
         **/
        private void setExternalIpv4( String ifName ) {
                byte [] ip=null;
                InetAddress ia = (InetAddress) ifaceMapNameAddress.get( ifName );
                this.canonicalHostName = ia.getCanonicalHostName();
                if ( ia != null ) {
                        ip = ia.getAddress();
                }
                if (isIPv4Address( ip )) {
				// mind the '/' in the output of InetAddress.toString() !
                        this.extIfIpv4Address = splitStringAddress( ia.toString() );
                }
                // the following should not happen since in we discover only ipv4 in the constructor
                if (isIPv6Address( ip )) {
				// mind the '/' in the output of InetAddress.toString() !
                        this.extIfIpv6Address = splitStringAddress( ia.toString() );
                }
		    
        }


	  /**
	   * Return the IP address which can be used from outside of my network.
         * (In case of a public IP, getExternalIfIpv4() and getInternalIfIpv4() return a same result).
         * The IP corresponds to the network interface automatically selected.
	   * If an IP has been set in config file, this is the IP returned.
	   * Otherwise, returns the hostname if available, or the ip of the selected interface.
	   * If no network interface is available, returns null.
	   **/ 
	  public String getExternalIfIpv4( ) {
            
                if (extIfIpv4Address == null ) {
                        // it has never been set, guess the address
                        this.setExternalIpv4( this.selectedIface );
                        return( this.extIfIpv4Address );
                }
                // check if the address has already been assigned incorrectly
                if (extIfIpv4Address.equals( INVALID_TAG)) {
                        return( null );
                }
                else // return what was stored
                        return( this.extIfIpv4Address );
        }


        public String getExternalIfIpv6( ) {
                if	(this.extIfIpv6Address ==null)
                        return( null );
		    else 
				return( splitStringAddress( this.extIfIpv6Address ));
	  }

	  /**
	   * Return the IP address which can be used from inside my network.
         * The IP corresponds to the network interface automatically selected.
	   * If an IP has been set in config file, this is the IP returned.
	   * Otherwise, returns the hostname if available, or the ip of the selected interface.
	   * If no network interface is available, returns null.
	   **/ 
	  public String getInternalIfIpv4( ) {
            
                if (intIfIpv4Address == null ) {
                      // it has never been set, guess the address
                      InetAddress ia =  ifaceMapNameAddress.get((String) selectedIface );
                      this.intIfIpv4Address = splitStringAddress( ia.toString() );
                      return( this.intIfIpv4Address );
                }
                else // return what was stored
                      return( this.intIfIpv4Address );
        }


	  public String getCanonicalHostName() {
		    return( canonicalHostName );
	  }

	  private static boolean isIPv6Address(byte[] address) {
		    try {
				java.net.Inet6Address.getByAddress(address);
				return true;
		    } catch(Exception e) {
				return false;
		    }
	  }

	  private static boolean isIPv4Address(byte[] address) {
		    return address.length == 4;
	  }

	 /**
	  * main. Just tot test and see how to use class.
	  **/
	  public static void main(String[] args) {

                String prefIface=null;
                String prefIp=null;

                if (args.length>=1) {
                  prefIface=args[0];
                  if (args.length>=2) {
                        prefIp=args[1];
                  }
                }
		    NetIface extIf  = new NetIface( prefIface, prefIp );
                String ifName =  extIf.getIfName() ;
                String ifIntAddr =  extIf.getInternalIfIpv4() ;
                String ifExtAddr =  extIf.getExternalIfIpv4();

                if (ifName == null) {
                     System.out.println("[Error] setting interface name: '"+extIf.getErrorIfaceMsg()+"'");
                     //extIf.getAllIface( true );
                     System.out.println("[Info] this interface has properties: "+extIf.getIfaceType( prefIface ));
                }
                // both internal and external address should be set in any case
                if ((ifExtAddr == null) || (ifIntAddr==null)) {
                     System.out.println("[Error] setting interface address: '"+extIf.getErrorAddrMsg()+"'");
                     //extIf.getAllIfaceType();
                }

                //else
		    System.out.println("* selected interface: [" + ifName  + "] int/ext ipv4: ["+ifIntAddr+"/"+ifExtAddr+"]");
	  }

}
