/**
 * $Id:
 * DiskCache is an object associated to the File Transfer Server, whose
 * role is to handle persistent files on peeers.
 *
 * @author: genaud
 *
 */

package p2pmpi.common;

import p2pmpi.common.*;

import java.io.File;
import java.io.InputStream;
import java.util.Vector;


public class DiskCache {
	private long DISK_CACHE_SIZE_LIMIT= 200 * 1024 *1024; // (200MB)
	private final String DEFAULT_DISK_CACHE_DIRNAME= "datacache";
    private String diskCacheDir;
	private long diskCacheSizeBytes=0;
	private File diskCache = null;
	private static String hostIP = null; // my host IP, i.e the one other can use to contact me
	private static String osname=OsInfo.getName();


	/**
	 * Constructor with dirname and size. 
       * @param dir the directory under which cache is to be created or read. It is currently the temp dir.
       * @param diskCacheDir the name of the cache directory. Should be read from config file.
       * @param diskCacheSize the size requested for the cache (set by the user in config file).
	 */
	public DiskCache( String dir, String diskCacheDir,  String diskCacheSize ) {

            if (diskCacheDir != null) 
                  this.diskCache = new File( dir + File.separatorChar + diskCacheDir );
            else
      		this.diskCache = new File( dir + File.separatorChar + DEFAULT_DISK_CACHE_DIRNAME );

		// First set the cache size as specified in config file, with a maximum.
		try {
			this.diskCacheSizeBytes = Math.min( 
					Long.parseLong(diskCacheSize)*1024*1024,   // convert MBs
					DISK_CACHE_SIZE_LIMIT);

		} catch ( NumberFormatException e ) {
			System.err.println("[DiskCache] bad cache size specification.");
			this.diskCacheSizeBytes=0;
		}
		// if the cache does not exist yet, try create it
		if (!this.diskCache.exists()) {
			if ( this.diskCache.mkdirs() ) { // true if the dir and subdirs where created
				// ok
			}
			else {
				System.out.println("[DiskCache] error creating cache "+ this.diskCache);
				this.diskCache = null;
				this.diskCacheSizeBytes = 0;
			}
		}
	}
	/**
	 * Constructor with dir only 
	 */
	public DiskCache( String dir, String diskCacheDir ) {

            if (diskCacheDir != null) 
                  this.diskCache = new File( dir + File.separatorChar + diskCacheDir );
            else
      		this.diskCache = new File( dir + File.separatorChar + DEFAULT_DISK_CACHE_DIRNAME );

		// if the cache does not exist yet, try create it
		if (!this.diskCache.exists()) {
			if ( this.diskCache.mkdirs() ) { // true if the dir and subdirs where created
				// ok
			}
			else {
				System.out.println("[DiskCache] error creating cache "+ this.diskCache);
				this.diskCache = null;
				this.diskCacheSizeBytes = 0;
			}
		}
	}


	/**
	 * Return the disk cache size in bytes (converted from the user specification in config file)
	 */
	public long getSize() {
		return( this.diskCacheSizeBytes );
	}

	/**
	 * Return the Disk Cache as a dir 
	 */
	public File getDir() { 
		return( this.diskCache );
	}

	/**
	 * Return the total size occupied by files in cache 
	 */
	public long getUsedSize( ) {
		long cacheUsed=0; 
		File[ ] files = getDir().listFiles();

		for (int i=0; i<files.length; i++ ) {
			cacheUsed += files[i].length();
		}
		return( cacheUsed );
	}

	/**
	 * Return true if the space remaining in the cache is greater or equal to size
	 * (assuming not deleting any file in cache to make extra space).
	 * @param size size in bytes that we try to make fit in the cache
	 */
	public boolean canContain( long size ) {
		return ( this.getUsedSize() + size < this.getSize() );
	}

	/**
	 * Search files in cache which have the same md5sum as the one specified. 
	 * @param the specified md5sum 
	 * @return null if no match was found, or a vector of the filenames (short name) which match the MD5
	 */
	public Vector<String> searchByMd5( String md5sum ) {
		Vector<String> matches = new Vector<String>();
		File[ ] files = getDir().listFiles();

		for (int i=0; i<files.length; i++ ) {
			if ( md5sum.equals( Md5sum.md5sum(files[i] ))) {
				matches.add( files[i].getName() );
			}
		}
		return ( matches ); 
	}

	/**
	 * List of md5sums of files in cache 
	 * @return a Vector of md5sums of files in cache 
	 */
	public Vector<String> getMd5s( ) {
		Vector<String> sums = new Vector<String>();
		File[ ] files = getDir().listFiles();

		for (int i=0; i<files.length; i++ ) 
			sums.add( Md5sum.md5sum(files[i] ));
		return ( sums ); 
	}



	/**
	 * reclaim space in cache using a LRU policy
	 *
	 * @param size size in bytes to reclaim
	 * @return true if the space reclaimed has been granted, false otherwise
	 */
	private boolean reclaimLRU( long size ) {

		if ( size > this.getSize()) { // reclaim more than the cache capacity
			return( false );
		}
		long occupied = this.getUsedSize();
		do {
			long minTime = Long.MAX_VALUE;
			// get the oldest file and subtract it's size from the occupied and delete it from cache.
			File[] files = getDir().listFiles();
			int k=0;
			for(int j=0; j<files.length;j++) {
				if (minTime > files[j].lastModified()) {
					minTime = files[j].lastModified();
					k = j;
				}
			}
			occupied -= files[k].length();
			files[k].delete();
		} while ( occupied + size > this.getSize());
		return( true );
	}

	/**
	 * write a stream of bytes to the cache under a given filename,
	 * using a LRU (Least Recently Used) policy if it is needed.
	 *
	 * @param stream the input stream
	 * @param filename the filename to save the bytes stream under
	 * @return true if it could write the file in the cache, false if it failed. 
	 **/
	public boolean writeToLRU( InputStream stream, String filename, long filesize ) {

		if ( ! this.canContain( filesize )) {
			if ( ! this.reclaimLRU( filesize )) {
				System.err.println("[DiskCache] reclaimed "+filesize +
						" bytes which is greater than cache capacity ("
						+ this.getSize()+")");
				return( false );
			}
		}
		// at this point we should have enough space in cache to write
		StreamSaver sv = new StreamSaver();
		long bytesRead = sv.saveNetworkInputStream( getDir(), filename, filesize, stream);
		if (bytesRead != filesize) {
			System.out.println("[FT] internal error for file "+filename+": advertized="+
					filesize +", read="+bytesRead);
			return( false );
		}
		return( true );

	}

	/**
	 * makes available the files which are present in cache to temp/<uuid> 
	 * by copying the files in case of windows systems(except vista) because
	 * of the lack of linking facility and creating the link in temp/<uuid>
	 * for other OSes.
	 * 
	 * @param appDir: where files has to be made available before execution
	 * @param fileCharacteristicsVector: vector consisting of characteristics of files
	 * 
	 * @return integer 0 if it is successful in publishing files else -1
	 **/

	public int publishFiles(File appDir, Vector<FileCharacteristics> fileCharacs) {
		String osname = OsInfo.getName();
		int numFile = fileCharacs.size();
		if(OsInfo.isWindows(osname) & (!OsInfo.isWindowsVista(osname))) {
			// copy the files which are to be cached from datacache to tmp/<uuid>
			String[] fileNames = this.diskCache.list();
			for(int i=0; i<numFile; i++) {
				if(fileCharacs.elementAt(i).isToBeCached()) {
					for(int j=0; j<fileNames.length; j++) {
						if(fileCharacs.elementAt(i).getMd5sum().equals(Md5sum.md5sum(new File(this.diskCache+File.separator+fileNames[j])))) {
							// copy the file using fileCopier class.
							try {
								FileCopier fc = new NioCopier();
								fc.copy(new File( this.diskCache,fileNames[j]), new File(appDir,fileNames[j]));
							} catch (Exception e) {
								e.printStackTrace();
							}

						}
					}
				}
			}
		} else {
			for(int i=0; i<numFile; i++) {
				if(fileCharacs.elementAt(i).isToBeCached()) {
					int returnvalue= OsInfo.mklink(appDir, this.diskCache, fileCharacs.elementAt(i).getOnlyFileName()) ;
					if(returnvalue == -1) {
						System.err.println("** [Error] link could not be made to the files in data cache.");
						return -1;
					}
				}
			}

		}
		return 0;
	}
}


