/*
 * RefreshDisplay : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka and Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * RefreshDisplay.java
 * revised : july 31, 2007, 12:15:43 (UTC+0200) (Ghazi's original code restructuration)
 *
 * @author Stephane Genaud
 *
 * RefreshDisplay is a thread continuously displaying available information about peers.
 * It first instantiates a StatQuery object, then goes in a wait() state until 
 * either a refresh is explicitely asked ("Reload" button in GUI mode) or we have
 * reached refresh timeout. 
 **/


package p2pmpi.visu;

import p2pmpi.message.MessageCmd;
import p2pmpi.message.MPDMessage;
import p2pmpi.message.StatQueryMessage;
import p2pmpi.message.StatInfoMessage;
import p2pmpi.common.*;
import p2pmpi.mpd.*;

import java.util.*;
import java.io.*;
import java.net.*;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.*;
import javax.swing.JRadioButton;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.JButton;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JCheckBox;
import javax.swing.JList;
import javax.swing.JTable;
import javax.swing.JScrollPane;
import javax.swing.table.TableModel;
import javax.swing.table.AbstractTableModel;

import org.apache.log4j.*;


public class RefreshDisplay implements Runnable {
	  private static Logger log;
	  Visu_Param window;
        private URI superNode; // supernode URI set by constructor
	  private boolean clearCacheStatus; // is query to use cache 
	  private StatQuery MPDReadyQuery = null;
	  private Thread proxyNotificationListener = null; // will listen to notifications
	  private int listenPort; // the port the listener thread is bound to
	  private ServerSocket listenServerSocket; 

	/** 
	 * constructor.
	 **/
      public RefreshDisplay( Visu_Param  window ) {

            log = Logger.getLogger("VISU");
	      log.debug("@ RefreshDisplay :: RefreshDisplay()");
            File configFile          = new File(System.getProperty("P2PMPI_HOME"), "P2P-MPI.conf");
            P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( configFile.toString() );
            URI supernode            = ppConf.getSuperNode();
            URI visuproxy            = ppConf.getVisuProxy();
    		String peerCache         = ppConf.getPeerCacheFile();
    		String externalIp        = ppConf.getExternalIP();
    		int minPort  		 = ppConf.getMinPort();
    		int maxPort              = ppConf.getMaxPort();

            this.window = window;


		this.listenServerSocket = getServerSocket( minPort, maxPort ); 
		if ( listenServerSocket != null ) {
			  this.listenPort = listenServerSocket.getLocalPort();
			  // creates a query object, passing all necessary networking contacts
			  this.MPDReadyQuery = new StatQuery( supernode, visuproxy, peerCache, externalIp, listenPort, window );
			  // could not bind a ServerSocket for listener thread. Don't start it. We'll try a direct connection to supernode.
			  Thread proxyNotificationListener = new ProxyNotificationListener( this.MPDReadyQuery, listenServerSocket );
			  log.debug("@ RefreshDisplay :: RefreshDisplay() --> starting proxyNotificationListener thread ... ");
			  proxyNotificationListener.start();
		}	
		else
			this.MPDReadyQuery = new StatQuery( supernode, visuproxy, peerCache, externalIp, -1, window );

	}

	/** 
	 * Stop the Refresh Display process.
	 * Actions to perform : 
	 * 1) close VisuProxy notifications listen port (might be bound)
	 * 2) Instruct StatQuery object to save its DOM cache to file. 
	 * 3) Instruct StatQuery to unregister to visuproxy if needed. 
	 **/
	public void quit( ) {
		  try {
		  	this.listenServerSocket.close();
		  } catch (Exception e) {}
		  this.MPDReadyQuery.writeCache();
		  this.MPDReadyQuery.unregister();
	}

	/**
	 * set clearCache status
	 **/
	public void setClearCache( boolean b ) {
		  clearCacheStatus = b;
	}

	/**
	 * getServerSocket() : intializes a valid server socket for client to communicate.
	 *
	 * @param minPort min value in allowed port range to communicate in application 
	 * @param maxPort max value in allowed port range to communicate in application
	 * @param seed a number to intializes the random choice of ports in the range 
	 * @return a ServerSocket allocated on a free port (get the port with getLocalPort())
	 *         or null if no port could be allocated in [minPort-maxPort].
	 **/
	private ServerSocket getServerSocket ( int minPort, int maxPort ) {
		  ServerSocket reservedSocket=null;
		  if ( maxPort - minPort > 0 ) {
		  Random r = new Random();
		  int randomStart = r.nextInt(maxPort - minPort) + minPort; //avoid duplicate port

		  for(int i = randomStart; i <= maxPort; i++) {
			    try {
					reservedSocket = new ServerSocket(i);
					break;
			    } catch (Exception e) {
					reservedSocket = null;
					continue;
			    }
		  }
		  if (reservedSocket == null) {
			    for(int i = minPort; i <= maxPort; i++) {
					try {
						  reservedSocket = new ServerSocket(i);
						  break;
					} catch (Exception e) {
						  reservedSocket = null;
						  continue;
					}
			    }
			    if (reservedSocket == null) {
					System.err.println("** [Error] No allowed ports are available in ["+ minPort+"--"+maxPort+"]");
			    }
		  }
		 }
		 else {
			String m  = new String("[Error] Visu client: port range [MIN_PORT - MAX_PORT] in configuration is invalid."); 
			log.error( m );
			System.err.println( m );
		  }
		  return( reservedSocket );
	}


	/**
	 * Put thread in waiting state (for a Reload or Timeout refresh) 
	 **/
	public synchronized void monitorReloadOrRefresh( long timeout ){
		  
	    log.debug("@ RefreshDisplay :: monitorReloadOrRefresh() --> wait(" + timeout +") \n");
          if (timeout == 0) {
			notify();  // unlock wait
	    }
	    else {
	    	try {
			wait( timeout ); // until the "reload" button pressed emits a notify, or sleep() ends emitting a notify as well
	    	} catch(InterruptedException e) { 
			e.printStackTrace(); 
	    	}
	   }
	}

	/**
	 * Run 
	 **/
	public void run( ) {

		  int delay=0; // time to set up first data in adjacent thread
		  int peerListLength;

		  
		  // if we connected to VisuProxy, we do not need "Reload" or "Refresh" settings. Hide them.
		  //if ( MPDReadyQuery.isConnectedToVisuProxy( ) ) {
		  //	  window.setREloadCommandsVisible( false );
		  //}

		  // now, enter query inifinite loop.
		  while ( true ) {	
			    if ( MPDReadyQuery.query( !clearCacheStatus )) { 

					log.debug("@ RefreshDisplay :: run() --> window.peers.size()="+window.peers.size() );
					//we change the status of each peer and we clear the task Table
					((taskTableModel)window.jtaskTable.getModel()).clear();
					for(int i=0;i<window.peers.size();i++){
						  ((PeerNode) window.peers.get(i)).setStatus("Scanning...");
						  // Here we clean the task table of the peer (for the new scan), 
						  ((PeerNode) window.peers.get(i)).Tasks.clear(); 
					}//and we clean the parent.infoMsgs Vector

					window.infoMsgs.removeAllElements();

					//we also clear the taskComboBox
					if (window.jc2==null)
						  window.jc2 = new NetMap(window);

					window.jc2.clearAllComboTask();
					window.jc2.addComboTask("Disponibility");

					log.debug("@ RefreshDisplay::run() --> set scan result timestamp "+new java.util.Date() );
					window.jLabel6.setText("Scan results at " + new java.util.Date());

					if (window.fen != null) {
						  window.fen.refresh();
						  log.debug("@ RefreshDisplay::run() --> window.fen.refresh()" );
					}
					if (window.textMode && window.refreshRate > 0)
						  delay =  window.refreshRate * 1000;
					else {
						  delay = Integer.parseInt(window.refresh.getText())* 1000;
						  log.debug("@ RefreshDisplay::run() --> refresh.getText()="+ delay);
					}
					if (window.IsLogging)
						  window.writeLog();

					log.debug("@ RefreshDisplay :: run() --> wait("+ delay +")");
					monitorReloadOrRefresh( (long)delay );

			    }
			    else {
					monitorReloadOrRefresh( (long)delay );
					log.debug("@ RefreshDisplay :: run() --> peerList is empty.");
			    }
			    // instruct the query object to update its peer list 
			    log.debug("@ RefreshDisplay :: run() -> MPDReadyQuery.updatePeerList( ) "); 
			    if ( (peerListLength = this.MPDReadyQuery.updatePeerList( )) == -1 ) {
					String msg = "[visu client] unable to get information about connected peers. Exiting.";
					log.error( msg );
					System.err.println( msg );
					try {
						  this.listenServerSocket.close();
					} catch (Exception e) {}

					System.exit( 1 );
			    }

		  }
	}
}
 
