/* A simple MPI example program that uses MPI_Scatterv.                 */

/* The program should be run with exactly 8 processes.                  */
/* Process zero initializes a matrix of 8 by 8 integers and distributes */
/* the lower triangular part of the matrix to the processes using       */
/* MPI_Scatter. The processes receive different number of elements:     */
/* process 0 gets one element, process 1 gets 2 elements and process i  */
/* gets i+1 elements. The processes (except process 0) then send a      */
/* message back containing the hostname of the machine the process runs */
/* on and its own part of the distributed array. Process 0 receives     */
/* these messages and prints out the received values, formatted as a    */
/* lower triangular matrix.                                             */

/* Compile the program with 'mpicc scatterv.c -o scatterv'              */
/* Run the program with 'mpirun -machinefile hostfile -np 8 scatterv'   */

import p2pmpi.mpi.*;

public class ScatterV {
	public static void main(String[] args)
	{

		int MAXPROC = 8;    /* Max number of procsses */
		int NAMELEN = 80;   /* Max length of machine name */
		int LENGTH  = 8;    /*  Size of matrix in LENGTH * LENGTH */
  		int i, j, np, me, count;
  		final int nametag  = 42;    /* Tag value for sending name */
  		final int datatag  = 43;    /* Tag value for sending data */
  		final int root = 0;         /* Root process in scatter */

  		Status status;          /* Status object for receive */

		//  char myname[NAMELEN];             /* Local host name string */
 		// char hostname[MAXPROC][NAMELEN];  /* Received host names */

  		int[] x = new int[LENGTH*LENGTH];        /* Send buffer */
  		int[] y = new int[LENGTH];                /* Receive buffer */

  		int[] sendcount;
		int[] displs;      /* Arrays for sendcounts and displacements */

		MPI.Init(args);
		np = MPI.COMM_WORLD.Size();
		me = MPI.COMM_WORLD.Rank();

  		//gethostname(&myname, NAMELEN);    /* Get host name */
		sendcount = new int[LENGTH];
		displs	  = new int[LENGTH];

	 	/* Initialize sendcount and displacements arrays */
  		for(i = 0; i < LENGTH; i++) {
    			sendcount[i] = i+1;
    			displs[i] = i*LENGTH;
  		}

  		if (me == 0) {    /* Process 0 does this */
    			if (np != LENGTH) {
		    		/* Check that we have one process for each row in the matrix */
				System.out.println("You have to use " + LENGTH + " processes");
      				MPI.Finalize();
      				System.exit(0);
			}

    			/* Initialize the matrix x with values 0 .. LENGTH*LENGTH-1 */
    			for (i=0; i<LENGTH; i++) {
      				for (j=0; j<LENGTH; j++) {
					x[(i*LENGTH)+j] = i*LENGTH+j;
      				}
    			}

    			/* Print out the matrix before it is distributed */
    			System.out.println("The matrix is");
    			for (i=0; i<LENGTH; i++) {
      				for (j=0; j<LENGTH; j++) {
					System.out.printf("%4d", x[(i * LENGTH) + j]);
      				}
      				System.out.printf("\n");
    			}

    			System.out.printf("\nProcess %d on host is scatteringing the lower triangular part of the matrix x to all %d processes\n",  me, np);

    			/* Scatter the lower triangular part of array x to all proceses, place it in y */
			MPI.COMM_WORLD.Scatterv(x, 0, sendcount, displs, MPI.INT, y, 0, LENGTH, MPI.INT, root);

    			/* Process zero prints out own portion of the scattered array */
    			for (i=0; i<sendcount[me]; i++) {
      				System.out.printf("%4d", y[i]);
    			}
			System.out.println("");

    			/* Receive messages with hostname and the scattered data */
    			/* from all other processes */
    			for (i=1; i<np; i++) {
				status  = MPI.COMM_WORLD.Recv(y, 0, LENGTH, MPI.INT, i, datatag);
				count = status.Get_count(MPI.INT);	/* Get nr of elements in message */
      				/* Print out the elements we received from this process */
      				for (j=0; j<count; j++) {
					System.out.printf("%4d", y[j]);
      				}
      				for (j=0; j<LENGTH-count; j++) {     /* Format output to look like a matrix */
					System.out.printf("    ");
      				}
      				System.out.printf("  from process %d\n", i);
			}
   	 		System.out.printf("Ready\n");

  		} else { /* all other processes do this */

    			/* Check sanity of the user */
    			if (np != LENGTH) {
      				MPI.Finalize();
      				System.exit(0);
    			}

    			/* Receive the scattered matrix from process 0, place it in array y */
			MPI.COMM_WORLD.Scatterv(x, 0, sendcount, displs, MPI.INT, y, 0, LENGTH, MPI.INT, root);
    			//MPI_Scatterv(&x, sendcount, displs, MPI_INT, &y, LENGTH, MPI_INT, root, MPI_COMM_WORLD);
	
    			/* Send own name back to process 0 */
    			//MPI_Send (&myname, NAMELEN, MPI_CHAR, 0, nametag, MPI_COMM_WORLD);
    			/* Send the received array back to process 0 */
			MPI.COMM_WORLD.Send(y, 0, sendcount[me], MPI.INT, 0, datatag);
    			//MPI_Send (&y, sendcount[me], MPI_INT, 0, datatag, MPI_COMM_WORLD);
  		}

  		MPI.Finalize();
	}
}
