/*
 * ReservationServer : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka and Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * ReservationServer.java
 *
 * A Reservation Server handle either 
 * a) requests from MPDs to reserve a number of peers
 * b) requests from other Reservation servers to book their local resources.
 *
 * @author Choopan Rattanapoka, creation jul 2007
 * 
 **/


package p2pmpi.rs;

import java.io.*;
import java.net.*;
import java.util.*;
import p2pmpi.common.*;
import p2pmpi.message.*;
import org.apache.log4j.*;

public class ReservationServer {
	protected static Logger logRS;
      protected String mpdVer;
	protected int mpdPort;
	protected int rsPort;
	protected int numProcPerJob;
	protected int maxJob;
	protected volatile Vector<JobInfo> jobList;
	protected String hostAllow;
	protected String[] hostDeny;
	protected String usingIP;

	public static final int RESERVATION_OK     = 0;
	public static final int RESERVATION_NOK    = 1;
	public static final int RESERVATION_FAIL   = 2;

	public ReservationServer(int mpdPort, int rsPort, int numProcPerJob, int maxJob,
			         String hostDenyList, String hostAllow, String mpdVer) {
		logRS = Logger.getLogger("MPD");
            this.mpdVer        = mpdVer;
		this.mpdPort       = mpdPort;
		this.rsPort        = rsPort;
		this.maxJob        = maxJob;
		this.numProcPerJob = numProcPerJob;
		this.hostAllow     = hostAllow;
		logRS.debug("[RS] Host deny list = " + hostDenyList);
		this.hostDeny      =  parseHostDenyList(hostDenyList);
		jobList = new Vector<JobInfo>();

            NetIface extIf = new NetIface();
		this.usingIP = extIf.getExternalIfIpv4( );
		if ( this.usingIP == null ) {
			  logRS.error("[RS] Error: could not determine a working network interface. Exiting.");
			  System.err.println("[RS] Error: could not determine a working network interface. Exiting.");
			  System.exit(1);
		}

	}

	/**
	 * return an array of strings from a string
	 *
	 * @param s string of coma separated IP or networks 
	 * @return an array of the valid IP or networks 
	 **/
	protected String [] parseHostDenyList( String s ) {

		if (s==null)
			return (null);
		else {
			String [] atoms = s.split(",");
			String [] ipArray = new String [atoms.length];
			int idx=0;
			URI tmp;
			logRS.info("[RS] Deny List specified in config. file:" + s);
			for (int i=0;i<atoms.length;i++) {
				try {
					String atom = atoms[i].trim();
					tmp = new URI(atom);  // in fact, this does not make a strict check of a valid IP
					ipArray[idx] = new String( atom );
					logRS.info("[RS] - Requests from "+ipArray[idx]+" will be ignored.");
					idx++;
				} catch (URISyntaxException e) {
					logRS.debug(" [RS] [Error] " + atoms[i] + " is not a valid URI in configuration file.");
				}
			}
			logRS.info("[RS] Deny List parsed: " + ipArray.length + " hosts or networks.");
			return (ipArray);
		}
	}

	protected boolean isAllowed(String hostVisitor, String [] hostDeny) {
		if (hostDeny==null)
			return( true );
		for(int i=0;i< hostDeny.length; i ++) {
			if (hostVisitor.startsWith(hostDeny[i])) {
				// Deny work request because host is in host_deny
				logRS.info(" [Denied] " + hostVisitor + " is not allowed (from configuration file).");
				return(false);
			}
		}
		return(true);
	}


	public void start() {
		ServerSocket servSocket = null;
		Socket socket = null;
		try {
			servSocket = new ServerSocket(rsPort);
		} catch(Exception e) {
			logRS.info("** [RS Error] " + e.toString());
			System.exit(1);
		}

		while(true) {
			try {
				socket = servSocket.accept();
				Reservation r = new Reservation( socket, this.mpdVer );
				r.start();
			} catch (Exception e) {
				System.exit(1);
			}
		}

	}

	public class Reservation extends Thread {

		Socket socket;
		String mpdVer;

		public Reservation(Socket socket, String mpdVer ) {
			this.socket = socket;
			this.mpdVer = mpdVer;
		}

		//return number of element deleted
		public synchronized void cleanupJobList() {
			//Check expire reserved job
			JobInfo jobInfo;
			int maxList = jobList.size();
			int[] removeElement = new int[maxList];
			int numRemoveElement = 0;
			for(int i = 0; i < maxList; i++) {
				jobInfo = jobList.elementAt(i);
				if((jobInfo.getStatus() == JobInfo.JOB_RESERVED) ||
						(jobInfo.getStatus() == JobInfo.JOB_STARTING)) {
					if(jobInfo.isOverReservePeriod()) {
						removeElement[numRemoveElement] = i;
						numRemoveElement++;
					}
				}
			}

			//remove expire reserved job
			for(int i = 0; i < numRemoveElement; i++) {
				jobList.remove(removeElement[i]);
			}
		}


            /**
             * Check if a job can be reserved on this host.
             * @param requestIP the IP of the requester
             * @param mpdVersion the MPD version of the requester
             **/
		public synchronized boolean canReserve(String requestIP, String mpdVersion) {
			// Check allow node
			if(!isAllowed(requestIP, hostDeny)) {
				return false;
			}
                  // genaud: for the moment make a strong requirement that mpdVer must correspond.
                  if ( !(this.mpdVer.equals( mpdVersion ) )) {
                         return( false );
                  }

			cleanupJobList();

			if(maxJob <= 0)
				return true;

			if(jobList.size() + 1 <= maxJob) 
				return true;
			else
				return false;
		}

		public synchronized void cancelReservation(String hashID) {
			int size = jobList.size();
			for(int i = 0; i < size; i++) {
				JobInfo jobInfo = jobList.elementAt(i);
				if(jobInfo.getHashID().equals(hashID)) {
					jobList.remove(i);
					break;
				}
			}
		}

		public synchronized void updateStatus(String hashID) {
			int size = jobList.size();
			for(int i = 0; i < size; i++) {
				JobInfo jobInfo = jobList.elementAt(i);
				if(jobInfo.getHashID().equals(hashID)) {
					jobInfo.setStatus(JobInfo.JOB_RUNNING);
					break;
				}
			}
		}

		public synchronized boolean checkAndUpdateProcessStatus(String hashID) {
			boolean ret = false;
			int size = jobList.size();
			for(int i = 0; i < size; i++) {
				JobInfo jobInfo = jobList.elementAt(i);
				if(jobInfo.getHashID().equals(hashID)) {
					jobInfo.setStatus(JobInfo.JOB_STARTING);
					ret = true;
					break;
				}
			}
			return ret;
		}

		public void run() {
			InputStream in = null;
			OutputStream out = null;
			ObjectInputStream ois = null;
			ObjectOutputStream oos = null;
			Object oMsg = null;
			RSMessage rsMsg = null;
			int size;
			RSMessage resultMsg;
			String hashID;

			try {
				in = socket.getInputStream();
				out = socket.getOutputStream();
				ois = new ObjectInputStream(in);
				oMsg = ois.readObject();

				// check type of message
				if(oMsg instanceof RequestQuitMessage) {
					System.exit(0);
				} else if (oMsg instanceof RequestJobInfoMessage) {
					cleanupJobList();
					ReplyJobInfoMessage replyMsg = new ReplyJobInfoMessage(jobList);
					oos = new ObjectOutputStream(out);
					oos.writeObject(replyMsg);
					oos.flush();
					oos.close();
					out.close();	

				} else if (oMsg instanceof RSMessage) {
					rsMsg = (RSMessage)oMsg;
					long reservePeriod;

					switch(rsMsg.getCommand()) {
						case RSMessage.MPD_REQUEST_RESERVATION_TO_RS :
							Vector<HostEntry> list = rsMsg.getList();
							hashID          = rsMsg.getHashID();
							reservePeriod   = rsMsg.getReservePeriod();
							size = list.size();

							Vector<HostEntry> reservedList = new Vector<HostEntry>();
							Vector<HostEntry> deadList     = new Vector<HostEntry>();

							//Request to reserve a node
							logRS.info("[RS] RSMessage have list to reserver = " + size);
							for(int i = 0; i < size; i++) {
								HostEntry hostEntry = list.elementAt(i);
								int result = reserve(hashID, hostEntry, reservePeriod);
								switch(result) {
									case RESERVATION_OK :
										reservedList.add(hostEntry);
										break;
									case RESERVATION_NOK :
										//do nothing
										break;
									case RESERVATION_FAIL:
										deadList.add(hostEntry);
										break;
								}
							}	

							//Send result to MPD
							logRS.info("[RS] Done reserved with = " + reservedList.size());
							resultMsg = new RSMessage();
							resultMsg.setList(reservedList);
							resultMsg.setDeadList(deadList);
							oos = new ObjectOutputStream(out);
							oos.writeObject(resultMsg);
							oos.flush();
							oos.close();
							out.close();	

							selfReserve(hashID, reservePeriod);
							break;

						case RSMessage.RS_REQUEST_RESERVATION_TO_RS :
							resultMsg = new RSMessage();
							hashID      = rsMsg.getHashID();
							String requestIP = rsMsg.getIp();
                                          		String version = rsMsg.getMpdVer();
							reservePeriod = rsMsg.getReservePeriod();
							if( canReserve( requestIP, mpdVer )) {
								//Reserve
								JobInfo jInfo = new JobInfo(hashID, JobInfo.JOB_RESERVED, reservePeriod);
								jobList.add(jInfo);
								resultMsg.setReservationReply(RESERVATION_OK);
							} else {
								resultMsg.setReservationReply(RESERVATION_NOK);
							}

							oos = new ObjectOutputStream(out);
							oos.writeObject(resultMsg);
							oos.flush();
							oos.close();
							out.close();	
							break;

						//Update reservation status to RUNNING
						case RSMessage.MPI_UPDATE_STATUS_TO_RS :
							hashID = rsMsg.getHashID();
							updateStatus(hashID);
							break;


						//Check if job is reserved and update status to STARTING
						case RSMessage.MPD_UPDATE_STATUS_TO_RS :
							resultMsg = new RSMessage();
							resultMsg.setReserved(checkAndUpdateProcessStatus(rsMsg.getHashID()));
							oos = new ObjectOutputStream(out);
							oos.writeObject(resultMsg);
							oos.flush();
							oos.close();
							out.close();
							break;

						case RSMessage.MPD_CANCEL_RESERVATION_TO_RS:
							hashID = rsMsg.getHashID();
							Vector<HostEntry> cancelList = rsMsg.getList();
							if(cancelList != null) {
								size = cancelList.size();
								for(int i = 0; i < size; i++) {
									cancelReservationToRS(hashID, cancelList.elementAt(i));
								}
							} 
							break;

						case RSMessage.RS_CANCEL_RESERVATION_TO_RS:
							cancelReservation(rsMsg.getHashID());
							break;
					}
				}
				ois.close();
				in.close();
				socket.close();

			} catch (Exception e) {
				logRS.info("[RS] [Error]: " + e.toString() + ", " + e.getCause());
				return;
			}
		}

		public void cancelReservationToRS(String hashID, HostEntry hostEntry) {
			RSMessage cancelMsg = new RSMessage(RSMessage.RS_CANCEL_RESERVATION_TO_RS, hashID, usingIP, this.mpdVer );
			try {
				Socket socket = new Socket(hostEntry.getIp(), hostEntry.getRsPort());
				OutputStream os = socket.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(os);
				oos.writeObject(cancelMsg);
				oos.flush();
			} catch(Exception e) {
			}
		}

		public void selfReserve(String hashID, long reservePeriod) {
			JobInfo jInfo = new JobInfo(hashID, JobInfo.JOB_RESERVED, reservePeriod);
			jobList.add(jInfo);
		}

		public int reserve(String hashID, HostEntry hostEntry, long reservePeriod) {
			RSMessage msgRequest = new RSMessage(RSMessage.RS_REQUEST_RESERVATION_TO_RS, hashID, usingIP,this.mpdVer);
			msgRequest.setReservePeriod(reservePeriod);
			RSMessage msgReply   = new RSMessage();
			try {
				Socket socket = new Socket(hostEntry.getIp(), hostEntry.getRsPort());
				socket.setSoTimeout(1000);
				InputStream is  = socket.getInputStream();
				OutputStream os = socket.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(os);
				oos.writeObject(msgRequest);
				oos.flush();

				ObjectInputStream ois = new ObjectInputStream(is);
				Object oMsg = ois.readObject();
				if(msgReply instanceof RSMessage) {
					msgReply = (RSMessage)oMsg;
					int result = msgReply.getReservationReply();
					oos.close();
					ois.close();
					os.close();
					is.close();
					socket.close();
					return result;
	
				} else {
					oos.close();
					ois.close();
					os.close();
					is.close();
					socket.close();
					
					return RESERVATION_FAIL;
				}
			} catch (Exception e) {
				return RESERVATION_FAIL;
			}
		}
	}

	public static void main(String[] args) {
		Logger logRS = Logger.getLogger("MPD");
		// version should be the same for MPD and RS this they are build at the same time and go to the same jar.
		String version = Version.MAINVERSION + "." + Version.SUBVERSION; 
		logRS.info("[RS] Starting reservation service (version " + version +"). Reading configuration ...");
		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( logRS );
		ReservationServer rsServer = new ReservationServer(
				ppConf.getMPDPort(),
				ppConf.getRSPort(),
				ppConf.getNumProcPerJob(),
				ppConf.getMaxJob(),
				ppConf.getHostDeny(),
				ppConf.getHostAllow(),
				version);
		rsServer.start();
	}
	}
