package p2pmpi.mpi;

import p2pmpi.message.*;
import p2pmpi.common.*;
import p2pmpi.mpi.dev.*;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * P2P-MPI class. 
 *
 *  @author Choopan Rattanapoka
 */
public class MPI {
	/** 
	 * Primitive datatype
	 */
	public static Datatype NULL, BYTE, CHAR, SHORT, BOOLEAN, INT, LONG, FLOAT, DOUBLE, STRING, OBJECT;

	/** 
	 * Primitive datatype
	 */
	public static Datatype BYTE2, CHAR2, SHORT2, INT2, LONG2, FLOAT2, DOUBLE2;

	/** 
	 * Primitive datatype (not available yet)
	 */
	public static Datatype PACKED;

	/** 
	 * Primitive datatype constant 
	 */
	public static int  ANY_SOURCE;
	
	/** 
	 * Primitive datatype constant 
	 */
	public static int ANY_TAG;

	/**
	 * Return value of comparing MPI Group 
	 */
	public static int UNEQUAL, SIMILAR, IDENT;

	/**
	 * Undefined Rank 
	 */
	public static int UNDEFINED;
	
	/** 
	 * Primitive collective-operation
	 */
	public static Op MAX, MIN, SUM, PROD, BAND, MAXLOC, MINLOC;

	/**
	 * MPI_COMM_WORLD
	 */
	public static IntraComm COMM_WORLD;

	//static int nameServicePort;
	//static int myMpiPort;
	//static int sleepTime = 100;


	// Problem with STATIC ??
	protected static RankTable rankTable; 
	protected static MapRankTable mapRankTable;
	protected static String myHash = null;
	protected static int fdPort, mpdPort, rsPort, ftPort, myRank;
	protected static Device myDevice = null;
	protected static Random mpiRandom;
	//private static StatusUpdater statusMan  = null;
	protected static URI cmdURI = null;
	protected static URI ctrlURI = null;
	protected static URI dataURI = null;

	private static String hostIP=null;

	private static int t_sent = 1000;

	private static int dataPort;
	private static int ctrlPort;
	/**
	 * find which ip this host has, and stores it.
	 */
	private static void setLocalhostIP () {
		// Get Local IP address

		//-- first see if a fixed address has been set
		P2PMPI_ConfigFile conf = new P2PMPI_ConfigFile( );
		String confIP = conf.getExternalIP();

		//-- otherwise, ask the system which is our IP
		if(confIP == null) {
			InetAddress localhost = null;
			try {
				localhost = InetAddress.getLocalHost();
			} catch (Exception e) {
				System.err.println("** [ Error] Could not figure out local hostname.");
				System.exit(-1);
			}
			hostIP = localhost.getHostAddress();
		}
		else hostIP = confIP;
	}
	/**
	 * returns IP set.
	 */
	protected static String getLocalhostIP () {
		return ( hostIP );
	}


	static {
		// Get property from System
		/*
		int myRankInMpi = Integer.parseInt(System.getProperty("MPIRANK"));
		int myRankInList= Integer.parseInt(System.getProperty("LISTRANK"));
		*/
		//nameServicePort = Integer.parseInt(System.getProperty("NSPORT"));
		//commSize	= Integer.parseInt(System.getProperty("NUMPEERS"));

		// Initialized Basic Datatype
		NULL	= new Datatype(Datatype.NULL);
		BYTE 	= new Datatype(Datatype.BYTE);
		CHAR 	= new Datatype(Datatype.CHAR);
		SHORT 	= new Datatype(Datatype.SHORT);
		BOOLEAN = new Datatype(Datatype.BOOLEAN);
		INT	= new Datatype(Datatype.INT);
		LONG	= new Datatype(Datatype.LONG);
		FLOAT	= new Datatype(Datatype.FLOAT);
		DOUBLE	= new Datatype(Datatype.DOUBLE);
		PACKED 	= new Datatype(Datatype.PACKED);
		STRING  = new Datatype(Datatype.STRING);
		OBJECT	= new Datatype(Datatype.OBJECT);
		//OBJECT	= new Datatype(Datatype.JOBJECT);
		BYTE2	= BYTE.Contiguous(2);
		CHAR2	= CHAR.Contiguous(2);
		SHORT2	= SHORT.Contiguous(2);
		INT2	= INT.Contiguous(2);
		LONG2	= LONG.Contiguous(2);
		FLOAT2	= FLOAT.Contiguous(2);
		DOUBLE2	= DOUBLE.Contiguous(2);
		ANY_SOURCE 	= Tag.MPI_ANYSOURCE;
		ANY_TAG 	= Tag.MPI_ANYTAG;
		UNDEFINED	= -3;
		UNEQUAL		= 0;
		SIMILAR 	= 1;
		IDENT		= 2;

		// Initialized Basic Operation
		MAX 	= new Op(Op.MAX);
		MIN	= new Op(Op.MIN);
		SUM 	= new Op(Op.SUM);
		PROD	= new Op(Op.PROD);
		BAND	= new Op(Op.BAND);
		MAXLOC	= new Op(Op.MAXLOC);
		MINLOC	= new Op(Op.MINLOC);
	}


	/**
	 * Initialization the MPI program
	 *
	 * @param args arguments
	 *
	 * @return list of arguments of program
	 */
	public static String[] Init(String[] args) {
		String[] result = null;
		String runCmd = null;
		Socket conn = null;
		OutputStream os = null;
		ObjectOutputStream oos = null;
		InputStream in = null;
		ObjectInputStream ois = null;
		MPIMessage mpiMsg = null;

		// Get MPD Port from config file
		P2PMPI_ConfigFile conf = new P2PMPI_ConfigFile( );
		mpdPort = conf.getMPDPort();
		ftPort  = conf.getFTPort();
		rsPort  = conf.getRSPort();
		fdPort  = conf.getFDPort();
		int tGossip = conf.getTGossip();
            int tHang   = conf.getTHang();
		String gossipProtocol = conf.getGossipProtocol(); 
 

		// Get Local IP address
		setLocalhostIP();

		// Search available port
		int minPort = conf.getMinPort();
		int maxPort = conf.getMaxPort();
		//int myCmdPort = 0;
		//int myDataPort = 0;
		int myPort = 0;	
		//
		Random r;


		//usage: java app numpeer replicaNum dependancefile "runCmd args args"
		String mode = System.getProperty("mode");
		ServerSocket reservePort  = null;

		if(mode.equals("server")) {

			int c;
			int numPeers = -1;
			int maxWaitTime = -1;
			int numReplica = 1;
			String allocationMode = null;
			String fileDep = null;
			boolean interactive = true;
                  String msgIntArg = "must be followed by a positive integer number";
			//Getopt for argument
			Getopt g = new Getopt("P2P-MPI", args, "w:n:r:l:a:X:D:");
			g.setOpterr(false);

			String javaXOption = "";
			String javaDOption = "";
			String arg;
			while((c = g.getopt()) != -1) {
				switch(c) {
					case 'w' :
						arg = g.getOptarg();
						try {
							maxWaitTime = Integer.parseInt(arg);
						} catch (Exception e) {
							System.err.println("** [Error] -w  " + msgIntArg);
							System.exit(1);
						}
						if(maxWaitTime < -1) {
							System.err.println("** [Error] -w  " + msgIntArg);
							System.exit(1);
						}
					break;
	
					case 'n' :
						arg = g.getOptarg();
						try {
							numPeers = Integer.parseInt(arg);
						} catch (Exception e) {
							System.err.println("** [Error] -n  " + msgIntArg);
							System.exit(1);
						}
						if(numPeers < 1) {
							System.err.println("** [Error] -n  " + msgIntArg);
							System.exit(1);
						}
					break;
					case 'r' :
						arg = g.getOptarg();
						try {
							numReplica = Integer.parseInt(arg);
						} catch (Exception e) {
							System.err.println("** [Error] -r  " + msgIntArg);
							System.exit(1);
						}
						if(numReplica < 1) {
							System.err.println("** [Error] -r  " + msgIntArg);
							System.exit(1);
						}
					break;
					case 'l' :
						arg = g.getOptarg();
						fileDep = new String(arg);
					break;
					case 'a' : 
						arg = g.getOptarg();
						allocationMode = new String(arg);
						if(!allocationMode.equalsIgnoreCase("spread")
                                    && !allocationMode.equalsIgnoreCase("concentrate")) {
							System.err.println("** [Error] -a must be followed by \"spread\" or \"concentrate\"");
							System.exit(1);
						}
						break;
					case 'X' :
						arg = g.getOptarg();
						javaXOption = "-X"+arg+" ";
					break;

					case 'D' :
						arg = g.getOptarg();
						javaDOption = "-D"+arg+" ";
					break;
				}
			}
			Vector<String> vresult =  new Vector<String>();
			for(int i = g.getOptind(); i < args.length; i++) {
				vresult.addElement(args[i]);
			}

			//Java execution
			runCmd = new String(javaXOption + javaDOption + vresult.elementAt(0));

			//Form an return string[] of init
			int numargs = vresult.size();
			result = new String[numargs-1];
			for(int i = 0; i < numargs-1; i++) {
				result[i] = new String(vresult.elementAt(i+1));
				runCmd = runCmd + " " + result[i];
			}

			int minPeers   = numReplica;
			int maxPeers   = (numPeers - 1) * numReplica;	

			r = new Random();
			int randomStart = r.nextInt(maxPort - minPort) + minPort; //avoid duplicate port

			//Searching available port
			for(int i = randomStart; i <= maxPort; i++) {
				try {
					reservePort = new ServerSocket(i);
					myPort = i;
					break;
				} catch (Exception e) {
					reservePort = null;
					continue;
				}
			}
			if(reservePort == null) {
				for(int i = minPort; i <= maxPort; i++) {
					try {
						reservePort = new ServerSocket(i);
						myPort = i;
						break;
					} catch (Exception e) {
						reservePort = null;
						continue;
					}
				}
				if(reservePort == null) {
					System.err.println("No available ports in the range allowed [" + minPort + " - " + maxPort + "] to run this application");
					System.exit(1);
				}
			}
			myHash = HashGenerator.gen();
			mpiRandom = new Random(Long.parseLong(myHash));

			try {
				cmdURI = new URI("tcp://"+getLocalhostIP()+":"+myPort);
			} catch (Exception e) {
				//e.printStackTrace();
			}

			// Init MPI Device .............
			String dev=System.getProperty("device");
			if ( dev == null ) {
				System.err.println("[Error] : no proprty \"device\" defined. Use java -Ddevice. Exiting.");
				System.exit(1);
			}
			if( dev.equals("niodevice") ) {
				myDevice = new NIODevice();
			} else if( dev.equals("normal")) {
				myDevice = new NormalDevice();
			} else {
				myDevice = new NIODevice();
			}
			if(!myDevice.init()) {
				System.err.println("[Error] : MPI NIO device init failed");
				System.exit(1);
			}
			
			// Allocation data port
			if((dataPort = myDevice.initPort(minPort, maxPort, getLocalhostIP())) == -1) {
				System.err.println("No available ports in the range allowed [" + minPort + " - " + maxPort + "] to run this application");
				System.exit(1);
			}
			try {
				dataURI = new URI("tcp://"+getLocalhostIP()+":"+dataPort);
			} catch (Exception e) {
				//e.printStackTrace();
			}

			// Allocation ctrl port
			if((ctrlPort = myDevice.initPort(minPort, maxPort, getLocalhostIP())) == -1) {
				System.err.println("No available ports in the range allowed [" + minPort + " - " + maxPort + "] to run this application");
				System.exit(1);
			}
			try {
				ctrlURI = new URI("tcp://"+getLocalhostIP()+":"+ctrlPort);
			} catch (Exception e) {
				//e.printStackTrace();
			}



			////////////////////////////////////////////////////////////////////////
			// Stand alone node
			////////////////////////////////////////////////////////////////////////
			if(numPeers == 1) {
				//if CommSize=1 no need to contact FT, FD (But contact to MPD for mpistat!)
				//start MessageHandler with RankTable of 1 element
				rankTable = new RankTable();
				rankTable.addProcess(0, cmdURI, ctrlURI, dataURI, fdPort); // add self for rank 0   TODO: fix null here
				rankTable.setNumProcess(1);
				rankTable.setNumReplica(1);
				mapRankTable = new MapRankTable();
				mapRankTable.addMap(0, 0, 0);
			
				// register to MPD
				AppRegisterMessage appMsg = new AppRegisterMessage();
				appMsg.setID(myHash+"--0");
				appMsg.setRunCmd(runCmd);
				appMsg.setIPRank0( getLocalhostIP() );
				appMsg.setRunDir("");
				appMsg.setRank(0);
				appMsg.setPort(myPort);
				appMsg.setMPISize(numPeers);
				appMsg.setRealSize(rankTable.size());
				appMsg.setRankTable(rankTable);
	
				try {
					conn = new Socket("127.0.0.1", mpdPort);
					os = conn.getOutputStream();
					oos = new ObjectOutputStream(os);
					oos.writeObject(appMsg);
					oos.flush();

					in = conn.getInputStream();
					ois = new ObjectInputStream(in);
					PingReplyMessage replyMsg = (PingReplyMessage)ois.readObject();
					ois.close();
					oos.close();
					os.close();
					in.close();
					conn.close();
				} catch (Exception e) {
					System.err.println("** [Error] Please make sure that MPD is running.");
					//e.printStackTrace();
					System.exit(1);
				}

	                 	myDevice.connect(0, 0, 0, rankTable, numPeers, numReplica);
				myDevice.spawnMessageHandler(reservePort, numPeers, rankTable, 
						             myHash, mpdPort, fdPort,
							     conf.getTGossip(),
							     t_sent * rankTable.size(),
							     conf.getTHang(),
							     conf.getGossipProtocol());

				// Start device thread to listen communication and message
				// To receive data messages
				Thread devThread = new Thread(myDevice);
				devThread.setDaemon(true);
				devThread.start();

				COMM_WORLD = new IntraComm(myDevice, rankTable, 0, 0, numPeers, mapRankTable);

				// return new arguments
				return result;
			}
			////////////////////////////////////////////////////////////////////////
			// End : Stand alone node
			////////////////////////////////////////////////////////////////////////

			// Default -w is 3 seconds per node
			if(maxWaitTime < 0) {
				maxWaitTime = maxPeers*3;
			}

			// Construct a request message to demand node
			MPDMessage msg = new MPDMessage(MessageCmd.MPI_REQPEER, myHash);

			//Calculate Min/Max Peer
			msg.setOptionN(numPeers - 1);
			msg.setOptionR(numReplica);
			msg.setMinPeers(minPeers);
			msg.setMaxPeers(maxPeers);
			msg.setWaitTime(maxWaitTime);
			msg.setIP( getLocalhostIP() , myPort);
			msg.setRunCmd(runCmd);
			//allocation policy
			msg.setAllocationMode(allocationMode);

			///////////////////////////////////////////
			// Send a request message to local MPD
			///////////////////////////////////////////
			System.out.println("Searching nodes for a maximum of "+ maxWaitTime + " seconds ...");
			Socket requestConn = null;	
			try {
				requestConn = new Socket("127.0.0.1", mpdPort);
				os = requestConn.getOutputStream();
				in = requestConn.getInputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(msg);
				oos.flush();
				ois = new ObjectInputStream(in);
				FoundNodeMessage tmp = (FoundNodeMessage)ois.readObject();
				boolean canrun = tmp.isEnough();
				int foundNode  = tmp.getNumFoundNode();
				int availableSlot = tmp.getNumSlotAvailable();
				ois.close();
				oos.close();
				in.close();
				os.close();
				requestConn.close();

				if(!canrun) {
					if(availableSlot == 0) {
						System.err.println("** [Error] Not enough hosts (found only " + foundNode + ").");
					} else {
						System.err.println("** [Error] Not enough hosts (found only " + foundNode + " nodes that can provide only " + availableSlot + " slots).");
					}
					System.err.println("**         You may try to search longer with -w option.");
					System.exit(1);
				}

				if(!allocationMode.equalsIgnoreCase("gather")) {
					if(foundNode < maxPeers+1 ) {
						System.out.println("** [Warning] Found only " + foundNode + " host(s).");
				       		System.out.println("**           Several processes will be executed on a same host.");
					} 
				}

			} catch (Exception e) {
				System.err.println("** [Error] Cannot connect to local MPD (127.0.0.1:"+mpdPort+").");
				System.err.println("**         Make sure you have run \"mpiboot\"");
				//e.printStackTrace();
				System.exit(1);
			}

			System.out.println("Waiting reply from other nodes..");
			// Listen to an answer from remote MPD
			///////////////////////////////////////////
			conn = null;
			MPDMessage replyMsg = null;
			int replyNode = 0;
			URI[] uriList = new URI[maxPeers];
			int[] keyList = new int[maxPeers];

				
			try {
				reservePort.setSoTimeout(120000); //millis (30 seconds with no responds)
			} catch (Exception e) {
				System.err.println("** [Error] Cannot set socket timeout.");
				System.exit(1);
			}
			while(true) {
				try {
					conn = reservePort.accept();	
					in = conn.getInputStream();
					ois = new ObjectInputStream(in);
					replyMsg = (MPDMessage)ois.readObject();
					ois.close();
					in.close();
					conn.close();
				} catch (SocketTimeoutException se) {
					System.err.println("** [Error] No reply from at least one of the hosts.");
					System.err.println("**         Check no firewall blocks port " + reservePort.getLocalPort());
					System.exit(1);
				} catch (IOException e) { 
					  //System.out.println("** [Error] a peer was advertized but could not be contacted on given socket (" 
					//			+ replyMsg.getHost() + ":" + replyMsg.getFTPort() + ") **"); 
					  continue;
				} catch (ClassNotFoundException ce) {
					 // System.err.println("** [Error] a peer was advertized but could not be contacted on given socket (" 
					//			+ replyMsg.getHost() + ":" + replyMsg.getFTPort() + ") **"); 
					  continue;
				} catch (Exception e) {
					continue;
				}
				switch(replyMsg.getCmd()) {
					// Reply is ACCEPT work request
					///////////////////////////////
					case MessageCmd.MPD_ACCEPT :
						System.out.println("* got ACCEPT (FT "+replyMsg.getHost() + ":" + replyMsg.getFTPort()+
										",key="+replyMsg.getRank()+")");
						//System.out.println(" FD Port : " + replyMsg.getFDPort()); //TODO:REMOVE FDPort
						try {
							uriList[replyNode] = new URI("tcp://" + replyMsg.getHost() + ":" + replyMsg.getFTPort());
						} catch (Exception e) {
							//e.printStackTrace();
						}
						keyList[replyNode] = replyMsg.getRank();
	
						replyNode++;
					break;

					default :
						System.out.println("Unknown command");
					break;
				}

				if(replyNode >= maxPeers) 
			 		break;
			} 

			// Check If each Rank has at least 1 replica
			/////////////////////////////////////////////
			//System.out.println("DONE");

			///////////////////////////////////////////
			// Register to a File Transfer Service
			///////////////////////////////////////////
			// Open dependancy file and add it in fileList variable
			
			//HERE: $pwd/exec.class 	
			Vector<String> vectorFileList = new Vector<String>();
			if(fileDep == null) {
				String execName = new String(vresult.elementAt(0));
				File fileExecName = new File(System.getProperty("user.dir"), execName); 
				execName = new String(fileExecName.toString() + ".class");
				//System.out.println("Sending " + execName);
				vectorFileList.addElement(execName);	
			} else {
				File fileTransferList = new File(fileDep);	
				try {
					BufferedReader fileReader = new BufferedReader(new FileReader(fileTransferList));
					while(true) {
						String fileName = fileReader.readLine();
						if(fileName == null) {
							// EOF
							break;
						}
						File testFileExist = new File(fileName);
						if(!testFileExist.exists()) {
							System.err.println("** [Error] File : " + fileName + " in " + fileDep +
									   " does not exist");
							System.exit(1);
						}
						vectorFileList.addElement(fileName);
					}
				} catch (Exception e) {
					System.err.println("** [Error] Could not read file " + fileDep);
				        System.exit(1);
				}
			}
	
			int numFile = vectorFileList.size();
			String[] fileList = new String[numFile];
			for(int i = 0; i < numFile; i++) {
				fileList[i] = new  String(vectorFileList.elementAt(i));
			}

			// Construct FTMessage
			FTMessage ftMsg = new FTMessage(MessageCmd.FT_REGISTER, myHash);	
			ftMsg.setKeys(keyList);
			ftMsg.setURIs(uriList);
			ftMsg.setFileNames(fileList);
			ftMsg.setSize(replyNode);
			System.out.print("* Transfering files ... [ ");
			String [] shortFileNames = ftMsg.getOnlyFileNameList();
			for (int i=0;  i<shortFileNames.length; i++)
				System.out.print(shortFileNames[i]+" ");
			System.out.println("]");
			try {
				requestConn = new Socket("127.0.0.1", ftPort);
				os = requestConn.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(ftMsg);
				oos.flush();
				oos.close();
				os.close();
				requestConn.close();
			} catch (Exception e) {
				//e.printStackTrace();
			}
	
			System.out.println("* Waiting for processes to acknowledge file transfer ...");
			// Synchronous all MPI processes
			rankTable = new RankTable();
			rankTable.addProcess(0, cmdURI, ctrlURI, dataURI, fdPort); // add self for rank 0
			//Add Map for COMM_WORLD
			mapRankTable = new MapRankTable();
			mapRankTable.addMap(0, 0, 0);

			// TO_data = n*5000
			int readyNode = 0;
			try {
				reservePort.setSoTimeout(replyNode*5000); 
			} catch (Exception e) {
				System.err.println("** [Error] Cannot set socket timeout.");
				System.exit(1);
			}
			boolean receiveAllPeer = false;
			while(true) {
				try {
					System.out.println("* Wait for SYN message ...");
					conn = reservePort.accept();
					in = conn.getInputStream();
					ois = new ObjectInputStream(in);
					Object oMsg  = ois.readObject();

					if(oMsg instanceof MPIMessage) {
						mpiMsg = (MPIMessage)oMsg;
						ois.close();
						in.close();
						conn.close();
					} else if (oMsg instanceof OutputMessage) {
						OutputMessage outMsg = (OutputMessage)oMsg;
						System.out.println(">>>>" + outMsg.getOutput());
						ois.close();
						in.close();
						conn.close();
						continue;
					} else {
						continue;
					}
					 
				} catch (SocketTimeoutException se) {
					System.err.println("** [Error] Not enough peers to form an execution platform."); 
					System.err.println("**         At least one of the requested hosts is behind a firewall.");
					System.exit(1);
				} catch (IOException e) {
					continue;
				} catch (ClassNotFoundException ce) {
					continue;
				} 
				// Construct a table
				if(mpiMsg.getCmd() == MessageCmd.MPI_SYN1) {
					System.out.println("SYN:" + mpiMsg.getRank() + ":" + mpiMsg.getURI());
					rankTable.addProcess(mpiMsg.getRank(), mpiMsg.getURI(), mpiMsg.getCtrlURI(), mpiMsg.getDataURI(), mpiMsg.getFDPort());
					readyNode++ ;
					mapRankTable.addMap(mpiMsg.getRank(), mpiMsg.getRank(), readyNode); //ADD MAP
					if(readyNode >= replyNode)  {
						receiveAllPeer = true;
						break;
					}
				}
			}

			rankTable.setNumProcess(numPeers);
			rankTable.setNumReplica(numReplica);
			System.out.println("* Check Receive all peers");
			//TODO : If server socket times out 
			// Check if there is at least 1 node for a rank
			if(!receiveAllPeer) {
                        boolean[] rankExist = new boolean[numPeers];
				int tableSize = rankTable.size();
				//init rankExist with false
				for(int i = 0; i < numPeers; i++) {
					rankExist[i] = false;
				}
				//put true on the rank that exist
				for(int i = 0; i < tableSize; i++) {
					rankExist[rankTable.getRank(i)] = true;
				}
				//find if some rank is false (not exist)
				for(int i = 0; i < numPeers; i++) {
					if(rankExist[i] == false) {
						System.err.println("*** Not enough peer to form an execution platform. ***");
						System.exit(1);
					}
				}
			}		

			int numNode = rankTable.size();
			///////////// List MPI rank and master ////////////////////
			boolean[] masterRank = new boolean[numPeers];
			for(int i = 0; i < numPeers; i++) {
				masterRank[i] = false;
			}

			System.out.println("============================================");

			for(int i = 0; i < numNode; i++) {
				int tmpRank = rankTable.getRank(i);
				if(!masterRank[tmpRank]) {
					System.out.println("[Master of Rank "+tmpRank+"]: IP = " + rankTable.getCmdHost(i));
					masterRank[tmpRank] = true;
				} else {
					System.out.println("Rank "+tmpRank+": IP = " + rankTable.getCmdHost(i));
				}
			}
			System.out.println("============================================");
			//listIP.close();
			//////////////////////////////////////////////////////////////
			
			// Send List and rank in list to all processes (t0 -> heartbeat adjustment)
			long startSendRankTableTime = System.currentTimeMillis();


			System.out.println("* Using protocol = " + gossipProtocol );
			for(int i = 1; i < numNode; i++) { // Skip process 0
				mpiMsg = new MPIMessage(MessageCmd.MPI_SYN2);
				mpiMsg.setCommTable(rankTable);
				mpiMsg.setRank(rankTable.getRank(i));
				mpiMsg.setRankInList(i);
				mpiMsg.setSize(numPeers);
				// FD time adjustable
				mpiMsg.setTGossip( tGossip );
				mpiMsg.setTHang( tHang );
				mpiMsg.setTMargin(t_sent * rankTable.size());
				mpiMsg.setGossipProtocol( gossipProtocol );
				// Set time different to adjust FD heartbeat
				// delta t_i for adjusting heartbeat
				mpiMsg.setTDiff((int)(System.currentTimeMillis() - startSendRankTableTime));

				try {
					System.out.println("* send SYN2: " + rankTable.getCmdHost(i) + ":" + rankTable.getCmdPort(i));
					conn = new Socket(rankTable.getCmdHost(i), rankTable.getCmdPort(i));
					os = conn.getOutputStream();
					oos = new ObjectOutputStream(os);
					oos.writeObject(mpiMsg);
					oos.flush();
					oos.close();
					os.close();
					conn.close();
				} catch (Exception e) {
					//e.printStackTrace();
				}
			}
			
                 	myDevice.connect(0, 0, 0, rankTable, numPeers,  numReplica);
			myDevice.spawnMessageHandler(reservePort, numPeers, rankTable, 
					             myHash, mpdPort, fdPort,
						     conf.getTGossip(),
						     t_sent * rankTable.size(),
						     conf.getTHang(),
						     conf.getGossipProtocol());

			// Start device thread to listen communication and message
			// To receive data messages
			Thread devThread = new Thread(myDevice);
			devThread.setDaemon(true);
			devThread.start();
			// Create COMM_WORLD
			COMM_WORLD = new IntraComm(myDevice, rankTable, 0, 0, numPeers, mapRankTable);

			//Register to MPD
			AppRegisterMessage appMsg = new AppRegisterMessage();
			appMsg.setID(myHash+"--0");
			appMsg.setRunCmd(runCmd);
			appMsg.setIPRank0( getLocalhostIP() );
			appMsg.setRunDir(System.getProperty("user.dir"));
			appMsg.setRank(0);
			appMsg.setPort(myPort);
			appMsg.setMPISize(numPeers);
			appMsg.setRealSize(rankTable.size());
			appMsg.setRankTable(rankTable);

			try {
				conn = new Socket("127.0.0.1", mpdPort);
				os = conn.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(appMsg);
				oos.flush();

				in = conn.getInputStream();
				ois = new ObjectInputStream(in);
				PingReplyMessage repMsg = (PingReplyMessage)ois.readObject();
				ois.close();
				oos.close();
				os.close();
				in.close();
				conn.close();
			} catch (Exception e) {
				System.err.println("** [Error] Please make sure that MPD is running.");
				//e.printStackTrace();
				System.exit(1);
			}

			// Register to fault detection service
			System.out.println("* Create an gossip instance.");
			GossipMessage gossipMsg = new GossipMessage(MessageCmd.GOSSIP_REGISTER, myHash);
			gossipMsg.setMyRank(0); // rank in list
			gossipMsg.setMPIRank(0); // MPI rank
			gossipMsg.setMPIPort(myPort);
			gossipMsg.setURIList(rankTable.getFDURIs());
			gossipMsg.setTGossip(conf.getTGossip());
			gossipMsg.setTHang(conf.getTHang());
			gossipMsg.setTMargin(t_sent * rankTable.size());
			gossipMsg.setGossipProtocol(conf.getGossipProtocol());
			gossipMsg.setTDiff(System.currentTimeMillis(),
					   (int)(System.currentTimeMillis() - startSendRankTableTime));

			try {
				conn = new Socket("127.0.0.1", fdPort);
				os = conn.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(gossipMsg);
				oos.flush();
				oos.close();
				os.close();
				conn.close();
			} catch (Exception e) {
				System.err.println("** [Error] Fault Detection Service is not running");
				//e.printStackTrace();
				System.exit(1);
			}

			System.out.println("* Program is executing ...");

		///////////////////////////// Client Mode ///////////////////////////////////
		} else {

			MPIClient MPIWorker = new MPIClient( minPort, maxPort);
			result = new String[args.length];
			for(int i = 0; i < args.length; i++) {
				result[i] = new String(args[i]);
			}
		}

		//Change Reservation Status to RUNNING
		RSMessage rsMsg = new RSMessage(RSMessage.MPI_UPDATE_STATUS_TO_RS,
				                 myHash);
		try {
			conn = new Socket("127.0.0.1", rsPort);
			os = conn.getOutputStream();
			oos = new ObjectOutputStream(os);
			oos.writeObject(rsMsg);
			oos.flush();
			oos.close();
			os.close();
			conn.close();
		} catch (Exception e) {
			System.err.println("** [Error] Reservation Service is not running");
			//e.printStackTrace();
			System.exit(1);
		}

		while(!myDevice.isDone()){
			try {
				Thread.sleep(1000);
			} catch(Exception e) {}
		}
		//System.out.println("MPI Init : Start barrier");
		COMM_WORLD.Barrier();


		return result;
	}


	/**
	 * Returns a pseudorandom, uniformly distributed int value from this random number generator's sequence
	 */
	public static int Rand()
	{
		return mpiRandom.nextInt();
	}

	/**
	 * Returns a pseudorandom, uniformly distributed int value between 0 (inclusive) 
	 * and the specified value (exclusive), drawn from this random number generator's sequence.
	 */
	public static int Rand(int n) 
	{
		return mpiRandom.nextInt(n);
	}

	/**
	 * Finalization the MPI program
	 */
	public static void Finalize() {
		Socket conn = null;
		OutputStream os = null;
		ObjectOutputStream oos = null;

		COMM_WORLD.Barrier();

		if(rankTable.size() == 1) {
			// there is only 1 node so unregister running program to MPD
			// and quit
			if(myDevice != null) {
				myDevice.shutdown();
			}

			// Unregister to MPD 
			try {
				//System.out.println("Finalize HashKey = " + myHash);
				AppUnregisterMessage unRegMsg = new AppUnregisterMessage(myHash, myRank);
				conn = new Socket("127.0.0.1", mpdPort);
				os = conn.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(unRegMsg);
				oos.flush();
				oos.close();
				os.close();
				conn.close();
			} catch (Exception e) {
				System.out.println("MPD Service is not running");
				//e.printStackTrace();
			}
	
			return;
		}
	
		if(myDevice.getRankInList() == 0) {
		// if rank = 0 wait until get all terminated message
			myDevice.meTerminated();
			while(!myDevice.isAllTerminated()) {
				try {
					Thread.sleep(500);
				} catch (Exception e) {}
			}

			// send FinalizedConfirmMessage to the rest of procs
			myDevice.finalizeConfirm();
		} else {
			// if rank != 0 wait for a confirm message from rank 0 to exit
			//send FinalizedMessage to rank 0
			myDevice.finalize(myDevice.getRankInList());

			//wait for finalizedConfirmMessage
			while(!myDevice.isTerminated()) {
				try {
					Thread.sleep(500);
				} catch (Exception e) {}
			}
		}	
		

		//Unregister FD Service
		try {
			//System.out.println("Finalize HashKey = " + myHash);
			GossipMessage gossipMsg = new GossipMessage(MessageCmd.GOSSIP_UNREGISTER, myHash);
			int rankInList = myDevice.getRankInList();
			gossipMsg.setMyRank(rankInList);

			conn = new Socket("127.0.0.1", fdPort);
			os = conn.getOutputStream();
			oos = new ObjectOutputStream(os);
			oos.writeObject(gossipMsg);
			oos.flush();
			oos.close();
			os.close();
			conn.close();
		} catch (Exception e) {
			System.out.println("Fault Detection Service is not running");
			//e.printStackTrace();
		}

		// Unregister to MPD 
		try {
			//System.out.println("Finalize HashKey = " + myHash);
			AppUnregisterMessage unRegMsg = new AppUnregisterMessage(myHash, myRank);
			conn = new Socket("127.0.0.1", mpdPort);
			os = conn.getOutputStream();
			oos = new ObjectOutputStream(os);
			oos.writeObject(unRegMsg);
			oos.flush();
			oos.close();
			os.close();
			conn.close();
		} catch (Exception e) {
			System.out.println("MPD Service is not running");
			//e.printStackTrace();
		}
		// Check if any data messages are left: issue a warning
		int nbmsg=myDevice.getUnexpectedQueueSize();
		if (nbmsg !=0) {
		   System.err.println("** [rank "+ myDevice.getRank() + "] Warning : there was "+nbmsg+" message(s) left in reception buffer. **");
		}

		// Shutdown MessageHandle 
		if(myDevice != null) {
			myDevice.shutdown();
		}
		//
		//System.exit(1);
	}

	/**
	 * Returns an elapsed time on the calling processor (seconds)
	 *
	 * @return elapsed wallclock time in seconds since some time in the past
	 */
	public static double Wtime() {
		return (System.currentTimeMillis() / 1000.0);
	}


	/**
	 * Returns the resolution of Wtime
	 *
	 * @return resolution of wtime in seconds
	 */
	public static double Wtick() {
		double resolution = 0.0001;
		double tick = 0.0;
		double t;

		if(tick == 0.0) {
			tick = System.currentTimeMillis();
			tick = System.currentTimeMillis() - tick;

			for(int counter = 0; counter < 10; counter++) {
				t = System.currentTimeMillis();
				t = System.currentTimeMillis() - t;
				if( t < tick) {
					tick = t;
				}
			}

			tick = (tick > 0.0) ? tick : 1.0e-5;
		}
		return tick/1000.0;
	}

	/**
	 * Returns a local hostname - this one is part of the MPJ API
	 *
	 * @return local hostname
	 */
	public static String Get_processor_name() {
		try {
			return (InetAddress.getLocalHost().getHostName());
		} catch (Exception e) {
			return null;
		}		
	}
}
