/**
 * @author: choopan
 * @author: genaud
 *
**/
package p2pmpi.tools;

import p2pmpi.common.*;
import p2pmpi.message.*;
import java.util.*;
import java.net.*;
import java.io.*;

public class HostCacheDisplay {
	static private String dateTranslate(long time) {
		long tmp = time;

		int day = (int)(time/(84600*1000));
		tmp -= (day * (84600*1000));
		int hour = (int)(tmp/(3600*1000));
		tmp -= (hour * (3600*1000));
		int min = (int)(tmp/(60*1000));
		tmp -= (min * (60*1000));
		int sec = (int)(tmp/1000);
		tmp -= (sec * 1000);
		int ms = (int)tmp;

		return day+"D " + hour + ":" + min + ":" + sec + ":" + ms;
	}
	/**
	 * Given a textual IP, return a DNS name if possible, otherwise return
	 * the input IP.
	 * @param ip the IP address as a string.
	 **/
	private static String resolve( String ip ) {
		  String hostname=ip;
		  try {
			    hostname = InetAddress.getByName( ip ).getHostName();
		  }
		  catch (UnknownHostException e) { /* nothinh, leave IP */ } 
		  return( hostname );
	}

	/**
	 * return a blank string of length l, such that s.length() + l = maxlen.
	 * @param s the reference string to compute l
	 * @param maxlen the number of characters to reach
	 * @param blank a utility string containing enough spaces
	 **/
	private static String getFiller( String s, int maxlen, String blank) { 
		return( blank.substring(0,maxlen-s.length()+1));
	}

	/**
	 * Display contents of hostCache.
	 * @param ip IP as a String of MPD queried.
  	 * @param hostCache list of known hosts
 	 * @param numeric indicates if the IP are to be displayed under a numeric format or hostnames resolved.
	 **/ 

	static public void display(String ip, Vector<HostEntry> hostCache, boolean numeric) {
		String filler=null;
		String blank = new String();
		String hostLbl = "Host"; // header for column 'host'
		int maxlen=0;  // will contain longest hostname 

		if(hostCache == null) {
			return;
		}
		int size = hostCache.size();
		String hostnames[] = new String [ size ]; 

     		//-- computes header for a pretty display 
		// this loop computes the longuest hostname 
		if (!numeric) {
			for(int i = 0; i < size; i++) {
				hostnames[ i ] = resolve( hostCache.elementAt(i).getIp() );
				maxlen = hostnames[ i ].length() > maxlen ? hostnames[ i ].length() : maxlen;
			}
			//prevent exception when no host in cache 
			if(size == 0) 
				maxlen = 10;

			for (int i=0;i<maxlen+4;i++)
				blank += " "; 	
			filler = getFiller( hostLbl, maxlen, blank );
		}
		else {
			filler="\t\t";
		}
		System.out.println("Hostcache entry of MPD : " + ip + "\n");
		System.out.println( hostLbl+ filler + "MPD Port\tRTT(ms)\tAlive\tLast update");
		for(int i = 0; i < size; i++) {
			HostEntry host = hostCache.elementAt(i);
			String RTT;
			if(host.getRtt() == Integer.MAX_VALUE) {
				RTT = "unknown";
			} else {
				RTT = ""+host.getRtt();
			}
			if (numeric) {
				System.out.println( host.getIp() + "\t" + host.getMpdPort() + "\t\t" + RTT +"\t" + host.isAlive() + "\t" + dateTranslate(host.getLastUpdate()));
			}
			else {
				filler = getFiller( hostnames[i] , maxlen, blank );
				System.out.println( hostnames[i] + filler + host.getMpdPort() + "\t\t" + RTT +"\t" + host.isAlive() + "\t" + dateTranslate(host.getLastUpdate()));
			}
				
		}
		System.out.println("\nTotal : " + size + " MPD known.");
	}


	/**
	  * Get information about known peers by a MPD (ip,alive,rtt) at a given host, stored as a vector of host entries.
	  * 
	  * @param  ip the IP address or the hostname as a string, of the host to contact.
	  * @return a vector of entries or stops if a network error occurs
	  **/
	static public Vector<HostEntry> get( String ip ) {

		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( );
		int mpdPort 	= ppConf.getMPDPort();
		Vector<HostEntry> hostCache = null;

		try {
			Socket sock = new Socket( ip, mpdPort);
			InputStream in   = sock.getInputStream();
			OutputStream out = sock.getOutputStream();
			RequestHostCacheMessage requestMsg = new RequestHostCacheMessage();

			ObjectOutputStream oos = new ObjectOutputStream(out);
			oos.writeObject(requestMsg);
			oos.flush();

			ObjectInputStream ois = new ObjectInputStream(in);
			ReplyHostCacheMessage replyMsg = (ReplyHostCacheMessage)ois.readObject();
			hostCache = replyMsg.getHostCache();
			ois.close();
			oos.close();
			out.close();
			in.close();
			sock.close();
		} catch(Exception e) {
			System.out.println("[Error] : " + e.toString());
			System.exit(1);
		}
		return( hostCache );
	}

	/**
	 * main is parsing comd line arguments and contacts MPD.
	 * Waits for a least one IP as input parameter (127.0.01) for localhost.
	 * Accepts extra arguments -n and -h.
       **/
	static public void main(String[] args) {
		boolean numeric=false;
		String remoteHost="127.0.0.1"; // default is local query
		Vector<HostEntry> hostCache = null;
		for (int i=0; i<args.length; i++) {
			if (args[i].equals("-h")) {
				System.out.println("usage: mpihost [-h] [-n] [hostname]");
				System.out.println("  -h : this help message");
				System.out.println("  -n : display numeric IP addresses (no DNS resolution)");
				System.out.println("  hostname : IP or hostname. Without this argument, query localhost.");
				System.exit(0);
			}
			else {
				if (args[i].equals("-n")) 
					numeric=true;
				else 
					remoteHost=args[ i ];
			}
		}
		hostCache = HostCacheDisplay.get( remoteHost );
		HostCacheDisplay.display( remoteHost, hostCache, numeric );
	}
}
