/*
 * SuperNode : part of P2P-MPI (http://www.p2pmpi.org).
 * Compute various scores attached to the mesh cells, based on geometric
   information that rays bring when the traverse the cell.
 *
 * Copyright (C) 2007, Choopan Rattanapoka 
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 SuperNode stores information about hosts running MPD.
 It is a simple replacement to the JXTA SRDI previously used.
 Information is a list of 'HostEntry':
 HostEntry(String ip, int mpdPort, int fdPort, int ftPort, int rsPort, int numProcPerJob)

 @author Choopan Rattanapoka
*/
/*
 * SuperNode : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka and Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SuperNode.java
 * created : july 20, 2007
 *
 * @author Choopan Rattanapoka
 *
 **/


package p2pmpi.p2p;

import java.io.*;
import java.net.*;
import java.util.*;
import p2pmpi.p2p.message.*;
import p2pmpi.common.*;

import org.apache.log4j.*;


public class SuperNode extends Thread {

	private static Logger log;
	int port;
	HashMap<String, HostEntry> hostCache;

	public SuperNode(int port) {
		 log = Logger.getLogger("SUPERNODE");
		this.port = port;
		hostCache = new HashMap<String, HostEntry>();
	}

	public void run() {
		ServerSocket serverSocket = null;
		try {
			serverSocket = new ServerSocket(port);
		} catch(Exception e) {
			System.err.println("port:" + port + " is not available");
			System.exit(1);
		}

		try {
			OutputStream out = null;
			ObjectOutputStream oos = null;
			HostEntry host;
			while(true) {
				Socket sock = serverSocket.accept();
				InputStream in = sock.getInputStream();
				ObjectInputStream ois = new ObjectInputStream(in);
				Object oMsg = ois.readObject();

				if(oMsg instanceof RegisterMessage) { 
					host = ((RegisterMessage)oMsg).getHostEntry();
					log.debug("[SuperNode] register from " + host.toString());
					hostCache.put(host.toString(), host);
				} else if(oMsg instanceof UpdateMessage) {
					host = ((UpdateMessage)oMsg).getHostEntry();
					log.debug("[SuperNode] update from "+host.getIp());
					HostEntry entry = hostCache.get(host.toString());
						//System.out.println("update from" + host.toString());
					if(entry == null) {
						//never register but update ??
						// yes, in case SuperNode has been stopped and re-started
						// Consider update as a register query.
						log.debug("[SuperNode] converting update from " + host.getIp()+ " to register.");
						hostCache.put(host.toString(), host);
					} else {
						entry.setLastUpdate();
					}

				} else if(oMsg instanceof PeerRequestMessage) {
					Set set = hostCache.keySet();
					Iterator iter = set.iterator();
					PeerReplyMessage replyMsg = new PeerReplyMessage();
					while(iter.hasNext()) {
						host = hostCache.get(iter.next());
						if(host.isUpdate()) {
							replyMsg.add(host);
						}
					}
					//log.debug("[SuperNode] peer list request (|cache|="+hostCache.size()+")");
					out = sock.getOutputStream();
					oos = new ObjectOutputStream(out);
					oos.writeObject(replyMsg);
					oos.flush();
					oos.close();
					out.close();		
				} else if(oMsg instanceof UnregisterMessage) {
					host = ((UnregisterMessage)oMsg).getHostEntry();
					log.debug("[SuperNode] unregister from "+host.getIp());
					hostCache.remove(host.toString());

				} else if(oMsg instanceof QuitMessage) {
					System.exit(0);
				}

				ois.close();
				in.close();
				sock.close();
			}
		} catch(Exception e) {
			System.err.println("[Error] : " + e.toString());
			e.printStackTrace(); 
			System.exit(1);
		}


	}

	public static void main(String[] args) {
		//read config file for internal RDV port
		File configFile = new File(System.getProperty("P2PMPI_HOME"), "P2P-RDV.conf");
		GenericConfigFile conf = new GenericConfigFile(configFile.toString());
		int internalPort = Integer.parseInt(conf.getProperty("InternalPort"));

		SuperNode superNode = new SuperNode(internalPort);
		superNode.start();
	}

}
