/*
 * StatQuery : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka and Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/**
 * StatQuery.java
 *
 * StatQuery is created by a Visu object, and runs a thread that
 * queries known MPD (maintained in the member 'peerList').
 * Visu should create a StatQuery with an initial bootstrap list obtained from SuperNode. 
 * Upon receipt of a StatQueryMessage, a MPD sends back an StatInfoMessage with most of
 * its characteristics inside. 
 *
 * @author Ghazi Bouabene, creation jul 2005
 * @author Stephane Genaud, revision jul 24 14:15:12 CEST 2007
 * 
 **/

package p2pmpi.visu;

import p2pmpi.common.*;
import p2pmpi.tools.SupernodeHostCacheDisplay;
import p2pmpi.mpd.PeerCache;
import p2pmpi.message.StatQueryMessage;
import p2pmpi.message.StatInfoMessage;
import p2pmpi.p2p.message.RegisterMessage;
import p2pmpi.p2p.message.UnregisterMessage;

import java.io.*;
import java.net.*;
import java.util.*;

import com.touchgraph.graphlayout.Edge;
import com.touchgraph.graphlayout.GLPanel;
import com.touchgraph.graphlayout.Node;
import com.touchgraph.graphlayout.TGException;
import com.touchgraph.graphlayout.TGLensSet;
import com.touchgraph.graphlayout.TGPanel;
import java.awt.Color;
import java.awt.Image;

import org.apache.log4j.*;



public class StatQuery extends Thread {

	private static Logger log;
	Visu_Param parent;
	private URI supernode;
	private URI visuproxy;
      private Vector<HostEntry> peerList=null; // the list of host to scan (or get info from cache) 
	private PeerCache peerDiskCache=null; 
      private String   peerCachePath = null;
	private String   externalIp = null;
	private int listenPort;
      protected boolean connectedToProxy; // Could we register to proxy ?

	
	/**
	 * Constructor
	 *
	 **/	
	public StatQuery( URI supernode, URI visuproxy, String peerCachePath, String externalIp, int listenPort, Visu_Param parent ) {
		log = Logger.getLogger("VISU");
		this.parent = parent;
		this.peerCachePath = peerCachePath;
		this.peerDiskCache = new PeerCache( peerCachePath );
		this.supernode = supernode;
		this.visuproxy = visuproxy;
		this.externalIp = externalIp;
		this.listenPort = listenPort;
		if ( (listenPort == -1) || (visuproxy==null) ) {
			// the ServerSocket of listener thread could not be bound, or
                  // no VisuProxy found in conf. file 
			// => no use connecting to proxy
			this.connectedToProxy = false;
			log.debug( "@ StatQuery :: StatQuery() -> listenPort=-1 or visuproxy==null"); 
			System.out.println("Cannot connect to a VisuProxy. I will query SuperNode directly.");
		}
	 	else {	
			// now determine if we are able to use a Visu Proxy
			this.connectedToProxy = register( );
			log.debug( "@ StatQuery :: StatQuery() -> connectedToProxy == "+ connectedToProxy);
		}
	}


	/**
       * Is Query thread coonected to a Visu Proxy ? 
       **/
	 public boolean isConnectedToVisuProxy( ) {
			return( this.connectedToProxy );
	 }

	/**
	 * Update peer list contents by contacting supernode.
	 * @return the number of peers found, or -1 if SuperNode could not be contacted.
	 **/
	public int updatePeerList( ) {

		if ( !connectedToProxy ) {
		  log.debug("[visu client] contacting Supernode " 
					+ supernode.getHost() + ":" + supernode.getPort());
		  SupernodeHostCacheDisplay remoteSN = new SupernodeHostCacheDisplay();
		  this.peerList = remoteSN.query( supernode.getHost(), supernode.getPort() );

		  if ( this.peerList == null ) {
			    log.error("[visu client] could not contact Supernode "+supernode.toString()+" while not registered to a VisuProxy.");
			    return( -1 );
		  }
		  log.info("[visu client] Supernode returns "+this.peerList.size()+" connected hosts.");
		  return(this.peerList.size()); 
		}
		else return( 0 ); /* do nothing, wait to be notified by proxy */
	}

	/**
	 * Update the peerList from a given list.
	 **/	
	public void setPeerList( Vector<HostEntry> list ) {
		this.peerList = list;
	}


	/**
	 * Save peerDiskCache currently in memory (as a DOM document) to cache file.
	 **/	
	public void writeCache( ) { 
		log.debug("@ StatQuery::writeCache(\""+ this.peerCachePath+"\")");
		this.peerDiskCache.saveDocument( this.peerCachePath );
	}


	/**
	 * Guess which is my own IP, also considering configuration file. 
	 **/	
	public String getMyIp ( ) {
		String ip  = null ;
	      if ( this.externalIp == null) {
			  try {
				    ip = (InetAddress.getLocalHost()).getHostAddress();
			  } catch (Exception e) { 
				    log.error("Could not determine my own IP address ! Exiting.");
				    System.exit(1); 
			  }
			  return( ip );
		}
		else 
			  return ( externalIp );
	}

	/**
	 * Registers to the Proxy Visu.
	 * @return true if socket connected, false in case of network failure. 
	 **/	
	public boolean register( ) {
		    boolean registered = false;
		    HostEntry me = new HostEntry( getMyIp(), 0, 0, 0, 0, 0); // dummy fd,ft,rs, ... port numbers
		    RegisterMessage regMsg = new RegisterMessage( me , this.listenPort );

		    try {
		    		Socket s = new Socket( visuproxy.getHost(), visuproxy.getPort() );
				OutputStream out = s.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject( regMsg );
				oos.flush();
				oos.close();
				out.close();
				s.close();
		    		System.out.println("Registering to VisuProxy: " + visuproxy.toString());
				registered= true;
		    } catch (Exception e) {
				log.error("Could not connect to Visu Proxy "+ visuproxy.toString());
		    }
		    return( registered );
	}
	/**
	 * Unregister to the Proxy Visu.
	 * @return true if socket connected, false in case of network failure. 
	 **/	
	public boolean unregister( ) {
		
		    boolean registered = false;
		    // exit immediately if we were not registered ...
		    if ( ! connectedToProxy ) {
				return ( true );
		    }
		    // or really unregister.
		    HostEntry me = new HostEntry( getMyIp(), 0, 0, 0, 0, 0); // dummy fd,ft,rs, ... port numbers
		    UnregisterMessage unregMsg = new UnregisterMessage( me );

		    try {
		    		Socket s = new Socket( visuproxy.getHost(), visuproxy.getPort() );
				OutputStream out = s.getOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(out);
				oos.writeObject( unregMsg );
				oos.flush();
				oos.close();
				out.close();
				s.close();
				log.debug("Un-registering to visu proxy "+ visuproxy.toString());
				registered= true;
		    } catch (Exception e) {
				log.error("@ could not connect to visu proxy "+ visuproxy.toString() +" : "+e.getCause());
		    }
		    return( registered );
	}



	/**
	 * whois. Try to infer a sensible host name. 
	 * Do not make a real whois query (to expensive)
	 * @param queryIp IP as a String representation of host address
	 * @return a String array: if a reverse DNS query provides a fqdn, String[0] is 
                 host short name and String[1] is the domain name.
 		     If reverse DNS does not result in a canonical form, then String[0] is 
		     the IP address, and String[1] is "unregistered".
	 **/	
	private String [] whois(String queryIp) { 

		String [] names = new String[2]; // String[0] -> hostname String[1] -> domainName
		names[0] = queryIp;
		try {
			  // Get hostname by textual representation of IP address
			  InetAddress addr = InetAddress.getByName( queryIp );
			  String fqdn = addr.getCanonicalHostName();
			  if (fqdn.matches("\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}")) {
				    // returned ip, did not return an informative fqdn
				    names[1]  = "unregistered";
			  }
			  else {
				    int indx = fqdn.indexOf(".");
				    if (indx > 0 ) {
						names[0] = fqdn.substring( 0, indx );
						names[1] = fqdn.substring( indx + 1 );
				    }
				    else {
						names[0] = fqdn;
						names[1] = "localdomain";
				    }
			  }
		}
		catch (UnknownHostException e) { names[1] = "Unresolved"; }
		return( names );
	}
	/**
	 * Connect to a MPD to query its characteristics and return a StatInfoMessage.
	 * @param ip IP as a String representation of host address
	 * @param port port to connect to
	 * @return a StatInfoMessage containing the information, or null if an exception occured when connecting 
	 **/	
	private StatInfoMessage fillStatInfoMessageFromMpdQuery( String ip, int port) {

		  StatInfoMessage m = null;	

		  OutputStream out = null;
		  InputStream in = null;
		  ObjectOutputStream oos = null;
		  Socket s = null;
		  ObjectInputStream ois = null;
		  try {
			    StatQueryMessage queryMsg = new StatQueryMessage();
			    log.debug("@ StatQuery::fillStatInfoMessageFromMpdQuery("+ip +","+ port+")"); 
			    s = new Socket( );
			    SocketAddress sa  = new InetSocketAddress( ip , port);
			    s.connect( sa, 5000 );  // 5s timeout
			    in = s.getInputStream();
			    out = s.getOutputStream();
			    oos = new ObjectOutputStream(out);
			    oos.writeObject(queryMsg);
			    oos.flush();
			    ois = new ObjectInputStream(in);
			    m = (StatInfoMessage)ois.readObject();
			    ois.close();
			    oos.close();
			    in.close();
			    out.close();
			    s.close();

		  } catch (Exception e) { };
		  return( m );
	}
	/**
	 * Look into the peer cache to find a peer characteristics and return a StatInfoMessage.
	 * @param ip first part of id, which is the key to access peers in the cache
	 * @param port second part of id, which is the key to access peers in the cache
	 * @return a StatInfoMessage containing the information, or null if an exception occured when connecting 
	 **/	
	private StatInfoMessage fillStatInfoMessageFromPeerCache( String ip, int port) {

		  StatInfoMessage m = null;	
		  String id = new String( ip + ":" + port );
		  m = new StatInfoMessage (	
					ip, 
					port, 
					this.peerDiskCache.getMpdVer ( id ),
					this.peerDiskCache.getOsName ( id ),
					this.peerDiskCache.getArch   ( id ),
					this.peerDiskCache.getCpuName( id ),
					this.peerDiskCache.getCpuFreq( id ),
					this.peerDiskCache.getMem    ( id ));
		  return( m );

	}
	/**
	 * Add a StatInfoMessage object in the DOM document representing the cache.
	 * @param m the StatInfoMessage object to add. 
	 * @return true if all fields could be inserted, false otherwise.
	 **/	
	private boolean addCache( StatInfoMessage m ) {
		  boolean ok = true;
		  log.debug("@ StatQuery::addCache() --> starts.");
		  String id = new String( m.getIp() + ":" + m.getMpdPort() );
		  ok &= this.peerDiskCache.setMpdVer ( id, m.getMpdVer() );
		  ok &= this.peerDiskCache.setOsName ( id, m.getOsName () );
		  ok &= this.peerDiskCache.setArch   ( id, m.getArch ()   );
		  ok &= this.peerDiskCache.setCpuName( id, m.getCpuName ());
		  ok &= this.peerDiskCache.setCpuFreq( id, m.getCpuFreq() );
		  ok &= this.peerDiskCache.setMem    ( id, m.getMem ()    );
		  return( ok );
	}

	/**
	  * Thread code ....
	  **/ 	

	public void run(){
		 this.currentThread().setName("StatQuery thrd");
		query( true ); // query( useCache )
	}

	/**
	 * Query Peers characteristics and update 1) infoMsg in parent object 2) PeersNode in graphic object. 
	 * Query takes each host in this.peerList and display its characteristics.
	 * Query can get host information either from PeerCache (local disk cache) or through a TCP connection
       * to remote MPDs.
	 * @precond peerList != null
	 * @param useCache when true, first look in the cache. When false do not use cache.  
	 * @return true if a query was issued, false if no query was possible (empty peerList) 
	 **/
	public boolean query( boolean useCache ) {
		  StatInfoMessage infoMsg = null;
		  if ( this.peerList == null ) {
		  	log.debug("@ StatQuery::query() --> query starts and return immediately (peerList==null)"); 
			return( false );
		  }
		  log.debug("@ StatQuery::query() --> query starts useCache="+useCache); 
		  Iterator iter = this.peerList.iterator(); 
		  while ( iter.hasNext() ) {

			    Object entry  = iter.next();
			    String hostIP = ((HostEntry) entry).getIp();
			    int  hostPort = ((HostEntry) entry).getMpdPort(); 
		  	    log.debug("@ StatQuery::query() --> iter element "+hostIP); 

			    if ( useCache && peerDiskCache.hasId ( hostIP + ":" + hostPort  ) ) {
					log.debug("@ StatQuery::query() -> peerDiskCache.hasId( "+hostIP + ":" + hostPort+") == true"); 
					infoMsg = fillStatInfoMessageFromPeerCache( hostIP, hostPort );
			    }			
			    else {
					peerDiskCache.remove(  hostIP + ":" + hostPort );
					log.debug("@ StatQuery::query() --> peerDiskCache.hasId( "+hostIP + ":" + hostPort+") == false"); 
					infoMsg = fillStatInfoMessageFromMpdQuery( hostIP, hostPort );
			    		if ( infoMsg != null ) {
						addCache( infoMsg );
					}
			    }
			    if ( infoMsg != null ) {
					String [] fqdn = whois( hostIP ); // [0] -> short hostname, [1]-> domainname
					//String hostnameDisplayed = fqdn[0];
					//if (useCache)
					//	hostnameDisplayed += " *";   // to make evident cache is used for that host
					log.debug("@ StatQuery::query() --> whois(+" + hostIP + ")="+ 
						fqdn[0] + "." + fqdn[1] + "(os="+ infoMsg.getOsName()+")"); 

					//look if the Node is already known
					Boolean createNode = false;
					PeerNode p;
					if (( p = parent.jc2.findNode( hostIP )) == null)
						  createNode = true;
					log.debug("@ StatQuery::query() -->  node "+hostIP+ " creation == "+createNode);
					if ( createNode ) {
						  // create a new node
						  p = new PeerNode( hostIP, fqdn[0], parent);
					}
					// In all cases (new or existing node), set/refresh properties 
					//if (useCache)
						  p.setProperties( fqdn[0], infoMsg.getIp(), fqdn[1],  
									infoMsg.getCpuName(), infoMsg.getCpuFreq(), infoMsg.getMem(), 
									infoMsg.getOsName(), infoMsg.getArch()  );
					p.setStatus( "Available" );

					if ( createNode ) {
						  //we add this to parent.Peers vector
						  parent.peers.addElement( p );
						  //and to the JTable's data Vector
						  //((MyTableModel)parent.v.jdispTable.getModel()).addPeer(p);

						  PeerNode anchor=parent.jc2.findNode( fqdn[1] );
						  //here we add a node to the Graph
						  if ( anchor==null ){
							    try{
									anchor= new PeerNode( fqdn[1], fqdn[1] );
									anchor.setBackColor(Color.WHITE);
									anchor.setType(Node.TYPE_ELLIPSE);
									anchor.setTextColor(Color.BLACK);
									parent.jc2.addNode(anchor); 
							    } catch(com.touchgraph.graphlayout.TGException e){}
						  }
						  p.setType(Node.TYPE_ROUNDRECT);
						  p.setHint("IP: "+hostIP+"\nStatus: Available\nOS: ");
						  p.setBackColor(Color.decode("#4080A0"));
						  try {
							    parent.jc2.addNode(p);
							    Edge e = parent.jc2.addEdge(p, anchor, 2 * Edge.DEFAULT_LENGTH);
						  } catch ( com.touchgraph.graphlayout.TGException ei ){}
					}

					//-------- here we treat the task part ---------------
					int numProcesses=infoMsg.size();

					//on copie le infoMsg dans le visu parent
					log.debug("@ StatQuery::run() --> numProcesses=" + numProcesses); 
					boolean rankZero=false;
					for (int i=0; i < numProcesses; i++ ) {
						  ProcessInfo procInfo = infoMsg.getInfo(i);
						  if ( procInfo.getRank()==0 ) { //getRank semble toujours renvoyer 0
							    rankZero = true; 
						  }		
					}
					if (rankZero)
						  parent.infoMsgs.addElement(infoMsg);	
					parent.fillPeers();
			    } 
			    else { // infoMsg was null.
					log.debug("@ StatQuery::query() --> could not contact "+hostIP + ":" + hostPort
							    + " . Remove from peerList."); 
					iter.remove();
			    }
			    if(parent.fen!=null) {
					log.debug("@ StartQuery::query() --> graphic refresh() for "+ parent.fen); 
					parent.fen.refresh();
			    }
			    if (parent.jdispTable != null) {
					parent.refreshTables();
			    }
		  }
		  log.debug("@ StatQuery::query() --> query ends."); 
		  return( true );
	}
}
