package p2pmpi.mpi;

import p2pmpi.message.*;
import p2pmpi.common.*;
import p2pmpi.mpi.internal.*;
import java.net.*;
import java.io.*;
import java.util.*;


class MPIClient extends MPI {

	private int clientPort			= 0;
	private ServerSocket reservedSocket	= null ;

	/**
	 * getMyPort() : access method 
	 **/
	private int getMyPort () {
		return ( clientPort );
	}
	/**
	 * getReservedSocket() : access method 
	 **/
	private ServerSocket getReservedSocket() {
		return ( reservedSocket );
	}

	/**
	 * setPorts() : intializes a valid port and socket for client to communicate.
	 *
	 * @param minPort min value in allowaed port range to communicate in application 
	 * @param maxPort max value in allowaed port range to communicate in application
	 * @param seed a number to intializes the random choice of ports in the range 
	 **/
	private void setPorts ( int minPort, int maxPort, long seed) {
		mpiRandom = new Random( seed );
		Random r = new Random();
		int randomStart = r.nextInt(maxPort - minPort) + minPort; //avoid duplicate port

		for(int i = randomStart; i <= maxPort; i++) {
			try {
				reservedSocket = new ServerSocket(i);
				clientPort = i;
				break;
			} catch (Exception e) {
				reservedSocket = null;
				continue;
			}
		}
		if (reservedSocket == null) {
			for(int i = minPort; i <= maxPort; i++) {
				try {
					reservedSocket = new ServerSocket(i);
					clientPort = i;
					break;
				} catch (Exception e) {
					reservedSocket = null;
					continue;
				}
			}
			if (reservedSocket == null) {
				System.err.println("** [Error] No allowed ports are available for running this application");
				System.exit(1);
			}
		}
	}

	/**
	 * MPIClient constructor: does the whole work.
	 *
	 * @param minPort min value in allowaed port range to communicate in application 
	 * @param maxPort max value in allowaed port range to communicate in application
	 **/
	public MPIClient (int minPort, int maxPort) { 
		String rank0_host 	= System.getProperty("ip");
		int rank0_port		= Integer.parseInt(System.getProperty("port"));
		myRank			= Integer.parseInt(System.getProperty("rank"));
		myHash			= System.getProperty("hashkey");
		Socket sock			= null;
		ServerSocket reservePort = null;
		int myPort			= 0;
	
		// Client Mode
		
		// Allocation a port for MPI application. Initializezs clientPort and serverSocket.
		setPorts ( minPort, maxPort, Long.parseLong(myHash)+myRank);

		myPort = getMyPort();
		reservePort = getReservedSocket();

		msgHandle = new MessageHandler(reservePort, myHash, mpdPort, fdPort);
		// Change to daemon thread
		Thread msgHandleThread = new Thread(msgHandle);
		msgHandleThread.setDaemon(true);
		msgHandleThread.start();
		//new Thread(msgHandle).start();		

		// Send Synchronous message to rank 0 
		// tell rank0 how can connect to it
		//System.out.println("Sending SYN1 to Rank0");
		MPIMessage mpiMsg = null;
		mpiMsg = new MPIMessage(MessageCmd.MPI_SYN1);
		mpiMsg.setRank(myRank);
		mpiMsg.setFDPort(fdPort);
		mpiMsg.setURI( getLocalhostIP() , myPort);

		boolean synSuccess = false;
		OutputStream os = null;
		InputStream in = null;
		ObjectInputStream ois = null;
		ObjectOutputStream oos = null;

		for(int i = 0; i < 10; i++) {	
			try {
				sock = new Socket(rank0_host, rank0_port);
				os = sock.getOutputStream();
				oos = new ObjectOutputStream(os);
				oos.writeObject(mpiMsg);
				oos.flush();
				oos.close();
				os.close();
				synSuccess = true;
				break;
			} catch (Exception e) {
				//e.printStackTrace();
				try {
					Thread.sleep(1000);
				} catch (Exception ie) {}
			}
		}
		if(!synSuccess) {
			//Rank 0 quits before
			System.exit(1);
		}
		

		//System.out.println("Wait for RankTable");
		// BLOCK for List and its rank in List from Rank 0
		while(!msgHandle.isReady()) {
			try {
				Thread.sleep(500);
			} catch (Exception e) {}
		}

		long startDiffTime = System.currentTimeMillis();
		
		rankTable = msgHandle.getRankTable();
		int rankInList = msgHandle.getRankInList();
		mapRankTable = new MapRankTable();
		for(int i = 0; i < rankTable.size(); i++) {
			mapRankTable.addMap(rankTable.getRank(i), i);
		}

		// Create comm world
		COMM_WORLD = new IntraComm(	msgHandle,
							rankTable,
							msgHandle.getRank(),
							rankInList,
							msgHandle.getCommSize(),
							mapRankTable
				 	 );


		// register to MPD
		AppRegisterMessage appMsg = new AppRegisterMessage();
		appMsg.setID(myHash+"--"+myRank);
		appMsg.setRunCmd(System.getProperty("runCmd"));
		appMsg.setIPRank0(rank0_host);
		appMsg.setRunDir(System.getProperty("user.dir")); //Get PWD from System?
		appMsg.setRank(myRank);
		appMsg.setPort(myPort);
		appMsg.setMPISize(msgHandle.getCommSize());
		appMsg.setRealSize(rankTable.size());
		appMsg.setRankTable(rankTable);
	

		try {
			Socket conn = new Socket("127.0.0.1", mpdPort);
			os = conn.getOutputStream();
			oos = new ObjectOutputStream(os);
			oos.writeObject(appMsg);
			oos.flush();

                        in = conn.getInputStream();
			ois = new ObjectInputStream(in);
			PingReplyMessage replyMsg = (PingReplyMessage)ois.readObject();
			ois.close();
			oos.close();
			in.close();
			os.close();
			conn.close();
		} catch (Exception e) {
			System.err.println("** [Error] Make sure MPD is running");
			//e.printStackTrace();
			System.exit(-1);
		}

		// Register to fault detection service
		//System.out.println("Create an gossip instance !!");
		GossipMessage gossipMsg = new GossipMessage(MessageCmd.GOSSIP_REGISTER, myHash);
		gossipMsg.setMPIRank(myRank);
		gossipMsg.setMyRank(rankInList); // rank in list
		gossipMsg.setMPIPort(myPort);
		gossipMsg.setURIList(rankTable.getFDURIs());
		gossipMsg.setTGossip(msgHandle.getTGossip());
		gossipMsg.setTMargin(msgHandle.getTMargin());
		gossipMsg.setTHang(msgHandle.getTHang());
		gossipMsg.setGossipProtocol(msgHandle.getGossipProtocol());
		gossipMsg.setTDiff(System.currentTimeMillis(), (int)(System.currentTimeMillis() - startDiffTime + msgHandle.getTDiff()));

		try {
			sock = new Socket("127.0.0.1", fdPort);
			os = sock.getOutputStream();
			oos = new ObjectOutputStream(os);
			oos.writeObject(gossipMsg);
			oos.flush();
			oos.close();
			os.close();
			sock.close();
		} catch (Exception e) {
			System.out.println("Fault Detection Service is not running");
			//e.printStackTrace();
			System.exit(1);
		}
		//System.out.println("done!!!");
	}
}
