/* polyhedron.c
     COPYRIGHT
          Both this software and its documentation are

              Copyright 1993 by IRISA /Universite de Rennes I - France,
              Copyright 1995,1996 by BYU, Provo, Utah
                         all rights reserved.

          Permission is granted to copy, use, and distribute
          for any commercial or noncommercial purpose under the terms
          of the GNU General Public license, version 2, June 1991
          (see file : LICENSING).
*/

/*

1997/12/02 - Olivier Albiez
  Ce fichier contient les fonctions de la polylib de l'IRISA,
  passees en 64bits.
  La structure de la polylib a donc ete modifie pour permettre 
  le passage aux Value. La fonction Chernikova a ete reecrite.

*/

/*

1998/26/02 - Vincent Loechner
  Ajout de nombreuses fonctions, a la fin de ce fichier,
  pour les polyedres parametres 64 bits.
1998/16/03
  #define DEBUG  printf
  tests out of memory
  compatibilite avec la version de doran

*/

#undef POLY_DEBUG			/* debug printf: general functions */
#undef POLY_RR_DEBUG		/* debug printf: Remove Redundants */
#undef POLY_CH_DEBUG		/* debug printf: Chernikova */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "polylib.h"


#ifdef MAC_OS
  #define abs __abs
#endif

/*  errormsg1 is an external function which is usually supplied by the
    calling program (e.g. Domlib.c, ReadAlpha, etc...).
    See errormsg.c for an example of such a function.  */
void errormsg1(char *f , char *msgname, char *msg);

int Pol_status;                    /* error status after operations */


#define WSIZE (8*sizeof(int))


/*
 * Fonction utilitaires.
 *
 */


#define bexchange(a, b, t, l)\
{\
  memcpy((char *)(t), (char *)(a), (int)(l));\
  memcpy((char *)(a), (char *)(b), (int)(l));\
  memcpy((char *)(b), (char *)(t), (int)(l));\
}

#define exchange(a, b, t)\
{ (t)=(a); (a)=(b); (b)=(t); }



static void VVPd (p, length)
    Value *p;
    unsigned length;
{
  unsigned i;

  for (i=0; i< length; i++)
  {
    fprintf (stderr, " " VALUE_FMT " ", p[i]);
  }
  fprintf (stderr, "\n");
}

static void IVPd (p, length)
    int *p;
    unsigned length;
{
  unsigned i;

  for (i=0; i< length; i++)
  {
    fprintf (stderr, " %i ", p[i]);
  }
  fprintf (stderr, "\n");
}




/*
 * SatMatrix
 *
 */

/* The saturation matrix is an integer (int type) matrix, not
 * a Value matrix like the other ones.
 */

typedef struct
{
  unsigned int NbRows;
  unsigned int NbColumns;
  int **p;
  int *p_init;
} SatMatrix, *pSatMatrix; 


static pSatMatrix
SMAlloc (rows, cols)
    int rows, cols;
{
  int **q, *p, i;

  pSatMatrix result;

  result = (pSatMatrix) malloc (sizeof (SatMatrix));
  if( !result )
  {  errormsg1("SMAlloc", "outofmem", "out of memory space");
     return 0;
  }

  result->NbRows = rows;
  result->NbColumns = cols;
	if( rows==0 || cols==0 )
	{	result->p = NULL;
   	return result;
	}
  result->p = q = (int **)malloc (rows * sizeof(int *));
  if( !result->p )
  {  errormsg1("SMAlloc", "outofmem", "out of memory space");
     return 0;
  }
  result->p_init = p = (int *)malloc (rows * cols * sizeof (int));
  if( !result->p_init )
  {  errormsg1("SMAlloc", "outofmem", "out of memory space");
     return 0;
  }

  for (i=0; i < rows; i++)
  {
    *q++ = p;
    p += cols;
  }

  return result;
} /* SMAlloc */


static void 
SMFree (matrix)
    pSatMatrix matrix;
{
  free ((char *) matrix->p_init);
  free ((char *) matrix->p);
  free ((char *) matrix);
} /* SMFree */



#define SMVector_Copy(p1, p2, length) \
  memcpy((char *)(p2), (char *)(p1), (int)((length)*sizeof(int)))

#define SMVector_Init(p1, length) \
  memset((char *)(p1), 0, (int)((length)*sizeof(int)))


/* SatMatrix Print (debug) */
static void SMPd (Mat)
pSatMatrix Mat;
{
	int *p;
	int i, j;
	unsigned NbRows, NbColumns;

	fprintf(stderr,"%d %d\n", NbRows=Mat->NbRows, NbColumns=Mat->NbColumns);
	for (i=0;i<NbRows;i++) 
	{
		p=*(Mat->p+i);
		for (j=0;j<NbColumns;j++)
			fprintf(stderr, " %10X ", *p++);

		fprintf(stderr, "\n");
	}  
} /* SMPd */

/* int vector OR */
static void SatVector_Or(p1, p2, p3, length)
int *p1, *p2, *p3;
unsigned length;
{
	int *cp1, *cp2, *cp3;
	int i;

	cp1=p1;
	cp2=p2;
	cp3=p3;
	for (i=0;i<length;i++)
	{
		*cp3 = *cp1 | *cp2;
		cp3++;
		cp1++;
		cp2++;
	}
} /* SatVector_Or */




/*
 * Matrix
 *
 */

/* Matrix Print (debug) */
static void MPd (Mat)
Matrix *Mat;
{
  Matrix_Print (stderr, 0, Mat);
} /* MPd */


/*
 * Polyhedron
 *
 */

/* Combine --
     (p3) = linear combination of (p1) and (p2)
     such that (p3[pos]) is zero.
     First element of each vector is status word and is not
     changed in (p3).  (pos) may be 0 however.
     (length) does not include status word.1
*/
static void Combine(p1, p2, p3, pos, temp, length)
      Value *p1, *p2, *p3;
     int  pos;
     Value *temp;
     unsigned length;
{ 
  Value a, a1, a2;

  /* a1 = p1[pos] */
  value_assign (a1, p1[pos]); 

  /* a2 = p2[pos] */
  value_assign (a2, p2[pos]); 

  /* a  = Gcd (abs(a1), abs(a2)) */
  value_assign (a, Gcd (value_abs(a1), value_abs(a2)));

  /* a1 = a1/a */
  value_division (a1, a);

  /* a2 = a2/a */
  value_division (a2, a);

  Vector_Combine(p1+1, p2+1, p3+1, a2, value_uminus(a1), length);
  Vector_Normalize(p3+1, temp, length);
} /* Combine */




/* Transpose Sat --> Sat2 */
static pSatMatrix TransformSat(Mat, Ray, Sat)
Matrix *Mat, *Ray;
pSatMatrix Sat;
{ 
  int i, j, nc2;
  unsigned jx1, jx2, bx1, bx2;
  pSatMatrix Sat2;

  if (Mat->NbRows != 0) 
    nc2 = (Mat->NbRows-1) / (sizeof(int)*8) + 1;
  else                  
    nc2 = 0;

  Sat2 = SMAlloc(Ray->NbRows, nc2);
  SMVector_Init(Sat2->p_init, Ray->NbRows*nc2);

  for (i=0, jx1=0, bx1=MSB; i<Ray->NbRows; i++)
  { 
    for(j=0, jx2=0, bx2=MSB; j<Mat->NbRows; j++)
    { 
      if (Sat->p[j][jx1] & bx1) 
        Sat2->p[i][jx2] |= bx2;

      NEXT(jx2,bx2);
    }

    NEXT(jx1, bx1);
  }

  return Sat2;
}




/* RaySort -- Sorts the rays (Ray, Sat) into three tiers:
         NbBid       <= i < equal_bound : saturates the constraint
         equal_bound <= i < sup_bound   : verifies the constraint
         sup_bound   <= i < NbRay       : does not verify
   the tier order is chosen so non-verifying rays can be easily removed 
*/
static void RaySort (Ray, Sat, NbBid, NbRay, equal_bound, sup_bound,
                     RowSize1, RowSize2, Temp1, Temp2, bx, jx)
Matrix *Ray;
pSatMatrix Sat;
int NbBid, NbRay, *equal_bound, *sup_bound;
unsigned RowSize1, RowSize2;
Value *Temp1;
int *Temp2;
unsigned bx, jx;
{ 
  int inf_bound;
  Value **uni_eq, **uni_sup, **uni_inf;
  int **inc_eq, **inc_sup, **inc_inf;

      /* sort rays : NbBid <= i < equal_bound : saturates the constraint
         : equal_bound <= i < sup_bound : verifies the constraint
         : sup_bound <= i < NbRay : does not verify */
      /* the order is chosen so non-verifying rays can be easily removed */

      *sup_bound = *equal_bound = NbBid;
      uni_sup = uni_eq = Ray->p+NbBid;
      inc_sup = inc_eq = Sat->p+NbBid;
      inf_bound = NbRay;
      uni_inf = Ray->p+NbRay;
      inc_inf = Sat->p+NbRay;

      while (inf_bound>*sup_bound)
      { if (value_zero_p(**uni_sup))       /* status = satisfy */
        { bexchange(*uni_eq, *uni_sup, Temp1, RowSize1);
          bexchange(*inc_eq, *inc_sup, Temp2, RowSize2);
          (*equal_bound)++; uni_eq++; inc_eq++;
          (*sup_bound)++; uni_sup++; inc_sup++;
        }
        else
        { *((*inc_sup)+jx)|=bx;
          /* if (**uni_sup<0) */
          if (value_neg_p (**uni_sup))
          {    /* status = verify */
            inf_bound--; uni_inf--; inc_inf--;
            bexchange(*uni_inf, *uni_sup, Temp1, RowSize1);
            bexchange(*inc_inf, *inc_sup, Temp2, RowSize2);
          }
          else                  /* status = does not verify */
          { (*sup_bound)++; uni_sup++; inc_sup++;
          }
        }
      }
}



/* Chernikova -- computes the dual of (Mat) and places it in (Ray).
   (Mat) contains the equations (one per row).
   (Ray) contains the lineality space and ray space,
   (NbBid) is the number of lines in (Ray).
   Processes constraints from (Mat) starting at (FirstConstraint)
   and modifies (Ray) to reflect the constraints,
   creates (Sat) a boolean matrix dimensioned by NbRay X NbCon
       defined as Sat(i,j)==0 if Ray(i) saturates Eqn(j), 1 otherwise
   Returns 0 if successful.
*/
static int 
Chernikova (Mat, Ray, Sat, NbBid, NbMaxRays, FirstConstraint)
     Matrix *Mat, *Ray;
     pSatMatrix Sat;
     unsigned NbBid, NbMaxRays, FirstConstraint;
{
  unsigned NbRay, Dimension, NbConstraints, RowSize1, RowSize2, nc;
  int sup_bound, equal_bound, index_non_zero, bound;
  int i, j, k, l, redundant, rayonly, nbcommonconstraints;
  int *Temp2, aux;
  int *ip1, *ip2;
  unsigned bx, m, jx;
  Value *p1, *p2, *p3;
  Value *Temp1;

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: Input]\nRay = ");
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

  NbConstraints=Mat->NbRows;
  NbRay = Ray->NbRows;
  Dimension = Mat->NbColumns-1;         /* homogeneous dimension */
  nc=Sat->NbColumns;

  RowSize1=(Dimension+1) * sizeof(Value);
  Temp1=(Value *)malloc(RowSize1);
	if( !Temp1 )
	{	errormsg1("Chernikova", "outofmem", "out of memory space");
		return 0;
	}
  RowSize2=nc * sizeof(int);
  Temp2=(int *)malloc(RowSize2);
	if( !Temp2 )
	{	errormsg1("Chernikova", "outofmem", "out of memory space");
		return 0;
	}

  jx =   FirstConstraint/WSIZE;
  bx = MSB; bx >>= FirstConstraint%WSIZE;
  for (k=FirstConstraint; k<NbConstraints; k++)
  { 
    /* Set the status word of each ray[i] to ray[i] dot constraint[k] */
    /* This is equivalent to evaluating each ray by the constraint[k] */
    index_non_zero = NbRay;
    for (i=0; i<NbRay; i++)
    { 
      p1 = Ray->p[i]+1;
      p2 = Mat->p[k]+1;
      p3 = Ray->p[i];

      /* *p3 = *p1 * *p2 */
      value_assign(*p3, value_mult(*p1, *p2));
      p1++; p2++;

      for (j=1; j<Dimension; j++) 
      {
         /* *p3 +=  *p1 * *p2 */
	 value_addto(*p3, value_mult (*p1, *p2));
	 p1++; p2++;
      }

      if (value_notzero_p(*p3) && (i<index_non_zero)) 
        index_non_zero=i;
    }

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: A]\nRay = ");
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

    /* finds a bidirectional ray z such that cz <> 0 */
    if (index_non_zero<NbBid)
    { /* discards index_non_zero bidirectional ray */
      NbBid--;
      if (NbBid!=index_non_zero)
        bexchange(Ray->p[index_non_zero], Ray->p[NbBid], Temp1, RowSize1);

      /* Computes the new lineality space */
      for (i=0; i<NbBid; i++)
        if (value_notzero_p(Ray->p[i][0]))
          Combine(Ray->p[i], Ray->p[NbBid], Ray->p[i], 0, Temp1, Dimension);

      /* add the positive part of index_non_zero bidirectional ray to
         the set of unidirectional rays */
      if (value_neg_p(Ray->p[NbBid][0])) {
        p1=Ray->p[NbBid]; 
        for (j=0; j<Dimension+1; j++) 
	 { 
           /* *p1 = - *p1 */
	   value_assign(*p1, value_uminus(*p1));
	   p1++; 
	 }
      }

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: B]\nRay = ");
  Ray->NbRows=NbRay;
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

      /* Computes the new pointed cone */
      for (i=NbBid+1; i<NbRay; i++)
        if (value_notzero_p(Ray->p[i][0]))
          Combine(Ray->p[i], Ray->p[NbBid], Ray->p[i], 0, Temp1, Dimension);

      /* add the new ray */
      if (value_notzero_p(Mat->p[k][0])) /* Constraint is an inequality */
      { 
        for (j=0; j<nc; j++)
        {
          Sat->p[NbBid][j] = 0; /* sat vec for new ray */
        }

        /* new ray saturates everything except last inequality */
        Sat->p[NbBid][jx] |= bx;
      }
      else /* Constraint is an equality */
      { NbRay--;
        Vector_Copy(Ray->p[NbRay], Ray->p[NbBid], Dimension+1);
        SMVector_Copy(Sat->p[NbRay], Sat->p[NbBid], nc);
      }
#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: C]\nRay = ");
  Ray->NbRows=NbRay;
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

    } else {
      RaySort (Ray, Sat, NbBid, NbRay, &equal_bound, &sup_bound,
        RowSize1, RowSize2, Temp1, Temp2, bx, jx);
        /* sorts Uni into R0, R+, R- */
        /*        Ray 
NbRay-> bound-> ________
                |  R-  |        R- ==> ray.eq < 0  (outside domain)
          sup-> |------|
                |  R+  |        R+ ==> ray.eq > 0  (inside domain)
        equal-> |------|
                |  R0  |        R0 ==> ray.eq = 0  (on face of domain)
        NbBid-> |______|
        */

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: D]\nRay = ");
  Ray->NbRows=NbRay;
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

      /* Computes only the new pointed cone */
      bound=NbRay;
      for (i=equal_bound; i<sup_bound; i++)     /* for all pairs of */
        for(j=sup_bound; j<bound; j++) {        /* R- and R+ */
          /*-----------------------------------------------------------------*/
          /* compute and count the set of constraints saturated by R+ and R- */
          /* includes equalities, inequalities and the positivity constraint */
          /*-----------------------------------------------------------------*/
	  nbcommonconstraints = 0;
          for (l=0; l<jx; l++) 
	   {
	     aux = Temp2[l] = Sat->p[i][l] | Sat->p[j][l];
	     for (m=MSB; m!=0; m>>=1) if (!(aux&m)) nbcommonconstraints++;
          }

	  aux = Temp2[jx] =  Sat->p[i][jx] | Sat->p[j][jx];
	  for (m=MSB; m!=bx; m>>=1) if (!(aux&m)) nbcommonconstraints++;

          if (value_zero_p(Ray->p[i][Dimension]) && 
              value_zero_p(Ray->p[j][Dimension]))
            nbcommonconstraints++;      /* account for pos constr */

          /*-----------------------------------------------------------------*/
          /* Adjacency Test : is combination [R-,R+] a non redundant ray?    */
          /*-----------------------------------------------------------------*/
          if (nbcommonconstraints+NbBid>=Dimension-2) /* dimensionality check*/
          { /* check whether a ray m saturates the same set of constraints */
            redundant=0;
            rayonly = (value_zero_p(Ray->p[i][Dimension]) && 
                       value_zero_p(Ray->p[j][Dimension]));

            for (m=NbBid; m<bound; m++) if ((m!=i)&&(m!=j))
            { /* 2 rays (r+ r-) are never made redundant by a vertex   */
              /* because the positivity constraint saturates both rays */
              /* but not the vertex                                    */
              if (rayonly && value_notzero_p(Ray->p[m][Dimension])) continue;

              /* (r+ r-) is redundant if there does NOT exist an equation */
              /* which saturates both r+ and r- but not rm */
              ip1 = Temp2;
              ip2 = Sat->p[m];
              for (l=0; l<=jx; l++,ip2++,ip1++)
		if (*ip2 & ~*ip1) break;

              if (l>jx)
              { 
                redundant=1;
                break;
              }
            }

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: E]\nRay = ");
  Ray->NbRows=NbRay;
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif


            /*------------------------------------------------------------*/
            /* Add new ray generated by [R+,R-]                           */
            /*------------------------------------------------------------*/
            if (!redundant)
            { if (NbRay==NbMaxRays)
              {
						errormsg1("chernikova", "outofmem", "out of table space");
                  return 1;
              }
              /* computes the new ray */
              Combine(Ray->p[j], Ray->p[i], Ray->p[NbRay],0,Temp1, Dimension);
              SatVector_Or(Sat->p[j], 
                        Sat->p[i], 
                        Sat->p[NbRay], 
                        nc);

              Sat->p[NbRay][jx] &= ~bx;
              NbRay++;
           }
          }
        }
#ifdef POLY_CH_DEBUG
  fprintf(stderr, 
    "[Chernikova: F]\n"
    "sup_bound=%d\n"
    "equal_bound=%d\n"
    "bound=%d\n"
    "NbRay=%d\n"
    "Dimension = %d\n"
    "Ray = ",sup_bound,equal_bound,bound,NbRay, Dimension);
#endif
#ifdef POLY_CH_DEBUG
  Ray->NbRows=NbRay;
  fprintf(stderr, "[Chernikova: F]:\nRay = ");
  Matrix_Print(stderr, 0,Ray);
#endif


      /* Eliminates all non extremal rays */
      /* j = (Mat->p[k][0]) ? */
      j = (value_notzero_p(Mat->p[k][0])) ? 
        sup_bound : equal_bound;

      i = NbRay;
#ifdef POLY_CH_DEBUG
fprintf(stderr, "i = %d\nj = %d \n", i, j);
#endif
      while ((j<bound)&&(i>bound)) {
        i--;
        Vector_Copy(Ray->p[i], Ray->p[j], Dimension+1);
        SMVector_Copy(Sat->p[i], Sat->p[j], nc);
        j++;
      }

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "i = %d\nj = %d \n", i, j);
  fprintf(stderr, 
    "[Chernikova: F]\n"
    "sup_bound=%d\n"
    "equal_bound=%d\n"
    "bound=%d\n"
    "NbRay=%d\n"
    "Dimension = %d\n"
    "Ray = ",sup_bound,equal_bound,bound,NbRay, Dimension);
#endif
#ifdef POLY_CH_DEBUG
  Ray->NbRows=NbRay;
  fprintf(stderr, "[Chernikova: G]\nRay = "); 
  Matrix_Print(stderr, 0,Ray);
#endif

      if (j==bound) NbRay=i;
      else NbRay=j;
    }
    NEXT(jx, bx);
  }
  free(Temp1);
  free(Temp2);

  Ray->NbRows=NbRay;
  Sat->NbRows=NbRay;

#ifdef POLY_CH_DEBUG
  fprintf(stderr, "[Chernikova: Output]\nRay = ");
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

  return 0;
} /* Chernikova */




/*******************************************************************/
/* Gauss --
   Computes a minimal system of equations.
   performs Gaussian elimination, detects contridiction.
   Given Mat, a matrix of constraints in which the first NbEq
      constraints are equations,
   Given NbEq, the number of equations in Mat
   Given Dimension, the dimension of the homonogenous system,
   Produce, col_of_rank, the vector of independent variables,
   Uses temp, a temporary vector,
   Returns the rank, if Rank=Dimension ... is there a contradiction
 */
int Gauss(Mat, NbEq, Dimension, temp)
Matrix *Mat;
int     NbEq, Dimension;
Value  *temp;
{
  int RowSize = Dimension * sizeof(Value);
  int *col_of_rank;
  int i, j, k, pivot, Rank;
  Value gcd, *cp;

  col_of_rank=(int *)malloc(RowSize);
	if( !col_of_rank )
	{	errormsg1("Gauss", "outofmem", "out of memory space");
		return 0;
	}
  Rank=0;

#ifdef POLY_DEBUG
  fprintf(stderr, "[Gauss : Input]\nRay =");
  Matrix_Print(stderr, 0,Mat);
#endif

  for (j=1; j<=Dimension; j++) /* for each column (except status) */ 
  { 
    for (i=Rank; i<NbEq; i++)  /* starting at diagonal, look down */

      /* if (Mat->p[i][j] != 0) */
      if (value_notzero_p (Mat->p[i][j])) 
        break;                      /* to find the first non zero */

    if (i!=NbEq)                               /* was one found ? */
    { 
      if (i!=Rank)            /* was it found below the diagonal? */
          bexchange(Mat->p[Rank]+1, Mat->p[i]+1, temp, RowSize);

      /* Normalize the pivot row */
      
      /* gcd = Vector_Gcd(Mat->p[Rank]+1, temp, Dimension) */
      value_assign(gcd, Vector_Gcd(Mat->p[Rank]+1, temp,Dimension)); 

      /* if (gcd >= 2) */
      if (value_ge (gcd, VALUE_CONST(2)))
      { 
        cp = &Mat->p[Rank][1];
        for (k=0; k<Dimension; k++)
        {
          /* *cp /= gcd */
          value_division (*cp, gcd);
          
          cp++;
        }
      }

      /* if (Mat->p[Rank][j] < 0) */
      if (value_neg_p (Mat->p[Rank][j]))
      { 
        cp = Mat->p[Rank]+1;

        for (k=0; k<Dimension; k++)
        { 
          /* *cp *= -1 */
          value_multiply (*cp, VALUE_MONE);
          
          cp++;
        }
      }
      /* end normalize */

      pivot=i;
      for (i=pivot+1; i<NbEq; i++) /* zero out the rest of the column */
      { 
        /* if (Mat->p[i][j] != 0) */
        if (value_notzero_p(Mat->p[i][j]))
          Combine(Mat->p[i], 
                  Mat->p[Rank], 
                  Mat->p[i], 
                  j, temp, Dimension);
      }
      col_of_rank[Rank]=j;
      Rank++;
    }
  } /* end of Gauss elimination */


  /* Back Substitution -- normalizes the system of equations */
  for (k=Rank-1; k>=0; k--)
  { 
    j = col_of_rank[k];

    /* Normalize the equations */
    for (i=0; i<k; i++)
    { 
      /* if (Mat->p[i][j] != 0) */
      if (value_notzero_p(Mat->p[i][j]))
        Combine(Mat->p[i], Mat->p[k], Mat->p[i], j, temp, Dimension);
    }
    /* Normalize the inequalities */
    for (i=NbEq; i<Mat->NbRows; i++)
    { 
      /* if (Mat->p[i][j] != 0) */
      if (value_notzero_p(Mat->p[i][j]))
        Combine(Mat->p[i], Mat->p[k], Mat->p[i], j, temp, Dimension);
    }
  }

  free(col_of_rank);

#ifdef POLY_DEBUG
  fprintf(stderr, "[Gauss : Output]\nRay =");
  Matrix_Print(stderr, 0,Mat);
#endif

  return Rank;
} /* Gauss */



/*******************************************************************/
/* Remove_Redundants --
   Given (Mat) a matrix of equations and inequalities, and
   given (Ray) a matrix of rays and lines
   given (Sat) the saturation matrix,
   Return a polyhedron, composed of Mat and Ray after reductions.
   Usually called as a follup to Chernikova to remove redundant
   equations.  (Chernikova ensures that there are no redundant
   lines and rays).
*/
static Polyhedron *Remove_Redundants(Mat, Ray, Sat, Filter)
Matrix   *Mat, *Ray;
pSatMatrix Sat;
unsigned        *Filter;
{ 
  int i, j, k;
  unsigned Dimension, nc, NbRay, NbConstraints, RowSize1, RowSize2,
           *Trace, *bx, *jx, Dim, b;
  unsigned NbBid, NbUni, NbEq, NbIneq;
  unsigned NbBid2, NbUni2, NbEq2, NbIneq2;
  int Redundant;
  int aux, *temp2;
  Polyhedron *Pol;


  Value *temp1;
  Value *p, *q;
  Value Status;

 
  Dimension = Mat->NbColumns-1;     /* homogeneous dimension */

  NbRay = Ray->NbRows;
  nc = Sat->NbColumns;
  NbConstraints = Mat->NbRows;

  RowSize1=(Dimension+1) * sizeof(Value);
  RowSize2=nc*sizeof(int);

  temp1=(Value *)malloc(RowSize1);
	if( !temp1 )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		return 0;
	}
  temp2=(int *)malloc(RowSize2);
	if( !temp2 )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		free(temp1);
		return 0;
	}

  /* introduce indirections into Sat matrix to simplify processing
     with Sat and allow easy exchanges of columns */
  bx = (unsigned *)malloc(NbConstraints * sizeof(unsigned));
	if( !bx )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		free(temp1); free(temp2);
		return 0;
	}
  jx = (unsigned *)malloc(NbConstraints * sizeof(unsigned));
	if( !jx )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		free(temp1); free(temp2); free(bx);
		return 0;
	}
  i = 0;
  b = MSB;
  for (j=0; j<NbConstraints; j++)
  { 
    jx[j] = i;
    bx[j] = b;
    NEXT(i,b);
  }

  /* reset the status numbers, count the vertices */
  aux = 0;
  for (i=0; i<NbRay; i++)
  {  
     /* Ray->p[i][0] = 0 */
     value_assign (Ray->p[i][0], VALUE_ZERO);

     /* if (Ray->p[i][Dimension] != 0) */
     if (value_notzero_p(Ray->p[i][Dimension]))
       aux++;        /* vertex */
  }
  if (!aux)     /* no vertices -- empty polyhedron */
  {   /* return an empty polyhedron */
      free(temp1); free(temp2); free(jx); free(bx);
      return Empty_Polyhedron(Dimension-1);
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Init]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRays =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 1: Compute status counts for both rays and inequalities */
  /* for each equation, count the number of vertices/rays saturated
     by that equation, put the result in the status words.
     At the same time, for each vertex/ray, count the number of
     constaints saturated by it. */
  NbEq=0;
  memset((char *)temp2, 0, RowSize2);                   /* for SIMPLIFY */

#ifdef POLY_RR_DEBUG
   fprintf (stderr, " j = ");
#endif

  for (j=0; j<NbConstraints; j++)
  {
#ifdef POLY_RR_DEBUG
     fprintf (stderr, " %i ", j);
     fflush (stderr);
#endif

    /* if (Filter && !Mat->p[j][0]) */
    if (Filter && value_zero_p(Mat->p[j][0])) 
      temp2[jx[j]] |= bx[j]; /* for SIMPLIFY */

    /* Mat->p[j][0]=0 */
    value_assign (Mat->p[j][0], VALUE_ZERO);

    /* identify and remove the positivity constraint 1>=0 */
    for (i=1, p = &Mat->p[j][1]; i<Dimension; i++)
    { 
      /* if (*p) */
      if (value_notzero_p(*p))
      {
        p++; 
        break;
      }
      else 
        p++;
    }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : IntoStep1]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf (stderr, " j = %i \n", j);
#endif

    if (i==Dimension) /* Constraint j is a positivity constraint */
    {   /* or is it 1==0 ??   -- lets check */
        for (i=0; i<NbRay; i++)
            if (!(Sat->p[i][jx[j]]&bx[j]))
            {
              /* Mat->p[j][0]++ */
              value_increment(Mat->p[j][0]);
            }
 
        /* if ((Mat->p[j][0] == NbRay) &&   it is an equality
               (Mat->p[j][Dimension] != 0))  and its not 0=0 */
        if ((value_eq (Mat->p[j][0], int_to_value(NbRay))) &&  
            (value_notzero_p (Mat->p[j][Dimension])))
        {  
            /* return an empty polyhedron */
            free(temp1); free(temp2); free(jx); free(bx);
            return Empty_Polyhedron(Dimension-1);
        }

        /* delete the positivity constraint */
        NbConstraints--;
        if (j==NbConstraints) continue;
        bexchange(Mat->p[j], Mat->p[NbConstraints], temp1, RowSize1);
        exchange(jx[j], jx[NbConstraints], aux);
        exchange(bx[j], bx[NbConstraints], aux);
        j--; continue;
    }

    /* count the incidences of saturations */
    for (i=0; i<NbRay; i++) 
      if (!(Sat->p[i][jx[j]]&bx[j]))
      {  
        /* Mat->p[j][0]++ */
        value_increment (Mat->p[j][0]);

        /* Ray->p[i][0]++ */
        value_increment (Ray->p[i][0]);
      }

    /* if (Mat->p[j][0]==NbRay) */ 
    if (value_eq (Mat->p[j][0], int_to_value(NbRay))) 
      NbEq++;    /* all vert/rays are saturated */
  }
  Mat->NbRows = NbConstraints;

  NbBid=0;
  for (i=0; i<NbRay; i++)
  {
    /* give rays credit for saturating the positivity constraint */

    /* if (Ray->p[i][Dimension]==0) */
    if (value_zero_p(Ray->p[i][Dimension]))
      /* Ray->p[i][0]++ */
      value_increment (Ray->p[i][0]);

    /* test for a line.  Note that a vertex cannot become a line because
       only rays satisfy the pos constr and can status == NbConstraints+1 */

    /* if (Ray->p[i][0]==NbConstraints+1) */
    if (value_eq(Ray->p[i][0], int_to_value(NbConstraints+1)))
      NbBid++;
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step1]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step2: Sort equations to top of Mat, inequalities afterward */
  /* Detection of implicit equations such as y>=3; y<=3 */
  /* Keep inequalities in same relative order for SIMPLIFY to work */
  for (i=0; i<NbEq; i++)
    /* if (Mat->p[i][0]!=NbRay) : vert/rays are NOT all saturated (inequality)*/
    if (value_ne(Mat->p[i][0], int_to_value(NbRay)))
    { 
      for (k=i+1; 
           value_ne (Mat->p[k][0], int_to_value(NbRay)) && k<NbConstraints; 
           k++)
        /* skip over inequalities -- find an equation */ ;

      if (k==NbConstraints) /* none found--program error (NbEq wrong)*/ break;
      /* slide inequalities up, move equality down (toward begining of array)*/
      memcpy(temp1, Mat->p[k], RowSize1);
      aux = jx[k];
      j   = bx[k];
      for (;k>i;k--)
      {  
         memcpy(Mat->p[k], Mat->p[k-1], RowSize1);
         jx[k] = jx[k-1];
         bx[k] = bx[k-1];
      }
      memcpy(Mat->p[i], temp1, RowSize1);
      jx[i] = aux;
      bx[i] = j;
    }

  if (Filter)                     /* for SIMPLIFY */
  for (i=0; i<NbEq; i++)
  {  Redundant = 0;
     for (j=i+1; j<NbEq; j++)
     {  for (k=0, p=&Mat->p[i][1], q=&Mat->p[j][1]; k<Dimension; k++,p++,q++)
        {  
          /* if (*p!=*q) */
          if (value_ne (*p, *q)) 
            break;
        }
        /* Redundant if both the same `and' Mat[j] was equation (temp2==1) */
        if (k==Dimension && (temp2[jx[j]] & bx[j])) 
        { 
          Redundant=1; 
          break;
        }
     }

     if (!Redundant) Filter[jx[i]] |= bx[i];  /* set flag */
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step2]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 3: Gaussian elimiation of Equalities and reduction of
             strength on rays */ 
  NbEq2 = Gauss(Mat, NbEq, Dimension, temp1);
  if (NbEq2>=Dimension)		/* there is a contradiction */
  {  /* return an empty polyhedron */
     free(temp1); free(temp2); free(jx); free(bx);
     return Empty_Polyhedron(Dimension-1);
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step3]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 4: Sort lines to top of Ray, rays afterward */
  /* Detection of implicit lines such as ray(1,2) and ray(-1,-2) */
  for (i=0, k=NbRay; i<NbBid && k>i; i++)
    /* if (Ray->p[i][0] != (NbConstraints+1)) :all constraints not saturated */
    if (value_ne (Ray->p[i][0], int_to_value(NbConstraints+1)) ) 
    { 
      /* while (--k >i && Ray->p[k][0]!=(NbConstraints+1)) ; */
      while (--k >i && value_ne (Ray->p[k][0], int_to_value(NbConstraints+1))) ;
      bexchange(Ray->p[i], Ray->p[k], temp1, RowSize1);
      bexchange(Sat->p[i], Sat->p[k], temp2, RowSize2);
    }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step4]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 5 : Gaussian Elimination of on Lineality Space
              and computation of dimension of Ray Space */
  NbBid2 = Gauss(Ray, NbBid, Dimension, temp1);

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : After Gauss]\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  if (NbBid2>=Dimension)
  {
		errormsg1("RemoveRedundants", "rmrdt", "dimension error");
		free(temp1); free(temp2); free(jx); free(bx);
		return Empty_Polyhedron(Dimension-1);
  }
  /* compute dimension of the ray space */
  Dim = Dimension-1-NbEq2-NbBid2;  /* Dim of non-homogen Ray Space */

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step5]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 6: First pass filter of inequalities and equation identification */
  NbIneq=0;
  for (j=0; j<NbConstraints; j++)
  {
    /* New positivity constraints may have been created by step 3.
       the Gaussian elimination -- must check for them */
    /* identify and remove the positivity constraint 1>=0 */
    for (i=1, p = &Mat->p[j][1]; i<Dimension; i++)
	    /* if (*p) */
	    if (value_notzero_p (*p))
	    {
	      p++; 
	      break;
	    }
	    else
	      p++;

    if (i==Dimension) /* Constraint j is a positivity constraint */
    {   /* or is it 1==0 ??   -- lets check */
        /* if ((Mat->p[j][0]==NbRay) &&   : it is an equality 
            (Mat->p[j][Dimension]!=0))    : and its not 0=0 */
        if ((value_eq (Mat->p[j][0], int_to_value(NbRay))) &&
            (value_notzero_p(Mat->p[j][Dimension])))

        {   /* return an empty polyhedron */
            free(temp1); free(temp2); free(jx); free(bx);
            return Empty_Polyhedron(Dimension-1);
        }

        /* set the positivity constraint redundant */
        /* Mat->p[j][0]=2 */
        value_assign (Mat->p[j][0], VALUE_CONST(2));
        continue;
    }

    /* Status = Mat->p[j][0] */
    value_assign(Status, Mat->p[j][0]);

    /* if (Status == 0) */
    if (value_zero_p(Status)) 
      /* Mat->p[j][0]=2 : redundant */
      value_assign (Mat->p[j][0], VALUE_CONST(2));
    /* else if (Status<Dim) */              
    else if (value_lt (Status, int_to_value (Dim)))
      /* Mat->p[j][0]=2 : redundant */
      value_assign (Mat->p[j][0], VALUE_CONST(2));
    /* else if (Status==NbRay) */
    else if (value_eq (Status, int_to_value (NbRay)))
      /* Mat->p[j][0]=0 : equality */
      value_assign (Mat->p[j][0], VALUE_ZERO);
    else                  
    { 
      NbIneq++; 
      /* Mat->p[j][0]=1 : inequality */
      value_assign (Mat->p[j][0], VALUE_ONE);
    }
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step6]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 7: First pass filter of rays, identification of lines */
  /*    If Dim=0, the origin in the universe will have status 0-- but don't
        delete it, please */
  NbUni=0;
  for (j=0; j<NbRay; j++)
  { 
    /* Status = Ray->p[j][0] */
    value_assign(Status, Ray->p[j][0]);

    /* if (Status < Dim) */
    if (value_lt (Status, int_to_value (Dim)))
      /* Ray->p[j][0]=2 : redundant */
      value_assign (Ray->p[j][0], VALUE_CONST(2));
    /* else if (Status == (NbConstraints+1)) */
    else if (value_eq (Status, int_to_value (NbConstraints+1))) 
      /* Ray->p[j][0]=0 : line */
      value_assign (Ray->p[j][0], VALUE_ZERO);
    else                     
    { 
      NbUni++; 
      /* Ray->p[j][0]=1 : ray */
      value_assign (Ray->p[j][0], VALUE_ONE);
    }
  }

  /* Allocate the polyhedron (using approximate sizes) */
  Pol = Polyhedron_Alloc(Dimension-1, NbIneq+NbEq2+1, NbUni+NbBid2);
	if (!Pol)
	{	errormsg1("Remove_redundants", "outofmem", "out of memory space");
		free(temp1); free(temp2); free(jx); free(bx);
		return 0;
	}
  Pol->NbBid = NbBid2;
  Pol->NbEq  = NbEq2;

  /* Partially fill the Polyhedron structure */
  if (NbBid2) Vector_Copy(Ray->p[0], Pol->Ray[0], (Dimension+1)*NbBid2);
  if (NbEq2)  Vector_Copy(Mat->p[0], Pol->Constraint[0], (Dimension+1)*NbEq2);

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step7]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif


  /* Step 8: Final pass filter of inequalities */
  /* Detection of redundants inequalities */
  /* redundant inequalitIes will include:
        1. inequalities which are always true, such as 1>=0,
        2. redundant inequalities such as y>=4 given y>=3, or
           x>=2 given x=1, etc.
        3. redundant inequalities such as x+y>=5 given x>=3 and y>=2 */
  /* every 'good' inequality must saturate at least Dimension rays 
     and be unique */
  Trace=(unsigned *)malloc(nc * sizeof(unsigned));
	if( !Trace )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		free(temp1); free(temp2); free(jx); free(bx);
		return 0;
	}
  /*                   j-->
                        _________________
                        |               |
                        |      Mat      |
                        |_______________|
                                     |
        _________        ____________V____
      i |       |        |            0  |
      | |  Ray  |        |          Sat  |
      V |       |        |            0  |
        |       |        |            0  |
        |_______|        |_______________|
                               -OR-
                        ________|________
                        |_____Trace_0___|
  */
  NbIneq2 = 0;
  for (j=NbEq; j<NbConstraints; j++)
  { 
    /* if (Mat->p[j][0]==1) : non-redundant inequality */
    if (value_one_p (Mat->p[j][0]))
    { 
      for (k=0; k<nc; k++) Trace[k]=0;         /* init Trace */
      /* compute Trace: the union of all rows of Sat where
			constraint j is saturated */
      for (i=NbBid; i<NbRay; i++) 
        /* if (Ray->p[i][0]==1) */
        if (value_one_p(Ray->p[i][0]))
        { 
          if (!(Sat->p[i][jx[j]]&bx[j])) 
          for (k=0; k<nc; k++) Trace[k] |= Sat->p[i][k];
        }

      /* only constraint j should saturate this set of vertices/rays.
	 if another non-redundant constraint also saturates this set,
         then constraint j is redundant */
      Redundant=0;
      for (i=NbEq; i<NbConstraints; i++) 
      {
        /* if ( (Mat->p[i][0] ==1) && (i!=j) && !(Trace[jx[i]] & bx[i]) ) */
        if (value_one_p (Mat->p[i][0]) && (i!=j) && !(Trace[jx[i]] & bx[i]) )
        {   Redundant=1;
            break;
        }
      }
      if (Redundant) Mat->p[j][0]=2;
      else
      { Vector_Copy(Mat->p[j], Pol->Constraint[NbEq2+NbIneq2], Dimension+1);
        if (Filter) Filter[jx[j]] |= bx[j];		/* for SIMPLIFY */
	NbIneq2++;
      }
    }
  }
  free(Trace);

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step8]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  /* Step 9: Final pass filter of rays */
  /* Detection of redundants rays */
  Trace=(unsigned *)malloc(NbRay * sizeof(unsigned));
	if( !Trace )
	{	errormsg1("Remove_Redundants", "outofmem", "out of memory space");
		free(bx); free(jx); free(temp2); free(temp1);
		return 0;
	}
  /* 			j-->
			_________________
			|		|
			|      Mat	|
			|_______________|

	_________	_________________  ___
      i	|	|	|	 	|  |T|
      |	|  Ray	|	|      Sat	|  |r|
      V	|	|	|	 	|  |a|
	|	|	|	 	|  |c|
	|	|------>|  0   0 0   0	|  |e|
	|_______|	|_______________|  |_|
  */
  NbUni2 = 0;
  aux = 0;	/* let aux be the number of rays not vertices */
  for (i=NbBid; i<NbRay; i++) {
    /* if (Ray->p[i][0]==1) */
    if (value_one_p (Ray->p[i][0]))
    { 
      /* if (Ray->p[i][Dimension]!=0) : vertex */
      if (value_notzero_p (Ray->p[i][Dimension]))
          for (k=NbBid; k<NbRay; k++) Trace[k]=0;	/* init Trace */
      else				/* ray */
	  /* include the positivity constraint incidences for rays:
             the positivity constraint saturates all rays and no vertices */
	  for (k=NbBid; k<NbRay; k++) 
            /* Trace[k]=(Ray->p[k][Dimension]!=0); */
            Trace[k] = (value_notzero_p (Ray->p[k][Dimension]));

      Redundant=0;
      { /* compute Trace: the union of all cols of Sat where
			  ray i is saturated */

        for (j=NbEq; j<NbConstraints; j++) 
          /* if (Mat->p[j][0]==1) : ineq */
          if (value_one_p (Mat->p[j][0]))
          { 
            if (!(Sat->p[i][jx[j]]&bx[j]))
              for (k=NbBid; k<NbRay; k++) Trace[k] |= Sat->p[k][jx[j]]&bx[j];
          }

        /* if ray i does not saturate any inequalities (other than
	   the positivity constraint, then it is the case that there is
	   only one inequality and that ray is its orthogonal */

        /* only ray i should saturate this set of inequalities.
	   if another non-redundant ray also saturates this set,
           then ray i is redundant */
        for (j=NbBid; j<NbRay; j++)
        { 
          /* if ( (Ray->p[j][0]==1) && (i!=j) && !Trace[j] ) */
          if (value_one_p (Ray->p[j][0]) && (i!=j) && !Trace[j] )
          {   Redundant=1;
              break;
          }
	}
      }
      if (Redundant)
        value_assign( Ray->p[i][0], int_to_value(2) );
      else
      { Vector_Copy(Ray->p[i], Pol->Ray[NbBid2+NbUni2], Dimension+1);
	NbUni2++;			    /* inc nb of rays */

	/* if (Ray->p[i][Dimension]==0) */ 
        if (value_zero_p (Ray->p[i][Dimension]))
          aux++; /* inc nb of rays not vertices */
      }
    }
  }
  if (aux>=Dim) /* include the positivity constraint */
  { Vector_Init(Pol->Constraint[NbEq2+NbIneq2], Dimension+1);
    value_assign( Pol->Constraint[NbEq2+NbIneq2][0], VALUE_ONE );
	 /* inequality */
    value_assign( Pol->Constraint[NbEq2+NbIneq2][Dimension], VALUE_ONE );
	 	/* 1>=0 */
    NbIneq2++;
  }

#ifdef POLY_RR_DEBUG
  fprintf(stderr, "[Remove_redundants : Step9]\nConstraints =");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nRay =");
  Matrix_Print(stderr, 0,Ray);
#endif

  free(Trace);
  free(bx);
  free(jx);
  free(temp2);
  free(temp1);

  Pol->NbConstraints = NbEq2 + NbIneq2;
  Pol->NbRays = NbBid2 + NbUni2;
  return Pol;

} /* Remove_Redundants */




/*
 * Polyhedron
 *
 */




Polyhedron*
Polyhedron_Alloc(Dimension, NbConstraints, NbRays)
     unsigned Dimension, NbConstraints, NbRays;
{ 
  Polyhedron *Pol;
  unsigned NbRows,NbColumns;
  int i;
  Value *p, **q;

  Pol=(Polyhedron *)malloc(sizeof(Polyhedron));
	if( !Pol )
	{	errormsg1("Polyhedron_Alloc", "outofmem", "out of memory space");
		return 0;
	}

  Pol->next          = (Polyhedron *)0;
  Pol->Dimension     = Dimension;
  Pol->NbConstraints = NbConstraints;
  Pol->NbRays        = NbRays;
  Pol->NbEq          = 0;
  Pol->NbBid         = 0;
  NbRows             = NbConstraints + NbRays;
  NbColumns          = Dimension + 2;
  
  q = (Value **)malloc(NbRows * sizeof(Value *));
	if( !q )
	{	errormsg1("Polyhedron_Alloc", "outofmem", "out of memory space");
		return 0;
	}
  p = (Value *)malloc(NbRows * NbColumns * sizeof(Value));
	if( !p )
	{	errormsg1("Polyhedron_Alloc", "outofmem", "out of memory space");
		return 0;
	}

  Pol->Constraint    = q;
  Pol->Ray           = q + NbConstraints;
  Pol->p_Init        = p;

  for (i=0;i<NbRows;i++)
  { 
    *q++ = p;
    p += NbColumns;
  }

  return Pol;
} /* Polyhedron_Alloc */




void 
Polyhedron_Free(Pol)
     Polyhedron *Pol;
{ 
	if( !Pol )
		return;
  free((char *) Pol->p_Init);
  free((char *) Pol->Constraint);
  free((char *) Pol);
} /* Polyhedron_Free */

void Domain_Free(Pol)
     Polyhedron *Pol;
{
	if( !Pol )
		return;
  if (Pol->next) Domain_Free(Pol->next);
  free((char *) Pol->p_Init);
  free((char *) Pol->Constraint);
  free((char *) Pol);
} /* Domain_Free */



void Polyhedron_Print(Dst, Format, Pol)
     FILE *Dst;
     char *Format;
     Polyhedron *Pol;
{ 
  unsigned Dimension, NbConstraints, NbRays;
  int      i, j;
  Value    *p;

  if (!Pol)
  { 
    fprintf(Dst, "<null polyhedron>\n");
    return;
  }

  Dimension     = Pol->Dimension + 2;  /* 1 for bid/uni, 1 for ray/pnt */
  NbConstraints = Pol->NbConstraints;
  NbRays        = Pol->NbRays;
  fprintf(Dst, "POLYHEDRON Dimension:%d\n", Pol->Dimension);
  fprintf(Dst,"           Constraints:%d  Equations:%d  Rays:%d  Lines:%d\n",
        Pol->NbConstraints, Pol->NbEq, Pol->NbRays, Pol->NbBid);

  fprintf(Dst,"Constraints %d %d\n", NbConstraints, Dimension);
  for (i=0;i<NbConstraints;i++) 
  {
    p=Pol->Constraint[i];
    /* if (*p) */
    if (value_notzero_p (*p))
      fprintf(Dst,"Inequality: [");
    else      
      fprintf(Dst,"Equality:   [");

    p++;

    for (j=1;j<Dimension;j++)
      (void) fprintf(Dst, Format, *p++);
    (void) fprintf(Dst," ]\n");
  }

  (void)fprintf(Dst, "Rays %d %d\n", NbRays, Dimension);
  for (i=0;i<NbRays;i++) 
  {
    p=Pol->Ray[i];
    /* if (*p) */
    if (value_notzero_p (*p))
    {   
        p++;
        /* if ( p[Dimension-2] ) */
        if (value_notzero_p (p[Dimension-2]))
          fprintf(Dst, "Vertex: [");
        else                  
          fprintf(Dst, "Ray:    [");
    }
    else
    {
        p++;
        fprintf(Dst, "Line:   [");
    }
    for (j=1; j < Dimension-1; j++) fprintf(Dst, Format, *p++);

    /* if (*p) */
    if (value_notzero_p (*p)) 
    {
      fprintf( Dst, " ]/" );
      fprintf( Dst, VALUE_FMT, *p );
      fprintf( Dst, "\n" );
    }
    else    
      fprintf(Dst, " ]\n");
  }

  if (Pol->next)
  {  
     fprintf(Dst, "UNION ");
     Polyhedron_Print(Dst, Format, Pol->next);
  };
} /* Polyhedron_Print */




void
PPd (Pol)
     Polyhedron *Pol;
{
  Polyhedron_Print (stderr, "%4d", Pol);
}




/* Empty_Polyhedron --
   Create and return an empty polyhedron of non-homo dimension 'Dimension'
*/
Polyhedron *Empty_Polyhedron(Dimension)
unsigned  Dimension;  /* non-homogeneous dimension */
{ 
  Polyhedron *Pol;
  int i;

  /* create an overconstrained system of equations:
        x=0, y=0, ... , z=0, 1=0
     and no rays
     Dimension of Ray Space = Dimension - NbBid - NbEq = -1 (empty)
  */

  Pol = Polyhedron_Alloc(Dimension, Dimension+1, 0);
	if (!Pol)
	{	errormsg1("Empty_Polyhedron", "outofmem", "out of memory space");
		return 0;
	}
  Vector_Init(Pol->Constraint[0], (Dimension+1)*(Dimension+2));

  for (i=0; i<=Dimension; i++)
  {
    /* Pol->Constraint[i][i+1]=1 */
    value_assign (Pol->Constraint[i][i+1], VALUE_ONE);
  }

  Pol->NbEq = Dimension+1;
  Pol->NbBid = 0;

  return Pol;
} /* Empty_Polyhedron */




/* Universe_Polyhedron --
   Create and return an universe polyhedron of non-homo 
   dimension 'Dimension'
*/
Polyhedron *Universe_Polyhedron(Dimension)
unsigned  Dimension;  /* non-homogeneous dimension */
{ 
  Polyhedron *Pol;
  int i;

  /* create an unconstrained system of equations:
        1>=0
     and vertex(0,0,0...) and line (1,0,0...) , (0,1,0,... ) and (0,0,1,...)
     Dimension of Ray Space = Dimension - NbBid - NbEq = 0
  */

  Pol = Polyhedron_Alloc(Dimension, 1, Dimension+1);
	if (!Pol)
	{	errormsg1("Universe_Polyhedron", "outofmem", "out of memory space");
		return 0;
	}
  Vector_Init(Pol->Constraint[0], (Dimension+2));

  /* Pol->Constraint[0][0] = 1 */
  value_assign (Pol->Constraint[0][0], VALUE_ONE);

  /* Pol->Constraint[0][Dimension+1] = 1 */
  value_assign (Pol->Constraint[0][Dimension+1], VALUE_ONE);

  Vector_Init(Pol->Ray[0], (Dimension+1)*(Dimension+2));

  for (i=0; i<=Dimension; i++)
  {
    /* Pol->Ray[i][i+1]=1 */
    value_assign (Pol->Ray[i][i+1], VALUE_ONE);
  }

  /* Pol->Ray[Dimension][0] = 1 :  vertex status */
  value_assign (Pol->Ray[Dimension][0], VALUE_ONE);

  Pol->NbEq = 0;
  Pol->NbBid = Dimension;
  return Pol;
} /* Universe_Polyhedron */




Polyhedron *Constraints2Polyhedron(Constraints, NbMaxRays)
Matrix *Constraints;
unsigned NbMaxRays;
{
  Polyhedron *Pol;
  Matrix *Ray;
  pSatMatrix Sat;
  unsigned Dimension, nc;
  int i;

  Dimension = Constraints->NbColumns - 1; /* homogeneous dimension */


  if (Constraints->NbRows==0)
  {  
    Pol = Universe_Polyhedron(Dimension-1);
    return Pol;
  }

/*   rather than add a 'positivity constraint', it is better to
     initialize the lineality space with line in each of the index
     dimensions, but no line in the lambda dimension. Then initialize
     the ray space with an origin at 0.  This is what you get anyway,
     after the positivity constraint has been processed by Chernikova
*/

  /* Initialize Ray Space to Universe */
  Ray = Matrix_Alloc(NbMaxRays, Dimension+1);
	if(!Ray)
	{	errormsg1("Constraints2Polyhedron","outofmem","out of memory space");
		return 0;
	}
  Vector_Init(Ray->p_Init, NbMaxRays*(Dimension+1) );

/*  Vector_Init(Ray->p_Init, Dimension*(Dimension+1) );
*/
  for (i=0; i<Dimension; i++)
  {
    /* Ray->p[i][i+1] = 1 */
    value_assign (Ray->p[i][i+1], VALUE_ONE);
  }

  /* Ray->p[Dimension-1][0] = 1 : mark for ray */
  value_assign (Ray->p[Dimension-1][0], VALUE_ONE);

  Ray->NbRows = Dimension; 

  /* Initialize the Sat Matrix */
  nc = (Constraints->NbRows - 1) / (sizeof(int)*8) + 1;
  Sat = SMAlloc(NbMaxRays, nc);
  SMVector_Init(Sat->p_init, Dimension*nc);
  Sat->NbRows = Dimension;

  Chernikova(Constraints, Ray, Sat, Dimension-1, NbMaxRays, 0);

#ifdef POLY_DEBUG
  fprintf(stderr, "[constraints2polyhedron]\nConstraints = ");
  Matrix_Print(stderr, 0,Constraints);
  fprintf(stderr, "\nRay = ");
  Matrix_Print(stderr, 0, Ray);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

  /* remove redundant equations and build the polyhedron */
  Pol = Remove_Redundants(Constraints, Ray, Sat, 0);

#ifdef POLY_DEBUG
  fprintf(stderr, "\nPol = ");
  Polyhedron_Print(stderr, "%4d", Pol);
#endif

  SMFree(Sat);
  Matrix_Free(Ray);



  return Pol;
} /* Constraints2Polyhedron */




Matrix *Polyhedron2Constraints(Pol)
Polyhedron *Pol;
{
  Matrix     *Mat;
  unsigned NbConstraints, Dimension;

  NbConstraints = Pol->NbConstraints;
  Dimension     = Pol->Dimension+2;
  Mat = Matrix_Alloc(NbConstraints, Dimension);
	if(!Mat)
	{	errormsg1("Polyhedron2Constraints", "outofmem", "out of memory space");
		return 0;
	}

  Vector_Copy(Pol->Constraint[0], Mat->p_Init, NbConstraints*Dimension);

  return Mat;
} /* Polyhedron2Constraints */




Polyhedron *Rays2Polyhedron(Ray, NbMaxRays)
Matrix     *Ray;
unsigned   NbMaxRays;
{
  Polyhedron *Pol;
  Matrix *Mat;
  pSatMatrix Sat, Sat2;
  unsigned Dimension, nc;
  int i;


  Dimension = Ray->NbColumns-1; /* homogeneous dimension */


  if (Ray->NbRows==0)
  {  
     Pol = Empty_Polyhedron(Dimension-1);
     return(Pol);
  }

  /* Initialize Mat to Empty Polyhedron */
  Mat = Matrix_Alloc(NbMaxRays, Dimension+1);
	if(!Mat)
	{	errormsg1("Rays2Polyhedron", "outofmem", "out of memory space");
		return 0;
	}
  Vector_Init( Mat->p_Init, NbMaxRays*(Dimension+1) );

  for (i=0; i<Dimension; i++) 
  {
    /* Mat->p[i][i+1]=1 */
    value_assign (Mat->p[i][i+1], VALUE_ONE);
  }

  Mat->NbRows = Dimension;

  nc = ( Ray->NbRows -1 ) / (sizeof(int)*8) + 1;
  Sat = SMAlloc(NbMaxRays, nc);
  SMVector_Init(Sat->p[0], Dimension*nc );
  Sat->NbRows = Dimension;

#ifdef POLY_DEBUG
  fprintf(stderr, "[ray2polyhedron: Before]\nRay = ");
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

  Chernikova(Ray, Mat, Sat, Dimension, NbMaxRays, 0);

#ifdef POLY_DEBUG
  fprintf(stderr, "[ray2polyhedron: After]\nRay = ");
  Matrix_Print(stderr, 0,Ray);
  fprintf(stderr, "\nConstraints = ");
  Matrix_Print(stderr, 0,Mat);
  fprintf(stderr, "\nSat = ");
  SMPd (Sat);
#endif

  Sat2 = TransformSat(Mat, Ray, Sat);

#ifdef POLY_DEBUG
  fprintf(stderr, "\nSat2 =");
  SMPd (Sat2);
#endif
  SMFree(Sat);

  Pol = Remove_Redundants(Mat, Ray, Sat2, 0);

#ifdef POLY_DEBUG
  fprintf(stderr, "\nPol = ");
  Polyhedron_Print(stderr, "%4d", Pol);
#endif
  SMFree(Sat2);
  Matrix_Free(Mat);

  return Pol;
} /* Rays2Polyhedron */









/************************************************************/
/*   Vin100's stuff, for the polyparam vertices to work     */
/************************************************************/


static SatMatrix *BuildSat(Mat, Ray, NbCon, NbMaxRays)
Matrix *Mat, *Ray;
unsigned NbCon, NbMaxRays;
{
	SatMatrix *Sat;
	int i, j, k, jx;
	Value *p1, *p2, *p3;
	unsigned Dimension, NbRay, bx, nc;

	NbRay = Ray->NbRows;
	Dimension = Mat->NbColumns-1;	/* homogeneous Dimension */
 
	/* build the Sat matrix */
	/* at same time, verify the integrity of the polyhedron */
	nc      = (Mat->NbRows - 1) / (sizeof(int)*8) + 1;
	Sat     = SMAlloc(NbMaxRays, nc);
	Sat->NbRows = NbRay;
	SMVector_Init(Sat->p_init, nc*NbRay);
	jx=0; bx=MSB;
	for (k=0; k<NbCon; k++)
	{
		for (i=0; i<NbRay; i++)
		{
			p1 = Ray->p[i]+1;
			p2 = Mat->p[k]+1;
			p3 = Ray->p[i];
			value_assign( *p3, VALUE_ZERO );
			for (j=0; j<Dimension; j++)
			{
				value_addto( *p3,  value_mult(*p1, *p2 ) );
				p1++; p2++;
			}
		}
		for (j=0; j<NbRay; j++)
		{
			if ( value_notzero_p(Ray->p[j][0]) )
				Sat->p[j][jx]|=bx;
		}
		NEXT(jx, bx);
	}

	return Sat;
}


Polyhedron *AddConstraints(Con2, NbCon2, Pol1, NbMaxRays)
     Value	*Con2;
     unsigned	NbCon2;
     Polyhedron *Pol1;
     unsigned   NbMaxRays;
{
	Polyhedron *Pol;
	Matrix   *Mat, *Ray;
	SatMatrix *Sat;
	unsigned NbRay, NbCon, NbCon1, NbEle1, Dimension;

	NbRay		= Pol1->NbRays;
	NbCon1	= Pol1->NbConstraints;
	NbCon		= NbCon1 + NbCon2;
	Dimension	= Pol1->Dimension+2;	/* homogeneous Dimension + Status */
	NbEle1	= NbCon1*Dimension;

	Mat = Matrix_Alloc(NbCon, Dimension);
	if(!Mat)
	{	errormsg1("AddConstraints", "outofmem", "out of memory space");
		return 0;
	}

	/* Set the constraints of Pol1 */
	Vector_Copy(Pol1->Constraint[0], Mat->p[0], NbEle1);

	/* Add the new constraints */
	Vector_Copy(Con2, Mat->p[NbCon1], NbCon2*Dimension);

	Ray = Matrix_Alloc(NbMaxRays, Dimension);
	if(!Ray)
	{	errormsg1("AddConstraints", "outofmem", "out of memory space");
		return 0;
	}
	Ray->NbRows = NbRay;
	Vector_Copy(Pol1->Ray[0], Ray->p[0], NbRay*Dimension);

	Sat = BuildSat(Mat, Ray, NbCon1, NbMaxRays);
	Pol_status = Chernikova(Mat, Ray, Sat, Pol1->NbBid, NbMaxRays, NbCon1);
	Pol = Remove_Redundants(Mat, Ray, Sat, 0);

	SMFree(Sat);
	Matrix_Free(Ray);
	Matrix_Free(Mat);

	return Pol;
} /* AddConstraints */

/* PolyhedronIncludes --
      Return 1 if Pol1 includes (covers) Pol2, 0 otherwise
*/
int	PolyhedronIncludes(Pol1, Pol2)
Polyhedron *Pol1, *Pol2;
{
	int Dimension = Pol1->Dimension + 1;
	int i, j, k;
	Value *p1, *p2, p3;
 
	for (k=0; k<Pol1->NbConstraints; k++)
	{
		for (i=0; i<Pol2->NbRays; i++)
		{
			p1 = Pol2->Ray[i]+1;
			p2 = Pol1->Constraint[k]+1;
			value_assign( p3, VALUE_ZERO );
			for (j=0; j<Dimension; j++)
			{
				value_addto( p3, value_mult(*p1, *p2) );
				p1++; p2++;
			}
			if ( value_neg_p(p3) ||
					( value_notzero_p(p3) && value_zero_p(Pol1->Constraint[k][0])) )
				return 0;
		}
	}
	return 1;
} /* PolyhedronIncludes */


/* AddPolyToDomain --
   add polyhedron p3 to domain p3beg, check for redundancies.
   if p3 is not used, it is removed from memory.
   return the new domain
*/
Polyhedron *AddPolyToDomain (p3, p3beg)
Polyhedron *p3, *p3beg;
{
	Polyhedron *p, *p3end = (Polyhedron *) 0;
	int Redundant;

	if (!p3) return p3beg;
	if (!p3beg)	return p3;

	/* check for empty p3's */
	if ( emptyQ(p3) )
	{
		Polyhedron_Free(p3);
		return p3beg;
	}
	if ( emptyQ(p3beg) )
	{
		Polyhedron_Free(p3beg);
		return p3;
	}

	/* test p3 against the domain */
	Redundant = 0;
	for (p=p3beg, p3beg=(Polyhedron *)0; p; p=p->next)
	{
		if ( PolyhedronIncludes(p3, p) ) continue;

		/* add old polyhedron p to the new domain list */
		if (!p3beg) p3beg = p; else p3end->next = p;
		p3end = p;

		if ( PolyhedronIncludes(p, p3) ) /* p covers p3 */
		{
			Redundant = 1;
			break;
		}
	}

	if (!Redundant)	/* the whole list has been checked */
	{
		/* add new polyhedron p3 to the new domain list */
		if (!p3beg) p3beg = p3; else p3end->next = p3;
	}
	else
	{
		/* the rest of the list is just inherited from p */
		Polyhedron_Free(p3);
	}
	return p3beg;
}



/* SubConstraint--
   Add the inverse of Con2 ( >0, >=0, <0, <=0 ) to Pol1
   Pass : 0 add ( -constraint -1 )
          1 add ( +constraint -1 )
	  2 add ( -constraint    )
	  3 add (  constraint    )
*/
Polyhedron *SubConstraint(Con2, Pol1, NbMaxRays, Pass)
     Value	*Con2;
     Polyhedron *Pol1;
     unsigned   NbMaxRays;
	  int Pass;
{
	Polyhedron *Pol;
	Matrix   *Mat, *Ray;
	SatMatrix *Sat;
	unsigned NbRay, NbCon, NbCon1, NbEle1, Dimension;
	int      i;

	/* if Con2 is the positivity constraint -- check out now */
	Dimension     = Pol1->Dimension+1;    /* homogeneous Dimension */
	for (i=1; i<Dimension; i++)
		if ( value_notzero_p(Con2[i]) ) break;
	if (i==Dimension) return (Polyhedron *)0;

	NbRay     = Pol1->NbRays;
	NbCon1    = Pol1->NbConstraints;
	NbCon     = NbCon1 + 1;
	Dimension = Pol1->Dimension+2;	/* homogeneous Dimension + Status */
	NbEle1    = NbCon1*Dimension;

	Mat = Matrix_Alloc(NbCon, Dimension);
	if(!Mat)
	{	errormsg1("SubConstraint", "outofmem", "out of memory space");
		return 0;
	}

	/* Set the constraints of Pol1 */
	Vector_Copy(Pol1->Constraint[0], Mat->p[0], NbEle1);

	/* Add the new constraint */
	value_assign( Mat->p[NbCon1][0], VALUE_ONE );
	if (!(Pass&1))
		for(i=1; i<Dimension; i++)
			value_assign( Mat->p[NbCon1][i], value_uminus(Con2[i]) );
	else
		for(i=1; i<Dimension; i++)
			value_assign( Mat->p[NbCon1][i], Con2[i] );
	if (!(Pass&2))
		value_decrement( Mat->p[NbCon1][Dimension-1] );

	Ray = Matrix_Alloc(NbMaxRays, Dimension);
	if(!Ray)
	{	errormsg1("SubConstraint", "outofmem", "out of memory space");
		return 0;
	}
	Ray->NbRows = NbRay;
	Vector_Copy(Pol1->Ray[0], Ray->p[0], NbRay*Dimension);

	Sat = BuildSat(Mat, Ray, NbCon1, NbMaxRays);
	Pol_status =
	Chernikova(Mat, Ray, Sat, Pol1->NbBid, NbMaxRays, NbCon1);
	Pol = Remove_Redundants(Mat, Ray, Sat, 0);
	SMFree(Sat);
	Matrix_Free(Ray);
	Matrix_Free(Mat);
	return Pol;
} /* SubConstraint */


Polyhedron *DomainIntersection(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned   NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;

	if (!Pol1 || !Pol2) return (Polyhedron*) 0;
	if (Pol1->Dimension != Pol2->Dimension)
	{
	   errormsg1( "DomainIntersection", "diffdim",
				"operation on different dimensions");
		return (Polyhedron*) 0;
	}

	d = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		for (p2=Pol2; p2; p2=p2->next)
		{
			p3 = AddConstraints(p2->Constraint[0],
						p2->NbConstraints, p1, NbMaxRays);

			d = AddPolyToDomain (p3, d);
		}
	}
	if (!d)
		return Empty_Polyhedron(Pol1->Dimension);
	else
		return d;

} /* DomainIntersection */




Matrix *Polyhedron2Rays(Pol)
Polyhedron *Pol;
{
	Matrix     *Ray;
	unsigned NbRays, Dimension;
	NbRays    = Pol->NbRays;
	Dimension = Pol->Dimension+2;		/* homogeneous Dimension + Status */
	Ray = Matrix_Alloc(NbRays, Dimension);
	if(!Ray)
	{	errormsg1("Polyhedron2Rays", "outofmem", "out of memory space");
		return 0;
	}
	Vector_Copy(Pol->Ray[0], Ray->p_Init, NbRays*Dimension);
	return Ray;
} /* Polyhedron2Rays */


Polyhedron *AddRays(Ray2, NbRay2, Pol1, NbMaxRays)
Value        *Ray2;
unsigned   NbRay2;
Polyhedron *Pol1;
unsigned   NbMaxRays;
{
	Polyhedron *Pol;
	Matrix   *Mat, *Ray;
	SatMatrix *Sat, *Sat2;
	unsigned NbCon, NbRay, NbRay1, NbEle1, Dimension;

	NbCon		= Pol1->NbConstraints;
	NbRay1	= Pol1->NbRays;
	NbRay		= NbRay1 + NbRay2;
	Dimension     = Pol1->Dimension+2;	/* homogeneous Dimension + Status */
	NbEle1	= NbRay1*Dimension;

	Ray = Matrix_Alloc(NbRay, Dimension);
	if(!Ray)
	{	errormsg1("AddRays", "outofmem", "out of memory space");
		return 0;
	}

	/* Set the Rays of Pol1 */
	Vector_Copy(Pol1->Ray[0], Ray->p_Init, NbEle1);

	/* Add the new Rays */
	Vector_Copy(Ray2, Ray->p_Init+NbEle1, NbRay2*Dimension);

	Mat = Matrix_Alloc(NbMaxRays, Dimension);
	if(!Mat)
	{	errormsg1("AddRays", "outofmem", "out of memory space");
		return 0;
	}
	Mat->NbRows = NbCon;
	Vector_Copy(Pol1->Constraint[0], Mat->p_Init, NbCon*Dimension);

	Sat = BuildSat(Ray, Mat, NbRay1, NbMaxRays);
	Pol_status =
	Chernikova(Ray, Mat, Sat, Pol1->NbEq, NbMaxRays, NbRay1);
	Sat2 = TransformSat(Mat, Ray, Sat);
	SMFree(Sat);
	Pol = Remove_Redundants(Mat, Ray, Sat2, 0);

	SMFree(Sat2);
	Matrix_Free(Mat);
	Matrix_Free(Ray);
	return Pol;
} /* AddRays */


Polyhedron *DomainAddRays(Pol1, Mat2, NbMaxRays)
     Polyhedron *Pol1;
     Matrix *Mat2;
     unsigned   NbMaxRays;
{
	Polyhedron *PolA, *PolEndA, *p1, *p2, *p3;
	int Redundant;

	if (!Pol1 ) return (Polyhedron*) 0;
	if (!Mat2 ) return Pol1;
	if (Pol1->Dimension != Mat2->NbColumns-2)
	{
		errormsg1(
				"DomainAddRays", "diffdim", "operation on different dimensions");
		return (Polyhedron*) 0;
	}

 	/* copy Pol1 to PolA */
	PolA = PolEndA = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		p3 = AddRays(Mat2->p[0], Mat2->NbRows, p1, NbMaxRays);
		/* does any component of PolA cover it ? */
		Redundant = 0;
		for (p2=PolA; p2; p2=p2->next)
		{
			if ( PolyhedronIncludes(p2, p1) ) /* p2 covers p1 */
			{
				Redundant = 1;
				break;
			}
		}
		if (!Redundant)
		{   /* add p1 to PolA */
			if (!PolEndA)
				PolEndA = PolA = p3;
			else
			{
				PolEndA->next = p3;
				PolEndA = PolEndA->next;
			}
		}
	}
	return PolA;

} /* DomainAddRays */



Polyhedron *Polyhedron_Copy (Pol)
    Polyhedron *Pol;
{
	Polyhedron *Pol1;

	if (!Pol) return (Polyhedron *)0;

	Pol1 = Polyhedron_Alloc(Pol->Dimension, Pol->NbConstraints, Pol->NbRays);
	if (!Pol1)
	{	errormsg1("Polyhedron_Copy", "outofmem", "out of memory space");
		return 0;
	}

	memcpy(Pol1->Constraint[0], Pol->Constraint[0],
			Pol->NbConstraints*(Pol->Dimension+2)*sizeof(Value) );
	memcpy(Pol1->Ray[0], Pol->Ray[0],
			Pol->NbRays*(Pol->Dimension+2)*sizeof(Value) );
	Pol1->NbBid = Pol->NbBid;
	Pol1->NbEq = Pol->NbEq;
	return Pol1;
}


Polyhedron *Domain_Copy(Pol)
Polyhedron *Pol;
{ Polyhedron *Pol1;
  if (!Pol) return (Polyhedron *) 0;
  Pol1 = Polyhedron_Copy(Pol);
  if (Pol->next) Pol1->next = Domain_Copy(Pol->next);
  return Pol1;
}



/* addToFilter -- Used for the DomSimplify procedure below */
static void addToFilter(k, Filter, Sat, tmpR, tmpC, NbRays, NbConstraints)
int k, NbRays, NbConstraints;
unsigned *Filter;
Value *tmpR, *tmpC;
SatMatrix *Sat;
{
	int kj, i,j, jx;
	unsigned kb, bx;

	/* remove constraint k */
	kj =   k/WSIZE; kb = MSB; kb >>= k%WSIZE;
	Filter[kj]|=kb;
	value_assign( tmpC[k], VALUE_MONE );

	/* remove rays excluded by constraint k */
	for(i=0; i<NbRays; i++)
		if ( value_posz_p(tmpR[i]) )
		{
			if (Sat->p[i][kj]&kb)
				value_decrement( tmpR[i] );  /* adjust included ray */
			else
			{
				/* constraint k excludes ray i -- delete ray i */
				value_assign( tmpR[i], VALUE_MONE );

				/* adjust non-deleted constraints */
				jx=0; bx=MSB;
				for(j=0; j<NbConstraints; j++)
				{
					if ( value_posz_p(tmpC[j]) && (Sat->p[i][jx]&bx) )
						value_decrement( tmpC[j] );
					NEXT(jx,bx);
				}
			}
		}

}


/* FindSimple --
   P1 and P2 is empty: what is the minimum set of constraints in P1
   that, when intersected with P2, gives the empty set */
static void     FindSimple(P1, P2, Filter, NbMaxRays)
Polyhedron *P1, *P2;
unsigned *Filter;
unsigned NbMaxRays;
{
	Matrix *Mat;
	SatMatrix *Sat;
	int i, j, k, jx, found;
	Value *p1, *p2, p3;
	unsigned Dimension, NbRays, NbConstraints, bx, nc;
	Value NbConstraintsLeft;
	Value *tmpC, *tmpR;
	Polyhedron *Pol, *Pol2;

	Dimension = P1->Dimension+2;       /* status + homogeneous Dimension */
	Mat = Matrix_Alloc(P1->NbConstraints, Dimension);
	if(!Mat)
	{	errormsg1("FindSimple", "outofmem", "out of memory space");
		return;
	}

	/* post constraints in P1 already included by Filter */
	jx = 0; bx = MSB; Mat->NbRows=0;
	value_assign( NbConstraintsLeft, VALUE_ZERO );
	for (k=0; k<P1->NbConstraints; k++)
	{
		if (Filter[jx]&bx)
		{
			Vector_Copy(P1->Constraint[k], Mat->p[Mat->NbRows], Dimension);
			Mat->NbRows++;
		}
		else
			value_increment( NbConstraintsLeft );
		NEXT(jx,bx);
	}
	Pol2 = P2;

	for (;;)
	{
		if (Mat->NbRows==0)
			Pol = Polyhedron_Copy(Pol2);
		else
		{
			Pol = AddConstraints(Mat->p_Init, Mat->NbRows, Pol2, NbMaxRays);
			if (Pol2 != P2) Polyhedron_Free(Pol2);
		}
		if (emptyQ(Pol))
		{
			Matrix_Free(Mat);
			Polyhedron_Free(Pol);
			return;
		}
		Mat->NbRows = 0;        /* Reset Mat */
		Pol2 = Pol;

		/* Its not enough-- find some more constraints */
		Dimension         = Pol->Dimension+1;       /* homogeneous Dimension */
		NbRays            = Pol->NbRays;
		NbConstraints     = P1->NbConstraints;
		tmpR = (Value *)malloc(NbRays*sizeof(Value));
		if( !tmpR )
		{	errormsg1("FindSimple", "outofmem", "out of memory space");
			return;
		}
		tmpC = (Value *)malloc(NbConstraints*sizeof(Value));
		if( !tmpC )
		{	errormsg1("FindSimple", "outofmem", "out of memory space");
			return;
		}
		memset((char *)tmpR, 0, NbRays*sizeof(Value));
		memset((char *)tmpC, 0, NbConstraints*sizeof(Value));

		/* build the Sat matrix */
		nc      = (NbConstraints - 1) / (sizeof(int)*8) + 1;
		Sat     = SMAlloc(NbRays, nc);
		Sat->NbRows = NbRays;
		SMVector_Init(Sat->p_init, nc*NbRays);

		jx=0; bx=MSB;
		for (k=0; k<NbConstraints; k++)
		{
			if (Filter[jx]&bx)
				value_assign( tmpC[k], VALUE_MONE );
			else
				for (i=0; i<NbRays; i++)
				{
					p1 = Pol->Ray[i]+1;
					p2 = P1->Constraint[k]+1;
					value_assign( p3, VALUE_ZERO );
					for (j=0; j<Dimension; j++)
					{
						value_addto( p3, value_mult( *p1, *p2 ));
						p1++; p2++;
					}
					if( value_zero_p(p3) ||
						(value_pos_p(p3) && value_notzero_p(P1->Constraint[k][0])) )
					{
						Sat->p[i][jx]|=bx;  /* constraint includes ray, set flag */
						value_increment( tmpR[i] );
						value_increment( tmpC[k] );
					}
				}
			NEXT(jx, bx);
		}

		do /* find all of the essential constraints */
		{
			found = 0;
			for(i=0; i<NbRays; i++)
				if( value_posz_p(tmpR[i]) )
				{
					if( value_eq( value_plus( tmpR[i],VALUE_ONE),
										NbConstraintsLeft ) )
					{
						/* Ray i is excluded by only one constraint... find it */
						jx = 0; bx = MSB;
						for(k=0; k<NbConstraints; k++)
						{
							if( value_posz_p(tmpC[k]) && ((Sat->p[i][jx]&bx)==0))
							{
								addToFilter(k, Filter, Sat, tmpR, tmpC,
											NbRays, NbConstraints);
								Vector_Copy(P1->Constraint[k],
											Mat->p[Mat->NbRows], Dimension+1);
								Mat->NbRows++;
								value_decrement( NbConstraintsLeft );
								found=1;
								break;
							}
							NEXT(jx,bx);
						}
						break;
					}
				}
		}
		while (found);

		if (!Mat->NbRows) /* Well then, just use a stupid heuristic */
		{		   /* find the constraint which excludes the most */
			Value cmax;
#ifndef LINEAR_VALUE_IS_CHARS
			/* the typechecking arithmetic lib does (of course) not
			   recognize this one... */
			value_assign( cmax, ~((Value)1<<(sizeof(Value)*8-1)) );
#else
value_assign( cmax, VALUE_ONE );
#endif
			j = -1;
			for(k=0; k<NbConstraints; k++)
				if ( value_posz_p(tmpC[k]) )
				{
					if ( value_gt(cmax,tmpC[k]) )
					{
						value_assign( cmax, tmpC[k] );
						j = k;
					}
				}

			if (j<0)
			{
				errormsg1("DomSimplify","logerror","logic error");
			}
			else
			{
				addToFilter(j, Filter, Sat, tmpR, tmpC, NbRays, NbConstraints);
				Vector_Copy(P1->Constraint[j],Mat->p[Mat->NbRows],Dimension+1);
				Mat->NbRows++;
				value_decrement( NbConstraintsLeft );
			}
		}
		SMFree(Sat);
		free(tmpC);
		free(tmpR);
	}
}



/* return 1 if intersection of Pol1 and Pol2 is not empty */
static int SimplifyConstraints(Pol1, Pol2, Filter, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned	*Filter;
     unsigned   NbMaxRays;
{
	Polyhedron *Pol;
	Matrix   *Mat, *Ray;
	SatMatrix *Sat;
	unsigned NbRay, NbCon, NbCon1, NbCon2, NbEle1, Dimension, notvid;

	NbRay         = Pol1->NbRays;
	NbCon1        = Pol1->NbConstraints;
	NbCon2        = Pol2->NbConstraints;
	NbCon         = NbCon1 + NbCon2;
	Dimension     = Pol1->Dimension+2;    /* homogeneous Dimension + Status */
	NbEle1        = NbCon1*Dimension;

	Mat = Matrix_Alloc(NbCon, Dimension);
	if(!Mat)
	{	errormsg1("SimplifyConstraints", "outofmem", "out of memory space");
		return 0;
	}

	/* Set the constraints of Pol1 */
	Vector_Copy(Pol1->Constraint[0], Mat->p_Init, NbEle1);

	/* Add the new constraints */
	Vector_Copy(Pol2->Constraint[0], Mat->p_Init+NbEle1, NbCon2*Dimension);

	Ray = Matrix_Alloc(NbMaxRays, Dimension);
	if(!Ray)
	{	errormsg1("SimplifyConstraints", "outofmem", "out of memory space");
		return 0;
	}
	Ray->NbRows = NbRay;
	Vector_Copy(Pol1->Ray[0], Ray->p_Init, NbRay*Dimension);

	Sat = BuildSat(Mat, Ray, NbCon1, NbMaxRays);
	Pol_status =
	Chernikova(Mat, Ray, Sat, Pol1->NbBid, NbMaxRays, NbCon1);
	Pol = Remove_Redundants(Mat, Ray, Sat, Filter);
	notvid = 1;
	if (Filter && emptyQ(Pol) ) 
	{
		notvid = 0;
		FindSimple(Pol1, Pol2, Filter, NbMaxRays);
	}
	Polyhedron_Free(Pol);
	SMFree(Sat);
	Matrix_Free(Ray);
	Matrix_Free(Mat);
	return notvid;
} /* SimplifyConstraints */


Polyhedron *DomainSimplify(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned   NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;
	unsigned k, jx, bx, nc, NbConstraints, RowSize, Dimension, *Filter, NbCon2;
	unsigned vid;
	Matrix *Constraints;

	if (!Pol1 || !Pol2) return Pol1;
	if (Pol1->Dimension != Pol2->Dimension)
	{
		errormsg1("DomainSimplify","diffdim","operation on different dimensions");
		return 0;
	}
	if (emptyQ(Pol1)||emptyQ(Pol2))
		return Empty_Polyhedron(Pol1->Dimension);

	NbCon2 = 0;
	for (p2=Pol2; p2; p2=p2->next)
		if (p2->NbConstraints > NbCon2)
			NbCon2 = p2->NbConstraints;

	Dimension = Pol1->Dimension+2;    /* allocation Dimension */
	d = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{ 
		NbConstraints = p1->NbConstraints;
		nc = (NbConstraints + NbCon2 - 1) / (sizeof(int)*8) + 1;
		RowSize = nc*sizeof(int);
		Filter  = (unsigned *)malloc(RowSize);
		if( !Filter )
		{	errormsg1("DomainSimplify", "outofmem", "out of memory space");
			return 0;
		}
		SMVector_Init(Filter, nc);

		/* Filter the constraints of p1 in context of p2(s) */
		vid = 1;
		for (p2=Pol2; p2; p2=p2->next)
		{
			if (SimplifyConstraints(p1, p2, Filter, NbMaxRays))
				vid=0;
			/* takes the union of all non redundant constraints */
		}

		if (!vid)
		{
			/* copy all non-redundant constraints to matrix */
			Constraints = Matrix_Alloc(NbConstraints, Dimension);
			if(!Constraints)
			{	errormsg1("DomainSimplify", "outofmem", "out of memory space");
				return 0;
			}
			Constraints->NbRows = 0;
			for (k=0, jx=0, bx=MSB; k<NbConstraints; k++)
			{
				if (Filter[jx]&bx) /* copy constraint */
				{
					Vector_Copy(p1->Constraint[k],
							Constraints->p[Constraints->NbRows],
							Dimension);
					Constraints->NbRows++;
				}
				NEXT(jx,bx);
			}
			p3 = Constraints2Polyhedron(Constraints,NbMaxRays);
			Matrix_Free(Constraints);
			d = AddPolyToDomain (p3, d);
		}
		free(Filter);
	}
	if (!d)
		return Empty_Polyhedron(Pol1->Dimension);
	else
		return d;
} /* DomainSimplify */




Polyhedron *DomainUnion(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned   NbMaxRays;
{
	Polyhedron *PolA, *PolEndA, *PolB, *PolEndB, *p1, *p2;
	int Redundant;

	if (!Pol1 || !Pol2) return (Polyhedron*) 0;
	if (Pol1->Dimension != Pol2->Dimension)
	{
		errormsg1("DomainUnion","diffdim","operation on different dimensions");
		return (Polyhedron*) 0;
	}

	/* copy Pol1 to PolA */
	PolA = PolEndA = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		/* does any component of Pol2 cover it ? */
		Redundant = 0;
		for (p2=Pol2; p2; p2=p2->next)
		{
			if ( PolyhedronIncludes(p2, p1) ) /* p2 covers p1 */
			{
				Redundant = 1;
				break;
			}
		}
		if (!Redundant)
		{
			/* add p1 to Pol */
			if (!PolEndA)
				PolEndA = PolA = Polyhedron_Copy(p1);
			else
			{
				PolEndA->next = Polyhedron_Copy(p1);
				PolEndA = PolEndA->next;
			}
		}
	}

	/* copy Pol2 to PolB */
	PolB = PolEndB = (Polyhedron *)0;
	for (p2=Pol2; p2; p2=p2->next)
	{
		/* does any component of Pol cover it ? */
		Redundant = 0;
		for (p1=PolA; p1; p1=p1->next)
		{
			if ( PolyhedronIncludes(p1, p2) ) /* p1 covers p2 */
			{
				Redundant = 1;
				break;
			}
		}
		if (!Redundant)
		{
			/* add p2 to PolB */
			if (!PolEndB)
				PolEndB = PolB = Polyhedron_Copy(p2);
			else
			{
				PolEndB->next = Polyhedron_Copy(p2);
				PolEndB = PolEndB->next;
			}
		}
	}

	if (!PolA) return PolB;
	PolEndA->next = PolB;
	return PolA;
} /* DomainUnion */


Polyhedron *DomainConvex(Pol1, NbMaxRays)
     Polyhedron *Pol1;
     unsigned NbMaxRays;
{
	Polyhedron *p, *Pol, *PolNew;

	if (!Pol1)
		return (Polyhedron*) 0;

	Pol = Polyhedron_Copy(Pol1);
	for (p=Pol1->next; p; p=p->next)
	{
		PolNew = AddRays(p->Ray[0], p->NbRays, Pol, NbMaxRays);
		Polyhedron_Free(Pol);
		Pol = PolNew;
	}
	return Pol;
} /* DomainConvex */



Polyhedron *DomainDifference(Pol1, Pol2, NbMaxRays)
     Polyhedron *Pol1, *Pol2;
     unsigned NbMaxRays;
{
	Polyhedron *p1, *p2, *p3, *d;
	int i;

	if (!Pol1 || !Pol2) return (Polyhedron*) 0;
	if (Pol1->Dimension != Pol2->Dimension)
	{
		errormsg1(
        "DomainDifference", "diffdim", "operation on different dimensions");
		return (Polyhedron*) 0;
	}

	d = (Polyhedron *)0;
	for (p2=Pol2; p2; p2=p2->next)
	{
		for (p1=Pol1; p1; p1=p1->next)
		{
			for (i=0; i<p2->NbConstraints; i++)
			{
				p3 = SubConstraint(p2->Constraint[i], p1, NbMaxRays,0);
				d = AddPolyToDomain (p3, d);
				if( value_notzero_p(p2->Constraint[i][0]) )
					continue;  /*inequality*/
				p3 = SubConstraint(p2->Constraint[i], p1, NbMaxRays,1);
				d = AddPolyToDomain (p3, d);
			}
		}
		Pol1 = d;
		d = (Polyhedron *)0;
	}
	if (!Pol1)
		return Empty_Polyhedron(Pol2->Dimension);
	else
		return Pol1;
} /* DomainDifference */


/*----------------------------------------------------------------------*/
/* align_context(D,n,MAXRAYS)                                           */
/*       D : input context domain (Polyhedron *)                        */
/*       n : the output dimension                                       */
/*       MAXRAYS : work space size                                      */
/* converts D to a Polyhedron * with dimension expanded by n            */
/*----------------------------------------------------------------------*/
Polyhedron *align_context(D,n,MAXRAYS)
     Polyhedron *D;
     int n;
     int MAXRAYS;
{
	Polyhedron *p, *p1, *result;
	int  i, j, k;
	Matrix *M;

	if (!D) return D;
	if (n<D->Dimension)
	{
		errormsg1(
        "align_context", "diffdim", "context dimension exceeds data");
		return D;
	}
	if (n==D->Dimension) return D;
	k = n-D->Dimension;

	result = NULL;
	p = NULL;
	for (; D; D=D->next)
	{
		p1 = p;
		M = Matrix_Alloc(D->NbConstraints, n+2);
		if(!M)
		{	errormsg1("align_context", "outofmem", "out of memory space");
			return 0;
		}
		Vector_Init( M->p[0], (D->NbConstraints)*(n+2) );

		for (i=0; i<D->NbConstraints; i++)
		{
			for (j=1; j<=D->Dimension+1;j++)
			{
				value_assign( M->p[i][k+j], D->Constraint[i][j] );
			}
			value_assign( M->p[i][0], D->Constraint[i][0] ); /* the status bit */
		}
		p = Constraints2Polyhedron(M, MAXRAYS);
		if (p1) p1->next = p; else result = p;
		Matrix_Free(M);
	}
	return(result);
}



/*----------------------------------------------------------------------*/
/* Polyhedron *Polyhedron_Scan(D, C, MAXRAYS)                           */
/*       D : Domain to be scanned (single polyhedron only)              */
/*       C : Context domain                                             */
/*       MAXRAYS : Workspace size                                       */
/* Returns a linked list of scan domains, outer loop first              */
/*----------------------------------------------------------------------*/
Polyhedron *Polyhedron_Scan(D, C, MAXRAYS)
Polyhedron *D, *C;
unsigned MAXRAYS;
{
   int i, j, dim, dim2,  numP;
   Matrix *M;
   Polyhedron *C1, *C2, *D1;
   Polyhedron *res, *last, *tmp;

   numP = C->Dimension;
   dim2 = D->Dimension;
   dim = D->Dimension - numP;
   res = last = (Polyhedron *) 0;
   if (dim==0) return (Polyhedron *)0;
   M   = Matrix_Alloc(dim2, dim2+2);
	if(!M)
	{	errormsg1("Polyhedron_Scan", "outofmem", "out of memory space");
		return 0;
	}
   C1  = align_context(C, dim2, MAXRAYS);
	if(!C1)
	{	return 0;
	}
   for (i=0; i<dim; i++)
   {  Vector_Init(M->p_Init, dim2*(dim2+2));
      for (j=i+1; j<dim; j++) {  value_assign(M->p[j-i-1][j+1], VALUE_ONE ); }
      M->NbRows = dim-i-1;
      D1 = M->NbRows ? DomainAddRays(D, M, MAXRAYS) : D;
      tmp = DomainSimplify(D1, C1, MAXRAYS);
      if (!last) res = last = tmp;
      else { last->next = tmp; last = tmp; }
      C2 = DomainIntersection(C1, D1, MAXRAYS);
      if (i>0) Polyhedron_Free(C1);
      C1 = C2;
      if (M->NbRows) Polyhedron_Free(D1);
   }
   Matrix_Free(M);
   return res;
}


/*---------------------------------------------------------------------*/
/* int lower_upper_bounds(pos,P,context,LBp,UBp)                       */
/*    pos : index position of current loop index (1..hdim-1)           */
/*    P: loop domain                                                   */
/*    context : context values for fixed indices                       */
/*    LBp, UBp : pointers to resulting bounds                          */
/* returns the flag = (UB_INFINITY, LB_INFINITY)                       */
/*---------------------------------------------------------------------*/
int lower_upper_bounds(pos, P, context, LBp, UBp)
Polyhedron *P;
int pos;
Value *context, *LBp, *UBp;
{
   Value LB, UB;
   int flag, i;
	Value n, n1, d;

	value_assign( LB, VALUE_ZERO );
	value_assign( UB, VALUE_ZERO );

   /* Compute Upper Bound and Lower Bound for current loop */
   flag = LB_INFINITY | UB_INFINITY;
   for (i=0; i<P->NbConstraints; i++)
   {
	   value_assign( d, P->Constraint[i][pos] );
      if ( value_zero_p(d) ) continue;

		value_assign( n, value_uminus(
			Inner_Product(&context[1],&(P->Constraint[i][1]), P->Dimension+1 )));

      /*---------------------------------------------------*/
      /* Compute n/d        n/d<0              n/d>0       */
      /*---------------------------------------------------*/
      /*  n%d == 0    floor   = n/d      floor   = n/d     */
      /*              ceiling = n/d      ceiling = n/d     */
      /*---------------------------------------------------*/
      /*  n%d != 0    floor   = n/d - 1  floor   = n/d     */
      /*              ceiling = n/d      ceiling = n/d + 1 */
      /*---------------------------------------------------*/

      /* check to see if constraint is inequality */
      /* if constraint is equality, both upper and lower bounds are fixed */
      if( value_zero_p(P->Constraint[i][0]) )	/* Equality */
      {
         /* if not integer, return 0; */
         if( value_notzero_p( value_mod(n,d) ) )
         {
         	value_assign( *LBp, VALUE_ONE );
         	value_assign( *UBp, VALUE_ZERO );	/* empty loop */
         	return 0;
         }
         value_assign( n1, value_div(n,d) );
         /* Upper and Lower bounds found */
         if( (flag&LB_INFINITY) || value_gt(n1,LB) )
				value_assign( LB, n1 );
         if( (flag&UB_INFINITY) || value_lt(n1,UB) )
				value_assign( UB, n1 );
         flag = 0;
      }

      if ( value_pos_p(d) )   /* Lower Bound */
      {  /* n1 = ceiling(n/d) */
         if ( value_pos_p(n) && value_notzero_p( value_mod(n,d) ) )
			{
				value_assign( n1, value_div(n,d) );
				value_addto( n1, VALUE_ONE );
			}
			else
				value_assign( n1, value_div(n,d) );
         if (flag&LB_INFINITY)
			{ value_assign(LB, n1); flag^=LB_INFINITY; }
         else if ( value_gt(n1,LB) )
				value_assign( LB, n1 );
      }

      if ( value_neg_p(d) )   /* Upper Bound */
      {  /* n1 = floor(n/d) */
         if ( value_pos_p(n) && value_notzero_p(value_mod(n,d)) )
			{
				value_assign( n1, value_div(n,d) );
				value_addto( n1, VALUE_MONE );
			}
			else
				value_assign( n1, value_div(n,d) );

         if (flag&UB_INFINITY)
			{	value_assign( UB, n1 ); flag^=UB_INFINITY; }
         else if ( value_lt(n1,UB) )
				value_assign( UB, n1 );
      }
   }

   if ((flag & LB_INFINITY)==0) value_assign( *LBp, LB );
   if ((flag & UB_INFINITY)==0) value_assign( *UBp, UB );
   return flag;
}




Polyhedron *Polyhedron_Preimage(Pol1, Func, NbMaxRays)
Polyhedron *Pol1;
Matrix     *Func;
unsigned   NbMaxRays;
{
  Matrix *Constraints;
  Polyhedron *Pol2;
  unsigned NbConstraints, Dimension1, Dimension2;
  int i, j, k;
  Value Sum;

  NbConstraints = Pol1->NbConstraints;
  Dimension1    = Pol1->Dimension+1;	/* homogeneous Dimension */
  Dimension2    = Func->NbColumns;	/* homogeneous Dimension */
  if (Dimension1!=(Func->NbRows))
  {   errormsg1("Polyhedron_Preimage", "dimincomp", "incompatable dimensions");
      return Empty_Polyhedron(Dimension2-1);
  }
/*      Dim1           Dim2            Dim2
	__k__          __j__           __j__	
  NbCon |   |  X   Dim1|   |  =  NbCon |   |
    i   |___|       k  |___|       i   |___|
     Pol1->Constraints Function        Constraints
*/
  Constraints = Matrix_Alloc(NbConstraints, Dimension2+1);
  if (!Constraints)
  {  errormsg1("Polyhedron_Preimage", "outofmem", "out of memory space\n");
     Pol_status = 1;
     return 0;
  }
  for (i=0; i<NbConstraints; i++)
  {
    Constraints->p[i][0] = Pol1->Constraint[i][0];
    for (j=0; j<Dimension2; j++)
	 {
      value_assign( Sum, VALUE_ZERO );
      for (k=0; k<Dimension1; k++)
        /* Sum+=Pol1->Constraint[i][k+1] * Func->p[k][j]; */
        value_addto( Sum, value_mult(Pol1->Constraint[i][k+1], Func->p[k][j]) );
      		/* k = Dimension1-1 :
		Perform as though last row of Func where [ 0 0 ... 0 1 ] */
		/*if(j==(Dimension2-1))Sum+=Pol1->Constraint[i][Dimension1];*/
      value_assign( Constraints->p[i][j+1], Sum );
    }
  }
  Pol2 = Constraints2Polyhedron(Constraints, NbMaxRays);
  Matrix_Free(Constraints);
  return Pol2;
} /* Polyhedron_Preimage */

Polyhedron *DomainPreimage(Pol1, Func, NbMaxRays)
Polyhedron *Pol1;
Matrix     *Func;
unsigned   NbMaxRays;
{
	Polyhedron *p, *p1, *d;

	if (!Pol1 || !Func) return (Polyhedron *) 0;
	d = (Polyhedron *) 0;
	for (p=Pol1; p; p=p->next)
	{
		p1 = Polyhedron_Preimage(p, Func, NbMaxRays);
		d = AddPolyToDomain (p1, d);
	} 
	return d;
}

Polyhedron *Polyhedron_Image(Pol1, Func, NbMaxRays)
     Polyhedron *Pol1;
     Matrix     *Func;
     unsigned   NbMaxRays;
{
  Matrix *Rays;
  Polyhedron *Pol2;
  unsigned NbRays, Dimension1, Dimension2;
  int i, j, k;
  Value Sum;

  NbRays     = Pol1->NbRays;
  Dimension1 = Pol1->Dimension+1;	/* homogeneous Dimension */
  Dimension2 = Func->NbRows;		/* homogeneous Dimension */
  if (Dimension1!=Func->NbColumns)
  {   errormsg1("Polyhedron_Image", "dimincomp", "incompatable dimensions");
      return Empty_Polyhedron(Dimension2-1);
  }
/*      Dim1     /      Dim1  \Transpose      Dim2
        __k__    |      __k__ |              __j__
  NbRays|   |  X | Dim2 |   | |     =  NbRays|   |
    i   |___|    |   j  |___| |          i   |___|
     Pol1->Rays  \       Func /               Rays
*/
  Rays = Matrix_Alloc(NbRays, Dimension2+1);
  if (!Rays)
  {  errormsg1("Polyhedron_Image", "outofmem", "out of memory space\n");
     return 0;
  }
  for (i=0; i<NbRays; i++)
  {
    Rays->p[i][0] = Pol1->Ray[i][0];
    for (j=0; j<Dimension2; j++)
	 {
      value_assign( Sum, VALUE_ZERO );
      for (k=0; k<Dimension1; k++)
        /* Sum+=Pol1->Ray[i][k+1] * Func->p[j][k]; */
        value_addto( Sum, value_mult(Pol1->Ray[i][k+1], Func->p[j][k]) );
      value_assign( Rays->p[i][j+1], Sum );
    }
	/* j = Dimension2 :
	Perform as though last row of Func where [ 0 0 ... 0 1 ] */
	/* Rays->p[i][Dimension2+1]=Pol1->Ray[i][Dimension1];*/
  }
  Pol2 = Rays2Polyhedron(Rays, NbMaxRays);
  Matrix_Free(Rays);
  return Pol2;
} /* Polyhedron_Image */

Polyhedron *DomainImage(Pol1, Func, NbMaxRays)
     Polyhedron *Pol1;
     Matrix     *Func;
     unsigned   NbMaxRays;
{
    Polyhedron *p, *p1, *d;

    if (!Pol1 || !Func) return (Polyhedron *) 0;
    d = (Polyhedron *) 0;
    for (p=Pol1; p; p=p->next)
    {   p1 = Polyhedron_Image(p, Func, NbMaxRays);
        d = AddPolyToDomain (p1, d);
    }
    return d;
}

Interval *DomainCost(Pol1, Cost)
Polyhedron *Pol1;
Value *Cost;
{
	int i, j, NbRay, Dim;
	Value *p1, *p2, p3, d, status;
	Value **Ray;
	Interval *I;

    Ray = Pol1->Ray;
    NbRay = Pol1->NbRays;
    Dim = Pol1->Dimension+1;		/* homogenous dimension */
    I = (Interval *) malloc (sizeof(Interval));
    if (!I)
    {  errormsg1("DomainCost", "outofmem", "out of memory space\n");
       return 0;
    }
    value_assign( I->MaxN, VALUE_MONE );
    value_assign( I->MaxD, VALUE_ZERO );		/* actual cost is Max/MaxD */
    I->MaxI = -1;
    value_assign( I->MinN, VALUE_ONE );
    value_assign( I->MinD, VALUE_ZERO );
    I->MinI = -1;

    /* Compute the cost of each ray[i] */
    for (i=0; i<NbRay; i++)
    { p1 = Ray[i];
		value_assign( status, *p1 );
		p1++;
      p2 = Cost;
		/* p3 = *p1++ * *p2++; */
      value_assign(p3, value_mult(*p1, *p2) );
		p1++; p2++;
      for (j=1; j<Dim; j++)
		{	/* p3 += *p1++ * *p2++; */
			value_addto( p3, value_mult(*p1, *p2) );
			p1++; p2++;
      }
		/* d = *--p1; */
		p1--;
		value_assign( d, *p1 );

      /* Test p3/d, MaxN/MaxD, MinN/MinD */
      if ( I->MaxI==-1 ||
           value_gt( value_mult(p3,I->MaxD), value_mult(I->MaxN,d) ) ||
           ( value_eq( value_mult(p3,I->MaxD), value_mult(I->MaxN,d) ) &&
			       value_eq(d,VALUE_ONE) && value_ne(I->MaxD,VALUE_ONE) ) )
      {
			value_assign(I->MaxN, p3);
			value_assign(I->MaxD, d);
			I->MaxI = i;
		}

      if ( I->MinI==-1 ||
           value_lt( value_mult(p3,I->MinD), value_mult(I->MinN,d) ) ||
           ( value_eq( value_mult(p3,I->MinD), value_mult(I->MinN,d) ) &&
			       value_eq(d,VALUE_ONE) && value_ne(I->MinD,VALUE_ONE) ) )
      {
			value_assign(I->MinN, p3);
			value_assign(I->MinD, d);
			I->MinI = i;
      }

      if ( value_zero_p(status) ) /* line , d is 0 */
      { if ( value_lt( value_mult(p3,I->MaxD), VALUE_ZERO ) )
        {
          value_assign( I->MaxN, value_uminus(p3) );
          value_assign( I->MaxD, VALUE_ZERO );
          I->MaxI = i;
        }
        if ( value_gt( value_mult(p3,I->MinD), VALUE_ZERO ) )
        {
          value_assign( I->MinN, value_uminus(p3) );
          value_assign( I->MinD, VALUE_ZERO );
          I->MinI = i;
        }
      }
    }
    return I;
}

Polyhedron *DomainAddConstraints(Pol1, Mat2, NbMaxRays)
     Polyhedron *Pol1;
     Matrix *Mat2;
     unsigned   NbMaxRays;
{
	Polyhedron *PolA, *PolEndA, *p1, *p2, *p3;
	int Redundant;

	if (!Pol1 ) return (Polyhedron*) 0;
	if (!Mat2 ) return Pol1;
	if (Pol1->Dimension != Mat2->NbColumns-2)
	{
		errormsg1( "DomainAddConstraints",
		            "diffdim", "operation on different dimensions");
		return (Polyhedron*) 0;
    }

	/* copy Pol1 to PolA */
	PolA = PolEndA = (Polyhedron *)0;
	for (p1=Pol1; p1; p1=p1->next)
	{
		p3 = AddConstraints(Mat2->p_Init, Mat2->NbRows, p1, NbMaxRays);
		/* does any component of PolA cover it ? */
		Redundant = 0;
		for (p2=PolA; p2; p2=p2->next)
		{
			if ( PolyhedronIncludes(p2, p1) ) /* p2 covers p1 */
			{
				Redundant = 1;
				break;
			}
		}
		if (!Redundant)
		{   /* add p1 to PolA */
			if (!PolEndA)
				PolEndA = PolA = p3;
			else
			{
				PolEndA->next = p3;
				PolEndA = PolEndA->next;
			}
		}
	}
	return PolA;
} /* DomainAddConstraints */


