/***********************************************************************/
/*                Ehrhart V4.00                                        */
/*                copyright 1997, Doran Wilde                          */
/*                copyright 1997, 1998, Vincent Loechner               */
/*       Permission is granted to copy, use, and distribute            */
/*       for any commercial or noncommercial purpose under the terms   */
/*       of the GNU General Public license, version 2, June 1991       */
/*       (see file : LICENSING).                                       */
/***********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>

/* #define EDEBUG 	*/		/* normal debug */
/* #define EDEBUG1	*/		/* prints enumeration points */
/* #define EDEBUG2	*/		/* prints domains */
/* #define EDEBUG3	*/		/* prints systems of equations that are solved */
/* #define EDEBUG4	*/		/* prints message for degree reduction */
/* #define EDEBUG5	*/		/* prints result before simplification */

/* Reduce the degree of resulting polynomials */
#define REDUCE_DEGREE

#include "arithmetique.h"
#include "polylib.h"
#include "ehrhart.h"

#ifdef EP_EVALUATION
#include "eval_ehrhart.h"
#endif

#ifndef ALL_OVERFLOW_WARNINGS
int overflow_warning_flag = 1;
#endif


/*-------------------------------------------------------------------*/
/* EHRHART POLYNOMIAL SYMBOLIC ALGEBRA SYSTEM                        */
/*-------------------------------------------------------------------*/ 
/*------------------------------------------------------------*/
/* enode *new_enode(type, size, pos)                          */
/*      type : enode type                                     */
/*      size : degree+1 for polynomial, period for periodic   */
/*      pos  : 1..nb_param, position of parameter             */
/* returns a newly allocated enode                            */
/* can be freed with a simple free(x)                         */
/*------------------------------------------------------------*/
enode *new_enode(type,size,pos)
enode_type type;
int size, pos;
{  enode *res;
   int i;
	
	if( size == 0 )
	{
		fprintf(stderr, "Allocating enode of size 0 !\n" );
		return NULL;
	}
   res = (enode *) malloc( sizeof(enode) + (size-1)*sizeof(evalue) );
   res->type = type;
   res->size = size;
   res->pos = pos;
   for (i=0; i<size; i++)
   {
	   value_assign( res->arr[i].d, VALUE_ZERO );
      value_assign( res->arr[i].x.n, VALUE_ZERO );
   }
   return res;
}

/*------------------------------------------------------------*/
/* free_evalue_refs(e)                                        */
/*     e : pointer to an evalue                               */
/* releases all memory referenced by e.  (recursive)          */
/*------------------------------------------------------------*/
void free_evalue_refs(e)
evalue *e;
{   enode *p;
    int i;

    if ( value_notzero_p(e->d) ) return;  /* its a constant */
    p = e->x.p;
    if (!p) return;	/* null pointer */
    for (i=0; i<p->size; i++)
    {  free_evalue_refs(&(p->arr[i]));
    }
    free(p);
}

/*------------------------------------------------------------*/
/* evalue_copy(e)                                             */
/*     e : pointer to an evalue                               */
/*------------------------------------------------------------*/
enode *ecopy(e)
enode *e;
{
	enode *res;
	int i;

	res = new_enode( e->type, e->size, e->pos );
	for( i=0 ; i<e->size ; ++i )
	{
		value_assign( res->arr[i].d, e->arr[i].d );
		if( value_zero_p(res->arr[i].d) )
			res->arr[i].x.p = ecopy( e->arr[i].x.p );
		else
			value_assign( res->arr[i].x.n, e->arr[i].x.n );
	}
	return( res );
}

/*------------------------------------------------------------*/
/* print_evalue(DST, e)                                       */
/*    DST : destination file                                  */
/*    e : pointer to evalue to be printed                     */
/* Returns nothing, prints the evalue to DST                  */
/*------------------------------------------------------------*/
void print_evalue(DST, e)
FILE *DST;
evalue *e;
{  
	if( value_notzero_p(e->d) )
	{
		if( value_notone_p(e->d) )
			fprintf(DST, VALUE_FMT"/"VALUE_FMT, e->x.n, e->d);
		else
			fprintf(DST, VALUE_FMT, e->x.n);
	}
	else
		print_enode(DST, e->x.p);
}

/*------------------------------------------------------------*/
/* print_enode(DST, p )                                       */
/*    DST : destination file                                  */
/*    p : pointer to enode  to be printed                     */
/* Returns nothing, prints the enode  to DST                  */
/*------------------------------------------------------------*/
void print_enode(DST, p)
FILE *DST;
enode *p;
{   int i;

    if (!p)
    {   fprintf(DST, "NULL");
        return;
    }
    if (p->type == evector)
    {   fprintf(DST, "{ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " }\n");
    }
    else if (p->type == polynomial)
    {   fprintf(DST, "( ");
        for (i=p->size-1; i>=0; i--)
        {  print_evalue(DST, &p->arr[i]);
           if (i==1) fprintf(DST, " * %c + ", PCHAR+p->pos);
           else if ( i>1 ) fprintf(DST, " * %c^%d + ", PCHAR+p->pos, i);
        }
        fprintf(DST, " )\n");
    }
    else if (p->type == periodic)
    {   fprintf(DST, "[ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " ]_%c", PCHAR+p->pos);
    }
}

/*------------------------------------------------------------*/
/* int eequal (e1, e2)                                        */
/*      e1, e2 : pointers to evalues                          */
/* returns 1 (true) if they are equal, 0 (false) if not       */
/*------------------------------------------------------------*/
static int eequal(e1, e2)
evalue *e1, *e2;
{  int i;
   enode *p1, *p2;

	if ( value_ne( e1->d, e2->d) )
		return 0;
   /* e1->d == e2->d */
   if ( value_notzero_p(e1->d) )
   {
	   if ( value_ne( e1->x.n, e2->x.n) )
			return 0;
      /* e1->d == e2->d != 0  AND e1->n == e2->n */
      return 1;
   }
   /* e1->d == e2->d == 0 */
   p1 = e1->x.p;
   p2 = e2->x.p;
   if (p1->type != p2->type) return 0;
   if (p1->size != p2->size) return 0;
   if (p1->pos  != p2->pos ) return 0;
   for (i=0; i<p1->size; i++)
      if (!eequal(&p1->arr[i], &p2->arr[i]) ) return 0;
   return 1;
}

/*------------------------------------------------------------*/
/* void reduce_evalue ( e )                                   */
/*       e : pointer to an evalue                             */
/*------------------------------------------------------------*/
static void reduce_evalue ( e )
evalue *e;
{  enode *p;
   int i, j, k;

	if ( value_notzero_p( e->d ) )
		return;	/* a rational number, its already reduced */
	p = e->x.p;

	/* first reduce the components of p */
	for (i=0; i<p->size; i++)
		reduce_evalue(&p->arr[i]);

   if (p->type==periodic)
   {  /* try to reduce the period */
      for (i=1; i<=(p->size)/2; i++)
      {  if ( (p->size % i)==0 )
         {  /* can we reduce the size to i ? */
            for (j=0; j<i; j++)
               for (k=j+i; k<e->x.p->size; k+=i)
                  if (!eequal(&p->arr[j], &p->arr[k]) ) goto you_lose;
            /* OK, lets do it */
            for (j=i; j<p->size; j++) free_evalue_refs(&p->arr[j]);
            p->size = i;
            break;
you_lose:   /* OK, lets not do it */
            continue;
         }
      }

      /* try to reduce its strength */
      if (p->size == 1)
      {
			memcpy( e, &p->arr[0], sizeof(evalue) );
         free(p);
      }
   }
   else if (p->type==polynomial)
   {
	   /* try to reduce the degree */
      for (i=p->size-1; i>=0; i--)
      {
		   if ( value_one_p(p->arr[i].d) && value_zero_p(p->arr[i].x.n) )
			  /* zero coefficient */
            continue;
         else
            break;
      }
      if (i==-1) p->size = 1;
      else if (i+1<p->size) p->size = i+1;

      /* try to reduce its strength */
      if (p->size == 1)
      {
			memcpy( e, &p->arr[0], sizeof(evalue) );
         free(p);
      }
   }
}

/*------------------------------------------------------------*/
/* void emul (e1, e2, res)                                    */
/*       e1 : pointer to an evalue                            */
/*       e2 : pointer to a constant evalue                    */
/*       res : pointer to result evalue = e1 * e2             */
/* multiplies two evalues and puts the result in res          */
/*------------------------------------------------------------*/
static void emul (e1, e2, res)
evalue *e1, *e2, *res;
{  enode *p;
   int i;
	Value g;

   if ( value_zero_p(e2->d) )
   {  fprintf(stderr, "emul: ?expecting constant value\n");
      return;
   }
   if ( value_notzero_p(e1->d) )
   {  /* product of two rational numbers */
      value_assign( res->d, value_mult(e1->d,e2->d) );
      value_assign( res->x.n, value_mult( e1->x.n, e2->x.n ) );
      value_assign( g, Gcd(res->x.n, res->d) );
      if ( value_notone_p(g) )
      {
		   value_division( res->d, g );
         value_division( res->x.n, g );
      }
   }
   else /* e1 is an expression */
   {
	   value_assign( res->d, VALUE_ZERO );
      p = e1->x.p;
      res->x.p = new_enode(p->type, p->size, p->pos);
      for (i=0; i<p->size; i++)
      {  emul(&p->arr[i], e2, &(res->x.p->arr[i]) ); }
   }
}

/*------------------------------------------------------------*/
/* void eadd (e1, res)                                        */
/*       e1 : an evalue                                       */
/*       res : result = res + e1                              */
/* adds one evalue to evalue 'res'                            */
/*------------------------------------------------------------*/
void eadd(e1, res)
evalue *e1, *res;
{  int i;
	Value g, m1,m2;

   if ( value_notzero_p(e1->d) && value_notzero_p(res->d) )
   {  /* add two rational numbers*/
      value_assign( m1, value_mult(e1->x.n, res->d) );
		value_assign( m2, value_mult(res->x.n, e1->d) );
		value_assign( res->x.n, value_plus(m1, m2) );
      value_assign( res->d, value_mult(e1->d, res->d) );
      value_assign( g, Gcd(res->x.n, res->d) );
      if ( value_notone_p(g) )
      {  
			value_division( res->d, g );
         value_division( res->x.n, g );
      }
      return;
   }
   else if ( value_notzero_p(e1->d) && value_zero_p(res->d) )
   {  if (res->x.p->type==polynomial)
      {  /* add the constant to the constant term */
         eadd(e1, &res->x.p->arr[0]);
         return;
      }
      else if (res->x.p->type==periodic)
      {  /* add the constant to all elements of periodic number */
         for (i=0; i<res->x.p->size; i++)
         {  eadd(e1, &res->x.p->arr[i]);
         }
         return;
      }
      else
      {  fprintf(stderr, "eadd: cannot add const with vector\n");
         return;
      }
   }
   else if ( value_zero_p(e1->d) && value_notzero_p(res->d) )
   {
		fprintf(stderr, "eadd: cannot add evalue to const\n");
		return;
   }
   else /* ((e1->d==0) && (res->d==0)) */
   {
		if ( (e1->x.p->type != res->x.p->type) ||
				(e1->x.p->pos  != res->x.p->pos ) )
      {  fprintf(stderr, "eadd: ?cannot add, incompatible types\n");
         return;
      }
      if (e1->x.p->size == res->x.p->size)
      {  for (i=0; i<res->x.p->size; i++)
         {  eadd(&e1->x.p->arr[i], &res->x.p->arr[i]);
         }
         return;
      }
      /* sizes are different*/
      if (res->x.p->type==polynomial)
      {
			/* VIN100: if e1-size > res-size you have to copy e1 in a
				new enode and add res to that new node. If you do not do
				that, you lose the the upper weight part of e1 ! */
			if( e1->x.p->size > res->x.p->size )
			{
				enode *tmp;
				tmp = ecopy( e1->x.p );
				for( i=0 ; i<res->x.p->size ; ++i )
				{	eadd(&res->x.p->arr[i], &tmp->arr[i]);
					free_evalue_refs(&res->x.p->arr[i]);
				}
				res->x.p = tmp;
			}
			else
			{
				for (i=0; i<e1->x.p->size ; i++)
				{  eadd(&e1->x.p->arr[i], &res->x.p->arr[i]);
				}
				return;
			}
		}
      else if (res->x.p->type==periodic)
      {  fprintf(stderr, "eadd: ?addition of different sized periodic nos\n");
         return;
      }
      else /* evector */
      {  fprintf(stderr, "eadd: ?cannot add vectors of different length\n");
         return;
      }
   }
}

/*------------------------------------------------------------*/
/* void edot (v1, v2, res)                                    */
/*       v1 : an enode (vector)                               */
/*       v2 : an enode (vector of constants)                  */
/*       res : result (evalue) = v1.v2 (dot product)          */
/* computes the inner product of two vectors. Result = res    */
/*------------------------------------------------------------*/
void edot(v1, v2, res)
enode *v1, *v2;
evalue *res;
{  int i;
   evalue tmp;

   if ((v1->type != evector) || (v2->type != evector))
   {  fprintf(stderr, "edot: ?expecting vectors\n");
      return;
   }
   if (v1->size != v2->size)
   {  fprintf(stderr, "edot: ? vector lengths do not agree\n");
      return;
   }
   if (v1->size<=0)
   {
	   value_assign( res->d, VALUE_ONE );	/* set result to 0/1 */
      value_assign( res->x.n, VALUE_ZERO );
      return;
   }

   /* vector v2 is expected to have only rational numbers in */
   /* the array.  No pointers. */

   emul(&v1->arr[0], &v2->arr[0], res);
   for (i=1; i<v1->size; i++)
   {  /* res = res + v1[i]*v2[i] */
      emul(&v1->arr[i], &v2->arr[i], &tmp);
      eadd(&tmp, res);
      free_evalue_refs(&tmp);
   }
}


/*--------------------------------------------------------------------*/
/* This procedure adds additional constraints to D so that as         */
/* each parameter is scanned, it will have a minimum of 'size' points */
/* If this is not possible, an empty polyhedron is returned           */
/*--------------------------------------------------------------------*/
Polyhedron *Polyhedron_Preprocess(D, size, MAXRAYS)
Polyhedron *D;
Value size;
unsigned MAXRAYS;
{  int p, p1, ub, lb;
	Value a, a1, b, b1, g, aa;
   int dim, con, new, needed;
   Value **C;
	Value *temp;
   Matrix *M;
   Polyhedron *D1;


   dim = D->Dimension;
   temp=(Value *)malloc((dim+1)*sizeof(Value));
   con = D->NbConstraints;
   M = Matrix_Alloc(MAXRAYS, dim+2);
   new = 0;
   C = D->Constraint;
   for (p=1; p<=dim; p++)
   {
      for (ub=0; ub<con; ub++)
      {
			value_assign( a, C[ub][p] );
		   if ( value_posz_p(a) ) continue;		/* not an upper bound */
         for (lb=0; lb<con; lb++)
         {
				value_assign( b, C[lb][p] );
			   if ( value_negz_p(b) ) continue;	/* not a lower bound */
            /* check if a new constraint is needed for this (ub,lb) pair */
            /* a constraint is only needed if a previously scanned */
            /* parameter (1..p-1) constrains this parameter (p) */
            needed=0;
            for (p1=1; p1<p; p1++)
            {  if ( value_notzero_p(C[ub][p1]) || value_notzero_p(C[lb][p1]) )
					{
						needed=1; break;
					}
            }
            if (!needed) continue;

            /* create new constraint: b*UB-a*LB >= a*b*size */
            value_assign( g, Gcd( value_abs(a), value_abs(b)) );
            value_assign( a1, value_div(a,g) );
            value_assign( b1, value_div(b,g) );
            value_assign( M->p[new][0], VALUE_ONE );
            Vector_Combine(&(C[ub][1]), &(C[lb][1]), &(M->p[new][1]),
                           b1,value_uminus(a1) ,dim+1);
				value_assign( aa, value_mult(a1,b1) );
				value_addto( M->p[new][dim+1], value_mult(aa,size) );
            Vector_Normalize(&(M->p[new][1]), temp, dim+1);
            new++;
         }
      }
   }
   D1 = AddConstraints(M->p_Init, new, D, MAXRAYS);
   Matrix_Free(M);
   free(temp);
   return D1;
}

/*---------------------------------------------------------------------*/
/* PROCEDURES TO COMPUTE ENUMERATION                                   */
/*---------------------------------------------------------------------*/
/* int count_points(pos,P,context)                                     */
/*    pos : index position of current loop index (0..hdim-1)           */
/*    P: loop domain                                                   */
/*    context : context values for fixed indices                       */
/* recursive procedure, recurs for each imbriquation                   */
/* returns the number of integer points in this polyhedron             */
/*---------------------------------------------------------------------*/
int count_points (pos, P, context)
int pos;
Polyhedron *P;
Value *context;
{
   Value LB, UB, k;
	int CNT;

	value_assign( LB, VALUE_ZERO );
	value_assign( UB, VALUE_ZERO );

   if ( lower_upper_bounds(pos, P, context, &LB, &UB) !=0 )
   /* Problem if UB or LB is INFINITY */
   {  fprintf(stderr, "count_points: ? infinite domain\n");
      return -1;
   }

#ifdef EDEBUG1
   if (!P->next)
   {  int i;
      for ( value_assign(k,LB); value_le(k,UB); value_increment(k) )
      {  fprintf(stderr, "(");
         for (i=1; i<pos; i++)
         {  fprintf(stderr, P_VALUE_FMT",", context[i]);
         }
         fprintf(stderr, P_VALUE_FMT")\n", k);
      }
   }
#endif

   value_assign( context[pos], VALUE_ZERO );
   if( value_lt(UB,LB) )
		return 0;
   if (!P->next)
	{
		Value m;
		value_assign( m, value_minus(UB,LB) );
		value_addto(m,VALUE_ONE);
		return ( VALUE_TO_INT(m) );
	}

   /*-----------------------------------------------------------------*/
   /* Optimization idea                                               */
   /*   If inner loops are not a function of k (the current index)    */
   /*   i.e. if P->Constraint[i][pos]==0 for all P following this and */
   /*        for all i,                                               */
   /*   Then CNT = (UB-LB+1)*count_points(pos+1, P->next, context)    */
   /*   (skip the for loop)                                           */
   /*-----------------------------------------------------------------*/

   CNT = 0;
   for ( value_assign(k,LB) ; value_le(k,UB) ; value_increment(k) )
   {
		int c;
      /* insert k in context */
      value_assign( context[pos], k );
      c = count_points( pos+1, P->next, context);
		if( c!=-1 )
      	CNT = CNT + c;
		else
		{
			CNT=0;
			break;
		}
   }

   /* reset context */
   value_assign( context[pos], VALUE_ZERO );
   return CNT;
}

/*------------------------------------------------------------*/
/* enode *P_Enum(L, LQ, context, pos, nb_param, dim, lcm)     */
/*     L : list of polyhedra for the loop nest                */
/*     LQ : list of polyhedra for the parameter loop nest     */
/*     pos : 1..nb_param, position of the parameter           */
/*     nb_param : number of parameters total                  */
/*     dim : total dimension of the polyhedron, param incl.   */
/*     lcm : the denominator of the polyhedron                */
/* Returns an enode tree representing the pseudo polynomial   */
/*   expression for the enumeration of the polyhedron.        */
/* A recursive procedure.                                     */
/*------------------------------------------------------------*/
static enode *P_Enum(L, LQ, context, pos, nb_param, dim, lcm)
Polyhedron *L, *LQ;
Value *context;
int pos, nb_param, dim;
Value lcm;
{  enode *res, *B, *C;
   int hdim, i, j, rank, flag;
	Value n, g, nLB, nUB, nlcm, noff, nexp, k1, nm, hdv, k;
   Value *temp;
   Matrix *A;

#ifdef EDEBUG
fprintf(stderr, "---------------------- begin P_Enum ---------------------\n");
fprintf(stderr, "Calling P_Enum with pos = %d\n", pos);
#endif

   /* Xavier Redon modification: handle the case when there is no parameter */
	if(nb_param==0)
	{
		res=new_enode(polynomial,1,0);
		value_assign( res->arr[0].d, VALUE_ONE );
		value_assign( res->arr[0].x.n, int_to_value(count_points(1,L,context)) );
		return res;
	}

   /* hdim is the degree of the polynomial + 1 */
   hdim = dim-nb_param+1;		/* homogenous dim w/o parameters */

   /* code to limit generation of equations to valid parameters only */
   /*----------------------------------------------------------------*/
   flag = lower_upper_bounds(pos, LQ, &context[dim-nb_param], &nLB, &nUB);
   if (flag & LB_INFINITY)
   {  if (!(flag & UB_INFINITY))
      {  /* compute nLB such that (nUB-nLB+1) >= (hdim*lcm) */
			value_assign( nLB, value_plus( nUB, VALUE_ONE ) );
			value_assign( hdv, int_to_value((int)hdim) );
			value_substract( nLB, value_mult( hdv,lcm) );
         /* only an upper limit: set lower limit */
			if( value_pos_p(nLB) )
				value_assign( nLB, VALUE_ZERO );
      }
      else
			value_assign( nLB, VALUE_ZERO );
			/* no upper nor lower limit: set lower limit to 0 */
   }

   /* if (nUB-nLB+1) < (hdim*lcm) then we have more unknowns than equations */
   /* We can: 1. Find more equations by changing the context parameters, or */
   /* 2. Assign extra unknowns values in such a way as to simplify result. */
   /* Think about ways to scan parameter space to get as much info out
      of it as possible */

#ifdef REDUCE_DEGREE
   if ( pos==1 && (flag & UB_INFINITY)==0 )
   /* only for finite upper bound on first parameter */
   /* NOTE: only first parameter because subsequent parameters may
            be artificially limited by the choice of the first parameter */
   {
		value_assign( n, value_minus(nUB,nLB) );
		value_increment( n );
		/* total number of samples>0 */
		if( value_neg_p(n) )
			i=0;
		else 
      {
			Value vi;
			if( value_notzero_p( value_mod(n,lcm) ) )
			{
				value_assign( vi, value_div(n,lcm ) );
				value_increment( vi );
				i = VALUE_TO_INT( vi );
			}
			else
			{
				value_assign( vi, value_div(n,lcm) );
				i =   VALUE_TO_INT( vi );	/* ceiling of n/lcm */
			}
		}
      /* reduce degree of polynomial based on number of sample points */
      if (i < hdim)
      {  hdim=i;
#ifdef EDEBUG4
fprintf(stdout, "Parameter #%d: LB="VALUE_FMT" UB="VALUE_FMT" lcm="VALUE_FMT" degree reduced to %d\n", pos, nLB, nUB, lcm, hdim-1);
#endif
      }
   }
#endif /* REDUCE_DEGREE */
   /* hdim is now set */

   /* allocate result structure */
   res = new_enode(polynomial, hdim, pos);
   for (i=0; i<hdim; i++)
   {
		int l;
		l = VALUE_TO_INT(lcm);
		res->arr[i].x.p = new_enode(periodic, l, pos);
   }

   /* utility arrays */
   temp=(Value *)malloc((2*hdim+2)*sizeof(Value));	/* used for Gauss */
   A = Matrix_Alloc(hdim, 2*hdim+1);		/* used for Gauss */
   B = new_enode(evector, hdim, 0);
   C = new_enode(evector, hdim, 0);

   /*----------------------------------------------------------------*/
   /*                                                                */
   /*                                0<-----+---k--------->          */
   /* |---------noff----------------->-nlcm->-------lcm---->         */
   /* |--- . . . -----|--------------|------+-------|------+-------|-*/
   /* 0          (q-1)*lcm         q*lcm    |  (q+1)*lcm   |         */
   /*                                      nLB          nLB+lcm      */
   /*                                                                */
   /*----------------------------------------------------------------*/
	if( value_neg_p(nLB) )
	{
		value_assign( nlcm, value_mod(nLB,lcm) );
		value_addto( nlcm, lcm );
	}
	else
	{
		value_assign( nlcm, value_mod(nLB,lcm) );
	}
	/* noff is a multiple of lcm */
	value_assign( noff, value_minus(nLB, nlcm) );
   for ( value_assign(k,nlcm) ; value_lt(k,value_plus(lcm,nlcm)) ;
				value_increment(k) )
   { 
#ifdef EDEBUG
fprintf(stderr,
 "Finding "VALUE_FMT"-th elements of periodic coefficients\n", k);
#endif
		value_assign( hdv, int_to_value(hdim) );
		value_assign( nm, value_mult(hdv,lcm) );
		value_addto( nm, nLB );
		i=0;
      for ( value_assign(n,value_plus(k,noff)) ; value_lt(n,nm);
				value_addto(n,lcm), i++)
      {
         /* n == i*lcm + k + noff;   */
         /* nlcm <= k < nlcm+lcm     */
         /* n mod lcm == k1 */

#ifdef ALL_OVERFLOW_WARNINGS
         if ( ((flag & UB_INFINITY)==0) && value_gt(n,nUB))
         {
				fprintf(stdout, "Domain Overflow: Parameter #%d:", pos );
				fprintf(stdout, "nLB="VALUE_FMT" n="VALUE_FMT" nUB="VALUE_FMT"\n",
						nLB, n, nUB);
			}
#else
			if ( overflow_warning_flag && ((flag & UB_INFINITY)==0) &&
					value_gt(n,nUB) )
			{
				fprintf(stdout, "\nWARNING: Parameter Domain Overflow." );
				fprintf(stdout, " Result may be incorrect on this domain.\n" );
				overflow_warning_flag = 0;
			}
#endif

         /* set parameter to n */
         value_assign( context[dim-nb_param+pos], n );
#ifdef EDEBUG1
fprintf(stderr,"%c = "VALUE_FMT" (hdim=%d, lcm="VALUE_FMT")\n",
	PCHAR+pos,n,hdim,lcm);
#endif
         /* setup B vector */
         if (pos==nb_param)
         {  /* call count */
            /* count can only be called when the context is fully specified */
            value_assign( B->arr[i].d, VALUE_ONE );
            value_assign( B->arr[i].x.n,
						int_to_value(count_points(1,L,context) ) );
#ifdef EDEBUG3
            for (j=1; j<pos; j++) fputs("   ", stdout);
            fprintf(stdout, "E(");
            for (j=1; j<nb_param; j++)
               fprintf(stdout, VALUE_FMT",", context[dim-nb_param+j]);
            fprintf(stdout, VALUE_FMT") = "VALUE_FMT" =", n, B->arr[i].x.n );
#endif
         }
         else /* count is a function of other parameters */
         {  /* call P_Enum recursively */
            value_assign( B->arr[i].d, VALUE_ZERO );
            B->arr[i].x.p =
                P_Enum(L,LQ->next,context,pos+1,nb_param,dim,lcm);
#ifdef EDEBUG3
            for (j=1; j<pos; j++) fputs("   ", stdout);
            fprintf(stdout, "E(");
            for (j=1; j<=pos; j++)
               fprintf(stdout, VALUE_FMT",", context[dim-nb_param+j]);
            for (j=pos+1; j<nb_param; j++)
               fprintf(stdout, "%c,", PCHAR + j);
            fprintf(stdout, "%c) = ", PCHAR + j);
            print_enode(stdout, B->arr[i].x.p);
            fprintf(stdout, " =");
#endif
         }

         /* create i-th equation*/
         /* K_0 + K_1 * n**1 + ... + K_dim * n**dim | Identity Matrix */
         value_assign( A->p[i][0], VALUE_ZERO );  /* status bit = equation */
         value_assign( nexp, VALUE_ONE );
         for (j=1; j<=hdim; j++)
         {  value_assign( A->p[i][j], nexp );
            value_assign( A->p[i][j+hdim], VALUE_ZERO );
#ifdef EDEBUG3
            fprintf(stdout, " + "VALUE_FMT" c%d", nexp, j);
#endif
            value_multiply( nexp, n );
         }
#ifdef EDEBUG3
			fprintf(stdout, "\n");
#endif
			value_assign( A->p[i][i+1+hdim], VALUE_ONE );
      }
      /* assertion check */
      if (i!=hdim)
         fprintf(stderr, "P_Enum: ?expecting i==hdim\n");

#ifdef EDEBUG
fprintf(stderr, "B (enode) =\n");
print_enode(stderr, B);
fprintf(stderr, "A (Before Gauss) =\n");
Matrix_Print(stderr, P_VALUE_FMT, A);
#endif
      /* solve hdim (=dim+1) equations with hdim unknowns, result in CNT */
      rank = Gauss (A, hdim, 2*hdim, temp);
#ifdef EDEBUG
fprintf(stderr, "A (After Gauss) =\n");
Matrix_Print(stderr, P_VALUE_FMT, A);
#endif
      /* assertion check */
      if (rank!=hdim)
      {  fprintf(stderr, "P_Enum: ?expecting rank==hdim\n");
      }

      /* if (rank < hdim) then set the last hdim-rank coefficients to ? */
      /* if (rank == 0) then set all coefficients to 0 */

      /* copy result as k1-th element of periodic numbers */
		if( value_lt(k,lcm) )
      	value_assign( k1, k );
		else
			value_assign( k1, value_minus(k,lcm) );

      for (i=0; i<rank; i++)
      {  /* set up coefficient vector C from i-th row of inverted matrix */
         for (j=0; j<rank; j++)
         {
			   value_assign( g, Gcd(A->p[i][i+1],A->p[i][j+1+hdim]) );
            value_assign( C->arr[j].d, value_div(A->p[i][i+1],g) );
            value_assign( C->arr[j].x.n, value_div(A->p[i][j+1+hdim],g) );
         }
#ifdef EDEBUG
fprintf(stderr, "C (enode) =\n");
print_enode(stderr, C);
#endif
         /* the i-th enode is the lcm-periodic coefficient of term n**i */
			edot(B, C, &(res->arr[i].x.p->arr[VALUE_TO_INT(k1)]) );
#ifdef EDEBUG
fprintf(stderr, "B.C (evalue)=\n");
print_evalue(stderr, &(res->arr[i].x.p->arr[VALUE_TO_INT(k1)]) );
fprintf(stderr, "\n");
#endif
      }
   }
#ifdef EDEBUG
fprintf(stderr, "res (enode) =\n");
print_enode(stderr, res);
fprintf(stderr, "--------------------- end P_Enum ------------------------\n");
#endif

   /* reset context */
   value_assign( context[dim-nb_param+pos], VALUE_ZERO );

   /* release memory */
   free(temp);
   Matrix_Free(A);
   free(B);
   free(C);

   return res;
}

/*----------------------------------------------------------------*/
/* Polyhedron_Enumerate(P, C, MAXRAYS)                            */
/*    P : Polyhedron to enumerate                                 */
/*    C : Context Domain                                          */
/*    MAXRAYS : size of workspace                                 */
/* Procedure to count points in a (fixed-parameter) polytope.     */
/* Returns a list of validity domains + evalues EP                */
/*----------------------------------------------------------------*/
Enumeration *Polyhedron_Enumerate(P, C, MAXRAYS)
Polyhedron *P;	/* Polyhedron to count */
Polyhedron *C;	/* parameter Context domain */
unsigned MAXRAYS;	/* workspace size */
{
   Polyhedron *L, *CQ, *CQ2, *LQ, *U;
   Param_Polyhedron *PP;
   Param_Domain   *Q;
   Param_Vertices *V;
   int i, j, ix, hdim, dim, nb_param, nbPV;
	Value k, lcm, m1, hdv;
   unsigned bx;
   Value *context;
	Enumeration *en, *res;

	res = NULL;

#ifdef EDEBUG2
fprintf(stderr, "C = \n");
Polyhedron_Print(stderr, P_VALUE_FMT, C);
#endif

   hdim		= P->Dimension + 1;
   dim		= P->Dimension;
   nb_param	= C->Dimension;

   /* Create a context vector size dim+2 */
   context = (Value *) malloc ( (hdim+1)*sizeof(Value) );
   /* Set it to all zeros */
   memset ((char *)context, 0, (hdim+1)*sizeof(Value) );
   /* Set context[hdim] = 1  (the constant) */
   value_assign( context[hdim], VALUE_ONE );
 
#ifdef EDEBUG2
	fprintf(stderr, "P = \n");
	Polyhedron_Print(stderr, P_VALUE_FMT, P);
#endif

	/* don't call Polyhedron2Param_Domain if there are no parameters */
	if( nb_param == 0 )
	{
		L = Polyhedron_Scan(P, C, MAXRAYS);
		res = (Enumeration *)malloc( sizeof(Enumeration) );
		res->next = NULL;
		res->ValidityDomain = Universe_Polyhedron(0); /* no parameters */
		res->EP.d = 0;

#ifndef USEALWAYSGMP
		CATCH(overflow_error)
		{
#ifdef GNUMP
#ifdef MPWARNING
			fprintf( stderr, "Enumerate: arithmetic overflow error.\n");
			fprintf( stderr, "Starting over in GNU-MP mode (may be slow).\n");
#endif /* MPWARNING */
#endif  /* GNUMP */
#endif /* USEALWAYSGMP */
#ifdef GNUMP
			/* Re-initialize context */
			memset ((char *)context, 0, (hdim+1)*sizeof(Value) );
			value_assign( context[hdim], VALUE_ONE );

			res->EP.x.p = P_Enum_mp( L, NULL, context, 1, 0, dim, 1);
#else /* GNUMP */
			fprintf( stderr, "Enumerate: arithmetic overflow error.\n");
			fprintf( stderr, "You should rebuild PolyLib using GNU-MP or increasing the size of integers.\n");
#endif  /* GNUMP */
#ifndef USEALWAYSGMP
		}
		TRY
		{
			res->EP.x.p = P_Enum( L, NULL, context, 1, 0, dim, 1);
			UNCATCH(overflow_error);
		}
#endif /* USEALWAYSGMP */


		Domain_Free( L );
#ifdef EPRINT
		fprintf(stdout, "\nEhrhart Polynomial:\n");
		print_evalue(stdout, &res->EP);
		fprintf( stdout, "\n" );
		/* sometimes the final \n lacks (when a single constant is printed) */
#endif
		return( res );
	}
	/* continue with the classical procedure */


   PP = Polyhedron2Param_Domain( P, C, MAXRAYS );
	if( !PP )
	{
#ifdef EPRINT
		fprintf(stdout, "\nEhrhart Polynomial:\n0\n");
#endif
		return( NULL );
   }
   nbPV = PP->nbV;

   for( Q=PP->D ; Q ; Q=Q->next )
   {
   	en = (Enumeration *)malloc( sizeof(Enumeration) );
		en->next = res;
		res = en;

		res->ValidityDomain = Q->Domain;

#ifdef EPRINT
		fprintf(stdout, "---------------------------------------\n" );
		fprintf(stdout, "Domain :\n");
		Print_Domain(stdout, Q->Domain);
#endif

#ifndef ALL_OVERFLOW_WARNINGS
		overflow_warning_flag = 1;
#endif

      /* compute "denominator of P" */
      /* lcm = Least Common Multiple of the denominators of the vertices of P */
      value_assign( lcm, VALUE_ONE );
#ifdef EPRINT
		fprintf(stdout, "Vertices :\n");
#endif
 
      for( i=0, ix=0, bx=MSB, V=PP->V ; V && i<nbPV ; i++, V=V->next )
      {
         if (Q->F[ix] & bx)
         {
#ifdef EPRINT
				Print_Vertex(stdout, V->Vertex );
				fprintf(stdout, "\n" );
#endif
            for( j=0; j<V->Vertex->NbRows; j++)
            {
					 /* a matrix */
				   value_assign(k, V->Vertex->p[j][V->Vertex->NbColumns-1] );
               if ( value_notzero_p(k) && value_notone_p(k) )
					{
						value_assign( m1, Gcd(lcm, k) );
						value_multiply( lcm, k );
						value_division( lcm, m1 );
					}
            }
         }
         NEXT(ix,bx);
      }
#ifdef EDEBUG
		fprintf(stderr, "Denominator = "VALUE_FMT"\n", lcm);
#endif

      CQ = DomainIntersection(C, Q->Domain, MAXRAYS);
#ifdef EDEBUG2
		fprintf(stderr, "CQ = \n");
		Polyhedron_Print(stderr, P_VALUE_FMT, CQ);
#endif

      /* before scanning, add constraints to ensure at least hdim*lcm */
      /* points in every dimension */
		value_assign( hdv, int_to_value((int)hdim) );
		value_assign( m1, value_mult(hdv,lcm) );
		value_substract( m1, int_to_value(2) );
      CQ2 = Polyhedron_Preprocess(CQ, m1, MAXRAYS);
      if (emptyQ(CQ2))
      {  fprintf(stdout, "Degenerate Domain. Can not continue.\n");
	      value_assign( res->EP.d, VALUE_ONE );
	      value_assign( res->EP.x.n, VALUE_ZERO );
		}
		else
		{
#ifdef EDEBUG2
			fprintf(stderr, "CQ2 = \n");
			Polyhedron_Print(stderr, P_VALUE_FMT, CQ2);
#endif

	      /* L is used in counting the number of points in the base cases */
	      L = Polyhedron_Scan(P, CQ, MAXRAYS);
	      U = Universe_Polyhedron(0);
	      /* LQ is used to scan the parameter space */
	      LQ = Polyhedron_Scan(CQ2, U, MAXRAYS); /* bounds on parameters */
	      Domain_Free(U);
	      Domain_Free(CQ);
	      Domain_Free(CQ2);

#ifdef EDEBUG2
			fprintf(stderr, "L = \n");
			Polyhedron_Print(stderr, P_VALUE_FMT, L);
			fprintf(stderr, "LQ = \n");
			Polyhedron_Print(stderr, P_VALUE_FMT, LQ);
#endif
#ifdef EDEBUG3
			fprintf(stdout, "\nSystem of Equations:\n");
#endif
	      value_assign( res->EP.d, VALUE_ZERO );

			/***  res->EP.x.p = P_Enum(L, LQ, context, 1, nb_param, dim, lcm); ***/
#ifndef USEALWAYSGMP
			CATCH(overflow_error)
			{
#ifdef GNUMP
#ifdef MPWARNING
				fprintf( stderr, "Enumerate: arithmetic overflow error.\n");
				fprintf( stderr, "Starting over in GNU-MP mode (may be slow).\n");
#endif /* MPWARNING */
#endif /* GNUMP */
#endif /* USEALWAYSGMP */
#ifdef GNUMP
				/* Re-initialize context */
				memset ((char *)context, 0, (hdim+1)*sizeof(Value) );
				value_assign( context[hdim], VALUE_ONE );
				res->EP.x.p = P_Enum_mp( L, LQ, context, 1, nb_param, dim, lcm);
#else  /* GNUMP */
				fprintf( stderr, "Enumerate: arithmetic overflow error.\n");
				fprintf( stderr, "You should rebuild PolyLib using GNU-MP or increasing the size of integers.\n");
#endif /* GNUMP */
#ifndef USEALWAYSGMP
			}
			TRY
			{
				res->EP.x.p = P_Enum( L, LQ, context, 1, nb_param, dim, lcm);
				UNCATCH(overflow_error);

			}
#endif /* USEALWAYSGMP */


			Domain_Free( L );
         Domain_Free( LQ );
#ifdef EDEBUG5
			fprintf(stdout, "\nEhrhart Polynomial (before simplification):\n");
			print_evalue(stdout, &res->EP);
#endif
	      reduce_evalue(&res->EP);
#ifdef EPRINT
			fprintf(stdout, "\nEhrhart Polynomial:\n");
			print_evalue(stdout, &res->EP);
			fprintf( stdout, "\n" );
		/* sometimes the final \n lacks (when a single constant is printed) */
#endif
		}
   }
   free( context );
	return res;
}


#ifdef EMAIN
int main()
{
	Matrix *C1, *P1;
	Polyhedron *C, *P;
	Enumeration *en;

#ifdef EP_EVALUATION
	Value *p;
	int k;
#endif

	P1 = Matrix_Read();
	C1 = Matrix_Read();
	if( C1->NbColumns < 2 )
	{	fprintf( stderr, "Not enough parameters !\n" );
		exit(0);
	}
	P = Constraints2Polyhedron(P1, 1024);
	C = Constraints2Polyhedron(C1, 1024);
	Matrix_Free( C1 );
	Matrix_Free( P1 );

	en = Polyhedron_Enumerate(P,C,1024);

#ifdef EP_EVALUATION
	if( !isatty( 0 ) )
		return 0 ;			/* no tty input -> no evaluation. */

	printf( "Evaluation of the Ehrhart polynomial :\n" );
	p = (Value *)malloc( sizeof(Value) * (C->Dimension) );
	FOREVER
	{
		fflush(stdin);
		printf( "Enter %d parameters : ", C->Dimension );
		for( k=0 ; k<C->Dimension ; ++k )
		{
			scanf( VALUE_FMT, &p[k] );
		}

		printf( "EP( "VALUE_FMT, p[0] );
		for( k=1 ; k<C->Dimension ; ++k )
			printf( ", "VALUE_FMT, p[k] );
		printf( " ) = "VALUE_FMT"\n",	compute_poly( en, p ) );
	}
#endif /* EP_EVALUATION */

	return 0;
}
#endif /* EMAIN */

