#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "stdansi.h"
#include "types.h"
#include "vector.h"

#define Vector_Copy(p1, p2, length) \
  memcpy((char *)(p2), (char *)(p1), (int)(length)*sizeof(int))

#define Vector_Init(p1, length) \
  memset((char *)(p1), 0, (int)(length)*sizeof(int))

#define Vector_Exchange(a, b, t, l)\
{\
  memcpy((char *)(t), (char *)(a), (l)*sizeof(int));\
  memcpy((char *)(a), (char *)(b), (l)*sizeof(int));\
  memcpy((char *)(b), (char *)(t), (l)*sizeof(int));\
}

/* basic hermite engine */
int hermite(Q, H)
Matrix *Q, *H;
{  int i, j, k, nc, nr, x, reduced;
   int rank, pivot, pivotrow;
   int *tmp, *tmp2;

   /* Computes form: A = QH */

   nc = H->NbColumns;
   nr = H->NbRows;
   tmp = (int *) malloc(nc * sizeof(int));
   tmp2 = (int *) malloc(nr * sizeof(int));

   rank = 0;
   for (k=0, reduced=0; k<nc && k<nr; k=reduced ? k : k+1, reduced=0)
   {  
      /* 1. find pivot row */
      pivot = abs(H->p[k][k]);
      pivotrow = k;
      for (i=k+1; i<nr; i++)
      {  x = abs(H->p[i][k]);
         if ( x!=0 && (x<pivot || pivot==0) ) {pivot = x; pivotrow = i;}
      }

      /* 2. bring pivot to diagonal */
      if (pivotrow != k)
      {  Vector_Exchange(H->p[pivotrow], H->p[k], tmp, nc );
         Vector_Exchange(Q->p[pivotrow], Q->p[k], tmp2, nr );
      }
      pivot = H->p[k][k];	/* actual ( no abs() ) pivot */

      if (pivot != 0)
      {  /* 3. Invert the row if pivot is negative */
         if (pivot < 0)
         {  pivot = -pivot;
            for (j=k; j<nc; j++) H->p[k][j] = -H->p[k][j];
            for (j=0; j<nr; j++) Q->p[k][j] = -Q->p[k][j];
         }

         /* 4. Reduce the column modulo the pivot */
         for (i=k+1; i<nr; i++)
         {  x = H->p[i][k];
            if (x)
	    {  /* floor[integer division] (corrected for neg x) */
               if ((x<0) && (x%pivot!=0)) x=(x/pivot)-1; else x/=pivot;
               for (j=k; j<nc; j++) H->p[i][j] -= x * H->p[k][j];
               for (j=0; j<nr; j++) Q->p[k][j] += x * Q->p[i][j];
               reduced = 1;
/*printf("pivotrow = %d, pivot = %d, x = %d -----------\n", pivotrow, pivot, x);
  Matrix_Print("%4d", Q);
  Matrix_Print("%4d", H);*/
            }
         }

         if (!reduced)     /* Last finish up this column */
         {  /* 5. Make pivot column positive (above pivot row) */
            rank++;
            for (i=0; i<k; i++)
            {  x = H->p[i][k];
               if (x)
               {  /* floor[integer division] (corrected for neg x) */
                  if ((x<0) && (x%pivot!=0)) x=(x/pivot)-1; else x/=pivot;
                  for (j=k; j<nc; j++) H->p[i][j] -= x * H->p[k][j];
                  for (j=0; j<nr; j++) Q->p[k][j] += x * Q->p[i][j];
/*printf("pivotrow = %d, pivot = %d, x = %d -----------\n", pivotrow, pivot, x);
  Matrix_Print("%4d", Q);
  Matrix_Print("%4d", H); */
               }
            }
         }
      }
   }
   free(tmp2); free(tmp);
   return rank;
}

void right_hermite(A, Qp, Hp)
Matrix *A, **Qp, **Hp;
{  Matrix *H, *Q;
   int i, j, nr, nc, tmp, rank;

   /* Computes form: A = QH */

   nc = A->NbColumns;
   nr = A->NbRows;

   /* H = A */
   *Hp = H = Matrix_Alloc(nr, nc);
   if (!H) return;

   Vector_Copy(A->p_Init, H->p_Init, nr*nc );

   /* Q = I */
   /* Actually I compute Q transpose... its easier */
   *Qp = Q = Matrix_Alloc(nr, nr);
   if (!Q) return;

   Vector_Init(Q->p_Init, nr*nr);               /* zero's */
   for (i=0; i<nr; i++) Q->p[i][i] = 1;         /* with diagonal of 1's */

   rank = hermite(Q,H);
   /* Q is returned transposed */

   /* Transpose Q */
   for (i=0; i<nr; i++)
      for (j=i+1; j<nr; j++)
      {  tmp = Q->p[i][j];
         Q->p[i][j] = Q->p[j][i];
         Q->p[j][i] = tmp;
      }

   /* ? What happens when A was not full rank ???? */
}

void left_hermite(A, Hp, Qp)
Matrix *A, **Hp, **Qp;
{  Matrix *H, *HT, *Q;
   int i, j, nc, nr;
   int rank;

   /* Computes left form: A = HQ 
                           T    T T
        using right form  A  = Q H  */

   nr = A->NbRows;
   nc = A->NbColumns;

   /* HT = A transpose */
   HT = Matrix_Alloc(nc, nr);
   if (!HT) return;
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         HT->p[j][i] = A->p[i][j];

   /* Q = I */
   *Qp = Q = Matrix_Alloc(nc, nc);
   if (!Q) return;
   Vector_Init(Q->p_Init, nc*nc);               /* zero's */
   for (i=0; i<nc; i++) Q->p[i][i] = 1;         /* with diagonal of 1's */

   rank = hermite(Q, HT);

   /* H = HT transpose */
   *Hp = H = Matrix_Alloc(nr, nc);
   if (!H) return;
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         H->p[i][j] = HT->p[j][i];
   Matrix_Free(HT);

   /* ? What happens when A was not full rank ???? */
}

/* doesnt work */
#ifdef pieceOfJunk
void smith(A)
Matrix *A;
{  Matrix *H, *HT, *QL, *QR;
   int i, j, nr, nc, tmp, rank;

   /* Computes form: A = QL H QR*/

   nc = A->NbColumns;
   nr = A->NbRows;

   /* H = A */
   H = Matrix_Alloc(nr, nc);
   Vector_Copy(A->p_Init, H->p_Init, nr*nc );

   /* QL = I */
   QL = Matrix_Alloc(nr, nr);
   Vector_Init(QL->p_Init, nr*nr);               /* zero's */
   for (i=0; i<nr; i++) QL->p[i][i] = 1;         /* with diagonal of 1's */

   rank = hermite(QL,H);
   /* QL is returned transposed */

   /* Transpose QL */
   for (i=0; i<nr; i++)
      for (j=i+1; j<nr; j++)
      {  tmp = QL->p[i][j];
         QL->p[i][j] = QL->p[j][i];
         QL->p[j][i] = tmp;
      }

   /* HT = H transpose */
   HT = Matrix_Alloc(nc, nr);
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         HT->p[j][i] = H->p[i][j];

   /* QR = I */
   QR = Matrix_Alloc(nc, nc);
   Vector_Init(QR->p_Init, nc*nc);               /* zero's */
   for (i=0; i<nc; i++) QR->p[i][i] = 1;         /* with diagonal of 1's */

   rank = hermite(QR, HT);

   /* H = HT transpose */
   H = Matrix_Alloc(nr, nc);
   for (i=0; i<nr; i++)
      for (j=0; j<nc; j++)
         H->p[i][j] = HT->p[j][i];
   Matrix_Free(HT);

   printf("Rank = %d\n", rank);
   Matrix_Print("%4d", QL);
   Matrix_Print("%4d", H);
   Matrix_Print("%4d", QR);
}
#endif

#ifdef DEBUGM
main()
{  Matrix *A,*H,*Q;
   A = Matrix_Read();

   left_hermite(A, &Q, &H);
   printf("Rank = %d\n", rank);
   Matrix_Print("%4d", Q);
   Matrix_Print("%4d", H);

   right_hermite(A, &H, &Q);
   printf("Rank = %d\n", rank);
   Matrix_Print("%4d", H);
   Matrix_Print("%4d", Q);

}
#endif
