/************************************************/
/*  eval_ehrhart.c                              */
/* functions to evaluate an Ehrhart polynomial. */
/* written by Emmanuel Jeannot (c) 1997.        */
/*  Emmanuel.Jeannot@ens-lyon.fr                */
/*  http://www.ens-lyon.fr/~ejeannot            */
/************************************************/

#include "eval_ehrhart.h"

#undef EVAL_EHRHART_DEBUG

/********************************************************/
/* function in domain                                   */
/*    check if the parameters in list_args              */
/*    verifies the constraints of Polyhedron P          */
/********************************************************/
int in_domain(Polyhedron *P,int *list_args){
  int col,row;
  int value; /*value of the constraint of a row when
	       parameters are instanciated*/

  /*P->Constraint constraint matrice of polyhedron P*/

  for(row=0;row<P->NbConstraints;row++){
    value=P->Constraint[row][P->Dimension+1]; /*constant part*/
    for(col=1;col<P->Dimension+1;col++)
      value+=P->Constraint[row][col]*list_args[col-1];

    if (P->Constraint[row][0]){
      if (value<0) 
	return 0; /*if value is not >=0 then this
		    constraint is not respected*/
    } else {
      if (value!=0)
	return 0; /*if value is not =0 then this
		    constraint is not respected*/
    }
  }
  
  /*if not retrun before this point => all 
    the constraints are respected*/
  return 1;
}

/****************************************************/
/* function compute enode                           */
/*     compute the value of enode p with parameters */
/*     list "list_args                              */
/*     compute the polynomial or the periodic       */
/****************************************************/

double compute_enode(enode *p, int *list_args){
  int i,param;
  double res=0.0;

    if (!p){
      return 0;
    }

    if (p->type == polynomial){
      param=list_args[p->pos-1];
      /*compute the polynomial using Horner's rule*/
      for (i=p->size-1; i>0; i--){
	res+=compute_evalue( &p->arr[i],list_args);
	res*=param;
      }
      res+=compute_evalue( &p->arr[0],list_args);
    }
    else if (p->type == periodic){
      param=list_args[p->pos-1];
      /*choose the right element of the periodic*/
      i=(param)%(p->size);/* not the Ehrhart convention but
			     the Clauss/Wilde/Loechner one*/
      res=compute_evalue(&p->arr[i],list_args);
    }
    return res;
}

/*************************************************/
/* return the value of Ehrhart Polynomial        */
/* It returns a double, because since it is      */
/* a recursive function, some intermediate value */
/* might not be integral                         */
/*************************************************/


double compute_evalue(evalue *e,int *list_args){
  double res;
  
  if (e->d != 0){   
    if (e->d != 1) 
      res=(double)e->x.n/(double)e->d;
    else 
      res=(double)e->x.n;
  } else 
    res=compute_enode(e->x.p,list_args);
  return res;
}


#ifdef EVAL_EHRHART_DEBUG
#define PCHAR 'O'
/*------------------------------------------------------------*/
/* print_evalue(DST, e)                                       */
/*    DST : destination file                                  */
/*    e : pointer to evalue to be printed                     */
/* Returns nothing, prints the evalue to DST                  */
/*                                       taken from ehrhart.c */
/*------------------------------------------------------------*/
void print_enode(FILE *DST, enode *p); /* forward declaration */
void print_evalue(DST, e)
FILE *DST;
evalue *e;
{  
    if (e->d != 0)
    {   if (e->d != 1) fprintf(DST, "%d/%d", e->x.n, e->d);
        else fprintf(DST, "%d", e->x.n);
        return;
    }
    else print_enode(DST, e->x.p);
}

/*------------------------------------------------------------*/
/* print_enode(DST, p )                                       */
/*    DST : destination file                                  */
/*    p : pointer to enode  to be printed                     */
/* Returns nothing, prints the enode  to DST                  */
/*                                       taken from ehrhart.c */
/*------------------------------------------------------------*/
void print_enode(DST, p)
FILE *DST;
enode *p;
{   int i;

    if (!p)
    {   fprintf(DST, "NULL");
        return;
    }
    if (p->type == evector)
    {   fprintf(DST, "{ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " }\n");
    }
    else if (p->type == polynomial)
    {   fprintf(DST, "( ");
        for (i=p->size-1; i>=0; i--)
        {  print_evalue(DST, &p->arr[i]);
           if (i==1) fprintf(DST, " * %c + ", PCHAR+p->pos);
           else if ( i>1 ) fprintf(DST, " * %c^%d + ", PCHAR+p->pos, i);
        }
        fprintf(DST, " )\n");
    }
    else if (p->type == periodic)
    {   fprintf(DST, "[ ");
        for (i=0; i<p->size; i++)
        {  print_evalue(DST, &p->arr[i]);
           if ( i!=(p->size-1) ) fprintf(DST, ", ");
        }
        fprintf(DST, " ]_%c", PCHAR+p->pos);
    }
}

/*----------------------------------------------------------------------*/
/* Print_Domain   taken from polyparam.c                                */
/*----------------------------------------------------------------------*/
void Print_Domain( DST, D )
FILE *DST;
Polyhedron *D;
{
   int l, v;
   int first;
   for( l=0 ; l<D->NbConstraints ; ++l )
   {
      fprintf(DST, "         " );
      first = 1;
      for( v=1 ; v<=D->Dimension ; ++v )
      {
         if(D->Constraint[l][v] )
         {
	   if( D->Constraint[l][v]==1 )
            {
               if( first ) fprintf(DST, "%c ", PCHAR+v );
               else fprintf(DST, "+ %c ", PCHAR+v );
	    }
            else if( D->Constraint[l][v]==-1 ) fprintf(DST, "- %c ", PCHAR+v );
            else
            {
               if( D->Constraint[l][v] > 0 && !first ) fprintf(DST, "+ " );
               fprintf(DST, "%d%c ", D->Constraint[l][v], PCHAR+v );
	    }
            first = 0;
         }
      }
      if( D->Constraint[l][v] )
      {
         if( D->Constraint[l][v] > 0 && !first ) fprintf(DST, "+" );
         fprintf(DST, " %d", D->Constraint[l][v] );
      }
      fprintf(DST, D->Constraint[l][0]?" >= 0":" = 0" );
      fprintf(DST, "\n" );
   }
   fprintf(DST, "\n");
}
#endif /* EVAL_EHRHART_DEBUG */

/****************************************************/
/* function compute_poly :                          */
/* Check for the good validity domain               */
/* return the number of point in the Polyhedron     */
/* Using the Ehrhart pseudo-polynomial              */
/****************************************************/
int compute_poly(Enumeration *en, int *list_args)
{
	if( !en )
		return 0;	/* no ehrhart polynomial */
	if(en->ValidityDomain)
	{
		if(!en->ValidityDomain->Dimension)
			return 0; /* no parameters, empty polyhedron */
	}
	else 
		return 0; /* no Validity Domain */
  
  
	while(en)
	{
		if(in_domain(en->ValidityDomain,list_args))
		{
#ifdef EVAL_EHRHART_DEBUG
			Print_Domain(stdout,en->ValidityDomain);
			print_evalue(stdout,&en->EP);
#endif
			return((int)compute_evalue(&en->EP,list_args));
		}
		else
			en=en->next;
	}

  return 0; /* no compatible domain with the arguments */

}
/****************** End Compute poly *******************************/

